var process = process || {env: {NODE_ENV: "development"}};
;(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
    typeof define === 'function' && define.amd ? define(factory) :
    global.moment = factory()
}(this, (function () { 'use strict';

    var hookCallback;

    function hooks () {
        return hookCallback.apply(null, arguments);
    }

    // This is done to register the method called with moment()
    // without creating circular dependencies.
    function setHookCallback (callback) {
        hookCallback = callback;
    }

    function isArray(input) {
        return input instanceof Array || Object.prototype.toString.call(input) === '[object Array]';
    }

    function isObject(input) {
        // IE8 will treat undefined and null as object if it wasn't for
        // input != null
        return input != null && Object.prototype.toString.call(input) === '[object Object]';
    }

    function isObjectEmpty(obj) {
        if (Object.getOwnPropertyNames) {
            return (Object.getOwnPropertyNames(obj).length === 0);
        } else {
            var k;
            for (k in obj) {
                if (obj.hasOwnProperty(k)) {
                    return false;
                }
            }
            return true;
        }
    }

    function isUndefined(input) {
        return input === void 0;
    }

    function isNumber(input) {
        return typeof input === 'number' || Object.prototype.toString.call(input) === '[object Number]';
    }

    function isDate(input) {
        return input instanceof Date || Object.prototype.toString.call(input) === '[object Date]';
    }

    function map(arr, fn) {
        var res = [], i;
        for (i = 0; i < arr.length; ++i) {
            res.push(fn(arr[i], i));
        }
        return res;
    }

    function hasOwnProp(a, b) {
        return Object.prototype.hasOwnProperty.call(a, b);
    }

    function extend(a, b) {
        for (var i in b) {
            if (hasOwnProp(b, i)) {
                a[i] = b[i];
            }
        }

        if (hasOwnProp(b, 'toString')) {
            a.toString = b.toString;
        }

        if (hasOwnProp(b, 'valueOf')) {
            a.valueOf = b.valueOf;
        }

        return a;
    }

    function createUTC (input, format, locale, strict) {
        return createLocalOrUTC(input, format, locale, strict, true).utc();
    }

    function defaultParsingFlags() {
        // We need to deep clone this object.
        return {
            empty           : false,
            unusedTokens    : [],
            unusedInput     : [],
            overflow        : -2,
            charsLeftOver   : 0,
            nullInput       : false,
            invalidMonth    : null,
            invalidFormat   : false,
            userInvalidated : false,
            iso             : false,
            parsedDateParts : [],
            meridiem        : null,
            rfc2822         : false,
            weekdayMismatch : false
        };
    }

    function getParsingFlags(m) {
        if (m._pf == null) {
            m._pf = defaultParsingFlags();
        }
        return m._pf;
    }

    var some;
    if (Array.prototype.some) {
        some = Array.prototype.some;
    } else {
        some = function (fun) {
            var t = Object(this);
            var len = t.length >>> 0;

            for (var i = 0; i < len; i++) {
                if (i in t && fun.call(this, t[i], i, t)) {
                    return true;
                }
            }

            return false;
        };
    }

    function isValid(m) {
        if (m._isValid == null) {
            var flags = getParsingFlags(m);
            var parsedParts = some.call(flags.parsedDateParts, function (i) {
                return i != null;
            });
            var isNowValid = !isNaN(m._d.getTime()) &&
                flags.overflow < 0 &&
                !flags.empty &&
                !flags.invalidMonth &&
                !flags.invalidWeekday &&
                !flags.weekdayMismatch &&
                !flags.nullInput &&
                !flags.invalidFormat &&
                !flags.userInvalidated &&
                (!flags.meridiem || (flags.meridiem && parsedParts));

            if (m._strict) {
                isNowValid = isNowValid &&
                    flags.charsLeftOver === 0 &&
                    flags.unusedTokens.length === 0 &&
                    flags.bigHour === undefined;
            }

            if (Object.isFrozen == null || !Object.isFrozen(m)) {
                m._isValid = isNowValid;
            }
            else {
                return isNowValid;
            }
        }
        return m._isValid;
    }

    function createInvalid (flags) {
        var m = createUTC(NaN);
        if (flags != null) {
            extend(getParsingFlags(m), flags);
        }
        else {
            getParsingFlags(m).userInvalidated = true;
        }

        return m;
    }

    // Plugins that add properties should also add the key here (null value),
    // so we can properly clone ourselves.
    var momentProperties = hooks.momentProperties = [];

    function copyConfig(to, from) {
        var i, prop, val;

        if (!isUndefined(from._isAMomentObject)) {
            to._isAMomentObject = from._isAMomentObject;
        }
        if (!isUndefined(from._i)) {
            to._i = from._i;
        }
        if (!isUndefined(from._f)) {
            to._f = from._f;
        }
        if (!isUndefined(from._l)) {
            to._l = from._l;
        }
        if (!isUndefined(from._strict)) {
            to._strict = from._strict;
        }
        if (!isUndefined(from._tzm)) {
            to._tzm = from._tzm;
        }
        if (!isUndefined(from._isUTC)) {
            to._isUTC = from._isUTC;
        }
        if (!isUndefined(from._offset)) {
            to._offset = from._offset;
        }
        if (!isUndefined(from._pf)) {
            to._pf = getParsingFlags(from);
        }
        if (!isUndefined(from._locale)) {
            to._locale = from._locale;
        }

        if (momentProperties.length > 0) {
            for (i = 0; i < momentProperties.length; i++) {
                prop = momentProperties[i];
                val = from[prop];
                if (!isUndefined(val)) {
                    to[prop] = val;
                }
            }
        }

        return to;
    }

    var updateInProgress = false;

    // Moment prototype object
    function Moment(config) {
        copyConfig(this, config);
        this._d = new Date(config._d != null ? config._d.getTime() : NaN);
        if (!this.isValid()) {
            this._d = new Date(NaN);
        }
        // Prevent infinite loop in case updateOffset creates new moment
        // objects.
        if (updateInProgress === false) {
            updateInProgress = true;
            hooks.updateOffset(this);
            updateInProgress = false;
        }
    }

    function isMoment (obj) {
        return obj instanceof Moment || (obj != null && obj._isAMomentObject != null);
    }

    function absFloor (number) {
        if (number < 0) {
            // -0 -> 0
            return Math.ceil(number) || 0;
        } else {
            return Math.floor(number);
        }
    }

    function toInt(argumentForCoercion) {
        var coercedNumber = +argumentForCoercion,
            value = 0;

        if (coercedNumber !== 0 && isFinite(coercedNumber)) {
            value = absFloor(coercedNumber);
        }

        return value;
    }

    // compare two arrays, return the number of differences
    function compareArrays(array1, array2, dontConvert) {
        var len = Math.min(array1.length, array2.length),
            lengthDiff = Math.abs(array1.length - array2.length),
            diffs = 0,
            i;
        for (i = 0; i < len; i++) {
            if ((dontConvert && array1[i] !== array2[i]) ||
                (!dontConvert && toInt(array1[i]) !== toInt(array2[i]))) {
                diffs++;
            }
        }
        return diffs + lengthDiff;
    }

    function warn(msg) {
        if (hooks.suppressDeprecationWarnings === false &&
                (typeof console !==  'undefined') && console.warn) {
            console.warn('Deprecation warning: ' + msg);
        }
    }

    function deprecate(msg, fn) {
        var firstTime = true;

        return extend(function () {
            if (hooks.deprecationHandler != null) {
                hooks.deprecationHandler(null, msg);
            }
            if (firstTime) {
                var args = [];
                var arg;
                for (var i = 0; i < arguments.length; i++) {
                    arg = '';
                    if (typeof arguments[i] === 'object') {
                        arg += '\n[' + i + '] ';
                        for (var key in arguments[0]) {
                            arg += key + ': ' + arguments[0][key] + ', ';
                        }
                        arg = arg.slice(0, -2); // Remove trailing comma and space
                    } else {
                        arg = arguments[i];
                    }
                    args.push(arg);
                }
                warn(msg + '\nArguments: ' + Array.prototype.slice.call(args).join('') + '\n' + (new Error()).stack);
                firstTime = false;
            }
            return fn.apply(this, arguments);
        }, fn);
    }

    var deprecations = {};

    function deprecateSimple(name, msg) {
        if (hooks.deprecationHandler != null) {
            hooks.deprecationHandler(name, msg);
        }
        if (!deprecations[name]) {
            warn(msg);
            deprecations[name] = true;
        }
    }

    hooks.suppressDeprecationWarnings = false;
    hooks.deprecationHandler = null;

    function isFunction(input) {
        return input instanceof Function || Object.prototype.toString.call(input) === '[object Function]';
    }

    function set (config) {
        var prop, i;
        for (i in config) {
            prop = config[i];
            if (isFunction(prop)) {
                this[i] = prop;
            } else {
                this['_' + i] = prop;
            }
        }
        this._config = config;
        // Lenient ordinal parsing accepts just a number in addition to
        // number + (possibly) stuff coming from _dayOfMonthOrdinalParse.
        // TODO: Remove "ordinalParse" fallback in next major release.
        this._dayOfMonthOrdinalParseLenient = new RegExp(
            (this._dayOfMonthOrdinalParse.source || this._ordinalParse.source) +
                '|' + (/\d{1,2}/).source);
    }

    function mergeConfigs(parentConfig, childConfig) {
        var res = extend({}, parentConfig), prop;
        for (prop in childConfig) {
            if (hasOwnProp(childConfig, prop)) {
                if (isObject(parentConfig[prop]) && isObject(childConfig[prop])) {
                    res[prop] = {};
                    extend(res[prop], parentConfig[prop]);
                    extend(res[prop], childConfig[prop]);
                } else if (childConfig[prop] != null) {
                    res[prop] = childConfig[prop];
                } else {
                    delete res[prop];
                }
            }
        }
        for (prop in parentConfig) {
            if (hasOwnProp(parentConfig, prop) &&
                    !hasOwnProp(childConfig, prop) &&
                    isObject(parentConfig[prop])) {
                // make sure changes to properties don't modify parent config
                res[prop] = extend({}, res[prop]);
            }
        }
        return res;
    }

    function Locale(config) {
        if (config != null) {
            this.set(config);
        }
    }

    var keys;

    if (Object.keys) {
        keys = Object.keys;
    } else {
        keys = function (obj) {
            var i, res = [];
            for (i in obj) {
                if (hasOwnProp(obj, i)) {
                    res.push(i);
                }
            }
            return res;
        };
    }

    var defaultCalendar = {
        sameDay : '[Today at] LT',
        nextDay : '[Tomorrow at] LT',
        nextWeek : 'dddd [at] LT',
        lastDay : '[Yesterday at] LT',
        lastWeek : '[Last] dddd [at] LT',
        sameElse : 'L'
    };

    function calendar (key, mom, now) {
        var output = this._calendar[key] || this._calendar['sameElse'];
        return isFunction(output) ? output.call(mom, now) : output;
    }

    var defaultLongDateFormat = {
        LTS  : 'h:mm:ss A',
        LT   : 'h:mm A',
        L    : 'MM/DD/YYYY',
        LL   : 'MMMM D, YYYY',
        LLL  : 'MMMM D, YYYY h:mm A',
        LLLL : 'dddd, MMMM D, YYYY h:mm A'
    };

    function longDateFormat (key) {
        var format = this._longDateFormat[key],
            formatUpper = this._longDateFormat[key.toUpperCase()];

        if (format || !formatUpper) {
            return format;
        }

        this._longDateFormat[key] = formatUpper.replace(/MMMM|MM|DD|dddd/g, function (val) {
            return val.slice(1);
        });

        return this._longDateFormat[key];
    }

    var defaultInvalidDate = 'Invalid date';

    function invalidDate () {
        return this._invalidDate;
    }

    var defaultOrdinal = '%d';
    var defaultDayOfMonthOrdinalParse = /\d{1,2}/;

    function ordinal (number) {
        return this._ordinal.replace('%d', number);
    }

    var defaultRelativeTime = {
        future : 'in %s',
        past   : '%s ago',
        s  : 'a few seconds',
        ss : '%d seconds',
        m  : 'a minute',
        mm : '%d minutes',
        h  : 'an hour',
        hh : '%d hours',
        d  : 'a day',
        dd : '%d days',
        M  : 'a month',
        MM : '%d months',
        y  : 'a year',
        yy : '%d years'
    };

    function relativeTime (number, withoutSuffix, string, isFuture) {
        var output = this._relativeTime[string];
        return (isFunction(output)) ?
            output(number, withoutSuffix, string, isFuture) :
            output.replace(/%d/i, number);
    }

    function pastFuture (diff, output) {
        var format = this._relativeTime[diff > 0 ? 'future' : 'past'];
        return isFunction(format) ? format(output) : format.replace(/%s/i, output);
    }

    var aliases = {};

    function addUnitAlias (unit, shorthand) {
        var lowerCase = unit.toLowerCase();
        aliases[lowerCase] = aliases[lowerCase + 's'] = aliases[shorthand] = unit;
    }

    function normalizeUnits(units) {
        return typeof units === 'string' ? aliases[units] || aliases[units.toLowerCase()] : undefined;
    }

    function normalizeObjectUnits(inputObject) {
        var normalizedInput = {},
            normalizedProp,
            prop;

        for (prop in inputObject) {
            if (hasOwnProp(inputObject, prop)) {
                normalizedProp = normalizeUnits(prop);
                if (normalizedProp) {
                    normalizedInput[normalizedProp] = inputObject[prop];
                }
            }
        }

        return normalizedInput;
    }

    var priorities = {};

    function addUnitPriority(unit, priority) {
        priorities[unit] = priority;
    }

    function getPrioritizedUnits(unitsObj) {
        var units = [];
        for (var u in unitsObj) {
            units.push({unit: u, priority: priorities[u]});
        }
        units.sort(function (a, b) {
            return a.priority - b.priority;
        });
        return units;
    }

    function zeroFill(number, targetLength, forceSign) {
        var absNumber = '' + Math.abs(number),
            zerosToFill = targetLength - absNumber.length,
            sign = number >= 0;
        return (sign ? (forceSign ? '+' : '') : '-') +
            Math.pow(10, Math.max(0, zerosToFill)).toString().substr(1) + absNumber;
    }

    var formattingTokens = /(\[[^\[]*\])|(\\)?([Hh]mm(ss)?|Mo|MM?M?M?|Do|DDDo|DD?D?D?|ddd?d?|do?|w[o|w]?|W[o|W]?|Qo?|YYYYYY|YYYYY|YYYY|YY|gg(ggg?)?|GG(GGG?)?|e|E|a|A|hh?|HH?|kk?|mm?|ss?|S{1,9}|x|X|zz?|ZZ?|.)/g;

    var localFormattingTokens = /(\[[^\[]*\])|(\\)?(LTS|LT|LL?L?L?|l{1,4})/g;

    var formatFunctions = {};

    var formatTokenFunctions = {};

    // token:    'M'
    // padded:   ['MM', 2]
    // ordinal:  'Mo'
    // callback: function () { this.month() + 1 }
    function addFormatToken (token, padded, ordinal, callback) {
        var func = callback;
        if (typeof callback === 'string') {
            func = function () {
                return this[callback]();
            };
        }
        if (token) {
            formatTokenFunctions[token] = func;
        }
        if (padded) {
            formatTokenFunctions[padded[0]] = function () {
                return zeroFill(func.apply(this, arguments), padded[1], padded[2]);
            };
        }
        if (ordinal) {
            formatTokenFunctions[ordinal] = function () {
                return this.localeData().ordinal(func.apply(this, arguments), token);
            };
        }
    }

    function removeFormattingTokens(input) {
        if (input.match(/\[[\s\S]/)) {
            return input.replace(/^\[|\]$/g, '');
        }
        return input.replace(/\\/g, '');
    }

    function makeFormatFunction(format) {
        var array = format.match(formattingTokens), i, length;

        for (i = 0, length = array.length; i < length; i++) {
            if (formatTokenFunctions[array[i]]) {
                array[i] = formatTokenFunctions[array[i]];
            } else {
                array[i] = removeFormattingTokens(array[i]);
            }
        }

        return function (mom) {
            var output = '', i;
            for (i = 0; i < length; i++) {
                output += isFunction(array[i]) ? array[i].call(mom, format) : array[i];
            }
            return output;
        };
    }

    // format date using native date object
    function formatMoment(m, format) {
        if (!m.isValid()) {
            return m.localeData().invalidDate();
        }

        format = expandFormat(format, m.localeData());
        formatFunctions[format] = formatFunctions[format] || makeFormatFunction(format);

        return formatFunctions[format](m);
    }

    function expandFormat(format, locale) {
        var i = 5;

        function replaceLongDateFormatTokens(input) {
            return locale.longDateFormat(input) || input;
        }

        localFormattingTokens.lastIndex = 0;
        while (i >= 0 && localFormattingTokens.test(format)) {
            format = format.replace(localFormattingTokens, replaceLongDateFormatTokens);
            localFormattingTokens.lastIndex = 0;
            i -= 1;
        }

        return format;
    }

    var match1         = /\d/;            //       0 - 9
    var match2         = /\d\d/;          //      00 - 99
    var match3         = /\d{3}/;         //     000 - 999
    var match4         = /\d{4}/;         //    0000 - 9999
    var match6         = /[+-]?\d{6}/;    // -999999 - 999999
    var match1to2      = /\d\d?/;         //       0 - 99
    var match3to4      = /\d\d\d\d?/;     //     999 - 9999
    var match5to6      = /\d\d\d\d\d\d?/; //   99999 - 999999
    var match1to3      = /\d{1,3}/;       //       0 - 999
    var match1to4      = /\d{1,4}/;       //       0 - 9999
    var match1to6      = /[+-]?\d{1,6}/;  // -999999 - 999999

    var matchUnsigned  = /\d+/;           //       0 - inf
    var matchSigned    = /[+-]?\d+/;      //    -inf - inf

    var matchOffset    = /Z|[+-]\d\d:?\d\d/gi; // +00:00 -00:00 +0000 -0000 or Z
    var matchShortOffset = /Z|[+-]\d\d(?::?\d\d)?/gi; // +00 -00 +00:00 -00:00 +0000 -0000 or Z

    var matchTimestamp = /[+-]?\d+(\.\d{1,3})?/; // 123456789 123456789.123

    // any word (or two) characters or numbers including two/three word month in arabic.
    // includes scottish gaelic two word and hyphenated months
    var matchWord = /[0-9]{0,256}['a-z\u00A0-\u05FF\u0700-\uD7FF\uF900-\uFDCF\uFDF0-\uFF07\uFF10-\uFFEF]{1,256}|[\u0600-\u06FF\/]{1,256}(\s*?[\u0600-\u06FF]{1,256}){1,2}/i;

    var regexes = {};

    function addRegexToken (token, regex, strictRegex) {
        regexes[token] = isFunction(regex) ? regex : function (isStrict, localeData) {
            return (isStrict && strictRegex) ? strictRegex : regex;
        };
    }

    function getParseRegexForToken (token, config) {
        if (!hasOwnProp(regexes, token)) {
            return new RegExp(unescapeFormat(token));
        }

        return regexes[token](config._strict, config._locale);
    }

    // Code from http://stackoverflow.com/questions/3561493/is-there-a-regexp-escape-function-in-javascript
    function unescapeFormat(s) {
        return regexEscape(s.replace('\\', '').replace(/\\(\[)|\\(\])|\[([^\]\[]*)\]|\\(.)/g, function (matched, p1, p2, p3, p4) {
            return p1 || p2 || p3 || p4;
        }));
    }

    function regexEscape(s) {
        return s.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
    }

    var tokens = {};

    function addParseToken (token, callback) {
        var i, func = callback;
        if (typeof token === 'string') {
            token = [token];
        }
        if (isNumber(callback)) {
            func = function (input, array) {
                array[callback] = toInt(input);
            };
        }
        for (i = 0; i < token.length; i++) {
            tokens[token[i]] = func;
        }
    }

    function addWeekParseToken (token, callback) {
        addParseToken(token, function (input, array, config, token) {
            config._w = config._w || {};
            callback(input, config._w, config, token);
        });
    }

    function addTimeToArrayFromToken(token, input, config) {
        if (input != null && hasOwnProp(tokens, token)) {
            tokens[token](input, config._a, config, token);
        }
    }

    var YEAR = 0;
    var MONTH = 1;
    var DATE = 2;
    var HOUR = 3;
    var MINUTE = 4;
    var SECOND = 5;
    var MILLISECOND = 6;
    var WEEK = 7;
    var WEEKDAY = 8;

    // FORMATTING

    addFormatToken('Y', 0, 0, function () {
        var y = this.year();
        return y <= 9999 ? '' + y : '+' + y;
    });

    addFormatToken(0, ['YY', 2], 0, function () {
        return this.year() % 100;
    });

    addFormatToken(0, ['YYYY',   4],       0, 'year');
    addFormatToken(0, ['YYYYY',  5],       0, 'year');
    addFormatToken(0, ['YYYYYY', 6, true], 0, 'year');

    // ALIASES

    addUnitAlias('year', 'y');

    // PRIORITIES

    addUnitPriority('year', 1);

    // PARSING

    addRegexToken('Y',      matchSigned);
    addRegexToken('YY',     match1to2, match2);
    addRegexToken('YYYY',   match1to4, match4);
    addRegexToken('YYYYY',  match1to6, match6);
    addRegexToken('YYYYYY', match1to6, match6);

    addParseToken(['YYYYY', 'YYYYYY'], YEAR);
    addParseToken('YYYY', function (input, array) {
        array[YEAR] = input.length === 2 ? hooks.parseTwoDigitYear(input) : toInt(input);
    });
    addParseToken('YY', function (input, array) {
        array[YEAR] = hooks.parseTwoDigitYear(input);
    });
    addParseToken('Y', function (input, array) {
        array[YEAR] = parseInt(input, 10);
    });

    // HELPERS

    function daysInYear(year) {
        return isLeapYear(year) ? 366 : 365;
    }

    function isLeapYear(year) {
        return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
    }

    // HOOKS

    hooks.parseTwoDigitYear = function (input) {
        return toInt(input) + (toInt(input) > 68 ? 1900 : 2000);
    };

    // MOMENTS

    var getSetYear = makeGetSet('FullYear', true);

    function getIsLeapYear () {
        return isLeapYear(this.year());
    }

    function makeGetSet (unit, keepTime) {
        return function (value) {
            if (value != null) {
                set$1(this, unit, value);
                hooks.updateOffset(this, keepTime);
                return this;
            } else {
                return get(this, unit);
            }
        };
    }

    function get (mom, unit) {
        return mom.isValid() ?
            mom._d['get' + (mom._isUTC ? 'UTC' : '') + unit]() : NaN;
    }

    function set$1 (mom, unit, value) {
        if (mom.isValid() && !isNaN(value)) {
            if (unit === 'FullYear' && isLeapYear(mom.year()) && mom.month() === 1 && mom.date() === 29) {
                mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value, mom.month(), daysInMonth(value, mom.month()));
            }
            else {
                mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value);
            }
        }
    }

    // MOMENTS

    function stringGet (units) {
        units = normalizeUnits(units);
        if (isFunction(this[units])) {
            return this[units]();
        }
        return this;
    }


    function stringSet (units, value) {
        if (typeof units === 'object') {
            units = normalizeObjectUnits(units);
            var prioritized = getPrioritizedUnits(units);
            for (var i = 0; i < prioritized.length; i++) {
                this[prioritized[i].unit](units[prioritized[i].unit]);
            }
        } else {
            units = normalizeUnits(units);
            if (isFunction(this[units])) {
                return this[units](value);
            }
        }
        return this;
    }

    function mod(n, x) {
        return ((n % x) + x) % x;
    }

    var indexOf;

    if (Array.prototype.indexOf) {
        indexOf = Array.prototype.indexOf;
    } else {
        indexOf = function (o) {
            // I know
            var i;
            for (i = 0; i < this.length; ++i) {
                if (this[i] === o) {
                    return i;
                }
            }
            return -1;
        };
    }

    function daysInMonth(year, month) {
        if (isNaN(year) || isNaN(month)) {
            return NaN;
        }
        var modMonth = mod(month, 12);
        year += (month - modMonth) / 12;
        return modMonth === 1 ? (isLeapYear(year) ? 29 : 28) : (31 - modMonth % 7 % 2);
    }

    // FORMATTING

    addFormatToken('M', ['MM', 2], 'Mo', function () {
        return this.month() + 1;
    });

    addFormatToken('MMM', 0, 0, function (format) {
        return this.localeData().monthsShort(this, format);
    });

    addFormatToken('MMMM', 0, 0, function (format) {
        return this.localeData().months(this, format);
    });

    // ALIASES

    addUnitAlias('month', 'M');

    // PRIORITY

    addUnitPriority('month', 8);

    // PARSING

    addRegexToken('M',    match1to2);
    addRegexToken('MM',   match1to2, match2);
    addRegexToken('MMM',  function (isStrict, locale) {
        return locale.monthsShortRegex(isStrict);
    });
    addRegexToken('MMMM', function (isStrict, locale) {
        return locale.monthsRegex(isStrict);
    });

    addParseToken(['M', 'MM'], function (input, array) {
        array[MONTH] = toInt(input) - 1;
    });

    addParseToken(['MMM', 'MMMM'], function (input, array, config, token) {
        var month = config._locale.monthsParse(input, token, config._strict);
        // if we didn't find a month name, mark the date as invalid.
        if (month != null) {
            array[MONTH] = month;
        } else {
            getParsingFlags(config).invalidMonth = input;
        }
    });

    // LOCALES

    var MONTHS_IN_FORMAT = /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?/;
    var defaultLocaleMonths = 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_');
    function localeMonths (m, format) {
        if (!m) {
            return isArray(this._months) ? this._months :
                this._months['standalone'];
        }
        return isArray(this._months) ? this._months[m.month()] :
            this._months[(this._months.isFormat || MONTHS_IN_FORMAT).test(format) ? 'format' : 'standalone'][m.month()];
    }

    var defaultLocaleMonthsShort = 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_');
    function localeMonthsShort (m, format) {
        if (!m) {
            return isArray(this._monthsShort) ? this._monthsShort :
                this._monthsShort['standalone'];
        }
        return isArray(this._monthsShort) ? this._monthsShort[m.month()] :
            this._monthsShort[MONTHS_IN_FORMAT.test(format) ? 'format' : 'standalone'][m.month()];
    }

    function handleStrictParse(monthName, format, strict) {
        var i, ii, mom, llc = monthName.toLocaleLowerCase();
        if (!this._monthsParse) {
            // this is not used
            this._monthsParse = [];
            this._longMonthsParse = [];
            this._shortMonthsParse = [];
            for (i = 0; i < 12; ++i) {
                mom = createUTC([2000, i]);
                this._shortMonthsParse[i] = this.monthsShort(mom, '').toLocaleLowerCase();
                this._longMonthsParse[i] = this.months(mom, '').toLocaleLowerCase();
            }
        }

        if (strict) {
            if (format === 'MMM') {
                ii = indexOf.call(this._shortMonthsParse, llc);
                return ii !== -1 ? ii : null;
            } else {
                ii = indexOf.call(this._longMonthsParse, llc);
                return ii !== -1 ? ii : null;
            }
        } else {
            if (format === 'MMM') {
                ii = indexOf.call(this._shortMonthsParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._longMonthsParse, llc);
                return ii !== -1 ? ii : null;
            } else {
                ii = indexOf.call(this._longMonthsParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._shortMonthsParse, llc);
                return ii !== -1 ? ii : null;
            }
        }
    }

    function localeMonthsParse (monthName, format, strict) {
        var i, mom, regex;

        if (this._monthsParseExact) {
            return handleStrictParse.call(this, monthName, format, strict);
        }

        if (!this._monthsParse) {
            this._monthsParse = [];
            this._longMonthsParse = [];
            this._shortMonthsParse = [];
        }

        // TODO: add sorting
        // Sorting makes sure if one month (or abbr) is a prefix of another
        // see sorting in computeMonthsParse
        for (i = 0; i < 12; i++) {
            // make the regex if we don't have it already
            mom = createUTC([2000, i]);
            if (strict && !this._longMonthsParse[i]) {
                this._longMonthsParse[i] = new RegExp('^' + this.months(mom, '').replace('.', '') + '$', 'i');
                this._shortMonthsParse[i] = new RegExp('^' + this.monthsShort(mom, '').replace('.', '') + '$', 'i');
            }
            if (!strict && !this._monthsParse[i]) {
                regex = '^' + this.months(mom, '') + '|^' + this.monthsShort(mom, '');
                this._monthsParse[i] = new RegExp(regex.replace('.', ''), 'i');
            }
            // test the regex
            if (strict && format === 'MMMM' && this._longMonthsParse[i].test(monthName)) {
                return i;
            } else if (strict && format === 'MMM' && this._shortMonthsParse[i].test(monthName)) {
                return i;
            } else if (!strict && this._monthsParse[i].test(monthName)) {
                return i;
            }
        }
    }

    // MOMENTS

    function setMonth (mom, value) {
        var dayOfMonth;

        if (!mom.isValid()) {
            // No op
            return mom;
        }

        if (typeof value === 'string') {
            if (/^\d+$/.test(value)) {
                value = toInt(value);
            } else {
                value = mom.localeData().monthsParse(value);
                // TODO: Another silent failure?
                if (!isNumber(value)) {
                    return mom;
                }
            }
        }

        dayOfMonth = Math.min(mom.date(), daysInMonth(mom.year(), value));
        mom._d['set' + (mom._isUTC ? 'UTC' : '') + 'Month'](value, dayOfMonth);
        return mom;
    }

    function getSetMonth (value) {
        if (value != null) {
            setMonth(this, value);
            hooks.updateOffset(this, true);
            return this;
        } else {
            return get(this, 'Month');
        }
    }

    function getDaysInMonth () {
        return daysInMonth(this.year(), this.month());
    }

    var defaultMonthsShortRegex = matchWord;
    function monthsShortRegex (isStrict) {
        if (this._monthsParseExact) {
            if (!hasOwnProp(this, '_monthsRegex')) {
                computeMonthsParse.call(this);
            }
            if (isStrict) {
                return this._monthsShortStrictRegex;
            } else {
                return this._monthsShortRegex;
            }
        } else {
            if (!hasOwnProp(this, '_monthsShortRegex')) {
                this._monthsShortRegex = defaultMonthsShortRegex;
            }
            return this._monthsShortStrictRegex && isStrict ?
                this._monthsShortStrictRegex : this._monthsShortRegex;
        }
    }

    var defaultMonthsRegex = matchWord;
    function monthsRegex (isStrict) {
        if (this._monthsParseExact) {
            if (!hasOwnProp(this, '_monthsRegex')) {
                computeMonthsParse.call(this);
            }
            if (isStrict) {
                return this._monthsStrictRegex;
            } else {
                return this._monthsRegex;
            }
        } else {
            if (!hasOwnProp(this, '_monthsRegex')) {
                this._monthsRegex = defaultMonthsRegex;
            }
            return this._monthsStrictRegex && isStrict ?
                this._monthsStrictRegex : this._monthsRegex;
        }
    }

    function computeMonthsParse () {
        function cmpLenRev(a, b) {
            return b.length - a.length;
        }

        var shortPieces = [], longPieces = [], mixedPieces = [],
            i, mom;
        for (i = 0; i < 12; i++) {
            // make the regex if we don't have it already
            mom = createUTC([2000, i]);
            shortPieces.push(this.monthsShort(mom, ''));
            longPieces.push(this.months(mom, ''));
            mixedPieces.push(this.months(mom, ''));
            mixedPieces.push(this.monthsShort(mom, ''));
        }
        // Sorting makes sure if one month (or abbr) is a prefix of another it
        // will match the longer piece.
        shortPieces.sort(cmpLenRev);
        longPieces.sort(cmpLenRev);
        mixedPieces.sort(cmpLenRev);
        for (i = 0; i < 12; i++) {
            shortPieces[i] = regexEscape(shortPieces[i]);
            longPieces[i] = regexEscape(longPieces[i]);
        }
        for (i = 0; i < 24; i++) {
            mixedPieces[i] = regexEscape(mixedPieces[i]);
        }

        this._monthsRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
        this._monthsShortRegex = this._monthsRegex;
        this._monthsStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
        this._monthsShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
    }

    function createDate (y, m, d, h, M, s, ms) {
        // can't just apply() to create a date:
        // https://stackoverflow.com/q/181348
        var date;
        // the date constructor remaps years 0-99 to 1900-1999
        if (y < 100 && y >= 0) {
            // preserve leap years using a full 400 year cycle, then reset
            date = new Date(y + 400, m, d, h, M, s, ms);
            if (isFinite(date.getFullYear())) {
                date.setFullYear(y);
            }
        } else {
            date = new Date(y, m, d, h, M, s, ms);
        }

        return date;
    }

    function createUTCDate (y) {
        var date;
        // the Date.UTC function remaps years 0-99 to 1900-1999
        if (y < 100 && y >= 0) {
            var args = Array.prototype.slice.call(arguments);
            // preserve leap years using a full 400 year cycle, then reset
            args[0] = y + 400;
            date = new Date(Date.UTC.apply(null, args));
            if (isFinite(date.getUTCFullYear())) {
                date.setUTCFullYear(y);
            }
        } else {
            date = new Date(Date.UTC.apply(null, arguments));
        }

        return date;
    }

    // start-of-first-week - start-of-year
    function firstWeekOffset(year, dow, doy) {
        var // first-week day -- which january is always in the first week (4 for iso, 1 for other)
            fwd = 7 + dow - doy,
            // first-week day local weekday -- which local weekday is fwd
            fwdlw = (7 + createUTCDate(year, 0, fwd).getUTCDay() - dow) % 7;

        return -fwdlw + fwd - 1;
    }

    // https://en.wikipedia.org/wiki/ISO_week_date#Calculating_a_date_given_the_year.2C_week_number_and_weekday
    function dayOfYearFromWeeks(year, week, weekday, dow, doy) {
        var localWeekday = (7 + weekday - dow) % 7,
            weekOffset = firstWeekOffset(year, dow, doy),
            dayOfYear = 1 + 7 * (week - 1) + localWeekday + weekOffset,
            resYear, resDayOfYear;

        if (dayOfYear <= 0) {
            resYear = year - 1;
            resDayOfYear = daysInYear(resYear) + dayOfYear;
        } else if (dayOfYear > daysInYear(year)) {
            resYear = year + 1;
            resDayOfYear = dayOfYear - daysInYear(year);
        } else {
            resYear = year;
            resDayOfYear = dayOfYear;
        }

        return {
            year: resYear,
            dayOfYear: resDayOfYear
        };
    }

    function weekOfYear(mom, dow, doy) {
        var weekOffset = firstWeekOffset(mom.year(), dow, doy),
            week = Math.floor((mom.dayOfYear() - weekOffset - 1) / 7) + 1,
            resWeek, resYear;

        if (week < 1) {
            resYear = mom.year() - 1;
            resWeek = week + weeksInYear(resYear, dow, doy);
        } else if (week > weeksInYear(mom.year(), dow, doy)) {
            resWeek = week - weeksInYear(mom.year(), dow, doy);
            resYear = mom.year() + 1;
        } else {
            resYear = mom.year();
            resWeek = week;
        }

        return {
            week: resWeek,
            year: resYear
        };
    }

    function weeksInYear(year, dow, doy) {
        var weekOffset = firstWeekOffset(year, dow, doy),
            weekOffsetNext = firstWeekOffset(year + 1, dow, doy);
        return (daysInYear(year) - weekOffset + weekOffsetNext) / 7;
    }

    // FORMATTING

    addFormatToken('w', ['ww', 2], 'wo', 'week');
    addFormatToken('W', ['WW', 2], 'Wo', 'isoWeek');

    // ALIASES

    addUnitAlias('week', 'w');
    addUnitAlias('isoWeek', 'W');

    // PRIORITIES

    addUnitPriority('week', 5);
    addUnitPriority('isoWeek', 5);

    // PARSING

    addRegexToken('w',  match1to2);
    addRegexToken('ww', match1to2, match2);
    addRegexToken('W',  match1to2);
    addRegexToken('WW', match1to2, match2);

    addWeekParseToken(['w', 'ww', 'W', 'WW'], function (input, week, config, token) {
        week[token.substr(0, 1)] = toInt(input);
    });

    // HELPERS

    // LOCALES

    function localeWeek (mom) {
        return weekOfYear(mom, this._week.dow, this._week.doy).week;
    }

    var defaultLocaleWeek = {
        dow : 0, // Sunday is the first day of the week.
        doy : 6  // The week that contains Jan 6th is the first week of the year.
    };

    function localeFirstDayOfWeek () {
        return this._week.dow;
    }

    function localeFirstDayOfYear () {
        return this._week.doy;
    }

    // MOMENTS

    function getSetWeek (input) {
        var week = this.localeData().week(this);
        return input == null ? week : this.add((input - week) * 7, 'd');
    }

    function getSetISOWeek (input) {
        var week = weekOfYear(this, 1, 4).week;
        return input == null ? week : this.add((input - week) * 7, 'd');
    }

    // FORMATTING

    addFormatToken('d', 0, 'do', 'day');

    addFormatToken('dd', 0, 0, function (format) {
        return this.localeData().weekdaysMin(this, format);
    });

    addFormatToken('ddd', 0, 0, function (format) {
        return this.localeData().weekdaysShort(this, format);
    });

    addFormatToken('dddd', 0, 0, function (format) {
        return this.localeData().weekdays(this, format);
    });

    addFormatToken('e', 0, 0, 'weekday');
    addFormatToken('E', 0, 0, 'isoWeekday');

    // ALIASES

    addUnitAlias('day', 'd');
    addUnitAlias('weekday', 'e');
    addUnitAlias('isoWeekday', 'E');

    // PRIORITY
    addUnitPriority('day', 11);
    addUnitPriority('weekday', 11);
    addUnitPriority('isoWeekday', 11);

    // PARSING

    addRegexToken('d',    match1to2);
    addRegexToken('e',    match1to2);
    addRegexToken('E',    match1to2);
    addRegexToken('dd',   function (isStrict, locale) {
        return locale.weekdaysMinRegex(isStrict);
    });
    addRegexToken('ddd',   function (isStrict, locale) {
        return locale.weekdaysShortRegex(isStrict);
    });
    addRegexToken('dddd',   function (isStrict, locale) {
        return locale.weekdaysRegex(isStrict);
    });

    addWeekParseToken(['dd', 'ddd', 'dddd'], function (input, week, config, token) {
        var weekday = config._locale.weekdaysParse(input, token, config._strict);
        // if we didn't get a weekday name, mark the date as invalid
        if (weekday != null) {
            week.d = weekday;
        } else {
            getParsingFlags(config).invalidWeekday = input;
        }
    });

    addWeekParseToken(['d', 'e', 'E'], function (input, week, config, token) {
        week[token] = toInt(input);
    });

    // HELPERS

    function parseWeekday(input, locale) {
        if (typeof input !== 'string') {
            return input;
        }

        if (!isNaN(input)) {
            return parseInt(input, 10);
        }

        input = locale.weekdaysParse(input);
        if (typeof input === 'number') {
            return input;
        }

        return null;
    }

    function parseIsoWeekday(input, locale) {
        if (typeof input === 'string') {
            return locale.weekdaysParse(input) % 7 || 7;
        }
        return isNaN(input) ? null : input;
    }

    // LOCALES
    function shiftWeekdays (ws, n) {
        return ws.slice(n, 7).concat(ws.slice(0, n));
    }

    var defaultLocaleWeekdays = 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_');
    function localeWeekdays (m, format) {
        var weekdays = isArray(this._weekdays) ? this._weekdays :
            this._weekdays[(m && m !== true && this._weekdays.isFormat.test(format)) ? 'format' : 'standalone'];
        return (m === true) ? shiftWeekdays(weekdays, this._week.dow)
            : (m) ? weekdays[m.day()] : weekdays;
    }

    var defaultLocaleWeekdaysShort = 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_');
    function localeWeekdaysShort (m) {
        return (m === true) ? shiftWeekdays(this._weekdaysShort, this._week.dow)
            : (m) ? this._weekdaysShort[m.day()] : this._weekdaysShort;
    }

    var defaultLocaleWeekdaysMin = 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_');
    function localeWeekdaysMin (m) {
        return (m === true) ? shiftWeekdays(this._weekdaysMin, this._week.dow)
            : (m) ? this._weekdaysMin[m.day()] : this._weekdaysMin;
    }

    function handleStrictParse$1(weekdayName, format, strict) {
        var i, ii, mom, llc = weekdayName.toLocaleLowerCase();
        if (!this._weekdaysParse) {
            this._weekdaysParse = [];
            this._shortWeekdaysParse = [];
            this._minWeekdaysParse = [];

            for (i = 0; i < 7; ++i) {
                mom = createUTC([2000, 1]).day(i);
                this._minWeekdaysParse[i] = this.weekdaysMin(mom, '').toLocaleLowerCase();
                this._shortWeekdaysParse[i] = this.weekdaysShort(mom, '').toLocaleLowerCase();
                this._weekdaysParse[i] = this.weekdays(mom, '').toLocaleLowerCase();
            }
        }

        if (strict) {
            if (format === 'dddd') {
                ii = indexOf.call(this._weekdaysParse, llc);
                return ii !== -1 ? ii : null;
            } else if (format === 'ddd') {
                ii = indexOf.call(this._shortWeekdaysParse, llc);
                return ii !== -1 ? ii : null;
            } else {
                ii = indexOf.call(this._minWeekdaysParse, llc);
                return ii !== -1 ? ii : null;
            }
        } else {
            if (format === 'dddd') {
                ii = indexOf.call(this._weekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._shortWeekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._minWeekdaysParse, llc);
                return ii !== -1 ? ii : null;
            } else if (format === 'ddd') {
                ii = indexOf.call(this._shortWeekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._weekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._minWeekdaysParse, llc);
                return ii !== -1 ? ii : null;
            } else {
                ii = indexOf.call(this._minWeekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._weekdaysParse, llc);
                if (ii !== -1) {
                    return ii;
                }
                ii = indexOf.call(this._shortWeekdaysParse, llc);
                return ii !== -1 ? ii : null;
            }
        }
    }

    function localeWeekdaysParse (weekdayName, format, strict) {
        var i, mom, regex;

        if (this._weekdaysParseExact) {
            return handleStrictParse$1.call(this, weekdayName, format, strict);
        }

        if (!this._weekdaysParse) {
            this._weekdaysParse = [];
            this._minWeekdaysParse = [];
            this._shortWeekdaysParse = [];
            this._fullWeekdaysParse = [];
        }

        for (i = 0; i < 7; i++) {
            // make the regex if we don't have it already

            mom = createUTC([2000, 1]).day(i);
            if (strict && !this._fullWeekdaysParse[i]) {
                this._fullWeekdaysParse[i] = new RegExp('^' + this.weekdays(mom, '').replace('.', '\\.?') + '$', 'i');
                this._shortWeekdaysParse[i] = new RegExp('^' + this.weekdaysShort(mom, '').replace('.', '\\.?') + '$', 'i');
                this._minWeekdaysParse[i] = new RegExp('^' + this.weekdaysMin(mom, '').replace('.', '\\.?') + '$', 'i');
            }
            if (!this._weekdaysParse[i]) {
                regex = '^' + this.weekdays(mom, '') + '|^' + this.weekdaysShort(mom, '') + '|^' + this.weekdaysMin(mom, '');
                this._weekdaysParse[i] = new RegExp(regex.replace('.', ''), 'i');
            }
            // test the regex
            if (strict && format === 'dddd' && this._fullWeekdaysParse[i].test(weekdayName)) {
                return i;
            } else if (strict && format === 'ddd' && this._shortWeekdaysParse[i].test(weekdayName)) {
                return i;
            } else if (strict && format === 'dd' && this._minWeekdaysParse[i].test(weekdayName)) {
                return i;
            } else if (!strict && this._weekdaysParse[i].test(weekdayName)) {
                return i;
            }
        }
    }

    // MOMENTS

    function getSetDayOfWeek (input) {
        if (!this.isValid()) {
            return input != null ? this : NaN;
        }
        var day = this._isUTC ? this._d.getUTCDay() : this._d.getDay();
        if (input != null) {
            input = parseWeekday(input, this.localeData());
            return this.add(input - day, 'd');
        } else {
            return day;
        }
    }

    function getSetLocaleDayOfWeek (input) {
        if (!this.isValid()) {
            return input != null ? this : NaN;
        }
        var weekday = (this.day() + 7 - this.localeData()._week.dow) % 7;
        return input == null ? weekday : this.add(input - weekday, 'd');
    }

    function getSetISODayOfWeek (input) {
        if (!this.isValid()) {
            return input != null ? this : NaN;
        }

        // behaves the same as moment#day except
        // as a getter, returns 7 instead of 0 (1-7 range instead of 0-6)
        // as a setter, sunday should belong to the previous week.

        if (input != null) {
            var weekday = parseIsoWeekday(input, this.localeData());
            return this.day(this.day() % 7 ? weekday : weekday - 7);
        } else {
            return this.day() || 7;
        }
    }

    var defaultWeekdaysRegex = matchWord;
    function weekdaysRegex (isStrict) {
        if (this._weekdaysParseExact) {
            if (!hasOwnProp(this, '_weekdaysRegex')) {
                computeWeekdaysParse.call(this);
            }
            if (isStrict) {
                return this._weekdaysStrictRegex;
            } else {
                return this._weekdaysRegex;
            }
        } else {
            if (!hasOwnProp(this, '_weekdaysRegex')) {
                this._weekdaysRegex = defaultWeekdaysRegex;
            }
            return this._weekdaysStrictRegex && isStrict ?
                this._weekdaysStrictRegex : this._weekdaysRegex;
        }
    }

    var defaultWeekdaysShortRegex = matchWord;
    function weekdaysShortRegex (isStrict) {
        if (this._weekdaysParseExact) {
            if (!hasOwnProp(this, '_weekdaysRegex')) {
                computeWeekdaysParse.call(this);
            }
            if (isStrict) {
                return this._weekdaysShortStrictRegex;
            } else {
                return this._weekdaysShortRegex;
            }
        } else {
            if (!hasOwnProp(this, '_weekdaysShortRegex')) {
                this._weekdaysShortRegex = defaultWeekdaysShortRegex;
            }
            return this._weekdaysShortStrictRegex && isStrict ?
                this._weekdaysShortStrictRegex : this._weekdaysShortRegex;
        }
    }

    var defaultWeekdaysMinRegex = matchWord;
    function weekdaysMinRegex (isStrict) {
        if (this._weekdaysParseExact) {
            if (!hasOwnProp(this, '_weekdaysRegex')) {
                computeWeekdaysParse.call(this);
            }
            if (isStrict) {
                return this._weekdaysMinStrictRegex;
            } else {
                return this._weekdaysMinRegex;
            }
        } else {
            if (!hasOwnProp(this, '_weekdaysMinRegex')) {
                this._weekdaysMinRegex = defaultWeekdaysMinRegex;
            }
            return this._weekdaysMinStrictRegex && isStrict ?
                this._weekdaysMinStrictRegex : this._weekdaysMinRegex;
        }
    }


    function computeWeekdaysParse () {
        function cmpLenRev(a, b) {
            return b.length - a.length;
        }

        var minPieces = [], shortPieces = [], longPieces = [], mixedPieces = [],
            i, mom, minp, shortp, longp;
        for (i = 0; i < 7; i++) {
            // make the regex if we don't have it already
            mom = createUTC([2000, 1]).day(i);
            minp = this.weekdaysMin(mom, '');
            shortp = this.weekdaysShort(mom, '');
            longp = this.weekdays(mom, '');
            minPieces.push(minp);
            shortPieces.push(shortp);
            longPieces.push(longp);
            mixedPieces.push(minp);
            mixedPieces.push(shortp);
            mixedPieces.push(longp);
        }
        // Sorting makes sure if one weekday (or abbr) is a prefix of another it
        // will match the longer piece.
        minPieces.sort(cmpLenRev);
        shortPieces.sort(cmpLenRev);
        longPieces.sort(cmpLenRev);
        mixedPieces.sort(cmpLenRev);
        for (i = 0; i < 7; i++) {
            shortPieces[i] = regexEscape(shortPieces[i]);
            longPieces[i] = regexEscape(longPieces[i]);
            mixedPieces[i] = regexEscape(mixedPieces[i]);
        }

        this._weekdaysRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
        this._weekdaysShortRegex = this._weekdaysRegex;
        this._weekdaysMinRegex = this._weekdaysRegex;

        this._weekdaysStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
        this._weekdaysShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
        this._weekdaysMinStrictRegex = new RegExp('^(' + minPieces.join('|') + ')', 'i');
    }

    // FORMATTING

    function hFormat() {
        return this.hours() % 12 || 12;
    }

    function kFormat() {
        return this.hours() || 24;
    }

    addFormatToken('H', ['HH', 2], 0, 'hour');
    addFormatToken('h', ['hh', 2], 0, hFormat);
    addFormatToken('k', ['kk', 2], 0, kFormat);

    addFormatToken('hmm', 0, 0, function () {
        return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2);
    });

    addFormatToken('hmmss', 0, 0, function () {
        return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2) +
            zeroFill(this.seconds(), 2);
    });

    addFormatToken('Hmm', 0, 0, function () {
        return '' + this.hours() + zeroFill(this.minutes(), 2);
    });

    addFormatToken('Hmmss', 0, 0, function () {
        return '' + this.hours() + zeroFill(this.minutes(), 2) +
            zeroFill(this.seconds(), 2);
    });

    function meridiem (token, lowercase) {
        addFormatToken(token, 0, 0, function () {
            return this.localeData().meridiem(this.hours(), this.minutes(), lowercase);
        });
    }

    meridiem('a', true);
    meridiem('A', false);

    // ALIASES

    addUnitAlias('hour', 'h');

    // PRIORITY
    addUnitPriority('hour', 13);

    // PARSING

    function matchMeridiem (isStrict, locale) {
        return locale._meridiemParse;
    }

    addRegexToken('a',  matchMeridiem);
    addRegexToken('A',  matchMeridiem);
    addRegexToken('H',  match1to2);
    addRegexToken('h',  match1to2);
    addRegexToken('k',  match1to2);
    addRegexToken('HH', match1to2, match2);
    addRegexToken('hh', match1to2, match2);
    addRegexToken('kk', match1to2, match2);

    addRegexToken('hmm', match3to4);
    addRegexToken('hmmss', match5to6);
    addRegexToken('Hmm', match3to4);
    addRegexToken('Hmmss', match5to6);

    addParseToken(['H', 'HH'], HOUR);
    addParseToken(['k', 'kk'], function (input, array, config) {
        var kInput = toInt(input);
        array[HOUR] = kInput === 24 ? 0 : kInput;
    });
    addParseToken(['a', 'A'], function (input, array, config) {
        config._isPm = config._locale.isPM(input);
        config._meridiem = input;
    });
    addParseToken(['h', 'hh'], function (input, array, config) {
        array[HOUR] = toInt(input);
        getParsingFlags(config).bigHour = true;
    });
    addParseToken('hmm', function (input, array, config) {
        var pos = input.length - 2;
        array[HOUR] = toInt(input.substr(0, pos));
        array[MINUTE] = toInt(input.substr(pos));
        getParsingFlags(config).bigHour = true;
    });
    addParseToken('hmmss', function (input, array, config) {
        var pos1 = input.length - 4;
        var pos2 = input.length - 2;
        array[HOUR] = toInt(input.substr(0, pos1));
        array[MINUTE] = toInt(input.substr(pos1, 2));
        array[SECOND] = toInt(input.substr(pos2));
        getParsingFlags(config).bigHour = true;
    });
    addParseToken('Hmm', function (input, array, config) {
        var pos = input.length - 2;
        array[HOUR] = toInt(input.substr(0, pos));
        array[MINUTE] = toInt(input.substr(pos));
    });
    addParseToken('Hmmss', function (input, array, config) {
        var pos1 = input.length - 4;
        var pos2 = input.length - 2;
        array[HOUR] = toInt(input.substr(0, pos1));
        array[MINUTE] = toInt(input.substr(pos1, 2));
        array[SECOND] = toInt(input.substr(pos2));
    });

    // LOCALES

    function localeIsPM (input) {
        // IE8 Quirks Mode & IE7 Standards Mode do not allow accessing strings like arrays
        // Using charAt should be more compatible.
        return ((input + '').toLowerCase().charAt(0) === 'p');
    }

    var defaultLocaleMeridiemParse = /[ap]\.?m?\.?/i;
    function localeMeridiem (hours, minutes, isLower) {
        if (hours > 11) {
            return isLower ? 'pm' : 'PM';
        } else {
            return isLower ? 'am' : 'AM';
        }
    }


    // MOMENTS

    // Setting the hour should keep the time, because the user explicitly
    // specified which hour they want. So trying to maintain the same hour (in
    // a new timezone) makes sense. Adding/subtracting hours does not follow
    // this rule.
    var getSetHour = makeGetSet('Hours', true);

    var baseConfig = {
        calendar: defaultCalendar,
        longDateFormat: defaultLongDateFormat,
        invalidDate: defaultInvalidDate,
        ordinal: defaultOrdinal,
        dayOfMonthOrdinalParse: defaultDayOfMonthOrdinalParse,
        relativeTime: defaultRelativeTime,

        months: defaultLocaleMonths,
        monthsShort: defaultLocaleMonthsShort,

        week: defaultLocaleWeek,

        weekdays: defaultLocaleWeekdays,
        weekdaysMin: defaultLocaleWeekdaysMin,
        weekdaysShort: defaultLocaleWeekdaysShort,

        meridiemParse: defaultLocaleMeridiemParse
    };

    // internal storage for locale config files
    var locales = {};
    var localeFamilies = {};
    var globalLocale;

    function normalizeLocale(key) {
        return key ? key.toLowerCase().replace('_', '-') : key;
    }

    // pick the locale from the array
    // try ['en-au', 'en-gb'] as 'en-au', 'en-gb', 'en', as in move through the list trying each
    // substring from most specific to least, but move to the next array item if it's a more specific variant than the current root
    function chooseLocale(names) {
        var i = 0, j, next, locale, split;

        while (i < names.length) {
            split = normalizeLocale(names[i]).split('-');
            j = split.length;
            next = normalizeLocale(names[i + 1]);
            next = next ? next.split('-') : null;
            while (j > 0) {
                locale = loadLocale(split.slice(0, j).join('-'));
                if (locale) {
                    return locale;
                }
                if (next && next.length >= j && compareArrays(split, next, true) >= j - 1) {
                    //the next array item is better than a shallower substring of this one
                    break;
                }
                j--;
            }
            i++;
        }
        return globalLocale;
    }

    function loadLocale(name) {
        var oldLocale = null;
        // TODO: Find a better way to register and load all the locales in Node
        if (!locales[name] && (typeof module !== 'undefined') &&
                module && module.exports) {
            try {
                oldLocale = globalLocale._abbr;
                var aliasedRequire = require;
                aliasedRequire('./locale/' + name);
                getSetGlobalLocale(oldLocale);
            } catch (e) {}
        }
        return locales[name];
    }

    // This function will load locale and then set the global locale.  If
    // no arguments are passed in, it will simply return the current global
    // locale key.
    function getSetGlobalLocale (key, values) {
        var data;
        if (key) {
            if (isUndefined(values)) {
                data = getLocale(key);
            }
            else {
                data = defineLocale(key, values);
            }

            if (data) {
                // moment.duration._locale = moment._locale = data;
                globalLocale = data;
            }
            else {
                if ((typeof console !==  'undefined') && console.warn) {
                    //warn user if arguments are passed but the locale could not be set
                    console.warn('Locale ' + key +  ' not found. Did you forget to load it?');
                }
            }
        }

        return globalLocale._abbr;
    }

    function defineLocale (name, config) {
        if (config !== null) {
            var locale, parentConfig = baseConfig;
            config.abbr = name;
            if (locales[name] != null) {
                deprecateSimple('defineLocaleOverride',
                        'use moment.updateLocale(localeName, config) to change ' +
                        'an existing locale. moment.defineLocale(localeName, ' +
                        'config) should only be used for creating a new locale ' +
                        'See http://momentjs.com/guides/#/warnings/define-locale/ for more info.');
                parentConfig = locales[name]._config;
            } else if (config.parentLocale != null) {
                if (locales[config.parentLocale] != null) {
                    parentConfig = locales[config.parentLocale]._config;
                } else {
                    locale = loadLocale(config.parentLocale);
                    if (locale != null) {
                        parentConfig = locale._config;
                    } else {
                        if (!localeFamilies[config.parentLocale]) {
                            localeFamilies[config.parentLocale] = [];
                        }
                        localeFamilies[config.parentLocale].push({
                            name: name,
                            config: config
                        });
                        return null;
                    }
                }
            }
            locales[name] = new Locale(mergeConfigs(parentConfig, config));

            if (localeFamilies[name]) {
                localeFamilies[name].forEach(function (x) {
                    defineLocale(x.name, x.config);
                });
            }

            // backwards compat for now: also set the locale
            // make sure we set the locale AFTER all child locales have been
            // created, so we won't end up with the child locale set.
            getSetGlobalLocale(name);


            return locales[name];
        } else {
            // useful for testing
            delete locales[name];
            return null;
        }
    }

    function updateLocale(name, config) {
        if (config != null) {
            var locale, tmpLocale, parentConfig = baseConfig;
            // MERGE
            tmpLocale = loadLocale(name);
            if (tmpLocale != null) {
                parentConfig = tmpLocale._config;
            }
            config = mergeConfigs(parentConfig, config);
            locale = new Locale(config);
            locale.parentLocale = locales[name];
            locales[name] = locale;

            // backwards compat for now: also set the locale
            getSetGlobalLocale(name);
        } else {
            // pass null for config to unupdate, useful for tests
            if (locales[name] != null) {
                if (locales[name].parentLocale != null) {
                    locales[name] = locales[name].parentLocale;
                } else if (locales[name] != null) {
                    delete locales[name];
                }
            }
        }
        return locales[name];
    }

    // returns locale data
    function getLocale (key) {
        var locale;

        if (key && key._locale && key._locale._abbr) {
            key = key._locale._abbr;
        }

        if (!key) {
            return globalLocale;
        }

        if (!isArray(key)) {
            //short-circuit everything else
            locale = loadLocale(key);
            if (locale) {
                return locale;
            }
            key = [key];
        }

        return chooseLocale(key);
    }

    function listLocales() {
        return keys(locales);
    }

    function checkOverflow (m) {
        var overflow;
        var a = m._a;

        if (a && getParsingFlags(m).overflow === -2) {
            overflow =
                a[MONTH]       < 0 || a[MONTH]       > 11  ? MONTH :
                a[DATE]        < 1 || a[DATE]        > daysInMonth(a[YEAR], a[MONTH]) ? DATE :
                a[HOUR]        < 0 || a[HOUR]        > 24 || (a[HOUR] === 24 && (a[MINUTE] !== 0 || a[SECOND] !== 0 || a[MILLISECOND] !== 0)) ? HOUR :
                a[MINUTE]      < 0 || a[MINUTE]      > 59  ? MINUTE :
                a[SECOND]      < 0 || a[SECOND]      > 59  ? SECOND :
                a[MILLISECOND] < 0 || a[MILLISECOND] > 999 ? MILLISECOND :
                -1;

            if (getParsingFlags(m)._overflowDayOfYear && (overflow < YEAR || overflow > DATE)) {
                overflow = DATE;
            }
            if (getParsingFlags(m)._overflowWeeks && overflow === -1) {
                overflow = WEEK;
            }
            if (getParsingFlags(m)._overflowWeekday && overflow === -1) {
                overflow = WEEKDAY;
            }

            getParsingFlags(m).overflow = overflow;
        }

        return m;
    }

    // Pick the first defined of two or three arguments.
    function defaults(a, b, c) {
        if (a != null) {
            return a;
        }
        if (b != null) {
            return b;
        }
        return c;
    }

    function currentDateArray(config) {
        // hooks is actually the exported moment object
        var nowValue = new Date(hooks.now());
        if (config._useUTC) {
            return [nowValue.getUTCFullYear(), nowValue.getUTCMonth(), nowValue.getUTCDate()];
        }
        return [nowValue.getFullYear(), nowValue.getMonth(), nowValue.getDate()];
    }

    // convert an array to a date.
    // the array should mirror the parameters below
    // note: all values past the year are optional and will default to the lowest possible value.
    // [year, month, day , hour, minute, second, millisecond]
    function configFromArray (config) {
        var i, date, input = [], currentDate, expectedWeekday, yearToUse;

        if (config._d) {
            return;
        }

        currentDate = currentDateArray(config);

        //compute day of the year from weeks and weekdays
        if (config._w && config._a[DATE] == null && config._a[MONTH] == null) {
            dayOfYearFromWeekInfo(config);
        }

        //if the day of the year is set, figure out what it is
        if (config._dayOfYear != null) {
            yearToUse = defaults(config._a[YEAR], currentDate[YEAR]);

            if (config._dayOfYear > daysInYear(yearToUse) || config._dayOfYear === 0) {
                getParsingFlags(config)._overflowDayOfYear = true;
            }

            date = createUTCDate(yearToUse, 0, config._dayOfYear);
            config._a[MONTH] = date.getUTCMonth();
            config._a[DATE] = date.getUTCDate();
        }

        // Default to current date.
        // * if no year, month, day of month are given, default to today
        // * if day of month is given, default month and year
        // * if month is given, default only year
        // * if year is given, don't default anything
        for (i = 0; i < 3 && config._a[i] == null; ++i) {
            config._a[i] = input[i] = currentDate[i];
        }

        // Zero out whatever was not defaulted, including time
        for (; i < 7; i++) {
            config._a[i] = input[i] = (config._a[i] == null) ? (i === 2 ? 1 : 0) : config._a[i];
        }

        // Check for 24:00:00.000
        if (config._a[HOUR] === 24 &&
                config._a[MINUTE] === 0 &&
                config._a[SECOND] === 0 &&
                config._a[MILLISECOND] === 0) {
            config._nextDay = true;
            config._a[HOUR] = 0;
        }

        config._d = (config._useUTC ? createUTCDate : createDate).apply(null, input);
        expectedWeekday = config._useUTC ? config._d.getUTCDay() : config._d.getDay();

        // Apply timezone offset from input. The actual utcOffset can be changed
        // with parseZone.
        if (config._tzm != null) {
            config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);
        }

        if (config._nextDay) {
            config._a[HOUR] = 24;
        }

        // check for mismatching day of week
        if (config._w && typeof config._w.d !== 'undefined' && config._w.d !== expectedWeekday) {
            getParsingFlags(config).weekdayMismatch = true;
        }
    }

    function dayOfYearFromWeekInfo(config) {
        var w, weekYear, week, weekday, dow, doy, temp, weekdayOverflow;

        w = config._w;
        if (w.GG != null || w.W != null || w.E != null) {
            dow = 1;
            doy = 4;

            // TODO: We need to take the current isoWeekYear, but that depends on
            // how we interpret now (local, utc, fixed offset). So create
            // a now version of current config (take local/utc/offset flags, and
            // create now).
            weekYear = defaults(w.GG, config._a[YEAR], weekOfYear(createLocal(), 1, 4).year);
            week = defaults(w.W, 1);
            weekday = defaults(w.E, 1);
            if (weekday < 1 || weekday > 7) {
                weekdayOverflow = true;
            }
        } else {
            dow = config._locale._week.dow;
            doy = config._locale._week.doy;

            var curWeek = weekOfYear(createLocal(), dow, doy);

            weekYear = defaults(w.gg, config._a[YEAR], curWeek.year);

            // Default to current week.
            week = defaults(w.w, curWeek.week);

            if (w.d != null) {
                // weekday -- low day numbers are considered next week
                weekday = w.d;
                if (weekday < 0 || weekday > 6) {
                    weekdayOverflow = true;
                }
            } else if (w.e != null) {
                // local weekday -- counting starts from beginning of week
                weekday = w.e + dow;
                if (w.e < 0 || w.e > 6) {
                    weekdayOverflow = true;
                }
            } else {
                // default to beginning of week
                weekday = dow;
            }
        }
        if (week < 1 || week > weeksInYear(weekYear, dow, doy)) {
            getParsingFlags(config)._overflowWeeks = true;
        } else if (weekdayOverflow != null) {
            getParsingFlags(config)._overflowWeekday = true;
        } else {
            temp = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy);
            config._a[YEAR] = temp.year;
            config._dayOfYear = temp.dayOfYear;
        }
    }

    // iso 8601 regex
    // 0000-00-00 0000-W00 or 0000-W00-0 + T + 00 or 00:00 or 00:00:00 or 00:00:00.000 + +00:00 or +0000 or +00)
    var extendedIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})-(?:\d\d-\d\d|W\d\d-\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?::\d\d(?::\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;
    var basicIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})(?:\d\d\d\d|W\d\d\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?:\d\d(?:\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;

    var tzRegex = /Z|[+-]\d\d(?::?\d\d)?/;

    var isoDates = [
        ['YYYYYY-MM-DD', /[+-]\d{6}-\d\d-\d\d/],
        ['YYYY-MM-DD', /\d{4}-\d\d-\d\d/],
        ['GGGG-[W]WW-E', /\d{4}-W\d\d-\d/],
        ['GGGG-[W]WW', /\d{4}-W\d\d/, false],
        ['YYYY-DDD', /\d{4}-\d{3}/],
        ['YYYY-MM', /\d{4}-\d\d/, false],
        ['YYYYYYMMDD', /[+-]\d{10}/],
        ['YYYYMMDD', /\d{8}/],
        // YYYYMM is NOT allowed by the standard
        ['GGGG[W]WWE', /\d{4}W\d{3}/],
        ['GGGG[W]WW', /\d{4}W\d{2}/, false],
        ['YYYYDDD', /\d{7}/]
    ];

    // iso time formats and regexes
    var isoTimes = [
        ['HH:mm:ss.SSSS', /\d\d:\d\d:\d\d\.\d+/],
        ['HH:mm:ss,SSSS', /\d\d:\d\d:\d\d,\d+/],
        ['HH:mm:ss', /\d\d:\d\d:\d\d/],
        ['HH:mm', /\d\d:\d\d/],
        ['HHmmss.SSSS', /\d\d\d\d\d\d\.\d+/],
        ['HHmmss,SSSS', /\d\d\d\d\d\d,\d+/],
        ['HHmmss', /\d\d\d\d\d\d/],
        ['HHmm', /\d\d\d\d/],
        ['HH', /\d\d/]
    ];

    var aspNetJsonRegex = /^\/?Date\((\-?\d+)/i;

    // date from iso format
    function configFromISO(config) {
        var i, l,
            string = config._i,
            match = extendedIsoRegex.exec(string) || basicIsoRegex.exec(string),
            allowTime, dateFormat, timeFormat, tzFormat;

        if (match) {
            getParsingFlags(config).iso = true;

            for (i = 0, l = isoDates.length; i < l; i++) {
                if (isoDates[i][1].exec(match[1])) {
                    dateFormat = isoDates[i][0];
                    allowTime = isoDates[i][2] !== false;
                    break;
                }
            }
            if (dateFormat == null) {
                config._isValid = false;
                return;
            }
            if (match[3]) {
                for (i = 0, l = isoTimes.length; i < l; i++) {
                    if (isoTimes[i][1].exec(match[3])) {
                        // match[2] should be 'T' or space
                        timeFormat = (match[2] || ' ') + isoTimes[i][0];
                        break;
                    }
                }
                if (timeFormat == null) {
                    config._isValid = false;
                    return;
                }
            }
            if (!allowTime && timeFormat != null) {
                config._isValid = false;
                return;
            }
            if (match[4]) {
                if (tzRegex.exec(match[4])) {
                    tzFormat = 'Z';
                } else {
                    config._isValid = false;
                    return;
                }
            }
            config._f = dateFormat + (timeFormat || '') + (tzFormat || '');
            configFromStringAndFormat(config);
        } else {
            config._isValid = false;
        }
    }

    // RFC 2822 regex: For details see https://tools.ietf.org/html/rfc2822#section-3.3
    var rfc2822 = /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),?\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|([+-]\d{4}))$/;

    function extractFromRFC2822Strings(yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
        var result = [
            untruncateYear(yearStr),
            defaultLocaleMonthsShort.indexOf(monthStr),
            parseInt(dayStr, 10),
            parseInt(hourStr, 10),
            parseInt(minuteStr, 10)
        ];

        if (secondStr) {
            result.push(parseInt(secondStr, 10));
        }

        return result;
    }

    function untruncateYear(yearStr) {
        var year = parseInt(yearStr, 10);
        if (year <= 49) {
            return 2000 + year;
        } else if (year <= 999) {
            return 1900 + year;
        }
        return year;
    }

    function preprocessRFC2822(s) {
        // Remove comments and folding whitespace and replace multiple-spaces with a single space
        return s.replace(/\([^)]*\)|[\n\t]/g, ' ').replace(/(\s\s+)/g, ' ').replace(/^\s\s*/, '').replace(/\s\s*$/, '');
    }

    function checkWeekday(weekdayStr, parsedInput, config) {
        if (weekdayStr) {
            // TODO: Replace the vanilla JS Date object with an indepentent day-of-week check.
            var weekdayProvided = defaultLocaleWeekdaysShort.indexOf(weekdayStr),
                weekdayActual = new Date(parsedInput[0], parsedInput[1], parsedInput[2]).getDay();
            if (weekdayProvided !== weekdayActual) {
                getParsingFlags(config).weekdayMismatch = true;
                config._isValid = false;
                return false;
            }
        }
        return true;
    }

    var obsOffsets = {
        UT: 0,
        GMT: 0,
        EDT: -4 * 60,
        EST: -5 * 60,
        CDT: -5 * 60,
        CST: -6 * 60,
        MDT: -6 * 60,
        MST: -7 * 60,
        PDT: -7 * 60,
        PST: -8 * 60
    };

    function calculateOffset(obsOffset, militaryOffset, numOffset) {
        if (obsOffset) {
            return obsOffsets[obsOffset];
        } else if (militaryOffset) {
            // the only allowed military tz is Z
            return 0;
        } else {
            var hm = parseInt(numOffset, 10);
            var m = hm % 100, h = (hm - m) / 100;
            return h * 60 + m;
        }
    }

    // date and time from ref 2822 format
    function configFromRFC2822(config) {
        var match = rfc2822.exec(preprocessRFC2822(config._i));
        if (match) {
            var parsedArray = extractFromRFC2822Strings(match[4], match[3], match[2], match[5], match[6], match[7]);
            if (!checkWeekday(match[1], parsedArray, config)) {
                return;
            }

            config._a = parsedArray;
            config._tzm = calculateOffset(match[8], match[9], match[10]);

            config._d = createUTCDate.apply(null, config._a);
            config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);

            getParsingFlags(config).rfc2822 = true;
        } else {
            config._isValid = false;
        }
    }

    // date from iso format or fallback
    function configFromString(config) {
        var matched = aspNetJsonRegex.exec(config._i);

        if (matched !== null) {
            config._d = new Date(+matched[1]);
            return;
        }

        configFromISO(config);
        if (config._isValid === false) {
            delete config._isValid;
        } else {
            return;
        }

        configFromRFC2822(config);
        if (config._isValid === false) {
            delete config._isValid;
        } else {
            return;
        }

        // Final attempt, use Input Fallback
        hooks.createFromInputFallback(config);
    }

    hooks.createFromInputFallback = deprecate(
        'value provided is not in a recognized RFC2822 or ISO format. moment construction falls back to js Date(), ' +
        'which is not reliable across all browsers and versions. Non RFC2822/ISO date formats are ' +
        'discouraged and will be removed in an upcoming major release. Please refer to ' +
        'http://momentjs.com/guides/#/warnings/js-date/ for more info.',
        function (config) {
            config._d = new Date(config._i + (config._useUTC ? ' UTC' : ''));
        }
    );

    // constant that refers to the ISO standard
    hooks.ISO_8601 = function () {};

    // constant that refers to the RFC 2822 form
    hooks.RFC_2822 = function () {};

    // date from string and format string
    function configFromStringAndFormat(config) {
        // TODO: Move this to another part of the creation flow to prevent circular deps
        if (config._f === hooks.ISO_8601) {
            configFromISO(config);
            return;
        }
        if (config._f === hooks.RFC_2822) {
            configFromRFC2822(config);
            return;
        }
        config._a = [];
        getParsingFlags(config).empty = true;

        // This array is used to make a Date, either with `new Date` or `Date.UTC`
        var string = '' + config._i,
            i, parsedInput, tokens, token, skipped,
            stringLength = string.length,
            totalParsedInputLength = 0;

        tokens = expandFormat(config._f, config._locale).match(formattingTokens) || [];

        for (i = 0; i < tokens.length; i++) {
            token = tokens[i];
            parsedInput = (string.match(getParseRegexForToken(token, config)) || [])[0];
            // console.log('token', token, 'parsedInput', parsedInput,
            //         'regex', getParseRegexForToken(token, config));
            if (parsedInput) {
                skipped = string.substr(0, string.indexOf(parsedInput));
                if (skipped.length > 0) {
                    getParsingFlags(config).unusedInput.push(skipped);
                }
                string = string.slice(string.indexOf(parsedInput) + parsedInput.length);
                totalParsedInputLength += parsedInput.length;
            }
            // don't parse if it's not a known token
            if (formatTokenFunctions[token]) {
                if (parsedInput) {
                    getParsingFlags(config).empty = false;
                }
                else {
                    getParsingFlags(config).unusedTokens.push(token);
                }
                addTimeToArrayFromToken(token, parsedInput, config);
            }
            else if (config._strict && !parsedInput) {
                getParsingFlags(config).unusedTokens.push(token);
            }
        }

        // add remaining unparsed input length to the string
        getParsingFlags(config).charsLeftOver = stringLength - totalParsedInputLength;
        if (string.length > 0) {
            getParsingFlags(config).unusedInput.push(string);
        }

        // clear _12h flag if hour is <= 12
        if (config._a[HOUR] <= 12 &&
            getParsingFlags(config).bigHour === true &&
            config._a[HOUR] > 0) {
            getParsingFlags(config).bigHour = undefined;
        }

        getParsingFlags(config).parsedDateParts = config._a.slice(0);
        getParsingFlags(config).meridiem = config._meridiem;
        // handle meridiem
        config._a[HOUR] = meridiemFixWrap(config._locale, config._a[HOUR], config._meridiem);

        configFromArray(config);
        checkOverflow(config);
    }


    function meridiemFixWrap (locale, hour, meridiem) {
        var isPm;

        if (meridiem == null) {
            // nothing to do
            return hour;
        }
        if (locale.meridiemHour != null) {
            return locale.meridiemHour(hour, meridiem);
        } else if (locale.isPM != null) {
            // Fallback
            isPm = locale.isPM(meridiem);
            if (isPm && hour < 12) {
                hour += 12;
            }
            if (!isPm && hour === 12) {
                hour = 0;
            }
            return hour;
        } else {
            // this is not supposed to happen
            return hour;
        }
    }

    // date from string and array of format strings
    function configFromStringAndArray(config) {
        var tempConfig,
            bestMoment,

            scoreToBeat,
            i,
            currentScore;

        if (config._f.length === 0) {
            getParsingFlags(config).invalidFormat = true;
            config._d = new Date(NaN);
            return;
        }

        for (i = 0; i < config._f.length; i++) {
            currentScore = 0;
            tempConfig = copyConfig({}, config);
            if (config._useUTC != null) {
                tempConfig._useUTC = config._useUTC;
            }
            tempConfig._f = config._f[i];
            configFromStringAndFormat(tempConfig);

            if (!isValid(tempConfig)) {
                continue;
            }

            // if there is any input that was not parsed add a penalty for that format
            currentScore += getParsingFlags(tempConfig).charsLeftOver;

            //or tokens
            currentScore += getParsingFlags(tempConfig).unusedTokens.length * 10;

            getParsingFlags(tempConfig).score = currentScore;

            if (scoreToBeat == null || currentScore < scoreToBeat) {
                scoreToBeat = currentScore;
                bestMoment = tempConfig;
            }
        }

        extend(config, bestMoment || tempConfig);
    }

    function configFromObject(config) {
        if (config._d) {
            return;
        }

        var i = normalizeObjectUnits(config._i);
        config._a = map([i.year, i.month, i.day || i.date, i.hour, i.minute, i.second, i.millisecond], function (obj) {
            return obj && parseInt(obj, 10);
        });

        configFromArray(config);
    }

    function createFromConfig (config) {
        var res = new Moment(checkOverflow(prepareConfig(config)));
        if (res._nextDay) {
            // Adding is smart enough around DST
            res.add(1, 'd');
            res._nextDay = undefined;
        }

        return res;
    }

    function prepareConfig (config) {
        var input = config._i,
            format = config._f;

        config._locale = config._locale || getLocale(config._l);

        if (input === null || (format === undefined && input === '')) {
            return createInvalid({nullInput: true});
        }

        if (typeof input === 'string') {
            config._i = input = config._locale.preparse(input);
        }

        if (isMoment(input)) {
            return new Moment(checkOverflow(input));
        } else if (isDate(input)) {
            config._d = input;
        } else if (isArray(format)) {
            configFromStringAndArray(config);
        } else if (format) {
            configFromStringAndFormat(config);
        }  else {
            configFromInput(config);
        }

        if (!isValid(config)) {
            config._d = null;
        }

        return config;
    }

    function configFromInput(config) {
        var input = config._i;
        if (isUndefined(input)) {
            config._d = new Date(hooks.now());
        } else if (isDate(input)) {
            config._d = new Date(input.valueOf());
        } else if (typeof input === 'string') {
            configFromString(config);
        } else if (isArray(input)) {
            config._a = map(input.slice(0), function (obj) {
                return parseInt(obj, 10);
            });
            configFromArray(config);
        } else if (isObject(input)) {
            configFromObject(config);
        } else if (isNumber(input)) {
            // from milliseconds
            config._d = new Date(input);
        } else {
            hooks.createFromInputFallback(config);
        }
    }

    function createLocalOrUTC (input, format, locale, strict, isUTC) {
        var c = {};

        if (locale === true || locale === false) {
            strict = locale;
            locale = undefined;
        }

        if ((isObject(input) && isObjectEmpty(input)) ||
                (isArray(input) && input.length === 0)) {
            input = undefined;
        }
        // object construction must be done this way.
        // https://github.com/moment/moment/issues/1423
        c._isAMomentObject = true;
        c._useUTC = c._isUTC = isUTC;
        c._l = locale;
        c._i = input;
        c._f = format;
        c._strict = strict;

        return createFromConfig(c);
    }

    function createLocal (input, format, locale, strict) {
        return createLocalOrUTC(input, format, locale, strict, false);
    }

    var prototypeMin = deprecate(
        'moment().min is deprecated, use moment.max instead. http://momentjs.com/guides/#/warnings/min-max/',
        function () {
            var other = createLocal.apply(null, arguments);
            if (this.isValid() && other.isValid()) {
                return other < this ? this : other;
            } else {
                return createInvalid();
            }
        }
    );

    var prototypeMax = deprecate(
        'moment().max is deprecated, use moment.min instead. http://momentjs.com/guides/#/warnings/min-max/',
        function () {
            var other = createLocal.apply(null, arguments);
            if (this.isValid() && other.isValid()) {
                return other > this ? this : other;
            } else {
                return createInvalid();
            }
        }
    );

    // Pick a moment m from moments so that m[fn](other) is true for all
    // other. This relies on the function fn to be transitive.
    //
    // moments should either be an array of moment objects or an array, whose
    // first element is an array of moment objects.
    function pickBy(fn, moments) {
        var res, i;
        if (moments.length === 1 && isArray(moments[0])) {
            moments = moments[0];
        }
        if (!moments.length) {
            return createLocal();
        }
        res = moments[0];
        for (i = 1; i < moments.length; ++i) {
            if (!moments[i].isValid() || moments[i][fn](res)) {
                res = moments[i];
            }
        }
        return res;
    }

    // TODO: Use [].sort instead?
    function min () {
        var args = [].slice.call(arguments, 0);

        return pickBy('isBefore', args);
    }

    function max () {
        var args = [].slice.call(arguments, 0);

        return pickBy('isAfter', args);
    }

    var now = function () {
        return Date.now ? Date.now() : +(new Date());
    };

    var ordering = ['year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond'];

    function isDurationValid(m) {
        for (var key in m) {
            if (!(indexOf.call(ordering, key) !== -1 && (m[key] == null || !isNaN(m[key])))) {
                return false;
            }
        }

        var unitHasDecimal = false;
        for (var i = 0; i < ordering.length; ++i) {
            if (m[ordering[i]]) {
                if (unitHasDecimal) {
                    return false; // only allow non-integers for smallest unit
                }
                if (parseFloat(m[ordering[i]]) !== toInt(m[ordering[i]])) {
                    unitHasDecimal = true;
                }
            }
        }

        return true;
    }

    function isValid$1() {
        return this._isValid;
    }

    function createInvalid$1() {
        return createDuration(NaN);
    }

    function Duration (duration) {
        var normalizedInput = normalizeObjectUnits(duration),
            years = normalizedInput.year || 0,
            quarters = normalizedInput.quarter || 0,
            months = normalizedInput.month || 0,
            weeks = normalizedInput.week || normalizedInput.isoWeek || 0,
            days = normalizedInput.day || 0,
            hours = normalizedInput.hour || 0,
            minutes = normalizedInput.minute || 0,
            seconds = normalizedInput.second || 0,
            milliseconds = normalizedInput.millisecond || 0;

        this._isValid = isDurationValid(normalizedInput);

        // representation for dateAddRemove
        this._milliseconds = +milliseconds +
            seconds * 1e3 + // 1000
            minutes * 6e4 + // 1000 * 60
            hours * 1000 * 60 * 60; //using 1000 * 60 * 60 instead of 36e5 to avoid floating point rounding errors https://github.com/moment/moment/issues/2978
        // Because of dateAddRemove treats 24 hours as different from a
        // day when working around DST, we need to store them separately
        this._days = +days +
            weeks * 7;
        // It is impossible to translate months into days without knowing
        // which months you are are talking about, so we have to store
        // it separately.
        this._months = +months +
            quarters * 3 +
            years * 12;

        this._data = {};

        this._locale = getLocale();

        this._bubble();
    }

    function isDuration (obj) {
        return obj instanceof Duration;
    }

    function absRound (number) {
        if (number < 0) {
            return Math.round(-1 * number) * -1;
        } else {
            return Math.round(number);
        }
    }

    // FORMATTING

    function offset (token, separator) {
        addFormatToken(token, 0, 0, function () {
            var offset = this.utcOffset();
            var sign = '+';
            if (offset < 0) {
                offset = -offset;
                sign = '-';
            }
            return sign + zeroFill(~~(offset / 60), 2) + separator + zeroFill(~~(offset) % 60, 2);
        });
    }

    offset('Z', ':');
    offset('ZZ', '');

    // PARSING

    addRegexToken('Z',  matchShortOffset);
    addRegexToken('ZZ', matchShortOffset);
    addParseToken(['Z', 'ZZ'], function (input, array, config) {
        config._useUTC = true;
        config._tzm = offsetFromString(matchShortOffset, input);
    });

    // HELPERS

    // timezone chunker
    // '+10:00' > ['10',  '00']
    // '-1530'  > ['-15', '30']
    var chunkOffset = /([\+\-]|\d\d)/gi;

    function offsetFromString(matcher, string) {
        var matches = (string || '').match(matcher);

        if (matches === null) {
            return null;
        }

        var chunk   = matches[matches.length - 1] || [];
        var parts   = (chunk + '').match(chunkOffset) || ['-', 0, 0];
        var minutes = +(parts[1] * 60) + toInt(parts[2]);

        return minutes === 0 ?
          0 :
          parts[0] === '+' ? minutes : -minutes;
    }

    // Return a moment from input, that is local/utc/zone equivalent to model.
    function cloneWithOffset(input, model) {
        var res, diff;
        if (model._isUTC) {
            res = model.clone();
            diff = (isMoment(input) || isDate(input) ? input.valueOf() : createLocal(input).valueOf()) - res.valueOf();
            // Use low-level api, because this fn is low-level api.
            res._d.setTime(res._d.valueOf() + diff);
            hooks.updateOffset(res, false);
            return res;
        } else {
            return createLocal(input).local();
        }
    }

    function getDateOffset (m) {
        // On Firefox.24 Date#getTimezoneOffset returns a floating point.
        // https://github.com/moment/moment/pull/1871
        return -Math.round(m._d.getTimezoneOffset() / 15) * 15;
    }

    // HOOKS

    // This function will be called whenever a moment is mutated.
    // It is intended to keep the offset in sync with the timezone.
    hooks.updateOffset = function () {};

    // MOMENTS

    // keepLocalTime = true means only change the timezone, without
    // affecting the local hour. So 5:31:26 +0300 --[utcOffset(2, true)]-->
    // 5:31:26 +0200 It is possible that 5:31:26 doesn't exist with offset
    // +0200, so we adjust the time as needed, to be valid.
    //
    // Keeping the time actually adds/subtracts (one hour)
    // from the actual represented time. That is why we call updateOffset
    // a second time. In case it wants us to change the offset again
    // _changeInProgress == true case, then we have to adjust, because
    // there is no such time in the given timezone.
    function getSetOffset (input, keepLocalTime, keepMinutes) {
        var offset = this._offset || 0,
            localAdjust;
        if (!this.isValid()) {
            return input != null ? this : NaN;
        }
        if (input != null) {
            if (typeof input === 'string') {
                input = offsetFromString(matchShortOffset, input);
                if (input === null) {
                    return this;
                }
            } else if (Math.abs(input) < 16 && !keepMinutes) {
                input = input * 60;
            }
            if (!this._isUTC && keepLocalTime) {
                localAdjust = getDateOffset(this);
            }
            this._offset = input;
            this._isUTC = true;
            if (localAdjust != null) {
                this.add(localAdjust, 'm');
            }
            if (offset !== input) {
                if (!keepLocalTime || this._changeInProgress) {
                    addSubtract(this, createDuration(input - offset, 'm'), 1, false);
                } else if (!this._changeInProgress) {
                    this._changeInProgress = true;
                    hooks.updateOffset(this, true);
                    this._changeInProgress = null;
                }
            }
            return this;
        } else {
            return this._isUTC ? offset : getDateOffset(this);
        }
    }

    function getSetZone (input, keepLocalTime) {
        if (input != null) {
            if (typeof input !== 'string') {
                input = -input;
            }

            this.utcOffset(input, keepLocalTime);

            return this;
        } else {
            return -this.utcOffset();
        }
    }

    function setOffsetToUTC (keepLocalTime) {
        return this.utcOffset(0, keepLocalTime);
    }

    function setOffsetToLocal (keepLocalTime) {
        if (this._isUTC) {
            this.utcOffset(0, keepLocalTime);
            this._isUTC = false;

            if (keepLocalTime) {
                this.subtract(getDateOffset(this), 'm');
            }
        }
        return this;
    }

    function setOffsetToParsedOffset () {
        if (this._tzm != null) {
            this.utcOffset(this._tzm, false, true);
        } else if (typeof this._i === 'string') {
            var tZone = offsetFromString(matchOffset, this._i);
            if (tZone != null) {
                this.utcOffset(tZone);
            }
            else {
                this.utcOffset(0, true);
            }
        }
        return this;
    }

    function hasAlignedHourOffset (input) {
        if (!this.isValid()) {
            return false;
        }
        input = input ? createLocal(input).utcOffset() : 0;

        return (this.utcOffset() - input) % 60 === 0;
    }

    function isDaylightSavingTime () {
        return (
            this.utcOffset() > this.clone().month(0).utcOffset() ||
            this.utcOffset() > this.clone().month(5).utcOffset()
        );
    }

    function isDaylightSavingTimeShifted () {
        if (!isUndefined(this._isDSTShifted)) {
            return this._isDSTShifted;
        }

        var c = {};

        copyConfig(c, this);
        c = prepareConfig(c);

        if (c._a) {
            var other = c._isUTC ? createUTC(c._a) : createLocal(c._a);
            this._isDSTShifted = this.isValid() &&
                compareArrays(c._a, other.toArray()) > 0;
        } else {
            this._isDSTShifted = false;
        }

        return this._isDSTShifted;
    }

    function isLocal () {
        return this.isValid() ? !this._isUTC : false;
    }

    function isUtcOffset () {
        return this.isValid() ? this._isUTC : false;
    }

    function isUtc () {
        return this.isValid() ? this._isUTC && this._offset === 0 : false;
    }

    // ASP.NET json date format regex
    var aspNetRegex = /^(\-|\+)?(?:(\d*)[. ])?(\d+)\:(\d+)(?:\:(\d+)(\.\d*)?)?$/;

    // from http://docs.closure-library.googlecode.com/git/closure_goog_date_date.js.source.html
    // somewhat more in line with 4.4.3.2 2004 spec, but allows decimal anywhere
    // and further modified to allow for strings containing both week and day
    var isoRegex = /^(-|\+)?P(?:([-+]?[0-9,.]*)Y)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)W)?(?:([-+]?[0-9,.]*)D)?(?:T(?:([-+]?[0-9,.]*)H)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)S)?)?$/;

    function createDuration (input, key) {
        var duration = input,
            // matching against regexp is expensive, do it on demand
            match = null,
            sign,
            ret,
            diffRes;

        if (isDuration(input)) {
            duration = {
                ms : input._milliseconds,
                d  : input._days,
                M  : input._months
            };
        } else if (isNumber(input)) {
            duration = {};
            if (key) {
                duration[key] = input;
            } else {
                duration.milliseconds = input;
            }
        } else if (!!(match = aspNetRegex.exec(input))) {
            sign = (match[1] === '-') ? -1 : 1;
            duration = {
                y  : 0,
                d  : toInt(match[DATE])                         * sign,
                h  : toInt(match[HOUR])                         * sign,
                m  : toInt(match[MINUTE])                       * sign,
                s  : toInt(match[SECOND])                       * sign,
                ms : toInt(absRound(match[MILLISECOND] * 1000)) * sign // the millisecond decimal point is included in the match
            };
        } else if (!!(match = isoRegex.exec(input))) {
            sign = (match[1] === '-') ? -1 : 1;
            duration = {
                y : parseIso(match[2], sign),
                M : parseIso(match[3], sign),
                w : parseIso(match[4], sign),
                d : parseIso(match[5], sign),
                h : parseIso(match[6], sign),
                m : parseIso(match[7], sign),
                s : parseIso(match[8], sign)
            };
        } else if (duration == null) {// checks for null or undefined
            duration = {};
        } else if (typeof duration === 'object' && ('from' in duration || 'to' in duration)) {
            diffRes = momentsDifference(createLocal(duration.from), createLocal(duration.to));

            duration = {};
            duration.ms = diffRes.milliseconds;
            duration.M = diffRes.months;
        }

        ret = new Duration(duration);

        if (isDuration(input) && hasOwnProp(input, '_locale')) {
            ret._locale = input._locale;
        }

        return ret;
    }

    createDuration.fn = Duration.prototype;
    createDuration.invalid = createInvalid$1;

    function parseIso (inp, sign) {
        // We'd normally use ~~inp for this, but unfortunately it also
        // converts floats to ints.
        // inp may be undefined, so careful calling replace on it.
        var res = inp && parseFloat(inp.replace(',', '.'));
        // apply sign while we're at it
        return (isNaN(res) ? 0 : res) * sign;
    }

    function positiveMomentsDifference(base, other) {
        var res = {};

        res.months = other.month() - base.month() +
            (other.year() - base.year()) * 12;
        if (base.clone().add(res.months, 'M').isAfter(other)) {
            --res.months;
        }

        res.milliseconds = +other - +(base.clone().add(res.months, 'M'));

        return res;
    }

    function momentsDifference(base, other) {
        var res;
        if (!(base.isValid() && other.isValid())) {
            return {milliseconds: 0, months: 0};
        }

        other = cloneWithOffset(other, base);
        if (base.isBefore(other)) {
            res = positiveMomentsDifference(base, other);
        } else {
            res = positiveMomentsDifference(other, base);
            res.milliseconds = -res.milliseconds;
            res.months = -res.months;
        }

        return res;
    }

    // TODO: remove 'name' arg after deprecation is removed
    function createAdder(direction, name) {
        return function (val, period) {
            var dur, tmp;
            //invert the arguments, but complain about it
            if (period !== null && !isNaN(+period)) {
                deprecateSimple(name, 'moment().' + name  + '(period, number) is deprecated. Please use moment().' + name + '(number, period). ' +
                'See http://momentjs.com/guides/#/warnings/add-inverted-param/ for more info.');
                tmp = val; val = period; period = tmp;
            }

            val = typeof val === 'string' ? +val : val;
            dur = createDuration(val, period);
            addSubtract(this, dur, direction);
            return this;
        };
    }

    function addSubtract (mom, duration, isAdding, updateOffset) {
        var milliseconds = duration._milliseconds,
            days = absRound(duration._days),
            months = absRound(duration._months);

        if (!mom.isValid()) {
            // No op
            return;
        }

        updateOffset = updateOffset == null ? true : updateOffset;

        if (months) {
            setMonth(mom, get(mom, 'Month') + months * isAdding);
        }
        if (days) {
            set$1(mom, 'Date', get(mom, 'Date') + days * isAdding);
        }
        if (milliseconds) {
            mom._d.setTime(mom._d.valueOf() + milliseconds * isAdding);
        }
        if (updateOffset) {
            hooks.updateOffset(mom, days || months);
        }
    }

    var add      = createAdder(1, 'add');
    var subtract = createAdder(-1, 'subtract');

    function getCalendarFormat(myMoment, now) {
        var diff = myMoment.diff(now, 'days', true);
        return diff < -6 ? 'sameElse' :
                diff < -1 ? 'lastWeek' :
                diff < 0 ? 'lastDay' :
                diff < 1 ? 'sameDay' :
                diff < 2 ? 'nextDay' :
                diff < 7 ? 'nextWeek' : 'sameElse';
    }

    function calendar$1 (time, formats) {
        // We want to compare the start of today, vs this.
        // Getting start-of-today depends on whether we're local/utc/offset or not.
        var now = time || createLocal(),
            sod = cloneWithOffset(now, this).startOf('day'),
            format = hooks.calendarFormat(this, sod) || 'sameElse';

        var output = formats && (isFunction(formats[format]) ? formats[format].call(this, now) : formats[format]);

        return this.format(output || this.localeData().calendar(format, this, createLocal(now)));
    }

    function clone () {
        return new Moment(this);
    }

    function isAfter (input, units) {
        var localInput = isMoment(input) ? input : createLocal(input);
        if (!(this.isValid() && localInput.isValid())) {
            return false;
        }
        units = normalizeUnits(units) || 'millisecond';
        if (units === 'millisecond') {
            return this.valueOf() > localInput.valueOf();
        } else {
            return localInput.valueOf() < this.clone().startOf(units).valueOf();
        }
    }

    function isBefore (input, units) {
        var localInput = isMoment(input) ? input : createLocal(input);
        if (!(this.isValid() && localInput.isValid())) {
            return false;
        }
        units = normalizeUnits(units) || 'millisecond';
        if (units === 'millisecond') {
            return this.valueOf() < localInput.valueOf();
        } else {
            return this.clone().endOf(units).valueOf() < localInput.valueOf();
        }
    }

    function isBetween (from, to, units, inclusivity) {
        var localFrom = isMoment(from) ? from : createLocal(from),
            localTo = isMoment(to) ? to : createLocal(to);
        if (!(this.isValid() && localFrom.isValid() && localTo.isValid())) {
            return false;
        }
        inclusivity = inclusivity || '()';
        return (inclusivity[0] === '(' ? this.isAfter(localFrom, units) : !this.isBefore(localFrom, units)) &&
            (inclusivity[1] === ')' ? this.isBefore(localTo, units) : !this.isAfter(localTo, units));
    }

    function isSame (input, units) {
        var localInput = isMoment(input) ? input : createLocal(input),
            inputMs;
        if (!(this.isValid() && localInput.isValid())) {
            return false;
        }
        units = normalizeUnits(units) || 'millisecond';
        if (units === 'millisecond') {
            return this.valueOf() === localInput.valueOf();
        } else {
            inputMs = localInput.valueOf();
            return this.clone().startOf(units).valueOf() <= inputMs && inputMs <= this.clone().endOf(units).valueOf();
        }
    }

    function isSameOrAfter (input, units) {
        return this.isSame(input, units) || this.isAfter(input, units);
    }

    function isSameOrBefore (input, units) {
        return this.isSame(input, units) || this.isBefore(input, units);
    }

    function diff (input, units, asFloat) {
        var that,
            zoneDelta,
            output;

        if (!this.isValid()) {
            return NaN;
        }

        that = cloneWithOffset(input, this);

        if (!that.isValid()) {
            return NaN;
        }

        zoneDelta = (that.utcOffset() - this.utcOffset()) * 6e4;

        units = normalizeUnits(units);

        switch (units) {
            case 'year': output = monthDiff(this, that) / 12; break;
            case 'month': output = monthDiff(this, that); break;
            case 'quarter': output = monthDiff(this, that) / 3; break;
            case 'second': output = (this - that) / 1e3; break; // 1000
            case 'minute': output = (this - that) / 6e4; break; // 1000 * 60
            case 'hour': output = (this - that) / 36e5; break; // 1000 * 60 * 60
            case 'day': output = (this - that - zoneDelta) / 864e5; break; // 1000 * 60 * 60 * 24, negate dst
            case 'week': output = (this - that - zoneDelta) / 6048e5; break; // 1000 * 60 * 60 * 24 * 7, negate dst
            default: output = this - that;
        }

        return asFloat ? output : absFloor(output);
    }

    function monthDiff (a, b) {
        // difference in months
        var wholeMonthDiff = ((b.year() - a.year()) * 12) + (b.month() - a.month()),
            // b is in (anchor - 1 month, anchor + 1 month)
            anchor = a.clone().add(wholeMonthDiff, 'months'),
            anchor2, adjust;

        if (b - anchor < 0) {
            anchor2 = a.clone().add(wholeMonthDiff - 1, 'months');
            // linear across the month
            adjust = (b - anchor) / (anchor - anchor2);
        } else {
            anchor2 = a.clone().add(wholeMonthDiff + 1, 'months');
            // linear across the month
            adjust = (b - anchor) / (anchor2 - anchor);
        }

        //check for negative zero, return zero if negative zero
        return -(wholeMonthDiff + adjust) || 0;
    }

    hooks.defaultFormat = 'YYYY-MM-DDTHH:mm:ssZ';
    hooks.defaultFormatUtc = 'YYYY-MM-DDTHH:mm:ss[Z]';

    function toString () {
        return this.clone().locale('en').format('ddd MMM DD YYYY HH:mm:ss [GMT]ZZ');
    }

    function toISOString(keepOffset) {
        if (!this.isValid()) {
            return null;
        }
        var utc = keepOffset !== true;
        var m = utc ? this.clone().utc() : this;
        if (m.year() < 0 || m.year() > 9999) {
            return formatMoment(m, utc ? 'YYYYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYYYY-MM-DD[T]HH:mm:ss.SSSZ');
        }
        if (isFunction(Date.prototype.toISOString)) {
            // native implementation is ~50x faster, use it when we can
            if (utc) {
                return this.toDate().toISOString();
            } else {
                return new Date(this.valueOf() + this.utcOffset() * 60 * 1000).toISOString().replace('Z', formatMoment(m, 'Z'));
            }
        }
        return formatMoment(m, utc ? 'YYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYY-MM-DD[T]HH:mm:ss.SSSZ');
    }

    /**
     * Return a human readable representation of a moment that can
     * also be evaluated to get a new moment which is the same
     *
     * @link https://nodejs.org/dist/latest/docs/api/util.html#util_custom_inspect_function_on_objects
     */
    function inspect () {
        if (!this.isValid()) {
            return 'moment.invalid(/* ' + this._i + ' */)';
        }
        var func = 'moment';
        var zone = '';
        if (!this.isLocal()) {
            func = this.utcOffset() === 0 ? 'moment.utc' : 'moment.parseZone';
            zone = 'Z';
        }
        var prefix = '[' + func + '("]';
        var year = (0 <= this.year() && this.year() <= 9999) ? 'YYYY' : 'YYYYYY';
        var datetime = '-MM-DD[T]HH:mm:ss.SSS';
        var suffix = zone + '[")]';

        return this.format(prefix + year + datetime + suffix);
    }

    function format (inputString) {
        if (!inputString) {
            inputString = this.isUtc() ? hooks.defaultFormatUtc : hooks.defaultFormat;
        }
        var output = formatMoment(this, inputString);
        return this.localeData().postformat(output);
    }

    function from (time, withoutSuffix) {
        if (this.isValid() &&
                ((isMoment(time) && time.isValid()) ||
                 createLocal(time).isValid())) {
            return createDuration({to: this, from: time}).locale(this.locale()).humanize(!withoutSuffix);
        } else {
            return this.localeData().invalidDate();
        }
    }

    function fromNow (withoutSuffix) {
        return this.from(createLocal(), withoutSuffix);
    }

    function to (time, withoutSuffix) {
        if (this.isValid() &&
                ((isMoment(time) && time.isValid()) ||
                 createLocal(time).isValid())) {
            return createDuration({from: this, to: time}).locale(this.locale()).humanize(!withoutSuffix);
        } else {
            return this.localeData().invalidDate();
        }
    }

    function toNow (withoutSuffix) {
        return this.to(createLocal(), withoutSuffix);
    }

    // If passed a locale key, it will set the locale for this
    // instance.  Otherwise, it will return the locale configuration
    // variables for this instance.
    function locale (key) {
        var newLocaleData;

        if (key === undefined) {
            return this._locale._abbr;
        } else {
            newLocaleData = getLocale(key);
            if (newLocaleData != null) {
                this._locale = newLocaleData;
            }
            return this;
        }
    }

    var lang = deprecate(
        'moment().lang() is deprecated. Instead, use moment().localeData() to get the language configuration. Use moment().locale() to change languages.',
        function (key) {
            if (key === undefined) {
                return this.localeData();
            } else {
                return this.locale(key);
            }
        }
    );

    function localeData () {
        return this._locale;
    }

    var MS_PER_SECOND = 1000;
    var MS_PER_MINUTE = 60 * MS_PER_SECOND;
    var MS_PER_HOUR = 60 * MS_PER_MINUTE;
    var MS_PER_400_YEARS = (365 * 400 + 97) * 24 * MS_PER_HOUR;

    // actual modulo - handles negative numbers (for dates before 1970):
    function mod$1(dividend, divisor) {
        return (dividend % divisor + divisor) % divisor;
    }

    function localStartOfDate(y, m, d) {
        // the date constructor remaps years 0-99 to 1900-1999
        if (y < 100 && y >= 0) {
            // preserve leap years using a full 400 year cycle, then reset
            return new Date(y + 400, m, d) - MS_PER_400_YEARS;
        } else {
            return new Date(y, m, d).valueOf();
        }
    }

    function utcStartOfDate(y, m, d) {
        // Date.UTC remaps years 0-99 to 1900-1999
        if (y < 100 && y >= 0) {
            // preserve leap years using a full 400 year cycle, then reset
            return Date.UTC(y + 400, m, d) - MS_PER_400_YEARS;
        } else {
            return Date.UTC(y, m, d);
        }
    }

    function startOf (units) {
        var time;
        units = normalizeUnits(units);
        if (units === undefined || units === 'millisecond' || !this.isValid()) {
            return this;
        }

        var startOfDate = this._isUTC ? utcStartOfDate : localStartOfDate;

        switch (units) {
            case 'year':
                time = startOfDate(this.year(), 0, 1);
                break;
            case 'quarter':
                time = startOfDate(this.year(), this.month() - this.month() % 3, 1);
                break;
            case 'month':
                time = startOfDate(this.year(), this.month(), 1);
                break;
            case 'week':
                time = startOfDate(this.year(), this.month(), this.date() - this.weekday());
                break;
            case 'isoWeek':
                time = startOfDate(this.year(), this.month(), this.date() - (this.isoWeekday() - 1));
                break;
            case 'day':
            case 'date':
                time = startOfDate(this.year(), this.month(), this.date());
                break;
            case 'hour':
                time = this._d.valueOf();
                time -= mod$1(time + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE), MS_PER_HOUR);
                break;
            case 'minute':
                time = this._d.valueOf();
                time -= mod$1(time, MS_PER_MINUTE);
                break;
            case 'second':
                time = this._d.valueOf();
                time -= mod$1(time, MS_PER_SECOND);
                break;
        }

        this._d.setTime(time);
        hooks.updateOffset(this, true);
        return this;
    }

    function endOf (units) {
        var time;
        units = normalizeUnits(units);
        if (units === undefined || units === 'millisecond' || !this.isValid()) {
            return this;
        }

        var startOfDate = this._isUTC ? utcStartOfDate : localStartOfDate;

        switch (units) {
            case 'year':
                time = startOfDate(this.year() + 1, 0, 1) - 1;
                break;
            case 'quarter':
                time = startOfDate(this.year(), this.month() - this.month() % 3 + 3, 1) - 1;
                break;
            case 'month':
                time = startOfDate(this.year(), this.month() + 1, 1) - 1;
                break;
            case 'week':
                time = startOfDate(this.year(), this.month(), this.date() - this.weekday() + 7) - 1;
                break;
            case 'isoWeek':
                time = startOfDate(this.year(), this.month(), this.date() - (this.isoWeekday() - 1) + 7) - 1;
                break;
            case 'day':
            case 'date':
                time = startOfDate(this.year(), this.month(), this.date() + 1) - 1;
                break;
            case 'hour':
                time = this._d.valueOf();
                time += MS_PER_HOUR - mod$1(time + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE), MS_PER_HOUR) - 1;
                break;
            case 'minute':
                time = this._d.valueOf();
                time += MS_PER_MINUTE - mod$1(time, MS_PER_MINUTE) - 1;
                break;
            case 'second':
                time = this._d.valueOf();
                time += MS_PER_SECOND - mod$1(time, MS_PER_SECOND) - 1;
                break;
        }

        this._d.setTime(time);
        hooks.updateOffset(this, true);
        return this;
    }

    function valueOf () {
        return this._d.valueOf() - ((this._offset || 0) * 60000);
    }

    function unix () {
        return Math.floor(this.valueOf() / 1000);
    }

    function toDate () {
        return new Date(this.valueOf());
    }

    function toArray () {
        var m = this;
        return [m.year(), m.month(), m.date(), m.hour(), m.minute(), m.second(), m.millisecond()];
    }

    function toObject () {
        var m = this;
        return {
            years: m.year(),
            months: m.month(),
            date: m.date(),
            hours: m.hours(),
            minutes: m.minutes(),
            seconds: m.seconds(),
            milliseconds: m.milliseconds()
        };
    }

    function toJSON () {
        // new Date(NaN).toJSON() === null
        return this.isValid() ? this.toISOString() : null;
    }

    function isValid$2 () {
        return isValid(this);
    }

    function parsingFlags () {
        return extend({}, getParsingFlags(this));
    }

    function invalidAt () {
        return getParsingFlags(this).overflow;
    }

    function creationData() {
        return {
            input: this._i,
            format: this._f,
            locale: this._locale,
            isUTC: this._isUTC,
            strict: this._strict
        };
    }

    // FORMATTING

    addFormatToken(0, ['gg', 2], 0, function () {
        return this.weekYear() % 100;
    });

    addFormatToken(0, ['GG', 2], 0, function () {
        return this.isoWeekYear() % 100;
    });

    function addWeekYearFormatToken (token, getter) {
        addFormatToken(0, [token, token.length], 0, getter);
    }

    addWeekYearFormatToken('gggg',     'weekYear');
    addWeekYearFormatToken('ggggg',    'weekYear');
    addWeekYearFormatToken('GGGG',  'isoWeekYear');
    addWeekYearFormatToken('GGGGG', 'isoWeekYear');

    // ALIASES

    addUnitAlias('weekYear', 'gg');
    addUnitAlias('isoWeekYear', 'GG');

    // PRIORITY

    addUnitPriority('weekYear', 1);
    addUnitPriority('isoWeekYear', 1);


    // PARSING

    addRegexToken('G',      matchSigned);
    addRegexToken('g',      matchSigned);
    addRegexToken('GG',     match1to2, match2);
    addRegexToken('gg',     match1to2, match2);
    addRegexToken('GGGG',   match1to4, match4);
    addRegexToken('gggg',   match1to4, match4);
    addRegexToken('GGGGG',  match1to6, match6);
    addRegexToken('ggggg',  match1to6, match6);

    addWeekParseToken(['gggg', 'ggggg', 'GGGG', 'GGGGG'], function (input, week, config, token) {
        week[token.substr(0, 2)] = toInt(input);
    });

    addWeekParseToken(['gg', 'GG'], function (input, week, config, token) {
        week[token] = hooks.parseTwoDigitYear(input);
    });

    // MOMENTS

    function getSetWeekYear (input) {
        return getSetWeekYearHelper.call(this,
                input,
                this.week(),
                this.weekday(),
                this.localeData()._week.dow,
                this.localeData()._week.doy);
    }

    function getSetISOWeekYear (input) {
        return getSetWeekYearHelper.call(this,
                input, this.isoWeek(), this.isoWeekday(), 1, 4);
    }

    function getISOWeeksInYear () {
        return weeksInYear(this.year(), 1, 4);
    }

    function getWeeksInYear () {
        var weekInfo = this.localeData()._week;
        return weeksInYear(this.year(), weekInfo.dow, weekInfo.doy);
    }

    function getSetWeekYearHelper(input, week, weekday, dow, doy) {
        var weeksTarget;
        if (input == null) {
            return weekOfYear(this, dow, doy).year;
        } else {
            weeksTarget = weeksInYear(input, dow, doy);
            if (week > weeksTarget) {
                week = weeksTarget;
            }
            return setWeekAll.call(this, input, week, weekday, dow, doy);
        }
    }

    function setWeekAll(weekYear, week, weekday, dow, doy) {
        var dayOfYearData = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy),
            date = createUTCDate(dayOfYearData.year, 0, dayOfYearData.dayOfYear);

        this.year(date.getUTCFullYear());
        this.month(date.getUTCMonth());
        this.date(date.getUTCDate());
        return this;
    }

    // FORMATTING

    addFormatToken('Q', 0, 'Qo', 'quarter');

    // ALIASES

    addUnitAlias('quarter', 'Q');

    // PRIORITY

    addUnitPriority('quarter', 7);

    // PARSING

    addRegexToken('Q', match1);
    addParseToken('Q', function (input, array) {
        array[MONTH] = (toInt(input) - 1) * 3;
    });

    // MOMENTS

    function getSetQuarter (input) {
        return input == null ? Math.ceil((this.month() + 1) / 3) : this.month((input - 1) * 3 + this.month() % 3);
    }

    // FORMATTING

    addFormatToken('D', ['DD', 2], 'Do', 'date');

    // ALIASES

    addUnitAlias('date', 'D');

    // PRIORITY
    addUnitPriority('date', 9);

    // PARSING

    addRegexToken('D',  match1to2);
    addRegexToken('DD', match1to2, match2);
    addRegexToken('Do', function (isStrict, locale) {
        // TODO: Remove "ordinalParse" fallback in next major release.
        return isStrict ?
          (locale._dayOfMonthOrdinalParse || locale._ordinalParse) :
          locale._dayOfMonthOrdinalParseLenient;
    });

    addParseToken(['D', 'DD'], DATE);
    addParseToken('Do', function (input, array) {
        array[DATE] = toInt(input.match(match1to2)[0]);
    });

    // MOMENTS

    var getSetDayOfMonth = makeGetSet('Date', true);

    // FORMATTING

    addFormatToken('DDD', ['DDDD', 3], 'DDDo', 'dayOfYear');

    // ALIASES

    addUnitAlias('dayOfYear', 'DDD');

    // PRIORITY
    addUnitPriority('dayOfYear', 4);

    // PARSING

    addRegexToken('DDD',  match1to3);
    addRegexToken('DDDD', match3);
    addParseToken(['DDD', 'DDDD'], function (input, array, config) {
        config._dayOfYear = toInt(input);
    });

    // HELPERS

    // MOMENTS

    function getSetDayOfYear (input) {
        var dayOfYear = Math.round((this.clone().startOf('day') - this.clone().startOf('year')) / 864e5) + 1;
        return input == null ? dayOfYear : this.add((input - dayOfYear), 'd');
    }

    // FORMATTING

    addFormatToken('m', ['mm', 2], 0, 'minute');

    // ALIASES

    addUnitAlias('minute', 'm');

    // PRIORITY

    addUnitPriority('minute', 14);

    // PARSING

    addRegexToken('m',  match1to2);
    addRegexToken('mm', match1to2, match2);
    addParseToken(['m', 'mm'], MINUTE);

    // MOMENTS

    var getSetMinute = makeGetSet('Minutes', false);

    // FORMATTING

    addFormatToken('s', ['ss', 2], 0, 'second');

    // ALIASES

    addUnitAlias('second', 's');

    // PRIORITY

    addUnitPriority('second', 15);

    // PARSING

    addRegexToken('s',  match1to2);
    addRegexToken('ss', match1to2, match2);
    addParseToken(['s', 'ss'], SECOND);

    // MOMENTS

    var getSetSecond = makeGetSet('Seconds', false);

    // FORMATTING

    addFormatToken('S', 0, 0, function () {
        return ~~(this.millisecond() / 100);
    });

    addFormatToken(0, ['SS', 2], 0, function () {
        return ~~(this.millisecond() / 10);
    });

    addFormatToken(0, ['SSS', 3], 0, 'millisecond');
    addFormatToken(0, ['SSSS', 4], 0, function () {
        return this.millisecond() * 10;
    });
    addFormatToken(0, ['SSSSS', 5], 0, function () {
        return this.millisecond() * 100;
    });
    addFormatToken(0, ['SSSSSS', 6], 0, function () {
        return this.millisecond() * 1000;
    });
    addFormatToken(0, ['SSSSSSS', 7], 0, function () {
        return this.millisecond() * 10000;
    });
    addFormatToken(0, ['SSSSSSSS', 8], 0, function () {
        return this.millisecond() * 100000;
    });
    addFormatToken(0, ['SSSSSSSSS', 9], 0, function () {
        return this.millisecond() * 1000000;
    });


    // ALIASES

    addUnitAlias('millisecond', 'ms');

    // PRIORITY

    addUnitPriority('millisecond', 16);

    // PARSING

    addRegexToken('S',    match1to3, match1);
    addRegexToken('SS',   match1to3, match2);
    addRegexToken('SSS',  match1to3, match3);

    var token;
    for (token = 'SSSS'; token.length <= 9; token += 'S') {
        addRegexToken(token, matchUnsigned);
    }

    function parseMs(input, array) {
        array[MILLISECOND] = toInt(('0.' + input) * 1000);
    }

    for (token = 'S'; token.length <= 9; token += 'S') {
        addParseToken(token, parseMs);
    }
    // MOMENTS

    var getSetMillisecond = makeGetSet('Milliseconds', false);

    // FORMATTING

    addFormatToken('z',  0, 0, 'zoneAbbr');
    addFormatToken('zz', 0, 0, 'zoneName');

    // MOMENTS

    function getZoneAbbr () {
        return this._isUTC ? 'UTC' : '';
    }

    function getZoneName () {
        return this._isUTC ? 'Coordinated Universal Time' : '';
    }

    var proto = Moment.prototype;

    proto.add               = add;
    proto.calendar          = calendar$1;
    proto.clone             = clone;
    proto.diff              = diff;
    proto.endOf             = endOf;
    proto.format            = format;
    proto.from              = from;
    proto.fromNow           = fromNow;
    proto.to                = to;
    proto.toNow             = toNow;
    proto.get               = stringGet;
    proto.invalidAt         = invalidAt;
    proto.isAfter           = isAfter;
    proto.isBefore          = isBefore;
    proto.isBetween         = isBetween;
    proto.isSame            = isSame;
    proto.isSameOrAfter     = isSameOrAfter;
    proto.isSameOrBefore    = isSameOrBefore;
    proto.isValid           = isValid$2;
    proto.lang              = lang;
    proto.locale            = locale;
    proto.localeData        = localeData;
    proto.max               = prototypeMax;
    proto.min               = prototypeMin;
    proto.parsingFlags      = parsingFlags;
    proto.set               = stringSet;
    proto.startOf           = startOf;
    proto.subtract          = subtract;
    proto.toArray           = toArray;
    proto.toObject          = toObject;
    proto.toDate            = toDate;
    proto.toISOString       = toISOString;
    proto.inspect           = inspect;
    proto.toJSON            = toJSON;
    proto.toString          = toString;
    proto.unix              = unix;
    proto.valueOf           = valueOf;
    proto.creationData      = creationData;
    proto.year       = getSetYear;
    proto.isLeapYear = getIsLeapYear;
    proto.weekYear    = getSetWeekYear;
    proto.isoWeekYear = getSetISOWeekYear;
    proto.quarter = proto.quarters = getSetQuarter;
    proto.month       = getSetMonth;
    proto.daysInMonth = getDaysInMonth;
    proto.week           = proto.weeks        = getSetWeek;
    proto.isoWeek        = proto.isoWeeks     = getSetISOWeek;
    proto.weeksInYear    = getWeeksInYear;
    proto.isoWeeksInYear = getISOWeeksInYear;
    proto.date       = getSetDayOfMonth;
    proto.day        = proto.days             = getSetDayOfWeek;
    proto.weekday    = getSetLocaleDayOfWeek;
    proto.isoWeekday = getSetISODayOfWeek;
    proto.dayOfYear  = getSetDayOfYear;
    proto.hour = proto.hours = getSetHour;
    proto.minute = proto.minutes = getSetMinute;
    proto.second = proto.seconds = getSetSecond;
    proto.millisecond = proto.milliseconds = getSetMillisecond;
    proto.utcOffset            = getSetOffset;
    proto.utc                  = setOffsetToUTC;
    proto.local                = setOffsetToLocal;
    proto.parseZone            = setOffsetToParsedOffset;
    proto.hasAlignedHourOffset = hasAlignedHourOffset;
    proto.isDST                = isDaylightSavingTime;
    proto.isLocal              = isLocal;
    proto.isUtcOffset          = isUtcOffset;
    proto.isUtc                = isUtc;
    proto.isUTC                = isUtc;
    proto.zoneAbbr = getZoneAbbr;
    proto.zoneName = getZoneName;
    proto.dates  = deprecate('dates accessor is deprecated. Use date instead.', getSetDayOfMonth);
    proto.months = deprecate('months accessor is deprecated. Use month instead', getSetMonth);
    proto.years  = deprecate('years accessor is deprecated. Use year instead', getSetYear);
    proto.zone   = deprecate('moment().zone is deprecated, use moment().utcOffset instead. http://momentjs.com/guides/#/warnings/zone/', getSetZone);
    proto.isDSTShifted = deprecate('isDSTShifted is deprecated. See http://momentjs.com/guides/#/warnings/dst-shifted/ for more information', isDaylightSavingTimeShifted);

    function createUnix (input) {
        return createLocal(input * 1000);
    }

    function createInZone () {
        return createLocal.apply(null, arguments).parseZone();
    }

    function preParsePostFormat (string) {
        return string;
    }

    var proto$1 = Locale.prototype;

    proto$1.calendar        = calendar;
    proto$1.longDateFormat  = longDateFormat;
    proto$1.invalidDate     = invalidDate;
    proto$1.ordinal         = ordinal;
    proto$1.preparse        = preParsePostFormat;
    proto$1.postformat      = preParsePostFormat;
    proto$1.relativeTime    = relativeTime;
    proto$1.pastFuture      = pastFuture;
    proto$1.set             = set;

    proto$1.months            =        localeMonths;
    proto$1.monthsShort       =        localeMonthsShort;
    proto$1.monthsParse       =        localeMonthsParse;
    proto$1.monthsRegex       = monthsRegex;
    proto$1.monthsShortRegex  = monthsShortRegex;
    proto$1.week = localeWeek;
    proto$1.firstDayOfYear = localeFirstDayOfYear;
    proto$1.firstDayOfWeek = localeFirstDayOfWeek;

    proto$1.weekdays       =        localeWeekdays;
    proto$1.weekdaysMin    =        localeWeekdaysMin;
    proto$1.weekdaysShort  =        localeWeekdaysShort;
    proto$1.weekdaysParse  =        localeWeekdaysParse;

    proto$1.weekdaysRegex       =        weekdaysRegex;
    proto$1.weekdaysShortRegex  =        weekdaysShortRegex;
    proto$1.weekdaysMinRegex    =        weekdaysMinRegex;

    proto$1.isPM = localeIsPM;
    proto$1.meridiem = localeMeridiem;

    function get$1 (format, index, field, setter) {
        var locale = getLocale();
        var utc = createUTC().set(setter, index);
        return locale[field](utc, format);
    }

    function listMonthsImpl (format, index, field) {
        if (isNumber(format)) {
            index = format;
            format = undefined;
        }

        format = format || '';

        if (index != null) {
            return get$1(format, index, field, 'month');
        }

        var i;
        var out = [];
        for (i = 0; i < 12; i++) {
            out[i] = get$1(format, i, field, 'month');
        }
        return out;
    }

    // ()
    // (5)
    // (fmt, 5)
    // (fmt)
    // (true)
    // (true, 5)
    // (true, fmt, 5)
    // (true, fmt)
    function listWeekdaysImpl (localeSorted, format, index, field) {
        if (typeof localeSorted === 'boolean') {
            if (isNumber(format)) {
                index = format;
                format = undefined;
            }

            format = format || '';
        } else {
            format = localeSorted;
            index = format;
            localeSorted = false;

            if (isNumber(format)) {
                index = format;
                format = undefined;
            }

            format = format || '';
        }

        var locale = getLocale(),
            shift = localeSorted ? locale._week.dow : 0;

        if (index != null) {
            return get$1(format, (index + shift) % 7, field, 'day');
        }

        var i;
        var out = [];
        for (i = 0; i < 7; i++) {
            out[i] = get$1(format, (i + shift) % 7, field, 'day');
        }
        return out;
    }

    function listMonths (format, index) {
        return listMonthsImpl(format, index, 'months');
    }

    function listMonthsShort (format, index) {
        return listMonthsImpl(format, index, 'monthsShort');
    }

    function listWeekdays (localeSorted, format, index) {
        return listWeekdaysImpl(localeSorted, format, index, 'weekdays');
    }

    function listWeekdaysShort (localeSorted, format, index) {
        return listWeekdaysImpl(localeSorted, format, index, 'weekdaysShort');
    }

    function listWeekdaysMin (localeSorted, format, index) {
        return listWeekdaysImpl(localeSorted, format, index, 'weekdaysMin');
    }

    getSetGlobalLocale('en', {
        dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (toInt(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        }
    });

    // Side effect imports

    hooks.lang = deprecate('moment.lang is deprecated. Use moment.locale instead.', getSetGlobalLocale);
    hooks.langData = deprecate('moment.langData is deprecated. Use moment.localeData instead.', getLocale);

    var mathAbs = Math.abs;

    function abs () {
        var data           = this._data;

        this._milliseconds = mathAbs(this._milliseconds);
        this._days         = mathAbs(this._days);
        this._months       = mathAbs(this._months);

        data.milliseconds  = mathAbs(data.milliseconds);
        data.seconds       = mathAbs(data.seconds);
        data.minutes       = mathAbs(data.minutes);
        data.hours         = mathAbs(data.hours);
        data.months        = mathAbs(data.months);
        data.years         = mathAbs(data.years);

        return this;
    }

    function addSubtract$1 (duration, input, value, direction) {
        var other = createDuration(input, value);

        duration._milliseconds += direction * other._milliseconds;
        duration._days         += direction * other._days;
        duration._months       += direction * other._months;

        return duration._bubble();
    }

    // supports only 2.0-style add(1, 's') or add(duration)
    function add$1 (input, value) {
        return addSubtract$1(this, input, value, 1);
    }

    // supports only 2.0-style subtract(1, 's') or subtract(duration)
    function subtract$1 (input, value) {
        return addSubtract$1(this, input, value, -1);
    }

    function absCeil (number) {
        if (number < 0) {
            return Math.floor(number);
        } else {
            return Math.ceil(number);
        }
    }

    function bubble () {
        var milliseconds = this._milliseconds;
        var days         = this._days;
        var months       = this._months;
        var data         = this._data;
        var seconds, minutes, hours, years, monthsFromDays;

        // if we have a mix of positive and negative values, bubble down first
        // check: https://github.com/moment/moment/issues/2166
        if (!((milliseconds >= 0 && days >= 0 && months >= 0) ||
                (milliseconds <= 0 && days <= 0 && months <= 0))) {
            milliseconds += absCeil(monthsToDays(months) + days) * 864e5;
            days = 0;
            months = 0;
        }

        // The following code bubbles up values, see the tests for
        // examples of what that means.
        data.milliseconds = milliseconds % 1000;

        seconds           = absFloor(milliseconds / 1000);
        data.seconds      = seconds % 60;

        minutes           = absFloor(seconds / 60);
        data.minutes      = minutes % 60;

        hours             = absFloor(minutes / 60);
        data.hours        = hours % 24;

        days += absFloor(hours / 24);

        // convert days to months
        monthsFromDays = absFloor(daysToMonths(days));
        months += monthsFromDays;
        days -= absCeil(monthsToDays(monthsFromDays));

        // 12 months -> 1 year
        years = absFloor(months / 12);
        months %= 12;

        data.days   = days;
        data.months = months;
        data.years  = years;

        return this;
    }

    function daysToMonths (days) {
        // 400 years have 146097 days (taking into account leap year rules)
        // 400 years have 12 months === 4800
        return days * 4800 / 146097;
    }

    function monthsToDays (months) {
        // the reverse of daysToMonths
        return months * 146097 / 4800;
    }

    function as (units) {
        if (!this.isValid()) {
            return NaN;
        }
        var days;
        var months;
        var milliseconds = this._milliseconds;

        units = normalizeUnits(units);

        if (units === 'month' || units === 'quarter' || units === 'year') {
            days = this._days + milliseconds / 864e5;
            months = this._months + daysToMonths(days);
            switch (units) {
                case 'month':   return months;
                case 'quarter': return months / 3;
                case 'year':    return months / 12;
            }
        } else {
            // handle milliseconds separately because of floating point math errors (issue #1867)
            days = this._days + Math.round(monthsToDays(this._months));
            switch (units) {
                case 'week'   : return days / 7     + milliseconds / 6048e5;
                case 'day'    : return days         + milliseconds / 864e5;
                case 'hour'   : return days * 24    + milliseconds / 36e5;
                case 'minute' : return days * 1440  + milliseconds / 6e4;
                case 'second' : return days * 86400 + milliseconds / 1000;
                // Math.floor prevents floating point math errors here
                case 'millisecond': return Math.floor(days * 864e5) + milliseconds;
                default: throw new Error('Unknown unit ' + units);
            }
        }
    }

    // TODO: Use this.as('ms')?
    function valueOf$1 () {
        if (!this.isValid()) {
            return NaN;
        }
        return (
            this._milliseconds +
            this._days * 864e5 +
            (this._months % 12) * 2592e6 +
            toInt(this._months / 12) * 31536e6
        );
    }

    function makeAs (alias) {
        return function () {
            return this.as(alias);
        };
    }

    var asMilliseconds = makeAs('ms');
    var asSeconds      = makeAs('s');
    var asMinutes      = makeAs('m');
    var asHours        = makeAs('h');
    var asDays         = makeAs('d');
    var asWeeks        = makeAs('w');
    var asMonths       = makeAs('M');
    var asQuarters     = makeAs('Q');
    var asYears        = makeAs('y');

    function clone$1 () {
        return createDuration(this);
    }

    function get$2 (units) {
        units = normalizeUnits(units);
        return this.isValid() ? this[units + 's']() : NaN;
    }

    function makeGetter(name) {
        return function () {
            return this.isValid() ? this._data[name] : NaN;
        };
    }

    var milliseconds = makeGetter('milliseconds');
    var seconds      = makeGetter('seconds');
    var minutes      = makeGetter('minutes');
    var hours        = makeGetter('hours');
    var days         = makeGetter('days');
    var months       = makeGetter('months');
    var years        = makeGetter('years');

    function weeks () {
        return absFloor(this.days() / 7);
    }

    var round = Math.round;
    var thresholds = {
        ss: 44,         // a few seconds to seconds
        s : 45,         // seconds to minute
        m : 45,         // minutes to hour
        h : 22,         // hours to day
        d : 26,         // days to month
        M : 11          // months to year
    };

    // helper function for moment.fn.from, moment.fn.fromNow, and moment.duration.fn.humanize
    function substituteTimeAgo(string, number, withoutSuffix, isFuture, locale) {
        return locale.relativeTime(number || 1, !!withoutSuffix, string, isFuture);
    }

    function relativeTime$1 (posNegDuration, withoutSuffix, locale) {
        var duration = createDuration(posNegDuration).abs();
        var seconds  = round(duration.as('s'));
        var minutes  = round(duration.as('m'));
        var hours    = round(duration.as('h'));
        var days     = round(duration.as('d'));
        var months   = round(duration.as('M'));
        var years    = round(duration.as('y'));

        var a = seconds <= thresholds.ss && ['s', seconds]  ||
                seconds < thresholds.s   && ['ss', seconds] ||
                minutes <= 1             && ['m']           ||
                minutes < thresholds.m   && ['mm', minutes] ||
                hours   <= 1             && ['h']           ||
                hours   < thresholds.h   && ['hh', hours]   ||
                days    <= 1             && ['d']           ||
                days    < thresholds.d   && ['dd', days]    ||
                months  <= 1             && ['M']           ||
                months  < thresholds.M   && ['MM', months]  ||
                years   <= 1             && ['y']           || ['yy', years];

        a[2] = withoutSuffix;
        a[3] = +posNegDuration > 0;
        a[4] = locale;
        return substituteTimeAgo.apply(null, a);
    }

    // This function allows you to set the rounding function for relative time strings
    function getSetRelativeTimeRounding (roundingFunction) {
        if (roundingFunction === undefined) {
            return round;
        }
        if (typeof(roundingFunction) === 'function') {
            round = roundingFunction;
            return true;
        }
        return false;
    }

    // This function allows you to set a threshold for relative time strings
    function getSetRelativeTimeThreshold (threshold, limit) {
        if (thresholds[threshold] === undefined) {
            return false;
        }
        if (limit === undefined) {
            return thresholds[threshold];
        }
        thresholds[threshold] = limit;
        if (threshold === 's') {
            thresholds.ss = limit - 1;
        }
        return true;
    }

    function humanize (withSuffix) {
        if (!this.isValid()) {
            return this.localeData().invalidDate();
        }

        var locale = this.localeData();
        var output = relativeTime$1(this, !withSuffix, locale);

        if (withSuffix) {
            output = locale.pastFuture(+this, output);
        }

        return locale.postformat(output);
    }

    var abs$1 = Math.abs;

    function sign(x) {
        return ((x > 0) - (x < 0)) || +x;
    }

    function toISOString$1() {
        // for ISO strings we do not use the normal bubbling rules:
        //  * milliseconds bubble up until they become hours
        //  * days do not bubble at all
        //  * months bubble up until they become years
        // This is because there is no context-free conversion between hours and days
        // (think of clock changes)
        // and also not between days and months (28-31 days per month)
        if (!this.isValid()) {
            return this.localeData().invalidDate();
        }

        var seconds = abs$1(this._milliseconds) / 1000;
        var days         = abs$1(this._days);
        var months       = abs$1(this._months);
        var minutes, hours, years;

        // 3600 seconds -> 60 minutes -> 1 hour
        minutes           = absFloor(seconds / 60);
        hours             = absFloor(minutes / 60);
        seconds %= 60;
        minutes %= 60;

        // 12 months -> 1 year
        years  = absFloor(months / 12);
        months %= 12;


        // inspired by https://github.com/dordille/moment-isoduration/blob/master/moment.isoduration.js
        var Y = years;
        var M = months;
        var D = days;
        var h = hours;
        var m = minutes;
        var s = seconds ? seconds.toFixed(3).replace(/\.?0+$/, '') : '';
        var total = this.asSeconds();

        if (!total) {
            // this is the same as C#'s (Noda) and python (isodate)...
            // but not other JS (goog.date)
            return 'P0D';
        }

        var totalSign = total < 0 ? '-' : '';
        var ymSign = sign(this._months) !== sign(total) ? '-' : '';
        var daysSign = sign(this._days) !== sign(total) ? '-' : '';
        var hmsSign = sign(this._milliseconds) !== sign(total) ? '-' : '';

        return totalSign + 'P' +
            (Y ? ymSign + Y + 'Y' : '') +
            (M ? ymSign + M + 'M' : '') +
            (D ? daysSign + D + 'D' : '') +
            ((h || m || s) ? 'T' : '') +
            (h ? hmsSign + h + 'H' : '') +
            (m ? hmsSign + m + 'M' : '') +
            (s ? hmsSign + s + 'S' : '');
    }

    var proto$2 = Duration.prototype;

    proto$2.isValid        = isValid$1;
    proto$2.abs            = abs;
    proto$2.add            = add$1;
    proto$2.subtract       = subtract$1;
    proto$2.as             = as;
    proto$2.asMilliseconds = asMilliseconds;
    proto$2.asSeconds      = asSeconds;
    proto$2.asMinutes      = asMinutes;
    proto$2.asHours        = asHours;
    proto$2.asDays         = asDays;
    proto$2.asWeeks        = asWeeks;
    proto$2.asMonths       = asMonths;
    proto$2.asQuarters     = asQuarters;
    proto$2.asYears        = asYears;
    proto$2.valueOf        = valueOf$1;
    proto$2._bubble        = bubble;
    proto$2.clone          = clone$1;
    proto$2.get            = get$2;
    proto$2.milliseconds   = milliseconds;
    proto$2.seconds        = seconds;
    proto$2.minutes        = minutes;
    proto$2.hours          = hours;
    proto$2.days           = days;
    proto$2.weeks          = weeks;
    proto$2.months         = months;
    proto$2.years          = years;
    proto$2.humanize       = humanize;
    proto$2.toISOString    = toISOString$1;
    proto$2.toString       = toISOString$1;
    proto$2.toJSON         = toISOString$1;
    proto$2.locale         = locale;
    proto$2.localeData     = localeData;

    proto$2.toIsoString = deprecate('toIsoString() is deprecated. Please use toISOString() instead (notice the capitals)', toISOString$1);
    proto$2.lang = lang;

    // Side effect imports

    // FORMATTING

    addFormatToken('X', 0, 0, 'unix');
    addFormatToken('x', 0, 0, 'valueOf');

    // PARSING

    addRegexToken('x', matchSigned);
    addRegexToken('X', matchTimestamp);
    addParseToken('X', function (input, array, config) {
        config._d = new Date(parseFloat(input, 10) * 1000);
    });
    addParseToken('x', function (input, array, config) {
        config._d = new Date(toInt(input));
    });

    // Side effect imports

    //! moment.js

    hooks.version = '2.24.0';

    setHookCallback(createLocal);

    hooks.fn                    = proto;
    hooks.min                   = min;
    hooks.max                   = max;
    hooks.now                   = now;
    hooks.utc                   = createUTC;
    hooks.unix                  = createUnix;
    hooks.months                = listMonths;
    hooks.isDate                = isDate;
    hooks.locale                = getSetGlobalLocale;
    hooks.invalid               = createInvalid;
    hooks.duration              = createDuration;
    hooks.isMoment              = isMoment;
    hooks.weekdays              = listWeekdays;
    hooks.parseZone             = createInZone;
    hooks.localeData            = getLocale;
    hooks.isDuration            = isDuration;
    hooks.monthsShort           = listMonthsShort;
    hooks.weekdaysMin           = listWeekdaysMin;
    hooks.defineLocale          = defineLocale;
    hooks.updateLocale          = updateLocale;
    hooks.locales               = listLocales;
    hooks.weekdaysShort         = listWeekdaysShort;
    hooks.normalizeUnits        = normalizeUnits;
    hooks.relativeTimeRounding  = getSetRelativeTimeRounding;
    hooks.relativeTimeThreshold = getSetRelativeTimeThreshold;
    hooks.calendarFormat        = getCalendarFormat;
    hooks.prototype             = proto;

    // currently HTML5 input type only supports 24-hour formats
    hooks.HTML5_FMT = {
        DATETIME_LOCAL: 'YYYY-MM-DDTHH:mm',             // <input type="datetime-local" />
        DATETIME_LOCAL_SECONDS: 'YYYY-MM-DDTHH:mm:ss',  // <input type="datetime-local" step="1" />
        DATETIME_LOCAL_MS: 'YYYY-MM-DDTHH:mm:ss.SSS',   // <input type="datetime-local" step="0.001" />
        DATE: 'YYYY-MM-DD',                             // <input type="date" />
        TIME: 'HH:mm',                                  // <input type="time" />
        TIME_SECONDS: 'HH:mm:ss',                       // <input type="time" step="1" />
        TIME_MS: 'HH:mm:ss.SSS',                        // <input type="time" step="0.001" />
        WEEK: 'GGGG-[W]WW',                             // <input type="week" />
        MONTH: 'YYYY-MM'                                // <input type="month" />
    };

    //! moment.js locale configuration

    hooks.defineLocale('af', {
        months : 'Januarie_Februarie_Maart_April_Mei_Junie_Julie_Augustus_September_Oktober_November_Desember'.split('_'),
        monthsShort : 'Jan_Feb_Mrt_Apr_Mei_Jun_Jul_Aug_Sep_Okt_Nov_Des'.split('_'),
        weekdays : 'Sondag_Maandag_Dinsdag_Woensdag_Donderdag_Vrydag_Saterdag'.split('_'),
        weekdaysShort : 'Son_Maa_Din_Woe_Don_Vry_Sat'.split('_'),
        weekdaysMin : 'So_Ma_Di_Wo_Do_Vr_Sa'.split('_'),
        meridiemParse: /vm|nm/i,
        isPM : function (input) {
            return /^nm$/i.test(input);
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 12) {
                return isLower ? 'vm' : 'VM';
            } else {
                return isLower ? 'nm' : 'NM';
            }
        },
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Vandag om] LT',
            nextDay : '[Môre om] LT',
            nextWeek : 'dddd [om] LT',
            lastDay : '[Gister om] LT',
            lastWeek : '[Laas] dddd [om] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'oor %s',
            past : '%s gelede',
            s : '\'n paar sekondes',
            ss : '%d sekondes',
            m : '\'n minuut',
            mm : '%d minute',
            h : '\'n uur',
            hh : '%d ure',
            d : '\'n dag',
            dd : '%d dae',
            M : '\'n maand',
            MM : '%d maande',
            y : '\'n jaar',
            yy : '%d jaar'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
        ordinal : function (number) {
            return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de'); // Thanks to Joris Röling : https://github.com/jjupiter
        },
        week : {
            dow : 1, // Maandag is die eerste dag van die week.
            doy : 4  // Die week wat die 4de Januarie bevat is die eerste week van die jaar.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ar-dz', {
        months : 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        monthsShort : 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'احد_اثنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'أح_إث_ثلا_أر_خم_جم_سب'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[اليوم على الساعة] LT',
            nextDay: '[غدا على الساعة] LT',
            nextWeek: 'dddd [على الساعة] LT',
            lastDay: '[أمس على الساعة] LT',
            lastWeek: 'dddd [على الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'في %s',
            past : 'منذ %s',
            s : 'ثوان',
            ss : '%d ثانية',
            m : 'دقيقة',
            mm : '%d دقائق',
            h : 'ساعة',
            hh : '%d ساعات',
            d : 'يوم',
            dd : '%d أيام',
            M : 'شهر',
            MM : '%d أشهر',
            y : 'سنة',
            yy : '%d سنوات'
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ar-kw', {
        months : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
        monthsShort : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
        weekdays : 'الأحد_الإتنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'احد_اتنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[اليوم على الساعة] LT',
            nextDay: '[غدا على الساعة] LT',
            nextWeek: 'dddd [على الساعة] LT',
            lastDay: '[أمس على الساعة] LT',
            lastWeek: 'dddd [على الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'في %s',
            past : 'منذ %s',
            s : 'ثوان',
            ss : '%d ثانية',
            m : 'دقيقة',
            mm : '%d دقائق',
            h : 'ساعة',
            hh : '%d ساعات',
            d : 'يوم',
            dd : '%d أيام',
            M : 'شهر',
            MM : '%d أشهر',
            y : 'سنة',
            yy : '%d سنوات'
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap = {
        '1': '1',
        '2': '2',
        '3': '3',
        '4': '4',
        '5': '5',
        '6': '6',
        '7': '7',
        '8': '8',
        '9': '9',
        '0': '0'
    }, pluralForm = function (n) {
        return n === 0 ? 0 : n === 1 ? 1 : n === 2 ? 2 : n % 100 >= 3 && n % 100 <= 10 ? 3 : n % 100 >= 11 ? 4 : 5;
    }, plurals = {
        s : ['أقل من ثانية', 'ثانية واحدة', ['ثانيتان', 'ثانيتين'], '%d ثوان', '%d ثانية', '%d ثانية'],
        m : ['أقل من دقيقة', 'دقيقة واحدة', ['دقيقتان', 'دقيقتين'], '%d دقائق', '%d دقيقة', '%d دقيقة'],
        h : ['أقل من ساعة', 'ساعة واحدة', ['ساعتان', 'ساعتين'], '%d ساعات', '%d ساعة', '%d ساعة'],
        d : ['أقل من يوم', 'يوم واحد', ['يومان', 'يومين'], '%d أيام', '%d يومًا', '%d يوم'],
        M : ['أقل من شهر', 'شهر واحد', ['شهران', 'شهرين'], '%d أشهر', '%d شهرا', '%d شهر'],
        y : ['أقل من عام', 'عام واحد', ['عامان', 'عامين'], '%d أعوام', '%d عامًا', '%d عام']
    }, pluralize = function (u) {
        return function (number, withoutSuffix, string, isFuture) {
            var f = pluralForm(number),
                str = plurals[u][pluralForm(number)];
            if (f === 2) {
                str = str[withoutSuffix ? 0 : 1];
            }
            return str.replace(/%d/i, number);
        };
    }, months$1 = [
        'يناير',
        'فبراير',
        'مارس',
        'أبريل',
        'مايو',
        'يونيو',
        'يوليو',
        'أغسطس',
        'سبتمبر',
        'أكتوبر',
        'نوفمبر',
        'ديسمبر'
    ];

    hooks.defineLocale('ar-ly', {
        months : months$1,
        monthsShort : months$1,
        weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'D/\u200FM/\u200FYYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        meridiemParse: /ص|م/,
        isPM : function (input) {
            return 'م' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ص';
            } else {
                return 'م';
            }
        },
        calendar : {
            sameDay: '[اليوم عند الساعة] LT',
            nextDay: '[غدًا عند الساعة] LT',
            nextWeek: 'dddd [عند الساعة] LT',
            lastDay: '[أمس عند الساعة] LT',
            lastWeek: 'dddd [عند الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'بعد %s',
            past : 'منذ %s',
            s : pluralize('s'),
            ss : pluralize('s'),
            m : pluralize('m'),
            mm : pluralize('m'),
            h : pluralize('h'),
            hh : pluralize('h'),
            d : pluralize('d'),
            dd : pluralize('d'),
            M : pluralize('M'),
            MM : pluralize('M'),
            y : pluralize('y'),
            yy : pluralize('y')
        },
        preparse: function (string) {
            return string.replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap[match];
            }).replace(/,/g, '،');
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ar-ma', {
        months : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
        monthsShort : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
        weekdays : 'الأحد_الإتنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'احد_اتنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[اليوم على الساعة] LT',
            nextDay: '[غدا على الساعة] LT',
            nextWeek: 'dddd [على الساعة] LT',
            lastDay: '[أمس على الساعة] LT',
            lastWeek: 'dddd [على الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'في %s',
            past : 'منذ %s',
            s : 'ثوان',
            ss : '%d ثانية',
            m : 'دقيقة',
            mm : '%d دقائق',
            h : 'ساعة',
            hh : '%d ساعات',
            d : 'يوم',
            dd : '%d أيام',
            M : 'شهر',
            MM : '%d أشهر',
            y : 'سنة',
            yy : '%d سنوات'
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$1 = {
        '1': '١',
        '2': '٢',
        '3': '٣',
        '4': '٤',
        '5': '٥',
        '6': '٦',
        '7': '٧',
        '8': '٨',
        '9': '٩',
        '0': '٠'
    }, numberMap = {
        '١': '1',
        '٢': '2',
        '٣': '3',
        '٤': '4',
        '٥': '5',
        '٦': '6',
        '٧': '7',
        '٨': '8',
        '٩': '9',
        '٠': '0'
    };

    hooks.defineLocale('ar-sa', {
        months : 'يناير_فبراير_مارس_أبريل_مايو_يونيو_يوليو_أغسطس_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        monthsShort : 'يناير_فبراير_مارس_أبريل_مايو_يونيو_يوليو_أغسطس_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        meridiemParse: /ص|م/,
        isPM : function (input) {
            return 'م' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ص';
            } else {
                return 'م';
            }
        },
        calendar : {
            sameDay: '[اليوم على الساعة] LT',
            nextDay: '[غدا على الساعة] LT',
            nextWeek: 'dddd [على الساعة] LT',
            lastDay: '[أمس على الساعة] LT',
            lastWeek: 'dddd [على الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'في %s',
            past : 'منذ %s',
            s : 'ثوان',
            ss : '%d ثانية',
            m : 'دقيقة',
            mm : '%d دقائق',
            h : 'ساعة',
            hh : '%d ساعات',
            d : 'يوم',
            dd : '%d أيام',
            M : 'شهر',
            MM : '%d أشهر',
            y : 'سنة',
            yy : '%d سنوات'
        },
        preparse: function (string) {
            return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
                return numberMap[match];
            }).replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$1[match];
            }).replace(/,/g, '،');
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ar-tn', {
        months: 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        monthsShort: 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
        weekdays: 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort: 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin: 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY HH:mm',
            LLLL: 'dddd D MMMM YYYY HH:mm'
        },
        calendar: {
            sameDay: '[اليوم على الساعة] LT',
            nextDay: '[غدا على الساعة] LT',
            nextWeek: 'dddd [على الساعة] LT',
            lastDay: '[أمس على الساعة] LT',
            lastWeek: 'dddd [على الساعة] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: 'في %s',
            past: 'منذ %s',
            s: 'ثوان',
            ss : '%d ثانية',
            m: 'دقيقة',
            mm: '%d دقائق',
            h: 'ساعة',
            hh: '%d ساعات',
            d: 'يوم',
            dd: '%d أيام',
            M: 'شهر',
            MM: '%d أشهر',
            y: 'سنة',
            yy: '%d سنوات'
        },
        week: {
            dow: 1, // Monday is the first day of the week.
            doy: 4 // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$2 = {
        '1': '١',
        '2': '٢',
        '3': '٣',
        '4': '٤',
        '5': '٥',
        '6': '٦',
        '7': '٧',
        '8': '٨',
        '9': '٩',
        '0': '٠'
    }, numberMap$1 = {
        '١': '1',
        '٢': '2',
        '٣': '3',
        '٤': '4',
        '٥': '5',
        '٦': '6',
        '٧': '7',
        '٨': '8',
        '٩': '9',
        '٠': '0'
    }, pluralForm$1 = function (n) {
        return n === 0 ? 0 : n === 1 ? 1 : n === 2 ? 2 : n % 100 >= 3 && n % 100 <= 10 ? 3 : n % 100 >= 11 ? 4 : 5;
    }, plurals$1 = {
        s : ['أقل من ثانية', 'ثانية واحدة', ['ثانيتان', 'ثانيتين'], '%d ثوان', '%d ثانية', '%d ثانية'],
        m : ['أقل من دقيقة', 'دقيقة واحدة', ['دقيقتان', 'دقيقتين'], '%d دقائق', '%d دقيقة', '%d دقيقة'],
        h : ['أقل من ساعة', 'ساعة واحدة', ['ساعتان', 'ساعتين'], '%d ساعات', '%d ساعة', '%d ساعة'],
        d : ['أقل من يوم', 'يوم واحد', ['يومان', 'يومين'], '%d أيام', '%d يومًا', '%d يوم'],
        M : ['أقل من شهر', 'شهر واحد', ['شهران', 'شهرين'], '%d أشهر', '%d شهرا', '%d شهر'],
        y : ['أقل من عام', 'عام واحد', ['عامان', 'عامين'], '%d أعوام', '%d عامًا', '%d عام']
    }, pluralize$1 = function (u) {
        return function (number, withoutSuffix, string, isFuture) {
            var f = pluralForm$1(number),
                str = plurals$1[u][pluralForm$1(number)];
            if (f === 2) {
                str = str[withoutSuffix ? 0 : 1];
            }
            return str.replace(/%d/i, number);
        };
    }, months$2 = [
        'يناير',
        'فبراير',
        'مارس',
        'أبريل',
        'مايو',
        'يونيو',
        'يوليو',
        'أغسطس',
        'سبتمبر',
        'أكتوبر',
        'نوفمبر',
        'ديسمبر'
    ];

    hooks.defineLocale('ar', {
        months : months$2,
        monthsShort : months$2,
        weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
        weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
        weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'D/\u200FM/\u200FYYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        meridiemParse: /ص|م/,
        isPM : function (input) {
            return 'م' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ص';
            } else {
                return 'م';
            }
        },
        calendar : {
            sameDay: '[اليوم عند الساعة] LT',
            nextDay: '[غدًا عند الساعة] LT',
            nextWeek: 'dddd [عند الساعة] LT',
            lastDay: '[أمس عند الساعة] LT',
            lastWeek: 'dddd [عند الساعة] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'بعد %s',
            past : 'منذ %s',
            s : pluralize$1('s'),
            ss : pluralize$1('s'),
            m : pluralize$1('m'),
            mm : pluralize$1('m'),
            h : pluralize$1('h'),
            hh : pluralize$1('h'),
            d : pluralize$1('d'),
            dd : pluralize$1('d'),
            M : pluralize$1('M'),
            MM : pluralize$1('M'),
            y : pluralize$1('y'),
            yy : pluralize$1('y')
        },
        preparse: function (string) {
            return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
                return numberMap$1[match];
            }).replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$2[match];
            }).replace(/,/g, '،');
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var suffixes = {
        1: '-inci',
        5: '-inci',
        8: '-inci',
        70: '-inci',
        80: '-inci',
        2: '-nci',
        7: '-nci',
        20: '-nci',
        50: '-nci',
        3: '-üncü',
        4: '-üncü',
        100: '-üncü',
        6: '-ncı',
        9: '-uncu',
        10: '-uncu',
        30: '-uncu',
        60: '-ıncı',
        90: '-ıncı'
    };

    hooks.defineLocale('az', {
        months : 'yanvar_fevral_mart_aprel_may_iyun_iyul_avqust_sentyabr_oktyabr_noyabr_dekabr'.split('_'),
        monthsShort : 'yan_fev_mar_apr_may_iyn_iyl_avq_sen_okt_noy_dek'.split('_'),
        weekdays : 'Bazar_Bazar ertəsi_Çərşənbə axşamı_Çərşənbə_Cümə axşamı_Cümə_Şənbə'.split('_'),
        weekdaysShort : 'Baz_BzE_ÇAx_Çər_CAx_Cüm_Şən'.split('_'),
        weekdaysMin : 'Bz_BE_ÇA_Çə_CA_Cü_Şə'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[bugün saat] LT',
            nextDay : '[sabah saat] LT',
            nextWeek : '[gələn həftə] dddd [saat] LT',
            lastDay : '[dünən] LT',
            lastWeek : '[keçən həftə] dddd [saat] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s sonra',
            past : '%s əvvəl',
            s : 'birneçə saniyə',
            ss : '%d saniyə',
            m : 'bir dəqiqə',
            mm : '%d dəqiqə',
            h : 'bir saat',
            hh : '%d saat',
            d : 'bir gün',
            dd : '%d gün',
            M : 'bir ay',
            MM : '%d ay',
            y : 'bir il',
            yy : '%d il'
        },
        meridiemParse: /gecə|səhər|gündüz|axşam/,
        isPM : function (input) {
            return /^(gündüz|axşam)$/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'gecə';
            } else if (hour < 12) {
                return 'səhər';
            } else if (hour < 17) {
                return 'gündüz';
            } else {
                return 'axşam';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(ıncı|inci|nci|üncü|ncı|uncu)/,
        ordinal : function (number) {
            if (number === 0) {  // special case for zero
                return number + '-ıncı';
            }
            var a = number % 10,
                b = number % 100 - a,
                c = number >= 100 ? 100 : null;
            return number + (suffixes[a] || suffixes[b] || suffixes[c]);
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function plural(word, num) {
        var forms = word.split('_');
        return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
    }
    function relativeTimeWithPlural(number, withoutSuffix, key) {
        var format = {
            'ss': withoutSuffix ? 'секунда_секунды_секунд' : 'секунду_секунды_секунд',
            'mm': withoutSuffix ? 'хвіліна_хвіліны_хвілін' : 'хвіліну_хвіліны_хвілін',
            'hh': withoutSuffix ? 'гадзіна_гадзіны_гадзін' : 'гадзіну_гадзіны_гадзін',
            'dd': 'дзень_дні_дзён',
            'MM': 'месяц_месяцы_месяцаў',
            'yy': 'год_гады_гадоў'
        };
        if (key === 'm') {
            return withoutSuffix ? 'хвіліна' : 'хвіліну';
        }
        else if (key === 'h') {
            return withoutSuffix ? 'гадзіна' : 'гадзіну';
        }
        else {
            return number + ' ' + plural(format[key], +number);
        }
    }

    hooks.defineLocale('be', {
        months : {
            format: 'студзеня_лютага_сакавіка_красавіка_траўня_чэрвеня_ліпеня_жніўня_верасня_кастрычніка_лістапада_снежня'.split('_'),
            standalone: 'студзень_люты_сакавік_красавік_травень_чэрвень_ліпень_жнівень_верасень_кастрычнік_лістапад_снежань'.split('_')
        },
        monthsShort : 'студ_лют_сак_крас_трав_чэрв_ліп_жнів_вер_каст_ліст_снеж'.split('_'),
        weekdays : {
            format: 'нядзелю_панядзелак_аўторак_сераду_чацвер_пятніцу_суботу'.split('_'),
            standalone: 'нядзеля_панядзелак_аўторак_серада_чацвер_пятніца_субота'.split('_'),
            isFormat: /\[ ?[Ууў] ?(?:мінулую|наступную)? ?\] ?dddd/
        },
        weekdaysShort : 'нд_пн_ат_ср_чц_пт_сб'.split('_'),
        weekdaysMin : 'нд_пн_ат_ср_чц_пт_сб'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY г.',
            LLL : 'D MMMM YYYY г., HH:mm',
            LLLL : 'dddd, D MMMM YYYY г., HH:mm'
        },
        calendar : {
            sameDay: '[Сёння ў] LT',
            nextDay: '[Заўтра ў] LT',
            lastDay: '[Учора ў] LT',
            nextWeek: function () {
                return '[У] dddd [ў] LT';
            },
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                    case 5:
                    case 6:
                        return '[У мінулую] dddd [ў] LT';
                    case 1:
                    case 2:
                    case 4:
                        return '[У мінулы] dddd [ў] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'праз %s',
            past : '%s таму',
            s : 'некалькі секунд',
            m : relativeTimeWithPlural,
            mm : relativeTimeWithPlural,
            h : relativeTimeWithPlural,
            hh : relativeTimeWithPlural,
            d : 'дзень',
            dd : relativeTimeWithPlural,
            M : 'месяц',
            MM : relativeTimeWithPlural,
            y : 'год',
            yy : relativeTimeWithPlural
        },
        meridiemParse: /ночы|раніцы|дня|вечара/,
        isPM : function (input) {
            return /^(дня|вечара)$/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'ночы';
            } else if (hour < 12) {
                return 'раніцы';
            } else if (hour < 17) {
                return 'дня';
            } else {
                return 'вечара';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(і|ы|га)/,
        ordinal: function (number, period) {
            switch (period) {
                case 'M':
                case 'd':
                case 'DDD':
                case 'w':
                case 'W':
                    return (number % 10 === 2 || number % 10 === 3) && (number % 100 !== 12 && number % 100 !== 13) ? number + '-і' : number + '-ы';
                case 'D':
                    return number + '-га';
                default:
                    return number;
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('bg', {
        months : 'януари_февруари_март_април_май_юни_юли_август_септември_октомври_ноември_декември'.split('_'),
        monthsShort : 'янр_фев_мар_апр_май_юни_юли_авг_сеп_окт_ное_дек'.split('_'),
        weekdays : 'неделя_понеделник_вторник_сряда_четвъртък_петък_събота'.split('_'),
        weekdaysShort : 'нед_пон_вто_сря_чет_пет_съб'.split('_'),
        weekdaysMin : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'D.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY H:mm',
            LLLL : 'dddd, D MMMM YYYY H:mm'
        },
        calendar : {
            sameDay : '[Днес в] LT',
            nextDay : '[Утре в] LT',
            nextWeek : 'dddd [в] LT',
            lastDay : '[Вчера в] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                    case 6:
                        return '[В изминалата] dddd [в] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[В изминалия] dddd [в] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'след %s',
            past : 'преди %s',
            s : 'няколко секунди',
            ss : '%d секунди',
            m : 'минута',
            mm : '%d минути',
            h : 'час',
            hh : '%d часа',
            d : 'ден',
            dd : '%d дни',
            M : 'месец',
            MM : '%d месеца',
            y : 'година',
            yy : '%d години'
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(ев|ен|ти|ви|ри|ми)/,
        ordinal : function (number) {
            var lastDigit = number % 10,
                last2Digits = number % 100;
            if (number === 0) {
                return number + '-ев';
            } else if (last2Digits === 0) {
                return number + '-ен';
            } else if (last2Digits > 10 && last2Digits < 20) {
                return number + '-ти';
            } else if (lastDigit === 1) {
                return number + '-ви';
            } else if (lastDigit === 2) {
                return number + '-ри';
            } else if (lastDigit === 7 || lastDigit === 8) {
                return number + '-ми';
            } else {
                return number + '-ти';
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('bm', {
        months : 'Zanwuyekalo_Fewuruyekalo_Marisikalo_Awirilikalo_Mɛkalo_Zuwɛnkalo_Zuluyekalo_Utikalo_Sɛtanburukalo_ɔkutɔburukalo_Nowanburukalo_Desanburukalo'.split('_'),
        monthsShort : 'Zan_Few_Mar_Awi_Mɛ_Zuw_Zul_Uti_Sɛt_ɔku_Now_Des'.split('_'),
        weekdays : 'Kari_Ntɛnɛn_Tarata_Araba_Alamisa_Juma_Sibiri'.split('_'),
        weekdaysShort : 'Kar_Ntɛ_Tar_Ara_Ala_Jum_Sib'.split('_'),
        weekdaysMin : 'Ka_Nt_Ta_Ar_Al_Ju_Si'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'MMMM [tile] D [san] YYYY',
            LLL : 'MMMM [tile] D [san] YYYY [lɛrɛ] HH:mm',
            LLLL : 'dddd MMMM [tile] D [san] YYYY [lɛrɛ] HH:mm'
        },
        calendar : {
            sameDay : '[Bi lɛrɛ] LT',
            nextDay : '[Sini lɛrɛ] LT',
            nextWeek : 'dddd [don lɛrɛ] LT',
            lastDay : '[Kunu lɛrɛ] LT',
            lastWeek : 'dddd [tɛmɛnen lɛrɛ] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s kɔnɔ',
            past : 'a bɛ %s bɔ',
            s : 'sanga dama dama',
            ss : 'sekondi %d',
            m : 'miniti kelen',
            mm : 'miniti %d',
            h : 'lɛrɛ kelen',
            hh : 'lɛrɛ %d',
            d : 'tile kelen',
            dd : 'tile %d',
            M : 'kalo kelen',
            MM : 'kalo %d',
            y : 'san kelen',
            yy : 'san %d'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$3 = {
        '1': '১',
        '2': '২',
        '3': '৩',
        '4': '৪',
        '5': '৫',
        '6': '৬',
        '7': '৭',
        '8': '৮',
        '9': '৯',
        '0': '০'
    },
    numberMap$2 = {
        '১': '1',
        '২': '2',
        '৩': '3',
        '৪': '4',
        '৫': '5',
        '৬': '6',
        '৭': '7',
        '৮': '8',
        '৯': '9',
        '০': '0'
    };

    hooks.defineLocale('bn', {
        months : 'জানুয়ারী_ফেব্রুয়ারি_মার্চ_এপ্রিল_মে_জুন_জুলাই_আগস্ট_সেপ্টেম্বর_অক্টোবর_নভেম্বর_ডিসেম্বর'.split('_'),
        monthsShort : 'জানু_ফেব_মার্চ_এপ্র_মে_জুন_জুল_আগ_সেপ্ট_অক্টো_নভে_ডিসে'.split('_'),
        weekdays : 'রবিবার_সোমবার_মঙ্গলবার_বুধবার_বৃহস্পতিবার_শুক্রবার_শনিবার'.split('_'),
        weekdaysShort : 'রবি_সোম_মঙ্গল_বুধ_বৃহস্পতি_শুক্র_শনি'.split('_'),
        weekdaysMin : 'রবি_সোম_মঙ্গ_বুধ_বৃহঃ_শুক্র_শনি'.split('_'),
        longDateFormat : {
            LT : 'A h:mm সময়',
            LTS : 'A h:mm:ss সময়',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm সময়',
            LLLL : 'dddd, D MMMM YYYY, A h:mm সময়'
        },
        calendar : {
            sameDay : '[আজ] LT',
            nextDay : '[আগামীকাল] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[গতকাল] LT',
            lastWeek : '[গত] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s পরে',
            past : '%s আগে',
            s : 'কয়েক সেকেন্ড',
            ss : '%d সেকেন্ড',
            m : 'এক মিনিট',
            mm : '%d মিনিট',
            h : 'এক ঘন্টা',
            hh : '%d ঘন্টা',
            d : 'এক দিন',
            dd : '%d দিন',
            M : 'এক মাস',
            MM : '%d মাস',
            y : 'এক বছর',
            yy : '%d বছর'
        },
        preparse: function (string) {
            return string.replace(/[১২৩৪৫৬৭৮৯০]/g, function (match) {
                return numberMap$2[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$3[match];
            });
        },
        meridiemParse: /রাত|সকাল|দুপুর|বিকাল|রাত/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if ((meridiem === 'রাত' && hour >= 4) ||
                    (meridiem === 'দুপুর' && hour < 5) ||
                    meridiem === 'বিকাল') {
                return hour + 12;
            } else {
                return hour;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'রাত';
            } else if (hour < 10) {
                return 'সকাল';
            } else if (hour < 17) {
                return 'দুপুর';
            } else if (hour < 20) {
                return 'বিকাল';
            } else {
                return 'রাত';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$4 = {
        '1': '༡',
        '2': '༢',
        '3': '༣',
        '4': '༤',
        '5': '༥',
        '6': '༦',
        '7': '༧',
        '8': '༨',
        '9': '༩',
        '0': '༠'
    },
    numberMap$3 = {
        '༡': '1',
        '༢': '2',
        '༣': '3',
        '༤': '4',
        '༥': '5',
        '༦': '6',
        '༧': '7',
        '༨': '8',
        '༩': '9',
        '༠': '0'
    };

    hooks.defineLocale('bo', {
        months : 'ཟླ་བ་དང་པོ_ཟླ་བ་གཉིས་པ_ཟླ་བ་གསུམ་པ_ཟླ་བ་བཞི་པ_ཟླ་བ་ལྔ་པ_ཟླ་བ་དྲུག་པ_ཟླ་བ་བདུན་པ_ཟླ་བ་བརྒྱད་པ_ཟླ་བ་དགུ་པ_ཟླ་བ་བཅུ་པ_ཟླ་བ་བཅུ་གཅིག་པ_ཟླ་བ་བཅུ་གཉིས་པ'.split('_'),
        monthsShort : 'ཟླ་བ་དང་པོ_ཟླ་བ་གཉིས་པ_ཟླ་བ་གསུམ་པ_ཟླ་བ་བཞི་པ_ཟླ་བ་ལྔ་པ_ཟླ་བ་དྲུག་པ_ཟླ་བ་བདུན་པ_ཟླ་བ་བརྒྱད་པ_ཟླ་བ་དགུ་པ_ཟླ་བ་བཅུ་པ_ཟླ་བ་བཅུ་གཅིག་པ_ཟླ་བ་བཅུ་གཉིས་པ'.split('_'),
        weekdays : 'གཟའ་ཉི་མ་_གཟའ་ཟླ་བ་_གཟའ་མིག་དམར་_གཟའ་ལྷག་པ་_གཟའ་ཕུར་བུ_གཟའ་པ་སངས་_གཟའ་སྤེན་པ་'.split('_'),
        weekdaysShort : 'ཉི་མ་_ཟླ་བ་_མིག་དམར་_ལྷག་པ་_ཕུར་བུ_པ་སངས་_སྤེན་པ་'.split('_'),
        weekdaysMin : 'ཉི་མ་_ཟླ་བ་_མིག་དམར་_ལྷག་པ་_ཕུར་བུ_པ་སངས་_སྤེན་པ་'.split('_'),
        longDateFormat : {
            LT : 'A h:mm',
            LTS : 'A h:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm',
            LLLL : 'dddd, D MMMM YYYY, A h:mm'
        },
        calendar : {
            sameDay : '[དི་རིང] LT',
            nextDay : '[སང་ཉིན] LT',
            nextWeek : '[བདུན་ཕྲག་རྗེས་མ], LT',
            lastDay : '[ཁ་སང] LT',
            lastWeek : '[བདུན་ཕྲག་མཐའ་མ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s ལ་',
            past : '%s སྔན་ལ',
            s : 'ལམ་སང',
            ss : '%d སྐར་ཆ།',
            m : 'སྐར་མ་གཅིག',
            mm : '%d སྐར་མ',
            h : 'ཆུ་ཚོད་གཅིག',
            hh : '%d ཆུ་ཚོད',
            d : 'ཉིན་གཅིག',
            dd : '%d ཉིན་',
            M : 'ཟླ་བ་གཅིག',
            MM : '%d ཟླ་བ',
            y : 'ལོ་གཅིག',
            yy : '%d ལོ'
        },
        preparse: function (string) {
            return string.replace(/[༡༢༣༤༥༦༧༨༩༠]/g, function (match) {
                return numberMap$3[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$4[match];
            });
        },
        meridiemParse: /མཚན་མོ|ཞོགས་ཀས|ཉིན་གུང|དགོང་དག|མཚན་མོ/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if ((meridiem === 'མཚན་མོ' && hour >= 4) ||
                    (meridiem === 'ཉིན་གུང' && hour < 5) ||
                    meridiem === 'དགོང་དག') {
                return hour + 12;
            } else {
                return hour;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'མཚན་མོ';
            } else if (hour < 10) {
                return 'ཞོགས་ཀས';
            } else if (hour < 17) {
                return 'ཉིན་གུང';
            } else if (hour < 20) {
                return 'དགོང་དག';
            } else {
                return 'མཚན་མོ';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function relativeTimeWithMutation(number, withoutSuffix, key) {
        var format = {
            'mm': 'munutenn',
            'MM': 'miz',
            'dd': 'devezh'
        };
        return number + ' ' + mutation(format[key], number);
    }
    function specialMutationForYears(number) {
        switch (lastNumber(number)) {
            case 1:
            case 3:
            case 4:
            case 5:
            case 9:
                return number + ' bloaz';
            default:
                return number + ' vloaz';
        }
    }
    function lastNumber(number) {
        if (number > 9) {
            return lastNumber(number % 10);
        }
        return number;
    }
    function mutation(text, number) {
        if (number === 2) {
            return softMutation(text);
        }
        return text;
    }
    function softMutation(text) {
        var mutationTable = {
            'm': 'v',
            'b': 'v',
            'd': 'z'
        };
        if (mutationTable[text.charAt(0)] === undefined) {
            return text;
        }
        return mutationTable[text.charAt(0)] + text.substring(1);
    }

    hooks.defineLocale('br', {
        months : 'Genver_C\'hwevrer_Meurzh_Ebrel_Mae_Mezheven_Gouere_Eost_Gwengolo_Here_Du_Kerzu'.split('_'),
        monthsShort : 'Gen_C\'hwe_Meu_Ebr_Mae_Eve_Gou_Eos_Gwe_Her_Du_Ker'.split('_'),
        weekdays : 'Sul_Lun_Meurzh_Merc\'her_Yaou_Gwener_Sadorn'.split('_'),
        weekdaysShort : 'Sul_Lun_Meu_Mer_Yao_Gwe_Sad'.split('_'),
        weekdaysMin : 'Su_Lu_Me_Mer_Ya_Gw_Sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'h[e]mm A',
            LTS : 'h[e]mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D [a viz] MMMM YYYY',
            LLL : 'D [a viz] MMMM YYYY h[e]mm A',
            LLLL : 'dddd, D [a viz] MMMM YYYY h[e]mm A'
        },
        calendar : {
            sameDay : '[Hiziv da] LT',
            nextDay : '[Warc\'hoazh da] LT',
            nextWeek : 'dddd [da] LT',
            lastDay : '[Dec\'h da] LT',
            lastWeek : 'dddd [paset da] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'a-benn %s',
            past : '%s \'zo',
            s : 'un nebeud segondennoù',
            ss : '%d eilenn',
            m : 'ur vunutenn',
            mm : relativeTimeWithMutation,
            h : 'un eur',
            hh : '%d eur',
            d : 'un devezh',
            dd : relativeTimeWithMutation,
            M : 'ur miz',
            MM : relativeTimeWithMutation,
            y : 'ur bloaz',
            yy : specialMutationForYears
        },
        dayOfMonthOrdinalParse: /\d{1,2}(añ|vet)/,
        ordinal : function (number) {
            var output = (number === 1) ? 'añ' : 'vet';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function translate(number, withoutSuffix, key) {
        var result = number + ' ';
        switch (key) {
            case 'ss':
                if (number === 1) {
                    result += 'sekunda';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'sekunde';
                } else {
                    result += 'sekundi';
                }
                return result;
            case 'm':
                return withoutSuffix ? 'jedna minuta' : 'jedne minute';
            case 'mm':
                if (number === 1) {
                    result += 'minuta';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'minute';
                } else {
                    result += 'minuta';
                }
                return result;
            case 'h':
                return withoutSuffix ? 'jedan sat' : 'jednog sata';
            case 'hh':
                if (number === 1) {
                    result += 'sat';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'sata';
                } else {
                    result += 'sati';
                }
                return result;
            case 'dd':
                if (number === 1) {
                    result += 'dan';
                } else {
                    result += 'dana';
                }
                return result;
            case 'MM':
                if (number === 1) {
                    result += 'mjesec';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'mjeseca';
                } else {
                    result += 'mjeseci';
                }
                return result;
            case 'yy':
                if (number === 1) {
                    result += 'godina';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'godine';
                } else {
                    result += 'godina';
                }
                return result;
        }
    }

    hooks.defineLocale('bs', {
        months : 'januar_februar_mart_april_maj_juni_juli_august_septembar_oktobar_novembar_decembar'.split('_'),
        monthsShort : 'jan._feb._mar._apr._maj._jun._jul._aug._sep._okt._nov._dec.'.split('_'),
        monthsParseExact: true,
        weekdays : 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
        weekdaysShort : 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
        weekdaysMin : 'ne_po_ut_sr_če_pe_su'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd, D. MMMM YYYY H:mm'
        },
        calendar : {
            sameDay  : '[danas u] LT',
            nextDay  : '[sutra u] LT',
            nextWeek : function () {
                switch (this.day()) {
                    case 0:
                        return '[u] [nedjelju] [u] LT';
                    case 3:
                        return '[u] [srijedu] [u] LT';
                    case 6:
                        return '[u] [subotu] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[jučer u] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                        return '[prošlu] dddd [u] LT';
                    case 6:
                        return '[prošle] [subote] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[prošli] dddd [u] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'za %s',
            past   : 'prije %s',
            s      : 'par sekundi',
            ss     : translate,
            m      : translate,
            mm     : translate,
            h      : translate,
            hh     : translate,
            d      : 'dan',
            dd     : translate,
            M      : 'mjesec',
            MM     : translate,
            y      : 'godinu',
            yy     : translate
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ca', {
        months : {
            standalone: 'gener_febrer_març_abril_maig_juny_juliol_agost_setembre_octubre_novembre_desembre'.split('_'),
            format: 'de gener_de febrer_de març_d\'abril_de maig_de juny_de juliol_d\'agost_de setembre_d\'octubre_de novembre_de desembre'.split('_'),
            isFormat: /D[oD]?(\s)+MMMM/
        },
        monthsShort : 'gen._febr._març_abr._maig_juny_jul._ag._set._oct._nov._des.'.split('_'),
        monthsParseExact : true,
        weekdays : 'diumenge_dilluns_dimarts_dimecres_dijous_divendres_dissabte'.split('_'),
        weekdaysShort : 'dg._dl._dt._dc._dj._dv._ds.'.split('_'),
        weekdaysMin : 'dg_dl_dt_dc_dj_dv_ds'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM [de] YYYY',
            ll : 'D MMM YYYY',
            LLL : 'D MMMM [de] YYYY [a les] H:mm',
            lll : 'D MMM YYYY, H:mm',
            LLLL : 'dddd D MMMM [de] YYYY [a les] H:mm',
            llll : 'ddd D MMM YYYY, H:mm'
        },
        calendar : {
            sameDay : function () {
                return '[avui a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            nextDay : function () {
                return '[demà a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            lastDay : function () {
                return '[ahir a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [passat a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'd\'aquí %s',
            past : 'fa %s',
            s : 'uns segons',
            ss : '%d segons',
            m : 'un minut',
            mm : '%d minuts',
            h : 'una hora',
            hh : '%d hores',
            d : 'un dia',
            dd : '%d dies',
            M : 'un mes',
            MM : '%d mesos',
            y : 'un any',
            yy : '%d anys'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(r|n|t|è|a)/,
        ordinal : function (number, period) {
            var output = (number === 1) ? 'r' :
                (number === 2) ? 'n' :
                (number === 3) ? 'r' :
                (number === 4) ? 't' : 'è';
            if (period === 'w' || period === 'W') {
                output = 'a';
            }
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var months$3 = 'leden_únor_březen_duben_květen_červen_červenec_srpen_září_říjen_listopad_prosinec'.split('_'),
        monthsShort = 'led_úno_bře_dub_kvě_čvn_čvc_srp_zář_říj_lis_pro'.split('_');

    var monthsParse = [/^led/i, /^úno/i, /^bře/i, /^dub/i, /^kvě/i, /^(čvn|červen$|června)/i, /^(čvc|červenec|července)/i, /^srp/i, /^zář/i, /^říj/i, /^lis/i, /^pro/i];
    // NOTE: 'červen' is substring of 'červenec'; therefore 'červenec' must precede 'červen' in the regex to be fully matched.
    // Otherwise parser matches '1. červenec' as '1. červen' + 'ec'.
    var monthsRegex$1 = /^(leden|únor|březen|duben|květen|červenec|července|červen|června|srpen|září|říjen|listopad|prosinec|led|úno|bře|dub|kvě|čvn|čvc|srp|zář|říj|lis|pro)/i;

    function plural$1(n) {
        return (n > 1) && (n < 5) && (~~(n / 10) !== 1);
    }
    function translate$1(number, withoutSuffix, key, isFuture) {
        var result = number + ' ';
        switch (key) {
            case 's':  // a few seconds / in a few seconds / a few seconds ago
                return (withoutSuffix || isFuture) ? 'pár sekund' : 'pár sekundami';
            case 'ss': // 9 seconds / in 9 seconds / 9 seconds ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'sekundy' : 'sekund');
                } else {
                    return result + 'sekundami';
                }
                break;
            case 'm':  // a minute / in a minute / a minute ago
                return withoutSuffix ? 'minuta' : (isFuture ? 'minutu' : 'minutou');
            case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'minuty' : 'minut');
                } else {
                    return result + 'minutami';
                }
                break;
            case 'h':  // an hour / in an hour / an hour ago
                return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
            case 'hh': // 9 hours / in 9 hours / 9 hours ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'hodiny' : 'hodin');
                } else {
                    return result + 'hodinami';
                }
                break;
            case 'd':  // a day / in a day / a day ago
                return (withoutSuffix || isFuture) ? 'den' : 'dnem';
            case 'dd': // 9 days / in 9 days / 9 days ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'dny' : 'dní');
                } else {
                    return result + 'dny';
                }
                break;
            case 'M':  // a month / in a month / a month ago
                return (withoutSuffix || isFuture) ? 'měsíc' : 'měsícem';
            case 'MM': // 9 months / in 9 months / 9 months ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'měsíce' : 'měsíců');
                } else {
                    return result + 'měsíci';
                }
                break;
            case 'y':  // a year / in a year / a year ago
                return (withoutSuffix || isFuture) ? 'rok' : 'rokem';
            case 'yy': // 9 years / in 9 years / 9 years ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$1(number) ? 'roky' : 'let');
                } else {
                    return result + 'lety';
                }
                break;
        }
    }

    hooks.defineLocale('cs', {
        months : months$3,
        monthsShort : monthsShort,
        monthsRegex : monthsRegex$1,
        monthsShortRegex : monthsRegex$1,
        // NOTE: 'červen' is substring of 'červenec'; therefore 'červenec' must precede 'červen' in the regex to be fully matched.
        // Otherwise parser matches '1. červenec' as '1. červen' + 'ec'.
        monthsStrictRegex : /^(leden|ledna|února|únor|březen|března|duben|dubna|květen|května|červenec|července|červen|června|srpen|srpna|září|říjen|října|listopadu|listopad|prosinec|prosince)/i,
        monthsShortStrictRegex : /^(led|úno|bře|dub|kvě|čvn|čvc|srp|zář|říj|lis|pro)/i,
        monthsParse : monthsParse,
        longMonthsParse : monthsParse,
        shortMonthsParse : monthsParse,
        weekdays : 'neděle_pondělí_úterý_středa_čtvrtek_pátek_sobota'.split('_'),
        weekdaysShort : 'ne_po_út_st_čt_pá_so'.split('_'),
        weekdaysMin : 'ne_po_út_st_čt_pá_so'.split('_'),
        longDateFormat : {
            LT: 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd D. MMMM YYYY H:mm',
            l : 'D. M. YYYY'
        },
        calendar : {
            sameDay: '[dnes v] LT',
            nextDay: '[zítra v] LT',
            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[v neděli v] LT';
                    case 1:
                    case 2:
                        return '[v] dddd [v] LT';
                    case 3:
                        return '[ve středu v] LT';
                    case 4:
                        return '[ve čtvrtek v] LT';
                    case 5:
                        return '[v pátek v] LT';
                    case 6:
                        return '[v sobotu v] LT';
                }
            },
            lastDay: '[včera v] LT',
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[minulou neděli v] LT';
                    case 1:
                    case 2:
                        return '[minulé] dddd [v] LT';
                    case 3:
                        return '[minulou středu v] LT';
                    case 4:
                    case 5:
                        return '[minulý] dddd [v] LT';
                    case 6:
                        return '[minulou sobotu v] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'za %s',
            past : 'před %s',
            s : translate$1,
            ss : translate$1,
            m : translate$1,
            mm : translate$1,
            h : translate$1,
            hh : translate$1,
            d : translate$1,
            dd : translate$1,
            M : translate$1,
            MM : translate$1,
            y : translate$1,
            yy : translate$1
        },
        dayOfMonthOrdinalParse : /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('cv', {
        months : 'кӑрлач_нарӑс_пуш_ака_май_ҫӗртме_утӑ_ҫурла_авӑн_юпа_чӳк_раштав'.split('_'),
        monthsShort : 'кӑр_нар_пуш_ака_май_ҫӗр_утӑ_ҫур_авн_юпа_чӳк_раш'.split('_'),
        weekdays : 'вырсарникун_тунтикун_ытларикун_юнкун_кӗҫнерникун_эрнекун_шӑматкун'.split('_'),
        weekdaysShort : 'выр_тун_ытл_юн_кӗҫ_эрн_шӑм'.split('_'),
        weekdaysMin : 'вр_тн_ыт_юн_кҫ_эр_шм'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD-MM-YYYY',
            LL : 'YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ]',
            LLL : 'YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ], HH:mm',
            LLLL : 'dddd, YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ], HH:mm'
        },
        calendar : {
            sameDay: '[Паян] LT [сехетре]',
            nextDay: '[Ыран] LT [сехетре]',
            lastDay: '[Ӗнер] LT [сехетре]',
            nextWeek: '[Ҫитес] dddd LT [сехетре]',
            lastWeek: '[Иртнӗ] dddd LT [сехетре]',
            sameElse: 'L'
        },
        relativeTime : {
            future : function (output) {
                var affix = /сехет$/i.exec(output) ? 'рен' : /ҫул$/i.exec(output) ? 'тан' : 'ран';
                return output + affix;
            },
            past : '%s каялла',
            s : 'пӗр-ик ҫеккунт',
            ss : '%d ҫеккунт',
            m : 'пӗр минут',
            mm : '%d минут',
            h : 'пӗр сехет',
            hh : '%d сехет',
            d : 'пӗр кун',
            dd : '%d кун',
            M : 'пӗр уйӑх',
            MM : '%d уйӑх',
            y : 'пӗр ҫул',
            yy : '%d ҫул'
        },
        dayOfMonthOrdinalParse: /\d{1,2}-мӗш/,
        ordinal : '%d-мӗш',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('cy', {
        months: 'Ionawr_Chwefror_Mawrth_Ebrill_Mai_Mehefin_Gorffennaf_Awst_Medi_Hydref_Tachwedd_Rhagfyr'.split('_'),
        monthsShort: 'Ion_Chwe_Maw_Ebr_Mai_Meh_Gor_Aws_Med_Hyd_Tach_Rhag'.split('_'),
        weekdays: 'Dydd Sul_Dydd Llun_Dydd Mawrth_Dydd Mercher_Dydd Iau_Dydd Gwener_Dydd Sadwrn'.split('_'),
        weekdaysShort: 'Sul_Llun_Maw_Mer_Iau_Gwe_Sad'.split('_'),
        weekdaysMin: 'Su_Ll_Ma_Me_Ia_Gw_Sa'.split('_'),
        weekdaysParseExact : true,
        // time formats are the same as en-gb
        longDateFormat: {
            LT: 'HH:mm',
            LTS : 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY HH:mm',
            LLLL: 'dddd, D MMMM YYYY HH:mm'
        },
        calendar: {
            sameDay: '[Heddiw am] LT',
            nextDay: '[Yfory am] LT',
            nextWeek: 'dddd [am] LT',
            lastDay: '[Ddoe am] LT',
            lastWeek: 'dddd [diwethaf am] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: 'mewn %s',
            past: '%s yn ôl',
            s: 'ychydig eiliadau',
            ss: '%d eiliad',
            m: 'munud',
            mm: '%d munud',
            h: 'awr',
            hh: '%d awr',
            d: 'diwrnod',
            dd: '%d diwrnod',
            M: 'mis',
            MM: '%d mis',
            y: 'blwyddyn',
            yy: '%d flynedd'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(fed|ain|af|il|ydd|ed|eg)/,
        // traditional ordinal numbers above 31 are not commonly used in colloquial Welsh
        ordinal: function (number) {
            var b = number,
                output = '',
                lookup = [
                    '', 'af', 'il', 'ydd', 'ydd', 'ed', 'ed', 'ed', 'fed', 'fed', 'fed', // 1af to 10fed
                    'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'fed' // 11eg to 20fed
                ];
            if (b > 20) {
                if (b === 40 || b === 50 || b === 60 || b === 80 || b === 100) {
                    output = 'fed'; // not 30ain, 70ain or 90ain
                } else {
                    output = 'ain';
                }
            } else if (b > 0) {
                output = lookup[b];
            }
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('da', {
        months : 'januar_februar_marts_april_maj_juni_juli_august_september_oktober_november_december'.split('_'),
        monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec'.split('_'),
        weekdays : 'søndag_mandag_tirsdag_onsdag_torsdag_fredag_lørdag'.split('_'),
        weekdaysShort : 'søn_man_tir_ons_tor_fre_lør'.split('_'),
        weekdaysMin : 'sø_ma_ti_on_to_fr_lø'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY HH:mm',
            LLLL : 'dddd [d.] D. MMMM YYYY [kl.] HH:mm'
        },
        calendar : {
            sameDay : '[i dag kl.] LT',
            nextDay : '[i morgen kl.] LT',
            nextWeek : 'på dddd [kl.] LT',
            lastDay : '[i går kl.] LT',
            lastWeek : '[i] dddd[s kl.] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'om %s',
            past : '%s siden',
            s : 'få sekunder',
            ss : '%d sekunder',
            m : 'et minut',
            mm : '%d minutter',
            h : 'en time',
            hh : '%d timer',
            d : 'en dag',
            dd : '%d dage',
            M : 'en måned',
            MM : '%d måneder',
            y : 'et år',
            yy : '%d år'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime(number, withoutSuffix, key, isFuture) {
        var format = {
            'm': ['eine Minute', 'einer Minute'],
            'h': ['eine Stunde', 'einer Stunde'],
            'd': ['ein Tag', 'einem Tag'],
            'dd': [number + ' Tage', number + ' Tagen'],
            'M': ['ein Monat', 'einem Monat'],
            'MM': [number + ' Monate', number + ' Monaten'],
            'y': ['ein Jahr', 'einem Jahr'],
            'yy': [number + ' Jahre', number + ' Jahren']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }

    hooks.defineLocale('de-at', {
        months : 'Jänner_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
        monthsShort : 'Jän._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
        monthsParseExact : true,
        weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
        weekdaysShort : 'So._Mo._Di._Mi._Do._Fr._Sa.'.split('_'),
        weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY HH:mm',
            LLLL : 'dddd, D. MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[heute um] LT [Uhr]',
            sameElse: 'L',
            nextDay: '[morgen um] LT [Uhr]',
            nextWeek: 'dddd [um] LT [Uhr]',
            lastDay: '[gestern um] LT [Uhr]',
            lastWeek: '[letzten] dddd [um] LT [Uhr]'
        },
        relativeTime : {
            future : 'in %s',
            past : 'vor %s',
            s : 'ein paar Sekunden',
            ss : '%d Sekunden',
            m : processRelativeTime,
            mm : '%d Minuten',
            h : processRelativeTime,
            hh : '%d Stunden',
            d : processRelativeTime,
            dd : processRelativeTime,
            M : processRelativeTime,
            MM : processRelativeTime,
            y : processRelativeTime,
            yy : processRelativeTime
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$1(number, withoutSuffix, key, isFuture) {
        var format = {
            'm': ['eine Minute', 'einer Minute'],
            'h': ['eine Stunde', 'einer Stunde'],
            'd': ['ein Tag', 'einem Tag'],
            'dd': [number + ' Tage', number + ' Tagen'],
            'M': ['ein Monat', 'einem Monat'],
            'MM': [number + ' Monate', number + ' Monaten'],
            'y': ['ein Jahr', 'einem Jahr'],
            'yy': [number + ' Jahre', number + ' Jahren']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }

    hooks.defineLocale('de-ch', {
        months : 'Januar_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
        monthsShort : 'Jan._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
        monthsParseExact : true,
        weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
        weekdaysShort : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
        weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY HH:mm',
            LLLL : 'dddd, D. MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[heute um] LT [Uhr]',
            sameElse: 'L',
            nextDay: '[morgen um] LT [Uhr]',
            nextWeek: 'dddd [um] LT [Uhr]',
            lastDay: '[gestern um] LT [Uhr]',
            lastWeek: '[letzten] dddd [um] LT [Uhr]'
        },
        relativeTime : {
            future : 'in %s',
            past : 'vor %s',
            s : 'ein paar Sekunden',
            ss : '%d Sekunden',
            m : processRelativeTime$1,
            mm : '%d Minuten',
            h : processRelativeTime$1,
            hh : '%d Stunden',
            d : processRelativeTime$1,
            dd : processRelativeTime$1,
            M : processRelativeTime$1,
            MM : processRelativeTime$1,
            y : processRelativeTime$1,
            yy : processRelativeTime$1
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$2(number, withoutSuffix, key, isFuture) {
        var format = {
            'm': ['eine Minute', 'einer Minute'],
            'h': ['eine Stunde', 'einer Stunde'],
            'd': ['ein Tag', 'einem Tag'],
            'dd': [number + ' Tage', number + ' Tagen'],
            'M': ['ein Monat', 'einem Monat'],
            'MM': [number + ' Monate', number + ' Monaten'],
            'y': ['ein Jahr', 'einem Jahr'],
            'yy': [number + ' Jahre', number + ' Jahren']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }

    hooks.defineLocale('de', {
        months : 'Januar_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
        monthsShort : 'Jan._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
        monthsParseExact : true,
        weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
        weekdaysShort : 'So._Mo._Di._Mi._Do._Fr._Sa.'.split('_'),
        weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY HH:mm',
            LLLL : 'dddd, D. MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[heute um] LT [Uhr]',
            sameElse: 'L',
            nextDay: '[morgen um] LT [Uhr]',
            nextWeek: 'dddd [um] LT [Uhr]',
            lastDay: '[gestern um] LT [Uhr]',
            lastWeek: '[letzten] dddd [um] LT [Uhr]'
        },
        relativeTime : {
            future : 'in %s',
            past : 'vor %s',
            s : 'ein paar Sekunden',
            ss : '%d Sekunden',
            m : processRelativeTime$2,
            mm : '%d Minuten',
            h : processRelativeTime$2,
            hh : '%d Stunden',
            d : processRelativeTime$2,
            dd : processRelativeTime$2,
            M : processRelativeTime$2,
            MM : processRelativeTime$2,
            y : processRelativeTime$2,
            yy : processRelativeTime$2
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var months$4 = [
        'ޖެނުއަރީ',
        'ފެބްރުއަރީ',
        'މާރިޗު',
        'އޭޕްރީލު',
        'މޭ',
        'ޖޫން',
        'ޖުލައި',
        'އޯގަސްޓު',
        'ސެޕްޓެމްބަރު',
        'އޮކްޓޯބަރު',
        'ނޮވެމްބަރު',
        'ޑިސެމްބަރު'
    ], weekdays = [
        'އާދިއްތަ',
        'ހޯމަ',
        'އަންގާރަ',
        'ބުދަ',
        'ބުރާސްފަތި',
        'ހުކުރު',
        'ހޮނިހިރު'
    ];

    hooks.defineLocale('dv', {
        months : months$4,
        monthsShort : months$4,
        weekdays : weekdays,
        weekdaysShort : weekdays,
        weekdaysMin : 'އާދި_ހޯމަ_އަން_ބުދަ_ބުރާ_ހުކު_ހޮނި'.split('_'),
        longDateFormat : {

            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'D/M/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        meridiemParse: /މކ|މފ/,
        isPM : function (input) {
            return 'މފ' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'މކ';
            } else {
                return 'މފ';
            }
        },
        calendar : {
            sameDay : '[މިއަދު] LT',
            nextDay : '[މާދަމާ] LT',
            nextWeek : 'dddd LT',
            lastDay : '[އިއްޔެ] LT',
            lastWeek : '[ފާއިތުވި] dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'ތެރޭގައި %s',
            past : 'ކުރިން %s',
            s : 'ސިކުންތުކޮޅެއް',
            ss : 'd% ސިކުންތު',
            m : 'މިނިޓެއް',
            mm : 'މިނިޓު %d',
            h : 'ގަޑިއިރެއް',
            hh : 'ގަޑިއިރު %d',
            d : 'ދުވަހެއް',
            dd : 'ދުވަސް %d',
            M : 'މަހެއް',
            MM : 'މަސް %d',
            y : 'އަހަރެއް',
            yy : 'އަހަރު %d'
        },
        preparse: function (string) {
            return string.replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/,/g, '،');
        },
        week : {
            dow : 7,  // Sunday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('el', {
        monthsNominativeEl : 'Ιανουάριος_Φεβρουάριος_Μάρτιος_Απρίλιος_Μάιος_Ιούνιος_Ιούλιος_Αύγουστος_Σεπτέμβριος_Οκτώβριος_Νοέμβριος_Δεκέμβριος'.split('_'),
        monthsGenitiveEl : 'Ιανουαρίου_Φεβρουαρίου_Μαρτίου_Απριλίου_Μαΐου_Ιουνίου_Ιουλίου_Αυγούστου_Σεπτεμβρίου_Οκτωβρίου_Νοεμβρίου_Δεκεμβρίου'.split('_'),
        months : function (momentToFormat, format) {
            if (!momentToFormat) {
                return this._monthsNominativeEl;
            } else if (typeof format === 'string' && /D/.test(format.substring(0, format.indexOf('MMMM')))) { // if there is a day number before 'MMMM'
                return this._monthsGenitiveEl[momentToFormat.month()];
            } else {
                return this._monthsNominativeEl[momentToFormat.month()];
            }
        },
        monthsShort : 'Ιαν_Φεβ_Μαρ_Απρ_Μαϊ_Ιουν_Ιουλ_Αυγ_Σεπ_Οκτ_Νοε_Δεκ'.split('_'),
        weekdays : 'Κυριακή_Δευτέρα_Τρίτη_Τετάρτη_Πέμπτη_Παρασκευή_Σάββατο'.split('_'),
        weekdaysShort : 'Κυρ_Δευ_Τρι_Τετ_Πεμ_Παρ_Σαβ'.split('_'),
        weekdaysMin : 'Κυ_Δε_Τρ_Τε_Πε_Πα_Σα'.split('_'),
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'μμ' : 'ΜΜ';
            } else {
                return isLower ? 'πμ' : 'ΠΜ';
            }
        },
        isPM : function (input) {
            return ((input + '').toLowerCase()[0] === 'μ');
        },
        meridiemParse : /[ΠΜ]\.?Μ?\.?/i,
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendarEl : {
            sameDay : '[Σήμερα {}] LT',
            nextDay : '[Αύριο {}] LT',
            nextWeek : 'dddd [{}] LT',
            lastDay : '[Χθες {}] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 6:
                        return '[το προηγούμενο] dddd [{}] LT';
                    default:
                        return '[την προηγούμενη] dddd [{}] LT';
                }
            },
            sameElse : 'L'
        },
        calendar : function (key, mom) {
            var output = this._calendarEl[key],
                hours = mom && mom.hours();
            if (isFunction(output)) {
                output = output.apply(mom);
            }
            return output.replace('{}', (hours % 12 === 1 ? 'στη' : 'στις'));
        },
        relativeTime : {
            future : 'σε %s',
            past : '%s πριν',
            s : 'λίγα δευτερόλεπτα',
            ss : '%d δευτερόλεπτα',
            m : 'ένα λεπτό',
            mm : '%d λεπτά',
            h : 'μία ώρα',
            hh : '%d ώρες',
            d : 'μία μέρα',
            dd : '%d μέρες',
            M : 'ένας μήνας',
            MM : '%d μήνες',
            y : 'ένας χρόνος',
            yy : '%d χρόνια'
        },
        dayOfMonthOrdinalParse: /\d{1,2}η/,
        ordinal: '%dη',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4st is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-SG', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-au', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-ca', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'YYYY-MM-DD',
            LL : 'MMMM D, YYYY',
            LLL : 'MMMM D, YYYY h:mm A',
            LLLL : 'dddd, MMMM D, YYYY h:mm A'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-gb', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-ie', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-il', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('en-nz', {
        months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
        weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
        weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
        weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'in %s',
            past : '%s ago',
            s : 'a few seconds',
            ss : '%d seconds',
            m : 'a minute',
            mm : '%d minutes',
            h : 'an hour',
            hh : '%d hours',
            d : 'a day',
            dd : '%d days',
            M : 'a month',
            MM : '%d months',
            y : 'a year',
            yy : '%d years'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('eo', {
        months : 'januaro_februaro_marto_aprilo_majo_junio_julio_aŭgusto_septembro_oktobro_novembro_decembro'.split('_'),
        monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aŭg_sep_okt_nov_dec'.split('_'),
        weekdays : 'dimanĉo_lundo_mardo_merkredo_ĵaŭdo_vendredo_sabato'.split('_'),
        weekdaysShort : 'dim_lun_mard_merk_ĵaŭ_ven_sab'.split('_'),
        weekdaysMin : 'di_lu_ma_me_ĵa_ve_sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'D[-a de] MMMM, YYYY',
            LLL : 'D[-a de] MMMM, YYYY HH:mm',
            LLLL : 'dddd, [la] D[-a de] MMMM, YYYY HH:mm'
        },
        meridiemParse: /[ap]\.t\.m/i,
        isPM: function (input) {
            return input.charAt(0).toLowerCase() === 'p';
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'p.t.m.' : 'P.T.M.';
            } else {
                return isLower ? 'a.t.m.' : 'A.T.M.';
            }
        },
        calendar : {
            sameDay : '[Hodiaŭ je] LT',
            nextDay : '[Morgaŭ je] LT',
            nextWeek : 'dddd [je] LT',
            lastDay : '[Hieraŭ je] LT',
            lastWeek : '[pasinta] dddd [je] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'post %s',
            past : 'antaŭ %s',
            s : 'sekundoj',
            ss : '%d sekundoj',
            m : 'minuto',
            mm : '%d minutoj',
            h : 'horo',
            hh : '%d horoj',
            d : 'tago',//ne 'diurno', ĉar estas uzita por proksimumo
            dd : '%d tagoj',
            M : 'monato',
            MM : '%d monatoj',
            y : 'jaro',
            yy : '%d jaroj'
        },
        dayOfMonthOrdinalParse: /\d{1,2}a/,
        ordinal : '%da',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortDot = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
        monthsShort$1 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

    var monthsParse$1 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
    var monthsRegex$2 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

    hooks.defineLocale('es-do', {
        months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortDot;
            } else if (/-MMM-/.test(format)) {
                return monthsShort$1[m.month()];
            } else {
                return monthsShortDot[m.month()];
            }
        },
        monthsRegex: monthsRegex$2,
        monthsShortRegex: monthsRegex$2,
        monthsStrictRegex: /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
        monthsShortStrictRegex: /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
        monthsParse: monthsParse$1,
        longMonthsParse: monthsParse$1,
        shortMonthsParse: monthsParse$1,
        weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
        weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
        weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY h:mm A',
            LLLL : 'dddd, D [de] MMMM [de] YYYY h:mm A'
        },
        calendar : {
            sameDay : function () {
                return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextDay : function () {
                return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastDay : function () {
                return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'en %s',
            past : 'hace %s',
            s : 'unos segundos',
            ss : '%d segundos',
            m : 'un minuto',
            mm : '%d minutos',
            h : 'una hora',
            hh : '%d horas',
            d : 'un día',
            dd : '%d días',
            M : 'un mes',
            MM : '%d meses',
            y : 'un año',
            yy : '%d años'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal : '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortDot$1 = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
        monthsShort$2 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

    var monthsParse$2 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
    var monthsRegex$3 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

    hooks.defineLocale('es-us', {
        months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortDot$1;
            } else if (/-MMM-/.test(format)) {
                return monthsShort$2[m.month()];
            } else {
                return monthsShortDot$1[m.month()];
            }
        },
        monthsRegex: monthsRegex$3,
        monthsShortRegex: monthsRegex$3,
        monthsStrictRegex: /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
        monthsShortStrictRegex: /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
        monthsParse: monthsParse$2,
        longMonthsParse: monthsParse$2,
        shortMonthsParse: monthsParse$2,
        weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
        weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
        weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'MM/DD/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY h:mm A',
            LLLL : 'dddd, D [de] MMMM [de] YYYY h:mm A'
        },
        calendar : {
            sameDay : function () {
                return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextDay : function () {
                return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastDay : function () {
                return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'en %s',
            past : 'hace %s',
            s : 'unos segundos',
            ss : '%d segundos',
            m : 'un minuto',
            mm : '%d minutos',
            h : 'una hora',
            hh : '%d horas',
            d : 'un día',
            dd : '%d días',
            M : 'un mes',
            MM : '%d meses',
            y : 'un año',
            yy : '%d años'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal : '%dº',
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortDot$2 = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
        monthsShort$3 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

    var monthsParse$3 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
    var monthsRegex$4 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

    hooks.defineLocale('es', {
        months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortDot$2;
            } else if (/-MMM-/.test(format)) {
                return monthsShort$3[m.month()];
            } else {
                return monthsShortDot$2[m.month()];
            }
        },
        monthsRegex : monthsRegex$4,
        monthsShortRegex : monthsRegex$4,
        monthsStrictRegex : /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
        monthsShortStrictRegex : /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
        monthsParse : monthsParse$3,
        longMonthsParse : monthsParse$3,
        shortMonthsParse : monthsParse$3,
        weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
        weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
        weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY H:mm',
            LLLL : 'dddd, D [de] MMMM [de] YYYY H:mm'
        },
        calendar : {
            sameDay : function () {
                return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextDay : function () {
                return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastDay : function () {
                return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'en %s',
            past : 'hace %s',
            s : 'unos segundos',
            ss : '%d segundos',
            m : 'un minuto',
            mm : '%d minutos',
            h : 'una hora',
            hh : '%d horas',
            d : 'un día',
            dd : '%d días',
            M : 'un mes',
            MM : '%d meses',
            y : 'un año',
            yy : '%d años'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal : '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$3(number, withoutSuffix, key, isFuture) {
        var format = {
            's' : ['mõne sekundi', 'mõni sekund', 'paar sekundit'],
            'ss': [number + 'sekundi', number + 'sekundit'],
            'm' : ['ühe minuti', 'üks minut'],
            'mm': [number + ' minuti', number + ' minutit'],
            'h' : ['ühe tunni', 'tund aega', 'üks tund'],
            'hh': [number + ' tunni', number + ' tundi'],
            'd' : ['ühe päeva', 'üks päev'],
            'M' : ['kuu aja', 'kuu aega', 'üks kuu'],
            'MM': [number + ' kuu', number + ' kuud'],
            'y' : ['ühe aasta', 'aasta', 'üks aasta'],
            'yy': [number + ' aasta', number + ' aastat']
        };
        if (withoutSuffix) {
            return format[key][2] ? format[key][2] : format[key][1];
        }
        return isFuture ? format[key][0] : format[key][1];
    }

    hooks.defineLocale('et', {
        months        : 'jaanuar_veebruar_märts_aprill_mai_juuni_juuli_august_september_oktoober_november_detsember'.split('_'),
        monthsShort   : 'jaan_veebr_märts_apr_mai_juuni_juuli_aug_sept_okt_nov_dets'.split('_'),
        weekdays      : 'pühapäev_esmaspäev_teisipäev_kolmapäev_neljapäev_reede_laupäev'.split('_'),
        weekdaysShort : 'P_E_T_K_N_R_L'.split('_'),
        weekdaysMin   : 'P_E_T_K_N_R_L'.split('_'),
        longDateFormat : {
            LT   : 'H:mm',
            LTS : 'H:mm:ss',
            L    : 'DD.MM.YYYY',
            LL   : 'D. MMMM YYYY',
            LLL  : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd, D. MMMM YYYY H:mm'
        },
        calendar : {
            sameDay  : '[Täna,] LT',
            nextDay  : '[Homme,] LT',
            nextWeek : '[Järgmine] dddd LT',
            lastDay  : '[Eile,] LT',
            lastWeek : '[Eelmine] dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s pärast',
            past   : '%s tagasi',
            s      : processRelativeTime$3,
            ss     : processRelativeTime$3,
            m      : processRelativeTime$3,
            mm     : processRelativeTime$3,
            h      : processRelativeTime$3,
            hh     : processRelativeTime$3,
            d      : processRelativeTime$3,
            dd     : '%d päeva',
            M      : processRelativeTime$3,
            MM     : processRelativeTime$3,
            y      : processRelativeTime$3,
            yy     : processRelativeTime$3
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('eu', {
        months : 'urtarrila_otsaila_martxoa_apirila_maiatza_ekaina_uztaila_abuztua_iraila_urria_azaroa_abendua'.split('_'),
        monthsShort : 'urt._ots._mar._api._mai._eka._uzt._abu._ira._urr._aza._abe.'.split('_'),
        monthsParseExact : true,
        weekdays : 'igandea_astelehena_asteartea_asteazkena_osteguna_ostirala_larunbata'.split('_'),
        weekdaysShort : 'ig._al._ar._az._og._ol._lr.'.split('_'),
        weekdaysMin : 'ig_al_ar_az_og_ol_lr'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'YYYY[ko] MMMM[ren] D[a]',
            LLL : 'YYYY[ko] MMMM[ren] D[a] HH:mm',
            LLLL : 'dddd, YYYY[ko] MMMM[ren] D[a] HH:mm',
            l : 'YYYY-M-D',
            ll : 'YYYY[ko] MMM D[a]',
            lll : 'YYYY[ko] MMM D[a] HH:mm',
            llll : 'ddd, YYYY[ko] MMM D[a] HH:mm'
        },
        calendar : {
            sameDay : '[gaur] LT[etan]',
            nextDay : '[bihar] LT[etan]',
            nextWeek : 'dddd LT[etan]',
            lastDay : '[atzo] LT[etan]',
            lastWeek : '[aurreko] dddd LT[etan]',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s barru',
            past : 'duela %s',
            s : 'segundo batzuk',
            ss : '%d segundo',
            m : 'minutu bat',
            mm : '%d minutu',
            h : 'ordu bat',
            hh : '%d ordu',
            d : 'egun bat',
            dd : '%d egun',
            M : 'hilabete bat',
            MM : '%d hilabete',
            y : 'urte bat',
            yy : '%d urte'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$5 = {
        '1': '۱',
        '2': '۲',
        '3': '۳',
        '4': '۴',
        '5': '۵',
        '6': '۶',
        '7': '۷',
        '8': '۸',
        '9': '۹',
        '0': '۰'
    }, numberMap$4 = {
        '۱': '1',
        '۲': '2',
        '۳': '3',
        '۴': '4',
        '۵': '5',
        '۶': '6',
        '۷': '7',
        '۸': '8',
        '۹': '9',
        '۰': '0'
    };

    hooks.defineLocale('fa', {
        months : 'ژانویه_فوریه_مارس_آوریل_مه_ژوئن_ژوئیه_اوت_سپتامبر_اکتبر_نوامبر_دسامبر'.split('_'),
        monthsShort : 'ژانویه_فوریه_مارس_آوریل_مه_ژوئن_ژوئیه_اوت_سپتامبر_اکتبر_نوامبر_دسامبر'.split('_'),
        weekdays : 'یک\u200cشنبه_دوشنبه_سه\u200cشنبه_چهارشنبه_پنج\u200cشنبه_جمعه_شنبه'.split('_'),
        weekdaysShort : 'یک\u200cشنبه_دوشنبه_سه\u200cشنبه_چهارشنبه_پنج\u200cشنبه_جمعه_شنبه'.split('_'),
        weekdaysMin : 'ی_د_س_چ_پ_ج_ش'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        meridiemParse: /قبل از ظهر|بعد از ظهر/,
        isPM: function (input) {
            return /بعد از ظهر/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'قبل از ظهر';
            } else {
                return 'بعد از ظهر';
            }
        },
        calendar : {
            sameDay : '[امروز ساعت] LT',
            nextDay : '[فردا ساعت] LT',
            nextWeek : 'dddd [ساعت] LT',
            lastDay : '[دیروز ساعت] LT',
            lastWeek : 'dddd [پیش] [ساعت] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'در %s',
            past : '%s پیش',
            s : 'چند ثانیه',
            ss : 'ثانیه d%',
            m : 'یک دقیقه',
            mm : '%d دقیقه',
            h : 'یک ساعت',
            hh : '%d ساعت',
            d : 'یک روز',
            dd : '%d روز',
            M : 'یک ماه',
            MM : '%d ماه',
            y : 'یک سال',
            yy : '%d سال'
        },
        preparse: function (string) {
            return string.replace(/[۰-۹]/g, function (match) {
                return numberMap$4[match];
            }).replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$5[match];
            }).replace(/,/g, '،');
        },
        dayOfMonthOrdinalParse: /\d{1,2}م/,
        ordinal : '%dم',
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12 // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var numbersPast = 'nolla yksi kaksi kolme neljä viisi kuusi seitsemän kahdeksan yhdeksän'.split(' '),
        numbersFuture = [
            'nolla', 'yhden', 'kahden', 'kolmen', 'neljän', 'viiden', 'kuuden',
            numbersPast[7], numbersPast[8], numbersPast[9]
        ];
    function translate$2(number, withoutSuffix, key, isFuture) {
        var result = '';
        switch (key) {
            case 's':
                return isFuture ? 'muutaman sekunnin' : 'muutama sekunti';
            case 'ss':
                return isFuture ? 'sekunnin' : 'sekuntia';
            case 'm':
                return isFuture ? 'minuutin' : 'minuutti';
            case 'mm':
                result = isFuture ? 'minuutin' : 'minuuttia';
                break;
            case 'h':
                return isFuture ? 'tunnin' : 'tunti';
            case 'hh':
                result = isFuture ? 'tunnin' : 'tuntia';
                break;
            case 'd':
                return isFuture ? 'päivän' : 'päivä';
            case 'dd':
                result = isFuture ? 'päivän' : 'päivää';
                break;
            case 'M':
                return isFuture ? 'kuukauden' : 'kuukausi';
            case 'MM':
                result = isFuture ? 'kuukauden' : 'kuukautta';
                break;
            case 'y':
                return isFuture ? 'vuoden' : 'vuosi';
            case 'yy':
                result = isFuture ? 'vuoden' : 'vuotta';
                break;
        }
        result = verbalNumber(number, isFuture) + ' ' + result;
        return result;
    }
    function verbalNumber(number, isFuture) {
        return number < 10 ? (isFuture ? numbersFuture[number] : numbersPast[number]) : number;
    }

    hooks.defineLocale('fi', {
        months : 'tammikuu_helmikuu_maaliskuu_huhtikuu_toukokuu_kesäkuu_heinäkuu_elokuu_syyskuu_lokakuu_marraskuu_joulukuu'.split('_'),
        monthsShort : 'tammi_helmi_maalis_huhti_touko_kesä_heinä_elo_syys_loka_marras_joulu'.split('_'),
        weekdays : 'sunnuntai_maanantai_tiistai_keskiviikko_torstai_perjantai_lauantai'.split('_'),
        weekdaysShort : 'su_ma_ti_ke_to_pe_la'.split('_'),
        weekdaysMin : 'su_ma_ti_ke_to_pe_la'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD.MM.YYYY',
            LL : 'Do MMMM[ta] YYYY',
            LLL : 'Do MMMM[ta] YYYY, [klo] HH.mm',
            LLLL : 'dddd, Do MMMM[ta] YYYY, [klo] HH.mm',
            l : 'D.M.YYYY',
            ll : 'Do MMM YYYY',
            lll : 'Do MMM YYYY, [klo] HH.mm',
            llll : 'ddd, Do MMM YYYY, [klo] HH.mm'
        },
        calendar : {
            sameDay : '[tänään] [klo] LT',
            nextDay : '[huomenna] [klo] LT',
            nextWeek : 'dddd [klo] LT',
            lastDay : '[eilen] [klo] LT',
            lastWeek : '[viime] dddd[na] [klo] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s päästä',
            past : '%s sitten',
            s : translate$2,
            ss : translate$2,
            m : translate$2,
            mm : translate$2,
            h : translate$2,
            hh : translate$2,
            d : translate$2,
            dd : translate$2,
            M : translate$2,
            MM : translate$2,
            y : translate$2,
            yy : translate$2
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('fo', {
        months : 'januar_februar_mars_apríl_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
        monthsShort : 'jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_'),
        weekdays : 'sunnudagur_mánadagur_týsdagur_mikudagur_hósdagur_fríggjadagur_leygardagur'.split('_'),
        weekdaysShort : 'sun_mán_týs_mik_hós_frí_ley'.split('_'),
        weekdaysMin : 'su_má_tý_mi_hó_fr_le'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D. MMMM, YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Í dag kl.] LT',
            nextDay : '[Í morgin kl.] LT',
            nextWeek : 'dddd [kl.] LT',
            lastDay : '[Í gjár kl.] LT',
            lastWeek : '[síðstu] dddd [kl] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'um %s',
            past : '%s síðani',
            s : 'fá sekund',
            ss : '%d sekundir',
            m : 'ein minuttur',
            mm : '%d minuttir',
            h : 'ein tími',
            hh : '%d tímar',
            d : 'ein dagur',
            dd : '%d dagar',
            M : 'ein mánaður',
            MM : '%d mánaðir',
            y : 'eitt ár',
            yy : '%d ár'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('fr-ca', {
        months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
        monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
        monthsParseExact : true,
        weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
        weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
        weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Aujourd’hui à] LT',
            nextDay : '[Demain à] LT',
            nextWeek : 'dddd [à] LT',
            lastDay : '[Hier à] LT',
            lastWeek : 'dddd [dernier à] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dans %s',
            past : 'il y a %s',
            s : 'quelques secondes',
            ss : '%d secondes',
            m : 'une minute',
            mm : '%d minutes',
            h : 'une heure',
            hh : '%d heures',
            d : 'un jour',
            dd : '%d jours',
            M : 'un mois',
            MM : '%d mois',
            y : 'un an',
            yy : '%d ans'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(er|e)/,
        ordinal : function (number, period) {
            switch (period) {
                // Words with masculine grammatical gender: mois, trimestre, jour
                default:
                case 'M':
                case 'Q':
                case 'D':
                case 'DDD':
                case 'd':
                    return number + (number === 1 ? 'er' : 'e');

                // Words with feminine grammatical gender: semaine
                case 'w':
                case 'W':
                    return number + (number === 1 ? 're' : 'e');
            }
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('fr-ch', {
        months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
        monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
        monthsParseExact : true,
        weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
        weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
        weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Aujourd’hui à] LT',
            nextDay : '[Demain à] LT',
            nextWeek : 'dddd [à] LT',
            lastDay : '[Hier à] LT',
            lastWeek : 'dddd [dernier à] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dans %s',
            past : 'il y a %s',
            s : 'quelques secondes',
            ss : '%d secondes',
            m : 'une minute',
            mm : '%d minutes',
            h : 'une heure',
            hh : '%d heures',
            d : 'un jour',
            dd : '%d jours',
            M : 'un mois',
            MM : '%d mois',
            y : 'un an',
            yy : '%d ans'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(er|e)/,
        ordinal : function (number, period) {
            switch (period) {
                // Words with masculine grammatical gender: mois, trimestre, jour
                default:
                case 'M':
                case 'Q':
                case 'D':
                case 'DDD':
                case 'd':
                    return number + (number === 1 ? 'er' : 'e');

                // Words with feminine grammatical gender: semaine
                case 'w':
                case 'W':
                    return number + (number === 1 ? 're' : 'e');
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('fr', {
        months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
        monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
        monthsParseExact : true,
        weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
        weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
        weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Aujourd’hui à] LT',
            nextDay : '[Demain à] LT',
            nextWeek : 'dddd [à] LT',
            lastDay : '[Hier à] LT',
            lastWeek : 'dddd [dernier à] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dans %s',
            past : 'il y a %s',
            s : 'quelques secondes',
            ss : '%d secondes',
            m : 'une minute',
            mm : '%d minutes',
            h : 'une heure',
            hh : '%d heures',
            d : 'un jour',
            dd : '%d jours',
            M : 'un mois',
            MM : '%d mois',
            y : 'un an',
            yy : '%d ans'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(er|)/,
        ordinal : function (number, period) {
            switch (period) {
                // TODO: Return 'e' when day of month > 1. Move this case inside
                // block for masculine words below.
                // See https://github.com/moment/moment/issues/3375
                case 'D':
                    return number + (number === 1 ? 'er' : '');

                // Words with masculine grammatical gender: mois, trimestre, jour
                default:
                case 'M':
                case 'Q':
                case 'DDD':
                case 'd':
                    return number + (number === 1 ? 'er' : 'e');

                // Words with feminine grammatical gender: semaine
                case 'w':
                case 'W':
                    return number + (number === 1 ? 're' : 'e');
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortWithDots = 'jan._feb._mrt._apr._mai_jun._jul._aug._sep._okt._nov._des.'.split('_'),
        monthsShortWithoutDots = 'jan_feb_mrt_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_');

    hooks.defineLocale('fy', {
        months : 'jannewaris_febrewaris_maart_april_maaie_juny_july_augustus_septimber_oktober_novimber_desimber'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortWithDots;
            } else if (/-MMM-/.test(format)) {
                return monthsShortWithoutDots[m.month()];
            } else {
                return monthsShortWithDots[m.month()];
            }
        },
        monthsParseExact : true,
        weekdays : 'snein_moandei_tiisdei_woansdei_tongersdei_freed_sneon'.split('_'),
        weekdaysShort : 'si._mo._ti._wo._to._fr._so.'.split('_'),
        weekdaysMin : 'Si_Mo_Ti_Wo_To_Fr_So'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD-MM-YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[hjoed om] LT',
            nextDay: '[moarn om] LT',
            nextWeek: 'dddd [om] LT',
            lastDay: '[juster om] LT',
            lastWeek: '[ôfrûne] dddd [om] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'oer %s',
            past : '%s lyn',
            s : 'in pear sekonden',
            ss : '%d sekonden',
            m : 'ien minút',
            mm : '%d minuten',
            h : 'ien oere',
            hh : '%d oeren',
            d : 'ien dei',
            dd : '%d dagen',
            M : 'ien moanne',
            MM : '%d moannen',
            y : 'ien jier',
            yy : '%d jierren'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
        ordinal : function (number) {
            return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration


    var months$5 = [
        'Eanáir', 'Feabhra', 'Márta', 'Aibreán', 'Bealtaine', 'Méitheamh', 'Iúil', 'Lúnasa', 'Meán Fómhair', 'Deaireadh Fómhair', 'Samhain', 'Nollaig'
    ];

    var monthsShort$4 = ['Eaná', 'Feab', 'Márt', 'Aibr', 'Beal', 'Méit', 'Iúil', 'Lúna', 'Meán', 'Deai', 'Samh', 'Noll'];

    var weekdays$1 = ['Dé Domhnaigh', 'Dé Luain', 'Dé Máirt', 'Dé Céadaoin', 'Déardaoin', 'Dé hAoine', 'Dé Satharn'];

    var weekdaysShort = ['Dom', 'Lua', 'Mái', 'Céa', 'Déa', 'hAo', 'Sat'];

    var weekdaysMin = ['Do', 'Lu', 'Má', 'Ce', 'Dé', 'hA', 'Sa'];

    hooks.defineLocale('ga', {
        months: months$5,
        monthsShort: monthsShort$4,
        monthsParseExact: true,
        weekdays: weekdays$1,
        weekdaysShort: weekdaysShort,
        weekdaysMin: weekdaysMin,
        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY HH:mm',
            LLLL: 'dddd, D MMMM YYYY HH:mm'
        },
        calendar: {
            sameDay: '[Inniu ag] LT',
            nextDay: '[Amárach ag] LT',
            nextWeek: 'dddd [ag] LT',
            lastDay: '[Inné aig] LT',
            lastWeek: 'dddd [seo caite] [ag] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: 'i %s',
            past: '%s ó shin',
            s: 'cúpla soicind',
            ss: '%d soicind',
            m: 'nóiméad',
            mm: '%d nóiméad',
            h: 'uair an chloig',
            hh: '%d uair an chloig',
            d: 'lá',
            dd: '%d lá',
            M: 'mí',
            MM: '%d mí',
            y: 'bliain',
            yy: '%d bliain'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(d|na|mh)/,
        ordinal: function (number) {
            var output = number === 1 ? 'd' : number % 10 === 2 ? 'na' : 'mh';
            return number + output;
        },
        week: {
            dow: 1, // Monday is the first day of the week.
            doy: 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var months$6 = [
        'Am Faoilleach', 'An Gearran', 'Am Màrt', 'An Giblean', 'An Cèitean', 'An t-Ògmhios', 'An t-Iuchar', 'An Lùnastal', 'An t-Sultain', 'An Dàmhair', 'An t-Samhain', 'An Dùbhlachd'
    ];

    var monthsShort$5 = ['Faoi', 'Gear', 'Màrt', 'Gibl', 'Cèit', 'Ògmh', 'Iuch', 'Lùn', 'Sult', 'Dàmh', 'Samh', 'Dùbh'];

    var weekdays$2 = ['Didòmhnaich', 'Diluain', 'Dimàirt', 'Diciadain', 'Diardaoin', 'Dihaoine', 'Disathairne'];

    var weekdaysShort$1 = ['Did', 'Dil', 'Dim', 'Dic', 'Dia', 'Dih', 'Dis'];

    var weekdaysMin$1 = ['Dò', 'Lu', 'Mà', 'Ci', 'Ar', 'Ha', 'Sa'];

    hooks.defineLocale('gd', {
        months : months$6,
        monthsShort : monthsShort$5,
        monthsParseExact : true,
        weekdays : weekdays$2,
        weekdaysShort : weekdaysShort$1,
        weekdaysMin : weekdaysMin$1,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[An-diugh aig] LT',
            nextDay : '[A-màireach aig] LT',
            nextWeek : 'dddd [aig] LT',
            lastDay : '[An-dè aig] LT',
            lastWeek : 'dddd [seo chaidh] [aig] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'ann an %s',
            past : 'bho chionn %s',
            s : 'beagan diogan',
            ss : '%d diogan',
            m : 'mionaid',
            mm : '%d mionaidean',
            h : 'uair',
            hh : '%d uairean',
            d : 'latha',
            dd : '%d latha',
            M : 'mìos',
            MM : '%d mìosan',
            y : 'bliadhna',
            yy : '%d bliadhna'
        },
        dayOfMonthOrdinalParse : /\d{1,2}(d|na|mh)/,
        ordinal : function (number) {
            var output = number === 1 ? 'd' : number % 10 === 2 ? 'na' : 'mh';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('gl', {
        months : 'xaneiro_febreiro_marzo_abril_maio_xuño_xullo_agosto_setembro_outubro_novembro_decembro'.split('_'),
        monthsShort : 'xan._feb._mar._abr._mai._xuñ._xul._ago._set._out._nov._dec.'.split('_'),
        monthsParseExact: true,
        weekdays : 'domingo_luns_martes_mércores_xoves_venres_sábado'.split('_'),
        weekdaysShort : 'dom._lun._mar._mér._xov._ven._sáb.'.split('_'),
        weekdaysMin : 'do_lu_ma_mé_xo_ve_sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY H:mm',
            LLLL : 'dddd, D [de] MMMM [de] YYYY H:mm'
        },
        calendar : {
            sameDay : function () {
                return '[hoxe ' + ((this.hours() !== 1) ? 'ás' : 'á') + '] LT';
            },
            nextDay : function () {
                return '[mañá ' + ((this.hours() !== 1) ? 'ás' : 'á') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [' + ((this.hours() !== 1) ? 'ás' : 'a') + '] LT';
            },
            lastDay : function () {
                return '[onte ' + ((this.hours() !== 1) ? 'á' : 'a') + '] LT';
            },
            lastWeek : function () {
                return '[o] dddd [pasado ' + ((this.hours() !== 1) ? 'ás' : 'a') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : function (str) {
                if (str.indexOf('un') === 0) {
                    return 'n' + str;
                }
                return 'en ' + str;
            },
            past : 'hai %s',
            s : 'uns segundos',
            ss : '%d segundos',
            m : 'un minuto',
            mm : '%d minutos',
            h : 'unha hora',
            hh : '%d horas',
            d : 'un día',
            dd : '%d días',
            M : 'un mes',
            MM : '%d meses',
            y : 'un ano',
            yy : '%d anos'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal : '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$4(number, withoutSuffix, key, isFuture) {
        var format = {
            's': ['thodde secondanim', 'thodde second'],
            'ss': [number + ' secondanim', number + ' second'],
            'm': ['eka mintan', 'ek minute'],
            'mm': [number + ' mintanim', number + ' mintam'],
            'h': ['eka voran', 'ek vor'],
            'hh': [number + ' voranim', number + ' voram'],
            'd': ['eka disan', 'ek dis'],
            'dd': [number + ' disanim', number + ' dis'],
            'M': ['eka mhoinean', 'ek mhoino'],
            'MM': [number + ' mhoineanim', number + ' mhoine'],
            'y': ['eka vorsan', 'ek voros'],
            'yy': [number + ' vorsanim', number + ' vorsam']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }

    hooks.defineLocale('gom-latn', {
        months : 'Janer_Febrer_Mars_Abril_Mai_Jun_Julai_Agost_Setembr_Otubr_Novembr_Dezembr'.split('_'),
        monthsShort : 'Jan._Feb._Mars_Abr._Mai_Jun_Jul._Ago._Set._Otu._Nov._Dez.'.split('_'),
        monthsParseExact : true,
        weekdays : 'Aitar_Somar_Mongllar_Budvar_Brestar_Sukrar_Son\'var'.split('_'),
        weekdaysShort : 'Ait._Som._Mon._Bud._Bre._Suk._Son.'.split('_'),
        weekdaysMin : 'Ai_Sm_Mo_Bu_Br_Su_Sn'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'A h:mm [vazta]',
            LTS : 'A h:mm:ss [vazta]',
            L : 'DD-MM-YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY A h:mm [vazta]',
            LLLL : 'dddd, MMMM[achea] Do, YYYY, A h:mm [vazta]',
            llll: 'ddd, D MMM YYYY, A h:mm [vazta]'
        },
        calendar : {
            sameDay: '[Aiz] LT',
            nextDay: '[Faleam] LT',
            nextWeek: '[Ieta to] dddd[,] LT',
            lastDay: '[Kal] LT',
            lastWeek: '[Fatlo] dddd[,] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : '%s',
            past : '%s adim',
            s : processRelativeTime$4,
            ss : processRelativeTime$4,
            m : processRelativeTime$4,
            mm : processRelativeTime$4,
            h : processRelativeTime$4,
            hh : processRelativeTime$4,
            d : processRelativeTime$4,
            dd : processRelativeTime$4,
            M : processRelativeTime$4,
            MM : processRelativeTime$4,
            y : processRelativeTime$4,
            yy : processRelativeTime$4
        },
        dayOfMonthOrdinalParse : /\d{1,2}(er)/,
        ordinal : function (number, period) {
            switch (period) {
                // the ordinal 'er' only applies to day of the month
                case 'D':
                    return number + 'er';
                default:
                case 'M':
                case 'Q':
                case 'DDD':
                case 'd':
                case 'w':
                case 'W':
                    return number;
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        },
        meridiemParse: /rati|sokalli|donparam|sanje/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'rati') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'sokalli') {
                return hour;
            } else if (meridiem === 'donparam') {
                return hour > 12 ? hour : hour + 12;
            } else if (meridiem === 'sanje') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'rati';
            } else if (hour < 12) {
                return 'sokalli';
            } else if (hour < 16) {
                return 'donparam';
            } else if (hour < 20) {
                return 'sanje';
            } else {
                return 'rati';
            }
        }
    });

    //! moment.js locale configuration

    var symbolMap$6 = {
            '1': '૧',
            '2': '૨',
            '3': '૩',
            '4': '૪',
            '5': '૫',
            '6': '૬',
            '7': '૭',
            '8': '૮',
            '9': '૯',
            '0': '૦'
        },
        numberMap$5 = {
            '૧': '1',
            '૨': '2',
            '૩': '3',
            '૪': '4',
            '૫': '5',
            '૬': '6',
            '૭': '7',
            '૮': '8',
            '૯': '9',
            '૦': '0'
        };

    hooks.defineLocale('gu', {
        months: 'જાન્યુઆરી_ફેબ્રુઆરી_માર્ચ_એપ્રિલ_મે_જૂન_જુલાઈ_ઑગસ્ટ_સપ્ટેમ્બર_ઑક્ટ્બર_નવેમ્બર_ડિસેમ્બર'.split('_'),
        monthsShort: 'જાન્યુ._ફેબ્રુ._માર્ચ_એપ્રિ._મે_જૂન_જુલા._ઑગ._સપ્ટે._ઑક્ટ્._નવે._ડિસે.'.split('_'),
        monthsParseExact: true,
        weekdays: 'રવિવાર_સોમવાર_મંગળવાર_બુધ્વાર_ગુરુવાર_શુક્રવાર_શનિવાર'.split('_'),
        weekdaysShort: 'રવિ_સોમ_મંગળ_બુધ્_ગુરુ_શુક્ર_શનિ'.split('_'),
        weekdaysMin: 'ર_સો_મં_બુ_ગુ_શુ_શ'.split('_'),
        longDateFormat: {
            LT: 'A h:mm વાગ્યે',
            LTS: 'A h:mm:ss વાગ્યે',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY, A h:mm વાગ્યે',
            LLLL: 'dddd, D MMMM YYYY, A h:mm વાગ્યે'
        },
        calendar: {
            sameDay: '[આજ] LT',
            nextDay: '[કાલે] LT',
            nextWeek: 'dddd, LT',
            lastDay: '[ગઇકાલે] LT',
            lastWeek: '[પાછલા] dddd, LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: '%s મા',
            past: '%s પેહલા',
            s: 'અમુક પળો',
            ss: '%d સેકંડ',
            m: 'એક મિનિટ',
            mm: '%d મિનિટ',
            h: 'એક કલાક',
            hh: '%d કલાક',
            d: 'એક દિવસ',
            dd: '%d દિવસ',
            M: 'એક મહિનો',
            MM: '%d મહિનો',
            y: 'એક વર્ષ',
            yy: '%d વર્ષ'
        },
        preparse: function (string) {
            return string.replace(/[૧૨૩૪૫૬૭૮૯૦]/g, function (match) {
                return numberMap$5[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$6[match];
            });
        },
        // Gujarati notation for meridiems are quite fuzzy in practice. While there exists
        // a rigid notion of a 'Pahar' it is not used as rigidly in modern Gujarati.
        meridiemParse: /રાત|બપોર|સવાર|સાંજ/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'રાત') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'સવાર') {
                return hour;
            } else if (meridiem === 'બપોર') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'સાંજ') {
                return hour + 12;
            }
        },
        meridiem: function (hour, minute, isLower) {
            if (hour < 4) {
                return 'રાત';
            } else if (hour < 10) {
                return 'સવાર';
            } else if (hour < 17) {
                return 'બપોર';
            } else if (hour < 20) {
                return 'સાંજ';
            } else {
                return 'રાત';
            }
        },
        week: {
            dow: 0, // Sunday is the first day of the week.
            doy: 6 // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('he', {
        months : 'ינואר_פברואר_מרץ_אפריל_מאי_יוני_יולי_אוגוסט_ספטמבר_אוקטובר_נובמבר_דצמבר'.split('_'),
        monthsShort : 'ינו׳_פבר׳_מרץ_אפר׳_מאי_יוני_יולי_אוג׳_ספט׳_אוק׳_נוב׳_דצמ׳'.split('_'),
        weekdays : 'ראשון_שני_שלישי_רביעי_חמישי_שישי_שבת'.split('_'),
        weekdaysShort : 'א׳_ב׳_ג׳_ד׳_ה׳_ו׳_ש׳'.split('_'),
        weekdaysMin : 'א_ב_ג_ד_ה_ו_ש'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D [ב]MMMM YYYY',
            LLL : 'D [ב]MMMM YYYY HH:mm',
            LLLL : 'dddd, D [ב]MMMM YYYY HH:mm',
            l : 'D/M/YYYY',
            ll : 'D MMM YYYY',
            lll : 'D MMM YYYY HH:mm',
            llll : 'ddd, D MMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[היום ב־]LT',
            nextDay : '[מחר ב־]LT',
            nextWeek : 'dddd [בשעה] LT',
            lastDay : '[אתמול ב־]LT',
            lastWeek : '[ביום] dddd [האחרון בשעה] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'בעוד %s',
            past : 'לפני %s',
            s : 'מספר שניות',
            ss : '%d שניות',
            m : 'דקה',
            mm : '%d דקות',
            h : 'שעה',
            hh : function (number) {
                if (number === 2) {
                    return 'שעתיים';
                }
                return number + ' שעות';
            },
            d : 'יום',
            dd : function (number) {
                if (number === 2) {
                    return 'יומיים';
                }
                return number + ' ימים';
            },
            M : 'חודש',
            MM : function (number) {
                if (number === 2) {
                    return 'חודשיים';
                }
                return number + ' חודשים';
            },
            y : 'שנה',
            yy : function (number) {
                if (number === 2) {
                    return 'שנתיים';
                } else if (number % 10 === 0 && number !== 10) {
                    return number + ' שנה';
                }
                return number + ' שנים';
            }
        },
        meridiemParse: /אחה"צ|לפנה"צ|אחרי הצהריים|לפני הצהריים|לפנות בוקר|בבוקר|בערב/i,
        isPM : function (input) {
            return /^(אחה"צ|אחרי הצהריים|בערב)$/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 5) {
                return 'לפנות בוקר';
            } else if (hour < 10) {
                return 'בבוקר';
            } else if (hour < 12) {
                return isLower ? 'לפנה"צ' : 'לפני הצהריים';
            } else if (hour < 18) {
                return isLower ? 'אחה"צ' : 'אחרי הצהריים';
            } else {
                return 'בערב';
            }
        }
    });

    //! moment.js locale configuration

    var symbolMap$7 = {
        '1': '१',
        '2': '२',
        '3': '३',
        '4': '४',
        '5': '५',
        '6': '६',
        '7': '७',
        '8': '८',
        '9': '९',
        '0': '०'
    },
    numberMap$6 = {
        '१': '1',
        '२': '2',
        '३': '3',
        '४': '4',
        '५': '5',
        '६': '6',
        '७': '7',
        '८': '8',
        '९': '9',
        '०': '0'
    };

    hooks.defineLocale('hi', {
        months : 'जनवरी_फ़रवरी_मार्च_अप्रैल_मई_जून_जुलाई_अगस्त_सितम्बर_अक्टूबर_नवम्बर_दिसम्बर'.split('_'),
        monthsShort : 'जन._फ़र._मार्च_अप्रै._मई_जून_जुल._अग._सित._अक्टू._नव._दिस.'.split('_'),
        monthsParseExact: true,
        weekdays : 'रविवार_सोमवार_मंगलवार_बुधवार_गुरूवार_शुक्रवार_शनिवार'.split('_'),
        weekdaysShort : 'रवि_सोम_मंगल_बुध_गुरू_शुक्र_शनि'.split('_'),
        weekdaysMin : 'र_सो_मं_बु_गु_शु_श'.split('_'),
        longDateFormat : {
            LT : 'A h:mm बजे',
            LTS : 'A h:mm:ss बजे',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm बजे',
            LLLL : 'dddd, D MMMM YYYY, A h:mm बजे'
        },
        calendar : {
            sameDay : '[आज] LT',
            nextDay : '[कल] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[कल] LT',
            lastWeek : '[पिछले] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s में',
            past : '%s पहले',
            s : 'कुछ ही क्षण',
            ss : '%d सेकंड',
            m : 'एक मिनट',
            mm : '%d मिनट',
            h : 'एक घंटा',
            hh : '%d घंटे',
            d : 'एक दिन',
            dd : '%d दिन',
            M : 'एक महीने',
            MM : '%d महीने',
            y : 'एक वर्ष',
            yy : '%d वर्ष'
        },
        preparse: function (string) {
            return string.replace(/[१२३४५६७८९०]/g, function (match) {
                return numberMap$6[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$7[match];
            });
        },
        // Hindi notation for meridiems are quite fuzzy in practice. While there exists
        // a rigid notion of a 'Pahar' it is not used as rigidly in modern Hindi.
        meridiemParse: /रात|सुबह|दोपहर|शाम/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'रात') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'सुबह') {
                return hour;
            } else if (meridiem === 'दोपहर') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'शाम') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'रात';
            } else if (hour < 10) {
                return 'सुबह';
            } else if (hour < 17) {
                return 'दोपहर';
            } else if (hour < 20) {
                return 'शाम';
            } else {
                return 'रात';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function translate$3(number, withoutSuffix, key) {
        var result = number + ' ';
        switch (key) {
            case 'ss':
                if (number === 1) {
                    result += 'sekunda';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'sekunde';
                } else {
                    result += 'sekundi';
                }
                return result;
            case 'm':
                return withoutSuffix ? 'jedna minuta' : 'jedne minute';
            case 'mm':
                if (number === 1) {
                    result += 'minuta';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'minute';
                } else {
                    result += 'minuta';
                }
                return result;
            case 'h':
                return withoutSuffix ? 'jedan sat' : 'jednog sata';
            case 'hh':
                if (number === 1) {
                    result += 'sat';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'sata';
                } else {
                    result += 'sati';
                }
                return result;
            case 'dd':
                if (number === 1) {
                    result += 'dan';
                } else {
                    result += 'dana';
                }
                return result;
            case 'MM':
                if (number === 1) {
                    result += 'mjesec';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'mjeseca';
                } else {
                    result += 'mjeseci';
                }
                return result;
            case 'yy':
                if (number === 1) {
                    result += 'godina';
                } else if (number === 2 || number === 3 || number === 4) {
                    result += 'godine';
                } else {
                    result += 'godina';
                }
                return result;
        }
    }

    hooks.defineLocale('hr', {
        months : {
            format: 'siječnja_veljače_ožujka_travnja_svibnja_lipnja_srpnja_kolovoza_rujna_listopada_studenoga_prosinca'.split('_'),
            standalone: 'siječanj_veljača_ožujak_travanj_svibanj_lipanj_srpanj_kolovoz_rujan_listopad_studeni_prosinac'.split('_')
        },
        monthsShort : 'sij._velj._ožu._tra._svi._lip._srp._kol._ruj._lis._stu._pro.'.split('_'),
        monthsParseExact: true,
        weekdays : 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
        weekdaysShort : 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
        weekdaysMin : 'ne_po_ut_sr_če_pe_su'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd, D. MMMM YYYY H:mm'
        },
        calendar : {
            sameDay  : '[danas u] LT',
            nextDay  : '[sutra u] LT',
            nextWeek : function () {
                switch (this.day()) {
                    case 0:
                        return '[u] [nedjelju] [u] LT';
                    case 3:
                        return '[u] [srijedu] [u] LT';
                    case 6:
                        return '[u] [subotu] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[jučer u] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                        return '[prošlu] dddd [u] LT';
                    case 6:
                        return '[prošle] [subote] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[prošli] dddd [u] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'za %s',
            past   : 'prije %s',
            s      : 'par sekundi',
            ss     : translate$3,
            m      : translate$3,
            mm     : translate$3,
            h      : translate$3,
            hh     : translate$3,
            d      : 'dan',
            dd     : translate$3,
            M      : 'mjesec',
            MM     : translate$3,
            y      : 'godinu',
            yy     : translate$3
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var weekEndings = 'vasárnap hétfőn kedden szerdán csütörtökön pénteken szombaton'.split(' ');
    function translate$4(number, withoutSuffix, key, isFuture) {
        var num = number;
        switch (key) {
            case 's':
                return (isFuture || withoutSuffix) ? 'néhány másodperc' : 'néhány másodperce';
            case 'ss':
                return num + (isFuture || withoutSuffix) ? ' másodperc' : ' másodperce';
            case 'm':
                return 'egy' + (isFuture || withoutSuffix ? ' perc' : ' perce');
            case 'mm':
                return num + (isFuture || withoutSuffix ? ' perc' : ' perce');
            case 'h':
                return 'egy' + (isFuture || withoutSuffix ? ' óra' : ' órája');
            case 'hh':
                return num + (isFuture || withoutSuffix ? ' óra' : ' órája');
            case 'd':
                return 'egy' + (isFuture || withoutSuffix ? ' nap' : ' napja');
            case 'dd':
                return num + (isFuture || withoutSuffix ? ' nap' : ' napja');
            case 'M':
                return 'egy' + (isFuture || withoutSuffix ? ' hónap' : ' hónapja');
            case 'MM':
                return num + (isFuture || withoutSuffix ? ' hónap' : ' hónapja');
            case 'y':
                return 'egy' + (isFuture || withoutSuffix ? ' év' : ' éve');
            case 'yy':
                return num + (isFuture || withoutSuffix ? ' év' : ' éve');
        }
        return '';
    }
    function week(isFuture) {
        return (isFuture ? '' : '[múlt] ') + '[' + weekEndings[this.day()] + '] LT[-kor]';
    }

    hooks.defineLocale('hu', {
        months : 'január_február_március_április_május_június_július_augusztus_szeptember_október_november_december'.split('_'),
        monthsShort : 'jan_feb_márc_ápr_máj_jún_júl_aug_szept_okt_nov_dec'.split('_'),
        weekdays : 'vasárnap_hétfő_kedd_szerda_csütörtök_péntek_szombat'.split('_'),
        weekdaysShort : 'vas_hét_kedd_sze_csüt_pén_szo'.split('_'),
        weekdaysMin : 'v_h_k_sze_cs_p_szo'.split('_'),
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'YYYY.MM.DD.',
            LL : 'YYYY. MMMM D.',
            LLL : 'YYYY. MMMM D. H:mm',
            LLLL : 'YYYY. MMMM D., dddd H:mm'
        },
        meridiemParse: /de|du/i,
        isPM: function (input) {
            return input.charAt(1).toLowerCase() === 'u';
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 12) {
                return isLower === true ? 'de' : 'DE';
            } else {
                return isLower === true ? 'du' : 'DU';
            }
        },
        calendar : {
            sameDay : '[ma] LT[-kor]',
            nextDay : '[holnap] LT[-kor]',
            nextWeek : function () {
                return week.call(this, true);
            },
            lastDay : '[tegnap] LT[-kor]',
            lastWeek : function () {
                return week.call(this, false);
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s múlva',
            past : '%s',
            s : translate$4,
            ss : translate$4,
            m : translate$4,
            mm : translate$4,
            h : translate$4,
            hh : translate$4,
            d : translate$4,
            dd : translate$4,
            M : translate$4,
            MM : translate$4,
            y : translate$4,
            yy : translate$4
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('hy-am', {
        months : {
            format: 'հունվարի_փետրվարի_մարտի_ապրիլի_մայիսի_հունիսի_հուլիսի_օգոստոսի_սեպտեմբերի_հոկտեմբերի_նոյեմբերի_դեկտեմբերի'.split('_'),
            standalone: 'հունվար_փետրվար_մարտ_ապրիլ_մայիս_հունիս_հուլիս_օգոստոս_սեպտեմբեր_հոկտեմբեր_նոյեմբեր_դեկտեմբեր'.split('_')
        },
        monthsShort : 'հնվ_փտր_մրտ_ապր_մյս_հնս_հլս_օգս_սպտ_հկտ_նմբ_դկտ'.split('_'),
        weekdays : 'կիրակի_երկուշաբթի_երեքշաբթի_չորեքշաբթի_հինգշաբթի_ուրբաթ_շաբաթ'.split('_'),
        weekdaysShort : 'կրկ_երկ_երք_չրք_հնգ_ուրբ_շբթ'.split('_'),
        weekdaysMin : 'կրկ_երկ_երք_չրք_հնգ_ուրբ_շբթ'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY թ.',
            LLL : 'D MMMM YYYY թ., HH:mm',
            LLLL : 'dddd, D MMMM YYYY թ., HH:mm'
        },
        calendar : {
            sameDay: '[այսօր] LT',
            nextDay: '[վաղը] LT',
            lastDay: '[երեկ] LT',
            nextWeek: function () {
                return 'dddd [օրը ժամը] LT';
            },
            lastWeek: function () {
                return '[անցած] dddd [օրը ժամը] LT';
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : '%s հետո',
            past : '%s առաջ',
            s : 'մի քանի վայրկյան',
            ss : '%d վայրկյան',
            m : 'րոպե',
            mm : '%d րոպե',
            h : 'ժամ',
            hh : '%d ժամ',
            d : 'օր',
            dd : '%d օր',
            M : 'ամիս',
            MM : '%d ամիս',
            y : 'տարի',
            yy : '%d տարի'
        },
        meridiemParse: /գիշերվա|առավոտվա|ցերեկվա|երեկոյան/,
        isPM: function (input) {
            return /^(ցերեկվա|երեկոյան)$/.test(input);
        },
        meridiem : function (hour) {
            if (hour < 4) {
                return 'գիշերվա';
            } else if (hour < 12) {
                return 'առավոտվա';
            } else if (hour < 17) {
                return 'ցերեկվա';
            } else {
                return 'երեկոյան';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}|\d{1,2}-(ին|րդ)/,
        ordinal: function (number, period) {
            switch (period) {
                case 'DDD':
                case 'w':
                case 'W':
                case 'DDDo':
                    if (number === 1) {
                        return number + '-ին';
                    }
                    return number + '-րդ';
                default:
                    return number;
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('id', {
        months : 'Januari_Februari_Maret_April_Mei_Juni_Juli_Agustus_September_Oktober_November_Desember'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_Mei_Jun_Jul_Agt_Sep_Okt_Nov_Des'.split('_'),
        weekdays : 'Minggu_Senin_Selasa_Rabu_Kamis_Jumat_Sabtu'.split('_'),
        weekdaysShort : 'Min_Sen_Sel_Rab_Kam_Jum_Sab'.split('_'),
        weekdaysMin : 'Mg_Sn_Sl_Rb_Km_Jm_Sb'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY [pukul] HH.mm',
            LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
        },
        meridiemParse: /pagi|siang|sore|malam/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'pagi') {
                return hour;
            } else if (meridiem === 'siang') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'sore' || meridiem === 'malam') {
                return hour + 12;
            }
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'pagi';
            } else if (hours < 15) {
                return 'siang';
            } else if (hours < 19) {
                return 'sore';
            } else {
                return 'malam';
            }
        },
        calendar : {
            sameDay : '[Hari ini pukul] LT',
            nextDay : '[Besok pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kemarin pukul] LT',
            lastWeek : 'dddd [lalu pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dalam %s',
            past : '%s yang lalu',
            s : 'beberapa detik',
            ss : '%d detik',
            m : 'semenit',
            mm : '%d menit',
            h : 'sejam',
            hh : '%d jam',
            d : 'sehari',
            dd : '%d hari',
            M : 'sebulan',
            MM : '%d bulan',
            y : 'setahun',
            yy : '%d tahun'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function plural$2(n) {
        if (n % 100 === 11) {
            return true;
        } else if (n % 10 === 1) {
            return false;
        }
        return true;
    }
    function translate$5(number, withoutSuffix, key, isFuture) {
        var result = number + ' ';
        switch (key) {
            case 's':
                return withoutSuffix || isFuture ? 'nokkrar sekúndur' : 'nokkrum sekúndum';
            case 'ss':
                if (plural$2(number)) {
                    return result + (withoutSuffix || isFuture ? 'sekúndur' : 'sekúndum');
                }
                return result + 'sekúnda';
            case 'm':
                return withoutSuffix ? 'mínúta' : 'mínútu';
            case 'mm':
                if (plural$2(number)) {
                    return result + (withoutSuffix || isFuture ? 'mínútur' : 'mínútum');
                } else if (withoutSuffix) {
                    return result + 'mínúta';
                }
                return result + 'mínútu';
            case 'hh':
                if (plural$2(number)) {
                    return result + (withoutSuffix || isFuture ? 'klukkustundir' : 'klukkustundum');
                }
                return result + 'klukkustund';
            case 'd':
                if (withoutSuffix) {
                    return 'dagur';
                }
                return isFuture ? 'dag' : 'degi';
            case 'dd':
                if (plural$2(number)) {
                    if (withoutSuffix) {
                        return result + 'dagar';
                    }
                    return result + (isFuture ? 'daga' : 'dögum');
                } else if (withoutSuffix) {
                    return result + 'dagur';
                }
                return result + (isFuture ? 'dag' : 'degi');
            case 'M':
                if (withoutSuffix) {
                    return 'mánuður';
                }
                return isFuture ? 'mánuð' : 'mánuði';
            case 'MM':
                if (plural$2(number)) {
                    if (withoutSuffix) {
                        return result + 'mánuðir';
                    }
                    return result + (isFuture ? 'mánuði' : 'mánuðum');
                } else if (withoutSuffix) {
                    return result + 'mánuður';
                }
                return result + (isFuture ? 'mánuð' : 'mánuði');
            case 'y':
                return withoutSuffix || isFuture ? 'ár' : 'ári';
            case 'yy':
                if (plural$2(number)) {
                    return result + (withoutSuffix || isFuture ? 'ár' : 'árum');
                }
                return result + (withoutSuffix || isFuture ? 'ár' : 'ári');
        }
    }

    hooks.defineLocale('is', {
        months : 'janúar_febrúar_mars_apríl_maí_júní_júlí_ágúst_september_október_nóvember_desember'.split('_'),
        monthsShort : 'jan_feb_mar_apr_maí_jún_júl_ágú_sep_okt_nóv_des'.split('_'),
        weekdays : 'sunnudagur_mánudagur_þriðjudagur_miðvikudagur_fimmtudagur_föstudagur_laugardagur'.split('_'),
        weekdaysShort : 'sun_mán_þri_mið_fim_fös_lau'.split('_'),
        weekdaysMin : 'Su_Má_Þr_Mi_Fi_Fö_La'.split('_'),
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY [kl.] H:mm',
            LLLL : 'dddd, D. MMMM YYYY [kl.] H:mm'
        },
        calendar : {
            sameDay : '[í dag kl.] LT',
            nextDay : '[á morgun kl.] LT',
            nextWeek : 'dddd [kl.] LT',
            lastDay : '[í gær kl.] LT',
            lastWeek : '[síðasta] dddd [kl.] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'eftir %s',
            past : 'fyrir %s síðan',
            s : translate$5,
            ss : translate$5,
            m : translate$5,
            mm : translate$5,
            h : 'klukkustund',
            hh : translate$5,
            d : translate$5,
            dd : translate$5,
            M : translate$5,
            MM : translate$5,
            y : translate$5,
            yy : translate$5
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('it-ch', {
        months : 'gennaio_febbraio_marzo_aprile_maggio_giugno_luglio_agosto_settembre_ottobre_novembre_dicembre'.split('_'),
        monthsShort : 'gen_feb_mar_apr_mag_giu_lug_ago_set_ott_nov_dic'.split('_'),
        weekdays : 'domenica_lunedì_martedì_mercoledì_giovedì_venerdì_sabato'.split('_'),
        weekdaysShort : 'dom_lun_mar_mer_gio_ven_sab'.split('_'),
        weekdaysMin : 'do_lu_ma_me_gi_ve_sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Oggi alle] LT',
            nextDay: '[Domani alle] LT',
            nextWeek: 'dddd [alle] LT',
            lastDay: '[Ieri alle] LT',
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[la scorsa] dddd [alle] LT';
                    default:
                        return '[lo scorso] dddd [alle] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : function (s) {
                return ((/^[0-9].+$/).test(s) ? 'tra' : 'in') + ' ' + s;
            },
            past : '%s fa',
            s : 'alcuni secondi',
            ss : '%d secondi',
            m : 'un minuto',
            mm : '%d minuti',
            h : 'un\'ora',
            hh : '%d ore',
            d : 'un giorno',
            dd : '%d giorni',
            M : 'un mese',
            MM : '%d mesi',
            y : 'un anno',
            yy : '%d anni'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal: '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('it', {
        months : 'gennaio_febbraio_marzo_aprile_maggio_giugno_luglio_agosto_settembre_ottobre_novembre_dicembre'.split('_'),
        monthsShort : 'gen_feb_mar_apr_mag_giu_lug_ago_set_ott_nov_dic'.split('_'),
        weekdays : 'domenica_lunedì_martedì_mercoledì_giovedì_venerdì_sabato'.split('_'),
        weekdaysShort : 'dom_lun_mar_mer_gio_ven_sab'.split('_'),
        weekdaysMin : 'do_lu_ma_me_gi_ve_sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Oggi alle] LT',
            nextDay: '[Domani alle] LT',
            nextWeek: 'dddd [alle] LT',
            lastDay: '[Ieri alle] LT',
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[la scorsa] dddd [alle] LT';
                    default:
                        return '[lo scorso] dddd [alle] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : function (s) {
                return ((/^[0-9].+$/).test(s) ? 'tra' : 'in') + ' ' + s;
            },
            past : '%s fa',
            s : 'alcuni secondi',
            ss : '%d secondi',
            m : 'un minuto',
            mm : '%d minuti',
            h : 'un\'ora',
            hh : '%d ore',
            d : 'un giorno',
            dd : '%d giorni',
            M : 'un mese',
            MM : '%d mesi',
            y : 'un anno',
            yy : '%d anni'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal: '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ja', {
        months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
        monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
        weekdays : '日曜日_月曜日_火曜日_水曜日_木曜日_金曜日_土曜日'.split('_'),
        weekdaysShort : '日_月_火_水_木_金_土'.split('_'),
        weekdaysMin : '日_月_火_水_木_金_土'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY/MM/DD',
            LL : 'YYYY年M月D日',
            LLL : 'YYYY年M月D日 HH:mm',
            LLLL : 'YYYY年M月D日 dddd HH:mm',
            l : 'YYYY/MM/DD',
            ll : 'YYYY年M月D日',
            lll : 'YYYY年M月D日 HH:mm',
            llll : 'YYYY年M月D日(ddd) HH:mm'
        },
        meridiemParse: /午前|午後/i,
        isPM : function (input) {
            return input === '午後';
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return '午前';
            } else {
                return '午後';
            }
        },
        calendar : {
            sameDay : '[今日] LT',
            nextDay : '[明日] LT',
            nextWeek : function (now) {
                if (now.week() < this.week()) {
                    return '[来週]dddd LT';
                } else {
                    return 'dddd LT';
                }
            },
            lastDay : '[昨日] LT',
            lastWeek : function (now) {
                if (this.week() < now.week()) {
                    return '[先週]dddd LT';
                } else {
                    return 'dddd LT';
                }
            },
            sameElse : 'L'
        },
        dayOfMonthOrdinalParse : /\d{1,2}日/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'DDD':
                    return number + '日';
                default:
                    return number;
            }
        },
        relativeTime : {
            future : '%s後',
            past : '%s前',
            s : '数秒',
            ss : '%d秒',
            m : '1分',
            mm : '%d分',
            h : '1時間',
            hh : '%d時間',
            d : '1日',
            dd : '%d日',
            M : '1ヶ月',
            MM : '%dヶ月',
            y : '1年',
            yy : '%d年'
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('jv', {
        months : 'Januari_Februari_Maret_April_Mei_Juni_Juli_Agustus_September_Oktober_Nopember_Desember'.split('_'),
        monthsShort : 'Jan_Feb_Mar_Apr_Mei_Jun_Jul_Ags_Sep_Okt_Nop_Des'.split('_'),
        weekdays : 'Minggu_Senen_Seloso_Rebu_Kemis_Jemuwah_Septu'.split('_'),
        weekdaysShort : 'Min_Sen_Sel_Reb_Kem_Jem_Sep'.split('_'),
        weekdaysMin : 'Mg_Sn_Sl_Rb_Km_Jm_Sp'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY [pukul] HH.mm',
            LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
        },
        meridiemParse: /enjing|siyang|sonten|ndalu/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'enjing') {
                return hour;
            } else if (meridiem === 'siyang') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'sonten' || meridiem === 'ndalu') {
                return hour + 12;
            }
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'enjing';
            } else if (hours < 15) {
                return 'siyang';
            } else if (hours < 19) {
                return 'sonten';
            } else {
                return 'ndalu';
            }
        },
        calendar : {
            sameDay : '[Dinten puniko pukul] LT',
            nextDay : '[Mbenjang pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kala wingi pukul] LT',
            lastWeek : 'dddd [kepengker pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'wonten ing %s',
            past : '%s ingkang kepengker',
            s : 'sawetawis detik',
            ss : '%d detik',
            m : 'setunggal menit',
            mm : '%d menit',
            h : 'setunggal jam',
            hh : '%d jam',
            d : 'sedinten',
            dd : '%d dinten',
            M : 'sewulan',
            MM : '%d wulan',
            y : 'setaun',
            yy : '%d taun'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ka', {
        months : {
            standalone: 'იანვარი_თებერვალი_მარტი_აპრილი_მაისი_ივნისი_ივლისი_აგვისტო_სექტემბერი_ოქტომბერი_ნოემბერი_დეკემბერი'.split('_'),
            format: 'იანვარს_თებერვალს_მარტს_აპრილის_მაისს_ივნისს_ივლისს_აგვისტს_სექტემბერს_ოქტომბერს_ნოემბერს_დეკემბერს'.split('_')
        },
        monthsShort : 'იან_თებ_მარ_აპრ_მაი_ივნ_ივლ_აგვ_სექ_ოქტ_ნოე_დეკ'.split('_'),
        weekdays : {
            standalone: 'კვირა_ორშაბათი_სამშაბათი_ოთხშაბათი_ხუთშაბათი_პარასკევი_შაბათი'.split('_'),
            format: 'კვირას_ორშაბათს_სამშაბათს_ოთხშაბათს_ხუთშაბათს_პარასკევს_შაბათს'.split('_'),
            isFormat: /(წინა|შემდეგ)/
        },
        weekdaysShort : 'კვი_ორშ_სამ_ოთხ_ხუთ_პარ_შაბ'.split('_'),
        weekdaysMin : 'კვ_ორ_სა_ოთ_ხუ_პა_შა'.split('_'),
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendar : {
            sameDay : '[დღეს] LT[-ზე]',
            nextDay : '[ხვალ] LT[-ზე]',
            lastDay : '[გუშინ] LT[-ზე]',
            nextWeek : '[შემდეგ] dddd LT[-ზე]',
            lastWeek : '[წინა] dddd LT-ზე',
            sameElse : 'L'
        },
        relativeTime : {
            future : function (s) {
                return (/(წამი|წუთი|საათი|წელი)/).test(s) ?
                    s.replace(/ი$/, 'ში') :
                    s + 'ში';
            },
            past : function (s) {
                if ((/(წამი|წუთი|საათი|დღე|თვე)/).test(s)) {
                    return s.replace(/(ი|ე)$/, 'ის წინ');
                }
                if ((/წელი/).test(s)) {
                    return s.replace(/წელი$/, 'წლის წინ');
                }
            },
            s : 'რამდენიმე წამი',
            ss : '%d წამი',
            m : 'წუთი',
            mm : '%d წუთი',
            h : 'საათი',
            hh : '%d საათი',
            d : 'დღე',
            dd : '%d დღე',
            M : 'თვე',
            MM : '%d თვე',
            y : 'წელი',
            yy : '%d წელი'
        },
        dayOfMonthOrdinalParse: /0|1-ლი|მე-\d{1,2}|\d{1,2}-ე/,
        ordinal : function (number) {
            if (number === 0) {
                return number;
            }
            if (number === 1) {
                return number + '-ლი';
            }
            if ((number < 20) || (number <= 100 && (number % 20 === 0)) || (number % 100 === 0)) {
                return 'მე-' + number;
            }
            return number + '-ე';
        },
        week : {
            dow : 1,
            doy : 7
        }
    });

    //! moment.js locale configuration

    var suffixes$1 = {
        0: '-ші',
        1: '-ші',
        2: '-ші',
        3: '-ші',
        4: '-ші',
        5: '-ші',
        6: '-шы',
        7: '-ші',
        8: '-ші',
        9: '-шы',
        10: '-шы',
        20: '-шы',
        30: '-шы',
        40: '-шы',
        50: '-ші',
        60: '-шы',
        70: '-ші',
        80: '-ші',
        90: '-шы',
        100: '-ші'
    };

    hooks.defineLocale('kk', {
        months : 'қаңтар_ақпан_наурыз_сәуір_мамыр_маусым_шілде_тамыз_қыркүйек_қазан_қараша_желтоқсан'.split('_'),
        monthsShort : 'қаң_ақп_нау_сәу_мам_мау_шіл_там_қыр_қаз_қар_жел'.split('_'),
        weekdays : 'жексенбі_дүйсенбі_сейсенбі_сәрсенбі_бейсенбі_жұма_сенбі'.split('_'),
        weekdaysShort : 'жек_дүй_сей_сәр_бей_жұм_сен'.split('_'),
        weekdaysMin : 'жк_дй_сй_ср_бй_жм_сн'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Бүгін сағат] LT',
            nextDay : '[Ертең сағат] LT',
            nextWeek : 'dddd [сағат] LT',
            lastDay : '[Кеше сағат] LT',
            lastWeek : '[Өткен аптаның] dddd [сағат] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s ішінде',
            past : '%s бұрын',
            s : 'бірнеше секунд',
            ss : '%d секунд',
            m : 'бір минут',
            mm : '%d минут',
            h : 'бір сағат',
            hh : '%d сағат',
            d : 'бір күн',
            dd : '%d күн',
            M : 'бір ай',
            MM : '%d ай',
            y : 'бір жыл',
            yy : '%d жыл'
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(ші|шы)/,
        ordinal : function (number) {
            var a = number % 10,
                b = number >= 100 ? 100 : null;
            return number + (suffixes$1[number] || suffixes$1[a] || suffixes$1[b]);
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$8 = {
        '1': '១',
        '2': '២',
        '3': '៣',
        '4': '៤',
        '5': '៥',
        '6': '៦',
        '7': '៧',
        '8': '៨',
        '9': '៩',
        '0': '០'
    }, numberMap$7 = {
        '១': '1',
        '២': '2',
        '៣': '3',
        '៤': '4',
        '៥': '5',
        '៦': '6',
        '៧': '7',
        '៨': '8',
        '៩': '9',
        '០': '0'
    };

    hooks.defineLocale('km', {
        months: 'មករា_កុម្ភៈ_មីនា_មេសា_ឧសភា_មិថុនា_កក្កដា_សីហា_កញ្ញា_តុលា_វិច្ឆិកា_ធ្នូ'.split(
            '_'
        ),
        monthsShort: 'មករា_កុម្ភៈ_មីនា_មេសា_ឧសភា_មិថុនា_កក្កដា_សីហា_កញ្ញា_តុលា_វិច្ឆិកា_ធ្នូ'.split(
            '_'
        ),
        weekdays: 'អាទិត្យ_ច័ន្ទ_អង្គារ_ពុធ_ព្រហស្បតិ៍_សុក្រ_សៅរ៍'.split('_'),
        weekdaysShort: 'អា_ច_អ_ព_ព្រ_សុ_ស'.split('_'),
        weekdaysMin: 'អា_ច_អ_ព_ព្រ_សុ_ស'.split('_'),
        weekdaysParseExact: true,
        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY HH:mm',
            LLLL: 'dddd, D MMMM YYYY HH:mm'
        },
        meridiemParse: /ព្រឹក|ល្ងាច/,
        isPM: function (input) {
            return input === 'ល្ងាច';
        },
        meridiem: function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ព្រឹក';
            } else {
                return 'ល្ងាច';
            }
        },
        calendar: {
            sameDay: '[ថ្ងៃនេះ ម៉ោង] LT',
            nextDay: '[ស្អែក ម៉ោង] LT',
            nextWeek: 'dddd [ម៉ោង] LT',
            lastDay: '[ម្សិលមិញ ម៉ោង] LT',
            lastWeek: 'dddd [សប្តាហ៍មុន] [ម៉ោង] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: '%sទៀត',
            past: '%sមុន',
            s: 'ប៉ុន្មានវិនាទី',
            ss: '%d វិនាទី',
            m: 'មួយនាទី',
            mm: '%d នាទី',
            h: 'មួយម៉ោង',
            hh: '%d ម៉ោង',
            d: 'មួយថ្ងៃ',
            dd: '%d ថ្ងៃ',
            M: 'មួយខែ',
            MM: '%d ខែ',
            y: 'មួយឆ្នាំ',
            yy: '%d ឆ្នាំ'
        },
        dayOfMonthOrdinalParse : /ទី\d{1,2}/,
        ordinal : 'ទី%d',
        preparse: function (string) {
            return string.replace(/[១២៣៤៥៦៧៨៩០]/g, function (match) {
                return numberMap$7[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$8[match];
            });
        },
        week: {
            dow: 1, // Monday is the first day of the week.
            doy: 4 // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$9 = {
        '1': '೧',
        '2': '೨',
        '3': '೩',
        '4': '೪',
        '5': '೫',
        '6': '೬',
        '7': '೭',
        '8': '೮',
        '9': '೯',
        '0': '೦'
    },
    numberMap$8 = {
        '೧': '1',
        '೨': '2',
        '೩': '3',
        '೪': '4',
        '೫': '5',
        '೬': '6',
        '೭': '7',
        '೮': '8',
        '೯': '9',
        '೦': '0'
    };

    hooks.defineLocale('kn', {
        months : 'ಜನವರಿ_ಫೆಬ್ರವರಿ_ಮಾರ್ಚ್_ಏಪ್ರಿಲ್_ಮೇ_ಜೂನ್_ಜುಲೈ_ಆಗಸ್ಟ್_ಸೆಪ್ಟೆಂಬರ್_ಅಕ್ಟೋಬರ್_ನವೆಂಬರ್_ಡಿಸೆಂಬರ್'.split('_'),
        monthsShort : 'ಜನ_ಫೆಬ್ರ_ಮಾರ್ಚ್_ಏಪ್ರಿಲ್_ಮೇ_ಜೂನ್_ಜುಲೈ_ಆಗಸ್ಟ್_ಸೆಪ್ಟೆಂ_ಅಕ್ಟೋ_ನವೆಂ_ಡಿಸೆಂ'.split('_'),
        monthsParseExact: true,
        weekdays : 'ಭಾನುವಾರ_ಸೋಮವಾರ_ಮಂಗಳವಾರ_ಬುಧವಾರ_ಗುರುವಾರ_ಶುಕ್ರವಾರ_ಶನಿವಾರ'.split('_'),
        weekdaysShort : 'ಭಾನು_ಸೋಮ_ಮಂಗಳ_ಬುಧ_ಗುರು_ಶುಕ್ರ_ಶನಿ'.split('_'),
        weekdaysMin : 'ಭಾ_ಸೋ_ಮಂ_ಬು_ಗು_ಶು_ಶ'.split('_'),
        longDateFormat : {
            LT : 'A h:mm',
            LTS : 'A h:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm',
            LLLL : 'dddd, D MMMM YYYY, A h:mm'
        },
        calendar : {
            sameDay : '[ಇಂದು] LT',
            nextDay : '[ನಾಳೆ] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[ನಿನ್ನೆ] LT',
            lastWeek : '[ಕೊನೆಯ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s ನಂತರ',
            past : '%s ಹಿಂದೆ',
            s : 'ಕೆಲವು ಕ್ಷಣಗಳು',
            ss : '%d ಸೆಕೆಂಡುಗಳು',
            m : 'ಒಂದು ನಿಮಿಷ',
            mm : '%d ನಿಮಿಷ',
            h : 'ಒಂದು ಗಂಟೆ',
            hh : '%d ಗಂಟೆ',
            d : 'ಒಂದು ದಿನ',
            dd : '%d ದಿನ',
            M : 'ಒಂದು ತಿಂಗಳು',
            MM : '%d ತಿಂಗಳು',
            y : 'ಒಂದು ವರ್ಷ',
            yy : '%d ವರ್ಷ'
        },
        preparse: function (string) {
            return string.replace(/[೧೨೩೪೫೬೭೮೯೦]/g, function (match) {
                return numberMap$8[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$9[match];
            });
        },
        meridiemParse: /ರಾತ್ರಿ|ಬೆಳಿಗ್ಗೆ|ಮಧ್ಯಾಹ್ನ|ಸಂಜೆ/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'ರಾತ್ರಿ') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'ಬೆಳಿಗ್ಗೆ') {
                return hour;
            } else if (meridiem === 'ಮಧ್ಯಾಹ್ನ') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'ಸಂಜೆ') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'ರಾತ್ರಿ';
            } else if (hour < 10) {
                return 'ಬೆಳಿಗ್ಗೆ';
            } else if (hour < 17) {
                return 'ಮಧ್ಯಾಹ್ನ';
            } else if (hour < 20) {
                return 'ಸಂಜೆ';
            } else {
                return 'ರಾತ್ರಿ';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}(ನೇ)/,
        ordinal : function (number) {
            return number + 'ನೇ';
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ko', {
        months : '1월_2월_3월_4월_5월_6월_7월_8월_9월_10월_11월_12월'.split('_'),
        monthsShort : '1월_2월_3월_4월_5월_6월_7월_8월_9월_10월_11월_12월'.split('_'),
        weekdays : '일요일_월요일_화요일_수요일_목요일_금요일_토요일'.split('_'),
        weekdaysShort : '일_월_화_수_목_금_토'.split('_'),
        weekdaysMin : '일_월_화_수_목_금_토'.split('_'),
        longDateFormat : {
            LT : 'A h:mm',
            LTS : 'A h:mm:ss',
            L : 'YYYY.MM.DD.',
            LL : 'YYYY년 MMMM D일',
            LLL : 'YYYY년 MMMM D일 A h:mm',
            LLLL : 'YYYY년 MMMM D일 dddd A h:mm',
            l : 'YYYY.MM.DD.',
            ll : 'YYYY년 MMMM D일',
            lll : 'YYYY년 MMMM D일 A h:mm',
            llll : 'YYYY년 MMMM D일 dddd A h:mm'
        },
        calendar : {
            sameDay : '오늘 LT',
            nextDay : '내일 LT',
            nextWeek : 'dddd LT',
            lastDay : '어제 LT',
            lastWeek : '지난주 dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s 후',
            past : '%s 전',
            s : '몇 초',
            ss : '%d초',
            m : '1분',
            mm : '%d분',
            h : '한 시간',
            hh : '%d시간',
            d : '하루',
            dd : '%d일',
            M : '한 달',
            MM : '%d달',
            y : '일 년',
            yy : '%d년'
        },
        dayOfMonthOrdinalParse : /\d{1,2}(일|월|주)/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'DDD':
                    return number + '일';
                case 'M':
                    return number + '월';
                case 'w':
                case 'W':
                    return number + '주';
                default:
                    return number;
            }
        },
        meridiemParse : /오전|오후/,
        isPM : function (token) {
            return token === '오후';
        },
        meridiem : function (hour, minute, isUpper) {
            return hour < 12 ? '오전' : '오후';
        }
    });

    //! moment.js locale configuration

    var symbolMap$a = {
        '1': '١',
        '2': '٢',
        '3': '٣',
        '4': '٤',
        '5': '٥',
        '6': '٦',
        '7': '٧',
        '8': '٨',
        '9': '٩',
        '0': '٠'
    }, numberMap$9 = {
        '١': '1',
        '٢': '2',
        '٣': '3',
        '٤': '4',
        '٥': '5',
        '٦': '6',
        '٧': '7',
        '٨': '8',
        '٩': '9',
        '٠': '0'
    },
    months$7 = [
        'کانونی دووەم',
        'شوبات',
        'ئازار',
        'نیسان',
        'ئایار',
        'حوزەیران',
        'تەمموز',
        'ئاب',
        'ئەیلوول',
        'تشرینی یەكەم',
        'تشرینی دووەم',
        'كانونی یەکەم'
    ];


    hooks.defineLocale('ku', {
        months : months$7,
        monthsShort : months$7,
        weekdays : 'یه‌كشه‌ممه‌_دووشه‌ممه‌_سێشه‌ممه‌_چوارشه‌ممه‌_پێنجشه‌ممه‌_هه‌ینی_شه‌ممه‌'.split('_'),
        weekdaysShort : 'یه‌كشه‌م_دووشه‌م_سێشه‌م_چوارشه‌م_پێنجشه‌م_هه‌ینی_شه‌ممه‌'.split('_'),
        weekdaysMin : 'ی_د_س_چ_پ_ه_ش'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        meridiemParse: /ئێواره‌|به‌یانی/,
        isPM: function (input) {
            return /ئێواره‌/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'به‌یانی';
            } else {
                return 'ئێواره‌';
            }
        },
        calendar : {
            sameDay : '[ئه‌مرۆ كاتژمێر] LT',
            nextDay : '[به‌یانی كاتژمێر] LT',
            nextWeek : 'dddd [كاتژمێر] LT',
            lastDay : '[دوێنێ كاتژمێر] LT',
            lastWeek : 'dddd [كاتژمێر] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'له‌ %s',
            past : '%s',
            s : 'چه‌ند چركه‌یه‌ك',
            ss : 'چركه‌ %d',
            m : 'یه‌ك خوله‌ك',
            mm : '%d خوله‌ك',
            h : 'یه‌ك كاتژمێر',
            hh : '%d كاتژمێر',
            d : 'یه‌ك ڕۆژ',
            dd : '%d ڕۆژ',
            M : 'یه‌ك مانگ',
            MM : '%d مانگ',
            y : 'یه‌ك ساڵ',
            yy : '%d ساڵ'
        },
        preparse: function (string) {
            return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
                return numberMap$9[match];
            }).replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$a[match];
            }).replace(/,/g, '،');
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12 // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var suffixes$2 = {
        0: '-чү',
        1: '-чи',
        2: '-чи',
        3: '-чү',
        4: '-чү',
        5: '-чи',
        6: '-чы',
        7: '-чи',
        8: '-чи',
        9: '-чу',
        10: '-чу',
        20: '-чы',
        30: '-чу',
        40: '-чы',
        50: '-чү',
        60: '-чы',
        70: '-чи',
        80: '-чи',
        90: '-чу',
        100: '-чү'
    };

    hooks.defineLocale('ky', {
        months : 'январь_февраль_март_апрель_май_июнь_июль_август_сентябрь_октябрь_ноябрь_декабрь'.split('_'),
        monthsShort : 'янв_фев_март_апр_май_июнь_июль_авг_сен_окт_ноя_дек'.split('_'),
        weekdays : 'Жекшемби_Дүйшөмбү_Шейшемби_Шаршемби_Бейшемби_Жума_Ишемби'.split('_'),
        weekdaysShort : 'Жек_Дүй_Шей_Шар_Бей_Жум_Ише'.split('_'),
        weekdaysMin : 'Жк_Дй_Шй_Шр_Бй_Жм_Иш'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Бүгүн саат] LT',
            nextDay : '[Эртең саат] LT',
            nextWeek : 'dddd [саат] LT',
            lastDay : '[Кечээ саат] LT',
            lastWeek : '[Өткөн аптанын] dddd [күнү] [саат] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s ичинде',
            past : '%s мурун',
            s : 'бирнече секунд',
            ss : '%d секунд',
            m : 'бир мүнөт',
            mm : '%d мүнөт',
            h : 'бир саат',
            hh : '%d саат',
            d : 'бир күн',
            dd : '%d күн',
            M : 'бир ай',
            MM : '%d ай',
            y : 'бир жыл',
            yy : '%d жыл'
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(чи|чы|чү|чу)/,
        ordinal : function (number) {
            var a = number % 10,
                b = number >= 100 ? 100 : null;
            return number + (suffixes$2[number] || suffixes$2[a] || suffixes$2[b]);
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$5(number, withoutSuffix, key, isFuture) {
        var format = {
            'm': ['eng Minutt', 'enger Minutt'],
            'h': ['eng Stonn', 'enger Stonn'],
            'd': ['een Dag', 'engem Dag'],
            'M': ['ee Mount', 'engem Mount'],
            'y': ['ee Joer', 'engem Joer']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }
    function processFutureTime(string) {
        var number = string.substr(0, string.indexOf(' '));
        if (eifelerRegelAppliesToNumber(number)) {
            return 'a ' + string;
        }
        return 'an ' + string;
    }
    function processPastTime(string) {
        var number = string.substr(0, string.indexOf(' '));
        if (eifelerRegelAppliesToNumber(number)) {
            return 'viru ' + string;
        }
        return 'virun ' + string;
    }
    /**
     * Returns true if the word before the given number loses the '-n' ending.
     * e.g. 'an 10 Deeg' but 'a 5 Deeg'
     *
     * @param number {integer}
     * @returns {boolean}
     */
    function eifelerRegelAppliesToNumber(number) {
        number = parseInt(number, 10);
        if (isNaN(number)) {
            return false;
        }
        if (number < 0) {
            // Negative Number --> always true
            return true;
        } else if (number < 10) {
            // Only 1 digit
            if (4 <= number && number <= 7) {
                return true;
            }
            return false;
        } else if (number < 100) {
            // 2 digits
            var lastDigit = number % 10, firstDigit = number / 10;
            if (lastDigit === 0) {
                return eifelerRegelAppliesToNumber(firstDigit);
            }
            return eifelerRegelAppliesToNumber(lastDigit);
        } else if (number < 10000) {
            // 3 or 4 digits --> recursively check first digit
            while (number >= 10) {
                number = number / 10;
            }
            return eifelerRegelAppliesToNumber(number);
        } else {
            // Anything larger than 4 digits: recursively check first n-3 digits
            number = number / 1000;
            return eifelerRegelAppliesToNumber(number);
        }
    }

    hooks.defineLocale('lb', {
        months: 'Januar_Februar_Mäerz_Abrëll_Mee_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
        monthsShort: 'Jan._Febr._Mrz._Abr._Mee_Jun._Jul._Aug._Sept._Okt._Nov._Dez.'.split('_'),
        monthsParseExact : true,
        weekdays: 'Sonndeg_Méindeg_Dënschdeg_Mëttwoch_Donneschdeg_Freideg_Samschdeg'.split('_'),
        weekdaysShort: 'So._Mé._Dë._Më._Do._Fr._Sa.'.split('_'),
        weekdaysMin: 'So_Mé_Dë_Më_Do_Fr_Sa'.split('_'),
        weekdaysParseExact : true,
        longDateFormat: {
            LT: 'H:mm [Auer]',
            LTS: 'H:mm:ss [Auer]',
            L: 'DD.MM.YYYY',
            LL: 'D. MMMM YYYY',
            LLL: 'D. MMMM YYYY H:mm [Auer]',
            LLLL: 'dddd, D. MMMM YYYY H:mm [Auer]'
        },
        calendar: {
            sameDay: '[Haut um] LT',
            sameElse: 'L',
            nextDay: '[Muer um] LT',
            nextWeek: 'dddd [um] LT',
            lastDay: '[Gëschter um] LT',
            lastWeek: function () {
                // Different date string for 'Dënschdeg' (Tuesday) and 'Donneschdeg' (Thursday) due to phonological rule
                switch (this.day()) {
                    case 2:
                    case 4:
                        return '[Leschten] dddd [um] LT';
                    default:
                        return '[Leschte] dddd [um] LT';
                }
            }
        },
        relativeTime : {
            future : processFutureTime,
            past : processPastTime,
            s : 'e puer Sekonnen',
            ss : '%d Sekonnen',
            m : processRelativeTime$5,
            mm : '%d Minutten',
            h : processRelativeTime$5,
            hh : '%d Stonnen',
            d : processRelativeTime$5,
            dd : '%d Deeg',
            M : processRelativeTime$5,
            MM : '%d Méint',
            y : processRelativeTime$5,
            yy : '%d Joer'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal: '%d.',
        week: {
            dow: 1, // Monday is the first day of the week.
            doy: 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('lo', {
        months : 'ມັງກອນ_ກຸມພາ_ມີນາ_ເມສາ_ພຶດສະພາ_ມິຖຸນາ_ກໍລະກົດ_ສິງຫາ_ກັນຍາ_ຕຸລາ_ພະຈິກ_ທັນວາ'.split('_'),
        monthsShort : 'ມັງກອນ_ກຸມພາ_ມີນາ_ເມສາ_ພຶດສະພາ_ມິຖຸນາ_ກໍລະກົດ_ສິງຫາ_ກັນຍາ_ຕຸລາ_ພະຈິກ_ທັນວາ'.split('_'),
        weekdays : 'ອາທິດ_ຈັນ_ອັງຄານ_ພຸດ_ພະຫັດ_ສຸກ_ເສົາ'.split('_'),
        weekdaysShort : 'ທິດ_ຈັນ_ອັງຄານ_ພຸດ_ພະຫັດ_ສຸກ_ເສົາ'.split('_'),
        weekdaysMin : 'ທ_ຈ_ອຄ_ພ_ພຫ_ສກ_ສ'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'ວັນdddd D MMMM YYYY HH:mm'
        },
        meridiemParse: /ຕອນເຊົ້າ|ຕອນແລງ/,
        isPM: function (input) {
            return input === 'ຕອນແລງ';
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ຕອນເຊົ້າ';
            } else {
                return 'ຕອນແລງ';
            }
        },
        calendar : {
            sameDay : '[ມື້ນີ້ເວລາ] LT',
            nextDay : '[ມື້ອື່ນເວລາ] LT',
            nextWeek : '[ວັນ]dddd[ໜ້າເວລາ] LT',
            lastDay : '[ມື້ວານນີ້ເວລາ] LT',
            lastWeek : '[ວັນ]dddd[ແລ້ວນີ້ເວລາ] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'ອີກ %s',
            past : '%sຜ່ານມາ',
            s : 'ບໍ່ເທົ່າໃດວິນາທີ',
            ss : '%d ວິນາທີ' ,
            m : '1 ນາທີ',
            mm : '%d ນາທີ',
            h : '1 ຊົ່ວໂມງ',
            hh : '%d ຊົ່ວໂມງ',
            d : '1 ມື້',
            dd : '%d ມື້',
            M : '1 ເດືອນ',
            MM : '%d ເດືອນ',
            y : '1 ປີ',
            yy : '%d ປີ'
        },
        dayOfMonthOrdinalParse: /(ທີ່)\d{1,2}/,
        ordinal : function (number) {
            return 'ທີ່' + number;
        }
    });

    //! moment.js locale configuration

    var units = {
        'ss' : 'sekundė_sekundžių_sekundes',
        'm' : 'minutė_minutės_minutę',
        'mm': 'minutės_minučių_minutes',
        'h' : 'valanda_valandos_valandą',
        'hh': 'valandos_valandų_valandas',
        'd' : 'diena_dienos_dieną',
        'dd': 'dienos_dienų_dienas',
        'M' : 'mėnuo_mėnesio_mėnesį',
        'MM': 'mėnesiai_mėnesių_mėnesius',
        'y' : 'metai_metų_metus',
        'yy': 'metai_metų_metus'
    };
    function translateSeconds(number, withoutSuffix, key, isFuture) {
        if (withoutSuffix) {
            return 'kelios sekundės';
        } else {
            return isFuture ? 'kelių sekundžių' : 'kelias sekundes';
        }
    }
    function translateSingular(number, withoutSuffix, key, isFuture) {
        return withoutSuffix ? forms(key)[0] : (isFuture ? forms(key)[1] : forms(key)[2]);
    }
    function special(number) {
        return number % 10 === 0 || (number > 10 && number < 20);
    }
    function forms(key) {
        return units[key].split('_');
    }
    function translate$6(number, withoutSuffix, key, isFuture) {
        var result = number + ' ';
        if (number === 1) {
            return result + translateSingular(number, withoutSuffix, key[0], isFuture);
        } else if (withoutSuffix) {
            return result + (special(number) ? forms(key)[1] : forms(key)[0]);
        } else {
            if (isFuture) {
                return result + forms(key)[1];
            } else {
                return result + (special(number) ? forms(key)[1] : forms(key)[2]);
            }
        }
    }
    hooks.defineLocale('lt', {
        months : {
            format: 'sausio_vasario_kovo_balandžio_gegužės_birželio_liepos_rugpjūčio_rugsėjo_spalio_lapkričio_gruodžio'.split('_'),
            standalone: 'sausis_vasaris_kovas_balandis_gegužė_birželis_liepa_rugpjūtis_rugsėjis_spalis_lapkritis_gruodis'.split('_'),
            isFormat: /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?|MMMM?(\[[^\[\]]*\]|\s)+D[oD]?/
        },
        monthsShort : 'sau_vas_kov_bal_geg_bir_lie_rgp_rgs_spa_lap_grd'.split('_'),
        weekdays : {
            format: 'sekmadienį_pirmadienį_antradienį_trečiadienį_ketvirtadienį_penktadienį_šeštadienį'.split('_'),
            standalone: 'sekmadienis_pirmadienis_antradienis_trečiadienis_ketvirtadienis_penktadienis_šeštadienis'.split('_'),
            isFormat: /dddd HH:mm/
        },
        weekdaysShort : 'Sek_Pir_Ant_Tre_Ket_Pen_Šeš'.split('_'),
        weekdaysMin : 'S_P_A_T_K_Pn_Š'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'YYYY [m.] MMMM D [d.]',
            LLL : 'YYYY [m.] MMMM D [d.], HH:mm [val.]',
            LLLL : 'YYYY [m.] MMMM D [d.], dddd, HH:mm [val.]',
            l : 'YYYY-MM-DD',
            ll : 'YYYY [m.] MMMM D [d.]',
            lll : 'YYYY [m.] MMMM D [d.], HH:mm [val.]',
            llll : 'YYYY [m.] MMMM D [d.], ddd, HH:mm [val.]'
        },
        calendar : {
            sameDay : '[Šiandien] LT',
            nextDay : '[Rytoj] LT',
            nextWeek : 'dddd LT',
            lastDay : '[Vakar] LT',
            lastWeek : '[Praėjusį] dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'po %s',
            past : 'prieš %s',
            s : translateSeconds,
            ss : translate$6,
            m : translateSingular,
            mm : translate$6,
            h : translateSingular,
            hh : translate$6,
            d : translateSingular,
            dd : translate$6,
            M : translateSingular,
            MM : translate$6,
            y : translateSingular,
            yy : translate$6
        },
        dayOfMonthOrdinalParse: /\d{1,2}-oji/,
        ordinal : function (number) {
            return number + '-oji';
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var units$1 = {
        'ss': 'sekundes_sekundēm_sekunde_sekundes'.split('_'),
        'm': 'minūtes_minūtēm_minūte_minūtes'.split('_'),
        'mm': 'minūtes_minūtēm_minūte_minūtes'.split('_'),
        'h': 'stundas_stundām_stunda_stundas'.split('_'),
        'hh': 'stundas_stundām_stunda_stundas'.split('_'),
        'd': 'dienas_dienām_diena_dienas'.split('_'),
        'dd': 'dienas_dienām_diena_dienas'.split('_'),
        'M': 'mēneša_mēnešiem_mēnesis_mēneši'.split('_'),
        'MM': 'mēneša_mēnešiem_mēnesis_mēneši'.split('_'),
        'y': 'gada_gadiem_gads_gadi'.split('_'),
        'yy': 'gada_gadiem_gads_gadi'.split('_')
    };
    /**
     * @param withoutSuffix boolean true = a length of time; false = before/after a period of time.
     */
    function format$1(forms, number, withoutSuffix) {
        if (withoutSuffix) {
            // E.g. "21 minūte", "3 minūtes".
            return number % 10 === 1 && number % 100 !== 11 ? forms[2] : forms[3];
        } else {
            // E.g. "21 minūtes" as in "pēc 21 minūtes".
            // E.g. "3 minūtēm" as in "pēc 3 minūtēm".
            return number % 10 === 1 && number % 100 !== 11 ? forms[0] : forms[1];
        }
    }
    function relativeTimeWithPlural$1(number, withoutSuffix, key) {
        return number + ' ' + format$1(units$1[key], number, withoutSuffix);
    }
    function relativeTimeWithSingular(number, withoutSuffix, key) {
        return format$1(units$1[key], number, withoutSuffix);
    }
    function relativeSeconds(number, withoutSuffix) {
        return withoutSuffix ? 'dažas sekundes' : 'dažām sekundēm';
    }

    hooks.defineLocale('lv', {
        months : 'janvāris_februāris_marts_aprīlis_maijs_jūnijs_jūlijs_augusts_septembris_oktobris_novembris_decembris'.split('_'),
        monthsShort : 'jan_feb_mar_apr_mai_jūn_jūl_aug_sep_okt_nov_dec'.split('_'),
        weekdays : 'svētdiena_pirmdiena_otrdiena_trešdiena_ceturtdiena_piektdiena_sestdiena'.split('_'),
        weekdaysShort : 'Sv_P_O_T_C_Pk_S'.split('_'),
        weekdaysMin : 'Sv_P_O_T_C_Pk_S'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY.',
            LL : 'YYYY. [gada] D. MMMM',
            LLL : 'YYYY. [gada] D. MMMM, HH:mm',
            LLLL : 'YYYY. [gada] D. MMMM, dddd, HH:mm'
        },
        calendar : {
            sameDay : '[Šodien pulksten] LT',
            nextDay : '[Rīt pulksten] LT',
            nextWeek : 'dddd [pulksten] LT',
            lastDay : '[Vakar pulksten] LT',
            lastWeek : '[Pagājušā] dddd [pulksten] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'pēc %s',
            past : 'pirms %s',
            s : relativeSeconds,
            ss : relativeTimeWithPlural$1,
            m : relativeTimeWithSingular,
            mm : relativeTimeWithPlural$1,
            h : relativeTimeWithSingular,
            hh : relativeTimeWithPlural$1,
            d : relativeTimeWithSingular,
            dd : relativeTimeWithPlural$1,
            M : relativeTimeWithSingular,
            MM : relativeTimeWithPlural$1,
            y : relativeTimeWithSingular,
            yy : relativeTimeWithPlural$1
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var translator = {
        words: { //Different grammatical cases
            ss: ['sekund', 'sekunda', 'sekundi'],
            m: ['jedan minut', 'jednog minuta'],
            mm: ['minut', 'minuta', 'minuta'],
            h: ['jedan sat', 'jednog sata'],
            hh: ['sat', 'sata', 'sati'],
            dd: ['dan', 'dana', 'dana'],
            MM: ['mjesec', 'mjeseca', 'mjeseci'],
            yy: ['godina', 'godine', 'godina']
        },
        correctGrammaticalCase: function (number, wordKey) {
            return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
        },
        translate: function (number, withoutSuffix, key) {
            var wordKey = translator.words[key];
            if (key.length === 1) {
                return withoutSuffix ? wordKey[0] : wordKey[1];
            } else {
                return number + ' ' + translator.correctGrammaticalCase(number, wordKey);
            }
        }
    };

    hooks.defineLocale('me', {
        months: 'januar_februar_mart_april_maj_jun_jul_avgust_septembar_oktobar_novembar_decembar'.split('_'),
        monthsShort: 'jan._feb._mar._apr._maj_jun_jul_avg._sep._okt._nov._dec.'.split('_'),
        monthsParseExact : true,
        weekdays: 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
        weekdaysShort: 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
        weekdaysMin: 'ne_po_ut_sr_če_pe_su'.split('_'),
        weekdaysParseExact : true,
        longDateFormat: {
            LT: 'H:mm',
            LTS : 'H:mm:ss',
            L: 'DD.MM.YYYY',
            LL: 'D. MMMM YYYY',
            LLL: 'D. MMMM YYYY H:mm',
            LLLL: 'dddd, D. MMMM YYYY H:mm'
        },
        calendar: {
            sameDay: '[danas u] LT',
            nextDay: '[sjutra u] LT',

            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[u] [nedjelju] [u] LT';
                    case 3:
                        return '[u] [srijedu] [u] LT';
                    case 6:
                        return '[u] [subotu] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[juče u] LT',
            lastWeek : function () {
                var lastWeekDays = [
                    '[prošle] [nedjelje] [u] LT',
                    '[prošlog] [ponedjeljka] [u] LT',
                    '[prošlog] [utorka] [u] LT',
                    '[prošle] [srijede] [u] LT',
                    '[prošlog] [četvrtka] [u] LT',
                    '[prošlog] [petka] [u] LT',
                    '[prošle] [subote] [u] LT'
                ];
                return lastWeekDays[this.day()];
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'za %s',
            past   : 'prije %s',
            s      : 'nekoliko sekundi',
            ss     : translator.translate,
            m      : translator.translate,
            mm     : translator.translate,
            h      : translator.translate,
            hh     : translator.translate,
            d      : 'dan',
            dd     : translator.translate,
            M      : 'mjesec',
            MM     : translator.translate,
            y      : 'godinu',
            yy     : translator.translate
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('mi', {
        months: 'Kohi-tāte_Hui-tanguru_Poutū-te-rangi_Paenga-whāwhā_Haratua_Pipiri_Hōngoingoi_Here-turi-kōkā_Mahuru_Whiringa-ā-nuku_Whiringa-ā-rangi_Hakihea'.split('_'),
        monthsShort: 'Kohi_Hui_Pou_Pae_Hara_Pipi_Hōngoi_Here_Mahu_Whi-nu_Whi-ra_Haki'.split('_'),
        monthsRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
        monthsStrictRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
        monthsShortRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
        monthsShortStrictRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,2}/i,
        weekdays: 'Rātapu_Mane_Tūrei_Wenerei_Tāite_Paraire_Hātarei'.split('_'),
        weekdaysShort: 'Ta_Ma_Tū_We_Tāi_Pa_Hā'.split('_'),
        weekdaysMin: 'Ta_Ma_Tū_We_Tāi_Pa_Hā'.split('_'),
        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY [i] HH:mm',
            LLLL: 'dddd, D MMMM YYYY [i] HH:mm'
        },
        calendar: {
            sameDay: '[i teie mahana, i] LT',
            nextDay: '[apopo i] LT',
            nextWeek: 'dddd [i] LT',
            lastDay: '[inanahi i] LT',
            lastWeek: 'dddd [whakamutunga i] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: 'i roto i %s',
            past: '%s i mua',
            s: 'te hēkona ruarua',
            ss: '%d hēkona',
            m: 'he meneti',
            mm: '%d meneti',
            h: 'te haora',
            hh: '%d haora',
            d: 'he ra',
            dd: '%d ra',
            M: 'he marama',
            MM: '%d marama',
            y: 'he tau',
            yy: '%d tau'
        },
        dayOfMonthOrdinalParse: /\d{1,2}º/,
        ordinal: '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('mk', {
        months : 'јануари_февруари_март_април_мај_јуни_јули_август_септември_октомври_ноември_декември'.split('_'),
        monthsShort : 'јан_фев_мар_апр_мај_јун_јул_авг_сеп_окт_ное_дек'.split('_'),
        weekdays : 'недела_понеделник_вторник_среда_четврток_петок_сабота'.split('_'),
        weekdaysShort : 'нед_пон_вто_сре_чет_пет_саб'.split('_'),
        weekdaysMin : 'нe_пo_вт_ср_че_пе_сa'.split('_'),
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'D.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY H:mm',
            LLLL : 'dddd, D MMMM YYYY H:mm'
        },
        calendar : {
            sameDay : '[Денес во] LT',
            nextDay : '[Утре во] LT',
            nextWeek : '[Во] dddd [во] LT',
            lastDay : '[Вчера во] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                    case 6:
                        return '[Изминатата] dddd [во] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[Изминатиот] dddd [во] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'после %s',
            past : 'пред %s',
            s : 'неколку секунди',
            ss : '%d секунди',
            m : 'минута',
            mm : '%d минути',
            h : 'час',
            hh : '%d часа',
            d : 'ден',
            dd : '%d дена',
            M : 'месец',
            MM : '%d месеци',
            y : 'година',
            yy : '%d години'
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(ев|ен|ти|ви|ри|ми)/,
        ordinal : function (number) {
            var lastDigit = number % 10,
                last2Digits = number % 100;
            if (number === 0) {
                return number + '-ев';
            } else if (last2Digits === 0) {
                return number + '-ен';
            } else if (last2Digits > 10 && last2Digits < 20) {
                return number + '-ти';
            } else if (lastDigit === 1) {
                return number + '-ви';
            } else if (lastDigit === 2) {
                return number + '-ри';
            } else if (lastDigit === 7 || lastDigit === 8) {
                return number + '-ми';
            } else {
                return number + '-ти';
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ml', {
        months : 'ജനുവരി_ഫെബ്രുവരി_മാർച്ച്_ഏപ്രിൽ_മേയ്_ജൂൺ_ജൂലൈ_ഓഗസ്റ്റ്_സെപ്റ്റംബർ_ഒക്ടോബർ_നവംബർ_ഡിസംബർ'.split('_'),
        monthsShort : 'ജനു._ഫെബ്രു._മാർ._ഏപ്രി._മേയ്_ജൂൺ_ജൂലൈ._ഓഗ._സെപ്റ്റ._ഒക്ടോ._നവം._ഡിസം.'.split('_'),
        monthsParseExact : true,
        weekdays : 'ഞായറാഴ്ച_തിങ്കളാഴ്ച_ചൊവ്വാഴ്ച_ബുധനാഴ്ച_വ്യാഴാഴ്ച_വെള്ളിയാഴ്ച_ശനിയാഴ്ച'.split('_'),
        weekdaysShort : 'ഞായർ_തിങ്കൾ_ചൊവ്വ_ബുധൻ_വ്യാഴം_വെള്ളി_ശനി'.split('_'),
        weekdaysMin : 'ഞാ_തി_ചൊ_ബു_വ്യാ_വെ_ശ'.split('_'),
        longDateFormat : {
            LT : 'A h:mm -നു',
            LTS : 'A h:mm:ss -നു',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm -നു',
            LLLL : 'dddd, D MMMM YYYY, A h:mm -നു'
        },
        calendar : {
            sameDay : '[ഇന്ന്] LT',
            nextDay : '[നാളെ] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[ഇന്നലെ] LT',
            lastWeek : '[കഴിഞ്ഞ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s കഴിഞ്ഞ്',
            past : '%s മുൻപ്',
            s : 'അൽപ നിമിഷങ്ങൾ',
            ss : '%d സെക്കൻഡ്',
            m : 'ഒരു മിനിറ്റ്',
            mm : '%d മിനിറ്റ്',
            h : 'ഒരു മണിക്കൂർ',
            hh : '%d മണിക്കൂർ',
            d : 'ഒരു ദിവസം',
            dd : '%d ദിവസം',
            M : 'ഒരു മാസം',
            MM : '%d മാസം',
            y : 'ഒരു വർഷം',
            yy : '%d വർഷം'
        },
        meridiemParse: /രാത്രി|രാവിലെ|ഉച്ച കഴിഞ്ഞ്|വൈകുന്നേരം|രാത്രി/i,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if ((meridiem === 'രാത്രി' && hour >= 4) ||
                    meridiem === 'ഉച്ച കഴിഞ്ഞ്' ||
                    meridiem === 'വൈകുന്നേരം') {
                return hour + 12;
            } else {
                return hour;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'രാത്രി';
            } else if (hour < 12) {
                return 'രാവിലെ';
            } else if (hour < 17) {
                return 'ഉച്ച കഴിഞ്ഞ്';
            } else if (hour < 20) {
                return 'വൈകുന്നേരം';
            } else {
                return 'രാത്രി';
            }
        }
    });

    //! moment.js locale configuration

    function translate$7(number, withoutSuffix, key, isFuture) {
        switch (key) {
            case 's':
                return withoutSuffix ? 'хэдхэн секунд' : 'хэдхэн секундын';
            case 'ss':
                return number + (withoutSuffix ? ' секунд' : ' секундын');
            case 'm':
            case 'mm':
                return number + (withoutSuffix ? ' минут' : ' минутын');
            case 'h':
            case 'hh':
                return number + (withoutSuffix ? ' цаг' : ' цагийн');
            case 'd':
            case 'dd':
                return number + (withoutSuffix ? ' өдөр' : ' өдрийн');
            case 'M':
            case 'MM':
                return number + (withoutSuffix ? ' сар' : ' сарын');
            case 'y':
            case 'yy':
                return number + (withoutSuffix ? ' жил' : ' жилийн');
            default:
                return number;
        }
    }

    hooks.defineLocale('mn', {
        months : 'Нэгдүгээр сар_Хоёрдугаар сар_Гуравдугаар сар_Дөрөвдүгээр сар_Тавдугаар сар_Зургадугаар сар_Долдугаар сар_Наймдугаар сар_Есдүгээр сар_Аравдугаар сар_Арван нэгдүгээр сар_Арван хоёрдугаар сар'.split('_'),
        monthsShort : '1 сар_2 сар_3 сар_4 сар_5 сар_6 сар_7 сар_8 сар_9 сар_10 сар_11 сар_12 сар'.split('_'),
        monthsParseExact : true,
        weekdays : 'Ням_Даваа_Мягмар_Лхагва_Пүрэв_Баасан_Бямба'.split('_'),
        weekdaysShort : 'Ням_Дав_Мяг_Лха_Пүр_Баа_Бям'.split('_'),
        weekdaysMin : 'Ня_Да_Мя_Лх_Пү_Ба_Бя'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'YYYY оны MMMMын D',
            LLL : 'YYYY оны MMMMын D HH:mm',
            LLLL : 'dddd, YYYY оны MMMMын D HH:mm'
        },
        meridiemParse: /ҮӨ|ҮХ/i,
        isPM : function (input) {
            return input === 'ҮХ';
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ҮӨ';
            } else {
                return 'ҮХ';
            }
        },
        calendar : {
            sameDay : '[Өнөөдөр] LT',
            nextDay : '[Маргааш] LT',
            nextWeek : '[Ирэх] dddd LT',
            lastDay : '[Өчигдөр] LT',
            lastWeek : '[Өнгөрсөн] dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s дараа',
            past : '%s өмнө',
            s : translate$7,
            ss : translate$7,
            m : translate$7,
            mm : translate$7,
            h : translate$7,
            hh : translate$7,
            d : translate$7,
            dd : translate$7,
            M : translate$7,
            MM : translate$7,
            y : translate$7,
            yy : translate$7
        },
        dayOfMonthOrdinalParse: /\d{1,2} өдөр/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'DDD':
                    return number + ' өдөр';
                default:
                    return number;
            }
        }
    });

    //! moment.js locale configuration

    var symbolMap$b = {
        '1': '१',
        '2': '२',
        '3': '३',
        '4': '४',
        '5': '५',
        '6': '६',
        '7': '७',
        '8': '८',
        '9': '९',
        '0': '०'
    },
    numberMap$a = {
        '१': '1',
        '२': '2',
        '३': '3',
        '४': '4',
        '५': '5',
        '६': '6',
        '७': '7',
        '८': '8',
        '९': '9',
        '०': '0'
    };

    function relativeTimeMr(number, withoutSuffix, string, isFuture)
    {
        var output = '';
        if (withoutSuffix) {
            switch (string) {
                case 's': output = 'काही सेकंद'; break;
                case 'ss': output = '%d सेकंद'; break;
                case 'm': output = 'एक मिनिट'; break;
                case 'mm': output = '%d मिनिटे'; break;
                case 'h': output = 'एक तास'; break;
                case 'hh': output = '%d तास'; break;
                case 'd': output = 'एक दिवस'; break;
                case 'dd': output = '%d दिवस'; break;
                case 'M': output = 'एक महिना'; break;
                case 'MM': output = '%d महिने'; break;
                case 'y': output = 'एक वर्ष'; break;
                case 'yy': output = '%d वर्षे'; break;
            }
        }
        else {
            switch (string) {
                case 's': output = 'काही सेकंदां'; break;
                case 'ss': output = '%d सेकंदां'; break;
                case 'm': output = 'एका मिनिटा'; break;
                case 'mm': output = '%d मिनिटां'; break;
                case 'h': output = 'एका तासा'; break;
                case 'hh': output = '%d तासां'; break;
                case 'd': output = 'एका दिवसा'; break;
                case 'dd': output = '%d दिवसां'; break;
                case 'M': output = 'एका महिन्या'; break;
                case 'MM': output = '%d महिन्यां'; break;
                case 'y': output = 'एका वर्षा'; break;
                case 'yy': output = '%d वर्षां'; break;
            }
        }
        return output.replace(/%d/i, number);
    }

    hooks.defineLocale('mr', {
        months : 'जानेवारी_फेब्रुवारी_मार्च_एप्रिल_मे_जून_जुलै_ऑगस्ट_सप्टेंबर_ऑक्टोबर_नोव्हेंबर_डिसेंबर'.split('_'),
        monthsShort: 'जाने._फेब्रु._मार्च._एप्रि._मे._जून._जुलै._ऑग._सप्टें._ऑक्टो._नोव्हें._डिसें.'.split('_'),
        monthsParseExact : true,
        weekdays : 'रविवार_सोमवार_मंगळवार_बुधवार_गुरूवार_शुक्रवार_शनिवार'.split('_'),
        weekdaysShort : 'रवि_सोम_मंगळ_बुध_गुरू_शुक्र_शनि'.split('_'),
        weekdaysMin : 'र_सो_मं_बु_गु_शु_श'.split('_'),
        longDateFormat : {
            LT : 'A h:mm वाजता',
            LTS : 'A h:mm:ss वाजता',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm वाजता',
            LLLL : 'dddd, D MMMM YYYY, A h:mm वाजता'
        },
        calendar : {
            sameDay : '[आज] LT',
            nextDay : '[उद्या] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[काल] LT',
            lastWeek: '[मागील] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future: '%sमध्ये',
            past: '%sपूर्वी',
            s: relativeTimeMr,
            ss: relativeTimeMr,
            m: relativeTimeMr,
            mm: relativeTimeMr,
            h: relativeTimeMr,
            hh: relativeTimeMr,
            d: relativeTimeMr,
            dd: relativeTimeMr,
            M: relativeTimeMr,
            MM: relativeTimeMr,
            y: relativeTimeMr,
            yy: relativeTimeMr
        },
        preparse: function (string) {
            return string.replace(/[१२३४५६७८९०]/g, function (match) {
                return numberMap$a[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$b[match];
            });
        },
        meridiemParse: /रात्री|सकाळी|दुपारी|सायंकाळी/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'रात्री') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'सकाळी') {
                return hour;
            } else if (meridiem === 'दुपारी') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'सायंकाळी') {
                return hour + 12;
            }
        },
        meridiem: function (hour, minute, isLower) {
            if (hour < 4) {
                return 'रात्री';
            } else if (hour < 10) {
                return 'सकाळी';
            } else if (hour < 17) {
                return 'दुपारी';
            } else if (hour < 20) {
                return 'सायंकाळी';
            } else {
                return 'रात्री';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ms-my', {
        months : 'Januari_Februari_Mac_April_Mei_Jun_Julai_Ogos_September_Oktober_November_Disember'.split('_'),
        monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ogs_Sep_Okt_Nov_Dis'.split('_'),
        weekdays : 'Ahad_Isnin_Selasa_Rabu_Khamis_Jumaat_Sabtu'.split('_'),
        weekdaysShort : 'Ahd_Isn_Sel_Rab_Kha_Jum_Sab'.split('_'),
        weekdaysMin : 'Ah_Is_Sl_Rb_Km_Jm_Sb'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY [pukul] HH.mm',
            LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
        },
        meridiemParse: /pagi|tengahari|petang|malam/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'pagi') {
                return hour;
            } else if (meridiem === 'tengahari') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'petang' || meridiem === 'malam') {
                return hour + 12;
            }
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'pagi';
            } else if (hours < 15) {
                return 'tengahari';
            } else if (hours < 19) {
                return 'petang';
            } else {
                return 'malam';
            }
        },
        calendar : {
            sameDay : '[Hari ini pukul] LT',
            nextDay : '[Esok pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kelmarin pukul] LT',
            lastWeek : 'dddd [lepas pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dalam %s',
            past : '%s yang lepas',
            s : 'beberapa saat',
            ss : '%d saat',
            m : 'seminit',
            mm : '%d minit',
            h : 'sejam',
            hh : '%d jam',
            d : 'sehari',
            dd : '%d hari',
            M : 'sebulan',
            MM : '%d bulan',
            y : 'setahun',
            yy : '%d tahun'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ms', {
        months : 'Januari_Februari_Mac_April_Mei_Jun_Julai_Ogos_September_Oktober_November_Disember'.split('_'),
        monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ogs_Sep_Okt_Nov_Dis'.split('_'),
        weekdays : 'Ahad_Isnin_Selasa_Rabu_Khamis_Jumaat_Sabtu'.split('_'),
        weekdaysShort : 'Ahd_Isn_Sel_Rab_Kha_Jum_Sab'.split('_'),
        weekdaysMin : 'Ah_Is_Sl_Rb_Km_Jm_Sb'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY [pukul] HH.mm',
            LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
        },
        meridiemParse: /pagi|tengahari|petang|malam/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'pagi') {
                return hour;
            } else if (meridiem === 'tengahari') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'petang' || meridiem === 'malam') {
                return hour + 12;
            }
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'pagi';
            } else if (hours < 15) {
                return 'tengahari';
            } else if (hours < 19) {
                return 'petang';
            } else {
                return 'malam';
            }
        },
        calendar : {
            sameDay : '[Hari ini pukul] LT',
            nextDay : '[Esok pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kelmarin pukul] LT',
            lastWeek : 'dddd [lepas pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'dalam %s',
            past : '%s yang lepas',
            s : 'beberapa saat',
            ss : '%d saat',
            m : 'seminit',
            mm : '%d minit',
            h : 'sejam',
            hh : '%d jam',
            d : 'sehari',
            dd : '%d hari',
            M : 'sebulan',
            MM : '%d bulan',
            y : 'setahun',
            yy : '%d tahun'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('mt', {
        months : 'Jannar_Frar_Marzu_April_Mejju_Ġunju_Lulju_Awwissu_Settembru_Ottubru_Novembru_Diċembru'.split('_'),
        monthsShort : 'Jan_Fra_Mar_Apr_Mej_Ġun_Lul_Aww_Set_Ott_Nov_Diċ'.split('_'),
        weekdays : 'Il-Ħadd_It-Tnejn_It-Tlieta_L-Erbgħa_Il-Ħamis_Il-Ġimgħa_Is-Sibt'.split('_'),
        weekdaysShort : 'Ħad_Tne_Tli_Erb_Ħam_Ġim_Sib'.split('_'),
        weekdaysMin : 'Ħa_Tn_Tl_Er_Ħa_Ġi_Si'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Illum fil-]LT',
            nextDay : '[Għada fil-]LT',
            nextWeek : 'dddd [fil-]LT',
            lastDay : '[Il-bieraħ fil-]LT',
            lastWeek : 'dddd [li għadda] [fil-]LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'f’ %s',
            past : '%s ilu',
            s : 'ftit sekondi',
            ss : '%d sekondi',
            m : 'minuta',
            mm : '%d minuti',
            h : 'siegħa',
            hh : '%d siegħat',
            d : 'ġurnata',
            dd : '%d ġranet',
            M : 'xahar',
            MM : '%d xhur',
            y : 'sena',
            yy : '%d sni'
        },
        dayOfMonthOrdinalParse : /\d{1,2}º/,
        ordinal: '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$c = {
        '1': '၁',
        '2': '၂',
        '3': '၃',
        '4': '၄',
        '5': '၅',
        '6': '၆',
        '7': '၇',
        '8': '၈',
        '9': '၉',
        '0': '၀'
    }, numberMap$b = {
        '၁': '1',
        '၂': '2',
        '၃': '3',
        '၄': '4',
        '၅': '5',
        '၆': '6',
        '၇': '7',
        '၈': '8',
        '၉': '9',
        '၀': '0'
    };

    hooks.defineLocale('my', {
        months: 'ဇန်နဝါရီ_ဖေဖော်ဝါရီ_မတ်_ဧပြီ_မေ_ဇွန်_ဇူလိုင်_သြဂုတ်_စက်တင်ဘာ_အောက်တိုဘာ_နိုဝင်ဘာ_ဒီဇင်ဘာ'.split('_'),
        monthsShort: 'ဇန်_ဖေ_မတ်_ပြီ_မေ_ဇွန်_လိုင်_သြ_စက်_အောက်_နို_ဒီ'.split('_'),
        weekdays: 'တနင်္ဂနွေ_တနင်္လာ_အင်္ဂါ_ဗုဒ္ဓဟူး_ကြာသပတေး_သောကြာ_စနေ'.split('_'),
        weekdaysShort: 'နွေ_လာ_ဂါ_ဟူး_ကြာ_သော_နေ'.split('_'),
        weekdaysMin: 'နွေ_လာ_ဂါ_ဟူး_ကြာ_သော_နေ'.split('_'),

        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'DD/MM/YYYY',
            LL: 'D MMMM YYYY',
            LLL: 'D MMMM YYYY HH:mm',
            LLLL: 'dddd D MMMM YYYY HH:mm'
        },
        calendar: {
            sameDay: '[ယနေ.] LT [မှာ]',
            nextDay: '[မနက်ဖြန်] LT [မှာ]',
            nextWeek: 'dddd LT [မှာ]',
            lastDay: '[မနေ.က] LT [မှာ]',
            lastWeek: '[ပြီးခဲ့သော] dddd LT [မှာ]',
            sameElse: 'L'
        },
        relativeTime: {
            future: 'လာမည့် %s မှာ',
            past: 'လွန်ခဲ့သော %s က',
            s: 'စက္ကန်.အနည်းငယ်',
            ss : '%d စက္ကန့်',
            m: 'တစ်မိနစ်',
            mm: '%d မိနစ်',
            h: 'တစ်နာရီ',
            hh: '%d နာရီ',
            d: 'တစ်ရက်',
            dd: '%d ရက်',
            M: 'တစ်လ',
            MM: '%d လ',
            y: 'တစ်နှစ်',
            yy: '%d နှစ်'
        },
        preparse: function (string) {
            return string.replace(/[၁၂၃၄၅၆၇၈၉၀]/g, function (match) {
                return numberMap$b[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$c[match];
            });
        },
        week: {
            dow: 1, // Monday is the first day of the week.
            doy: 4 // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('nb', {
        months : 'januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
        monthsShort : 'jan._feb._mars_april_mai_juni_juli_aug._sep._okt._nov._des.'.split('_'),
        monthsParseExact : true,
        weekdays : 'søndag_mandag_tirsdag_onsdag_torsdag_fredag_lørdag'.split('_'),
        weekdaysShort : 'sø._ma._ti._on._to._fr._lø.'.split('_'),
        weekdaysMin : 'sø_ma_ti_on_to_fr_lø'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY [kl.] HH:mm',
            LLLL : 'dddd D. MMMM YYYY [kl.] HH:mm'
        },
        calendar : {
            sameDay: '[i dag kl.] LT',
            nextDay: '[i morgen kl.] LT',
            nextWeek: 'dddd [kl.] LT',
            lastDay: '[i går kl.] LT',
            lastWeek: '[forrige] dddd [kl.] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'om %s',
            past : '%s siden',
            s : 'noen sekunder',
            ss : '%d sekunder',
            m : 'ett minutt',
            mm : '%d minutter',
            h : 'en time',
            hh : '%d timer',
            d : 'en dag',
            dd : '%d dager',
            M : 'en måned',
            MM : '%d måneder',
            y : 'ett år',
            yy : '%d år'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$d = {
        '1': '१',
        '2': '२',
        '3': '३',
        '4': '४',
        '5': '५',
        '6': '६',
        '7': '७',
        '8': '८',
        '9': '९',
        '0': '०'
    },
    numberMap$c = {
        '१': '1',
        '२': '2',
        '३': '3',
        '४': '4',
        '५': '5',
        '६': '6',
        '७': '7',
        '८': '8',
        '९': '9',
        '०': '0'
    };

    hooks.defineLocale('ne', {
        months : 'जनवरी_फेब्रुवरी_मार्च_अप्रिल_मई_जुन_जुलाई_अगष्ट_सेप्टेम्बर_अक्टोबर_नोभेम्बर_डिसेम्बर'.split('_'),
        monthsShort : 'जन._फेब्रु._मार्च_अप्रि._मई_जुन_जुलाई._अग._सेप्ट._अक्टो._नोभे._डिसे.'.split('_'),
        monthsParseExact : true,
        weekdays : 'आइतबार_सोमबार_मङ्गलबार_बुधबार_बिहिबार_शुक्रबार_शनिबार'.split('_'),
        weekdaysShort : 'आइत._सोम._मङ्गल._बुध._बिहि._शुक्र._शनि.'.split('_'),
        weekdaysMin : 'आ._सो._मं._बु._बि._शु._श.'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'Aको h:mm बजे',
            LTS : 'Aको h:mm:ss बजे',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, Aको h:mm बजे',
            LLLL : 'dddd, D MMMM YYYY, Aको h:mm बजे'
        },
        preparse: function (string) {
            return string.replace(/[१२३४५६७८९०]/g, function (match) {
                return numberMap$c[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$d[match];
            });
        },
        meridiemParse: /राति|बिहान|दिउँसो|साँझ/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'राति') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'बिहान') {
                return hour;
            } else if (meridiem === 'दिउँसो') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'साँझ') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 3) {
                return 'राति';
            } else if (hour < 12) {
                return 'बिहान';
            } else if (hour < 16) {
                return 'दिउँसो';
            } else if (hour < 20) {
                return 'साँझ';
            } else {
                return 'राति';
            }
        },
        calendar : {
            sameDay : '[आज] LT',
            nextDay : '[भोलि] LT',
            nextWeek : '[आउँदो] dddd[,] LT',
            lastDay : '[हिजो] LT',
            lastWeek : '[गएको] dddd[,] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%sमा',
            past : '%s अगाडि',
            s : 'केही क्षण',
            ss : '%d सेकेण्ड',
            m : 'एक मिनेट',
            mm : '%d मिनेट',
            h : 'एक घण्टा',
            hh : '%d घण्टा',
            d : 'एक दिन',
            dd : '%d दिन',
            M : 'एक महिना',
            MM : '%d महिना',
            y : 'एक बर्ष',
            yy : '%d बर्ष'
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortWithDots$1 = 'jan._feb._mrt._apr._mei_jun._jul._aug._sep._okt._nov._dec.'.split('_'),
        monthsShortWithoutDots$1 = 'jan_feb_mrt_apr_mei_jun_jul_aug_sep_okt_nov_dec'.split('_');

    var monthsParse$4 = [/^jan/i, /^feb/i, /^maart|mrt.?$/i, /^apr/i, /^mei$/i, /^jun[i.]?$/i, /^jul[i.]?$/i, /^aug/i, /^sep/i, /^okt/i, /^nov/i, /^dec/i];
    var monthsRegex$5 = /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december|jan\.?|feb\.?|mrt\.?|apr\.?|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i;

    hooks.defineLocale('nl-be', {
        months : 'januari_februari_maart_april_mei_juni_juli_augustus_september_oktober_november_december'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortWithDots$1;
            } else if (/-MMM-/.test(format)) {
                return monthsShortWithoutDots$1[m.month()];
            } else {
                return monthsShortWithDots$1[m.month()];
            }
        },

        monthsRegex: monthsRegex$5,
        monthsShortRegex: monthsRegex$5,
        monthsStrictRegex: /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december)/i,
        monthsShortStrictRegex: /^(jan\.?|feb\.?|mrt\.?|apr\.?|mei|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i,

        monthsParse : monthsParse$4,
        longMonthsParse : monthsParse$4,
        shortMonthsParse : monthsParse$4,

        weekdays : 'zondag_maandag_dinsdag_woensdag_donderdag_vrijdag_zaterdag'.split('_'),
        weekdaysShort : 'zo._ma._di._wo._do._vr._za.'.split('_'),
        weekdaysMin : 'zo_ma_di_wo_do_vr_za'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[vandaag om] LT',
            nextDay: '[morgen om] LT',
            nextWeek: 'dddd [om] LT',
            lastDay: '[gisteren om] LT',
            lastWeek: '[afgelopen] dddd [om] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'over %s',
            past : '%s geleden',
            s : 'een paar seconden',
            ss : '%d seconden',
            m : 'één minuut',
            mm : '%d minuten',
            h : 'één uur',
            hh : '%d uur',
            d : 'één dag',
            dd : '%d dagen',
            M : 'één maand',
            MM : '%d maanden',
            y : 'één jaar',
            yy : '%d jaar'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
        ordinal : function (number) {
            return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsShortWithDots$2 = 'jan._feb._mrt._apr._mei_jun._jul._aug._sep._okt._nov._dec.'.split('_'),
        monthsShortWithoutDots$2 = 'jan_feb_mrt_apr_mei_jun_jul_aug_sep_okt_nov_dec'.split('_');

    var monthsParse$5 = [/^jan/i, /^feb/i, /^maart|mrt.?$/i, /^apr/i, /^mei$/i, /^jun[i.]?$/i, /^jul[i.]?$/i, /^aug/i, /^sep/i, /^okt/i, /^nov/i, /^dec/i];
    var monthsRegex$6 = /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december|jan\.?|feb\.?|mrt\.?|apr\.?|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i;

    hooks.defineLocale('nl', {
        months : 'januari_februari_maart_april_mei_juni_juli_augustus_september_oktober_november_december'.split('_'),
        monthsShort : function (m, format) {
            if (!m) {
                return monthsShortWithDots$2;
            } else if (/-MMM-/.test(format)) {
                return monthsShortWithoutDots$2[m.month()];
            } else {
                return monthsShortWithDots$2[m.month()];
            }
        },

        monthsRegex: monthsRegex$6,
        monthsShortRegex: monthsRegex$6,
        monthsStrictRegex: /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december)/i,
        monthsShortStrictRegex: /^(jan\.?|feb\.?|mrt\.?|apr\.?|mei|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i,

        monthsParse : monthsParse$5,
        longMonthsParse : monthsParse$5,
        shortMonthsParse : monthsParse$5,

        weekdays : 'zondag_maandag_dinsdag_woensdag_donderdag_vrijdag_zaterdag'.split('_'),
        weekdaysShort : 'zo._ma._di._wo._do._vr._za.'.split('_'),
        weekdaysMin : 'zo_ma_di_wo_do_vr_za'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD-MM-YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[vandaag om] LT',
            nextDay: '[morgen om] LT',
            nextWeek: 'dddd [om] LT',
            lastDay: '[gisteren om] LT',
            lastWeek: '[afgelopen] dddd [om] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'over %s',
            past : '%s geleden',
            s : 'een paar seconden',
            ss : '%d seconden',
            m : 'één minuut',
            mm : '%d minuten',
            h : 'één uur',
            hh : '%d uur',
            d : 'één dag',
            dd : '%d dagen',
            M : 'één maand',
            MM : '%d maanden',
            y : 'één jaar',
            yy : '%d jaar'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
        ordinal : function (number) {
            return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('nn', {
        months : 'januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
        monthsShort : 'jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_'),
        weekdays : 'sundag_måndag_tysdag_onsdag_torsdag_fredag_laurdag'.split('_'),
        weekdaysShort : 'sun_mån_tys_ons_tor_fre_lau'.split('_'),
        weekdaysMin : 'su_må_ty_on_to_fr_lø'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY [kl.] H:mm',
            LLLL : 'dddd D. MMMM YYYY [kl.] HH:mm'
        },
        calendar : {
            sameDay: '[I dag klokka] LT',
            nextDay: '[I morgon klokka] LT',
            nextWeek: 'dddd [klokka] LT',
            lastDay: '[I går klokka] LT',
            lastWeek: '[Føregåande] dddd [klokka] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'om %s',
            past : '%s sidan',
            s : 'nokre sekund',
            ss : '%d sekund',
            m : 'eit minutt',
            mm : '%d minutt',
            h : 'ein time',
            hh : '%d timar',
            d : 'ein dag',
            dd : '%d dagar',
            M : 'ein månad',
            MM : '%d månader',
            y : 'eit år',
            yy : '%d år'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$e = {
        '1': '੧',
        '2': '੨',
        '3': '੩',
        '4': '੪',
        '5': '੫',
        '6': '੬',
        '7': '੭',
        '8': '੮',
        '9': '੯',
        '0': '੦'
    },
    numberMap$d = {
        '੧': '1',
        '੨': '2',
        '੩': '3',
        '੪': '4',
        '੫': '5',
        '੬': '6',
        '੭': '7',
        '੮': '8',
        '੯': '9',
        '੦': '0'
    };

    hooks.defineLocale('pa-in', {
        // There are months name as per Nanakshahi Calendar but they are not used as rigidly in modern Punjabi.
        months : 'ਜਨਵਰੀ_ਫ਼ਰਵਰੀ_ਮਾਰਚ_ਅਪ੍ਰੈਲ_ਮਈ_ਜੂਨ_ਜੁਲਾਈ_ਅਗਸਤ_ਸਤੰਬਰ_ਅਕਤੂਬਰ_ਨਵੰਬਰ_ਦਸੰਬਰ'.split('_'),
        monthsShort : 'ਜਨਵਰੀ_ਫ਼ਰਵਰੀ_ਮਾਰਚ_ਅਪ੍ਰੈਲ_ਮਈ_ਜੂਨ_ਜੁਲਾਈ_ਅਗਸਤ_ਸਤੰਬਰ_ਅਕਤੂਬਰ_ਨਵੰਬਰ_ਦਸੰਬਰ'.split('_'),
        weekdays : 'ਐਤਵਾਰ_ਸੋਮਵਾਰ_ਮੰਗਲਵਾਰ_ਬੁਧਵਾਰ_ਵੀਰਵਾਰ_ਸ਼ੁੱਕਰਵਾਰ_ਸ਼ਨੀਚਰਵਾਰ'.split('_'),
        weekdaysShort : 'ਐਤ_ਸੋਮ_ਮੰਗਲ_ਬੁਧ_ਵੀਰ_ਸ਼ੁਕਰ_ਸ਼ਨੀ'.split('_'),
        weekdaysMin : 'ਐਤ_ਸੋਮ_ਮੰਗਲ_ਬੁਧ_ਵੀਰ_ਸ਼ੁਕਰ_ਸ਼ਨੀ'.split('_'),
        longDateFormat : {
            LT : 'A h:mm ਵਜੇ',
            LTS : 'A h:mm:ss ਵਜੇ',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm ਵਜੇ',
            LLLL : 'dddd, D MMMM YYYY, A h:mm ਵਜੇ'
        },
        calendar : {
            sameDay : '[ਅਜ] LT',
            nextDay : '[ਕਲ] LT',
            nextWeek : '[ਅਗਲਾ] dddd, LT',
            lastDay : '[ਕਲ] LT',
            lastWeek : '[ਪਿਛਲੇ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s ਵਿੱਚ',
            past : '%s ਪਿਛਲੇ',
            s : 'ਕੁਝ ਸਕਿੰਟ',
            ss : '%d ਸਕਿੰਟ',
            m : 'ਇਕ ਮਿੰਟ',
            mm : '%d ਮਿੰਟ',
            h : 'ਇੱਕ ਘੰਟਾ',
            hh : '%d ਘੰਟੇ',
            d : 'ਇੱਕ ਦਿਨ',
            dd : '%d ਦਿਨ',
            M : 'ਇੱਕ ਮਹੀਨਾ',
            MM : '%d ਮਹੀਨੇ',
            y : 'ਇੱਕ ਸਾਲ',
            yy : '%d ਸਾਲ'
        },
        preparse: function (string) {
            return string.replace(/[੧੨੩੪੫੬੭੮੯੦]/g, function (match) {
                return numberMap$d[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$e[match];
            });
        },
        // Punjabi notation for meridiems are quite fuzzy in practice. While there exists
        // a rigid notion of a 'Pahar' it is not used as rigidly in modern Punjabi.
        meridiemParse: /ਰਾਤ|ਸਵੇਰ|ਦੁਪਹਿਰ|ਸ਼ਾਮ/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'ਰਾਤ') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'ਸਵੇਰ') {
                return hour;
            } else if (meridiem === 'ਦੁਪਹਿਰ') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'ਸ਼ਾਮ') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'ਰਾਤ';
            } else if (hour < 10) {
                return 'ਸਵੇਰ';
            } else if (hour < 17) {
                return 'ਦੁਪਹਿਰ';
            } else if (hour < 20) {
                return 'ਸ਼ਾਮ';
            } else {
                return 'ਰਾਤ';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var monthsNominative = 'styczeń_luty_marzec_kwiecień_maj_czerwiec_lipiec_sierpień_wrzesień_październik_listopad_grudzień'.split('_'),
        monthsSubjective = 'stycznia_lutego_marca_kwietnia_maja_czerwca_lipca_sierpnia_września_października_listopada_grudnia'.split('_');
    function plural$3(n) {
        return (n % 10 < 5) && (n % 10 > 1) && ((~~(n / 10) % 10) !== 1);
    }
    function translate$8(number, withoutSuffix, key) {
        var result = number + ' ';
        switch (key) {
            case 'ss':
                return result + (plural$3(number) ? 'sekundy' : 'sekund');
            case 'm':
                return withoutSuffix ? 'minuta' : 'minutę';
            case 'mm':
                return result + (plural$3(number) ? 'minuty' : 'minut');
            case 'h':
                return withoutSuffix  ? 'godzina'  : 'godzinę';
            case 'hh':
                return result + (plural$3(number) ? 'godziny' : 'godzin');
            case 'MM':
                return result + (plural$3(number) ? 'miesiące' : 'miesięcy');
            case 'yy':
                return result + (plural$3(number) ? 'lata' : 'lat');
        }
    }

    hooks.defineLocale('pl', {
        months : function (momentToFormat, format) {
            if (!momentToFormat) {
                return monthsNominative;
            } else if (format === '') {
                // Hack: if format empty we know this is used to generate
                // RegExp by moment. Give then back both valid forms of months
                // in RegExp ready format.
                return '(' + monthsSubjective[momentToFormat.month()] + '|' + monthsNominative[momentToFormat.month()] + ')';
            } else if (/D MMMM/.test(format)) {
                return monthsSubjective[momentToFormat.month()];
            } else {
                return monthsNominative[momentToFormat.month()];
            }
        },
        monthsShort : 'sty_lut_mar_kwi_maj_cze_lip_sie_wrz_paź_lis_gru'.split('_'),
        weekdays : 'niedziela_poniedziałek_wtorek_środa_czwartek_piątek_sobota'.split('_'),
        weekdaysShort : 'ndz_pon_wt_śr_czw_pt_sob'.split('_'),
        weekdaysMin : 'Nd_Pn_Wt_Śr_Cz_Pt_So'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Dziś o] LT',
            nextDay: '[Jutro o] LT',
            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[W niedzielę o] LT';

                    case 2:
                        return '[We wtorek o] LT';

                    case 3:
                        return '[W środę o] LT';

                    case 6:
                        return '[W sobotę o] LT';

                    default:
                        return '[W] dddd [o] LT';
                }
            },
            lastDay: '[Wczoraj o] LT',
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[W zeszłą niedzielę o] LT';
                    case 3:
                        return '[W zeszłą środę o] LT';
                    case 6:
                        return '[W zeszłą sobotę o] LT';
                    default:
                        return '[W zeszły] dddd [o] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'za %s',
            past : '%s temu',
            s : 'kilka sekund',
            ss : translate$8,
            m : translate$8,
            mm : translate$8,
            h : translate$8,
            hh : translate$8,
            d : '1 dzień',
            dd : '%d dni',
            M : 'miesiąc',
            MM : translate$8,
            y : 'rok',
            yy : translate$8
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('pt-br', {
        months : 'Janeiro_Fevereiro_Março_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro'.split('_'),
        monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
        weekdays : 'Domingo_Segunda-feira_Terça-feira_Quarta-feira_Quinta-feira_Sexta-feira_Sábado'.split('_'),
        weekdaysShort : 'Dom_Seg_Ter_Qua_Qui_Sex_Sáb'.split('_'),
        weekdaysMin : 'Do_2ª_3ª_4ª_5ª_6ª_Sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY [às] HH:mm',
            LLLL : 'dddd, D [de] MMMM [de] YYYY [às] HH:mm'
        },
        calendar : {
            sameDay: '[Hoje às] LT',
            nextDay: '[Amanhã às] LT',
            nextWeek: 'dddd [às] LT',
            lastDay: '[Ontem às] LT',
            lastWeek: function () {
                return (this.day() === 0 || this.day() === 6) ?
                    '[Último] dddd [às] LT' : // Saturday + Sunday
                    '[Última] dddd [às] LT'; // Monday - Friday
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'em %s',
            past : 'há %s',
            s : 'poucos segundos',
            ss : '%d segundos',
            m : 'um minuto',
            mm : '%d minutos',
            h : 'uma hora',
            hh : '%d horas',
            d : 'um dia',
            dd : '%d dias',
            M : 'um mês',
            MM : '%d meses',
            y : 'um ano',
            yy : '%d anos'
        },
        dayOfMonthOrdinalParse: /\d{1,2}º/,
        ordinal : '%dº'
    });

    //! moment.js locale configuration

    hooks.defineLocale('pt', {
        months : 'Janeiro_Fevereiro_Março_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro'.split('_'),
        monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
        weekdays : 'Domingo_Segunda-feira_Terça-feira_Quarta-feira_Quinta-feira_Sexta-feira_Sábado'.split('_'),
        weekdaysShort : 'Dom_Seg_Ter_Qua_Qui_Sex_Sáb'.split('_'),
        weekdaysMin : 'Do_2ª_3ª_4ª_5ª_6ª_Sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D [de] MMMM [de] YYYY',
            LLL : 'D [de] MMMM [de] YYYY HH:mm',
            LLLL : 'dddd, D [de] MMMM [de] YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Hoje às] LT',
            nextDay: '[Amanhã às] LT',
            nextWeek: 'dddd [às] LT',
            lastDay: '[Ontem às] LT',
            lastWeek: function () {
                return (this.day() === 0 || this.day() === 6) ?
                    '[Último] dddd [às] LT' : // Saturday + Sunday
                    '[Última] dddd [às] LT'; // Monday - Friday
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'em %s',
            past : 'há %s',
            s : 'segundos',
            ss : '%d segundos',
            m : 'um minuto',
            mm : '%d minutos',
            h : 'uma hora',
            hh : '%d horas',
            d : 'um dia',
            dd : '%d dias',
            M : 'um mês',
            MM : '%d meses',
            y : 'um ano',
            yy : '%d anos'
        },
        dayOfMonthOrdinalParse: /\d{1,2}º/,
        ordinal : '%dº',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function relativeTimeWithPlural$2(number, withoutSuffix, key) {
        var format = {
                'ss': 'secunde',
                'mm': 'minute',
                'hh': 'ore',
                'dd': 'zile',
                'MM': 'luni',
                'yy': 'ani'
            },
            separator = ' ';
        if (number % 100 >= 20 || (number >= 100 && number % 100 === 0)) {
            separator = ' de ';
        }
        return number + separator + format[key];
    }

    hooks.defineLocale('ro', {
        months : 'ianuarie_februarie_martie_aprilie_mai_iunie_iulie_august_septembrie_octombrie_noiembrie_decembrie'.split('_'),
        monthsShort : 'ian._febr._mart._apr._mai_iun._iul._aug._sept._oct._nov._dec.'.split('_'),
        monthsParseExact: true,
        weekdays : 'duminică_luni_marți_miercuri_joi_vineri_sâmbătă'.split('_'),
        weekdaysShort : 'Dum_Lun_Mar_Mie_Joi_Vin_Sâm'.split('_'),
        weekdaysMin : 'Du_Lu_Ma_Mi_Jo_Vi_Sâ'.split('_'),
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY H:mm',
            LLLL : 'dddd, D MMMM YYYY H:mm'
        },
        calendar : {
            sameDay: '[azi la] LT',
            nextDay: '[mâine la] LT',
            nextWeek: 'dddd [la] LT',
            lastDay: '[ieri la] LT',
            lastWeek: '[fosta] dddd [la] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'peste %s',
            past : '%s în urmă',
            s : 'câteva secunde',
            ss : relativeTimeWithPlural$2,
            m : 'un minut',
            mm : relativeTimeWithPlural$2,
            h : 'o oră',
            hh : relativeTimeWithPlural$2,
            d : 'o zi',
            dd : relativeTimeWithPlural$2,
            M : 'o lună',
            MM : relativeTimeWithPlural$2,
            y : 'un an',
            yy : relativeTimeWithPlural$2
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function plural$4(word, num) {
        var forms = word.split('_');
        return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
    }
    function relativeTimeWithPlural$3(number, withoutSuffix, key) {
        var format = {
            'ss': withoutSuffix ? 'секунда_секунды_секунд' : 'секунду_секунды_секунд',
            'mm': withoutSuffix ? 'минута_минуты_минут' : 'минуту_минуты_минут',
            'hh': 'час_часа_часов',
            'dd': 'день_дня_дней',
            'MM': 'месяц_месяца_месяцев',
            'yy': 'год_года_лет'
        };
        if (key === 'm') {
            return withoutSuffix ? 'минута' : 'минуту';
        }
        else {
            return number + ' ' + plural$4(format[key], +number);
        }
    }
    var monthsParse$6 = [/^янв/i, /^фев/i, /^мар/i, /^апр/i, /^ма[йя]/i, /^июн/i, /^июл/i, /^авг/i, /^сен/i, /^окт/i, /^ноя/i, /^дек/i];

    // http://new.gramota.ru/spravka/rules/139-prop : § 103
    // Сокращения месяцев: http://new.gramota.ru/spravka/buro/search-answer?s=242637
    // CLDR data:          http://www.unicode.org/cldr/charts/28/summary/ru.html#1753
    hooks.defineLocale('ru', {
        months : {
            format: 'января_февраля_марта_апреля_мая_июня_июля_августа_сентября_октября_ноября_декабря'.split('_'),
            standalone: 'январь_февраль_март_апрель_май_июнь_июль_август_сентябрь_октябрь_ноябрь_декабрь'.split('_')
        },
        monthsShort : {
            // по CLDR именно "июл." и "июн.", но какой смысл менять букву на точку ?
            format: 'янв._февр._мар._апр._мая_июня_июля_авг._сент._окт._нояб._дек.'.split('_'),
            standalone: 'янв._февр._март_апр._май_июнь_июль_авг._сент._окт._нояб._дек.'.split('_')
        },
        weekdays : {
            standalone: 'воскресенье_понедельник_вторник_среда_четверг_пятница_суббота'.split('_'),
            format: 'воскресенье_понедельник_вторник_среду_четверг_пятницу_субботу'.split('_'),
            isFormat: /\[ ?[Вв] ?(?:прошлую|следующую|эту)? ?\] ?dddd/
        },
        weekdaysShort : 'вс_пн_вт_ср_чт_пт_сб'.split('_'),
        weekdaysMin : 'вс_пн_вт_ср_чт_пт_сб'.split('_'),
        monthsParse : monthsParse$6,
        longMonthsParse : monthsParse$6,
        shortMonthsParse : monthsParse$6,

        // полные названия с падежами, по три буквы, для некоторых, по 4 буквы, сокращения с точкой и без точки
        monthsRegex: /^(январ[ья]|янв\.?|феврал[ья]|февр?\.?|марта?|мар\.?|апрел[ья]|апр\.?|ма[йя]|июн[ья]|июн\.?|июл[ья]|июл\.?|августа?|авг\.?|сентябр[ья]|сент?\.?|октябр[ья]|окт\.?|ноябр[ья]|нояб?\.?|декабр[ья]|дек\.?)/i,

        // копия предыдущего
        monthsShortRegex: /^(январ[ья]|янв\.?|феврал[ья]|февр?\.?|марта?|мар\.?|апрел[ья]|апр\.?|ма[йя]|июн[ья]|июн\.?|июл[ья]|июл\.?|августа?|авг\.?|сентябр[ья]|сент?\.?|октябр[ья]|окт\.?|ноябр[ья]|нояб?\.?|декабр[ья]|дек\.?)/i,

        // полные названия с падежами
        monthsStrictRegex: /^(январ[яь]|феврал[яь]|марта?|апрел[яь]|ма[яй]|июн[яь]|июл[яь]|августа?|сентябр[яь]|октябр[яь]|ноябр[яь]|декабр[яь])/i,

        // Выражение, которое соотвествует только сокращённым формам
        monthsShortStrictRegex: /^(янв\.|февр?\.|мар[т.]|апр\.|ма[яй]|июн[ья.]|июл[ья.]|авг\.|сент?\.|окт\.|нояб?\.|дек\.)/i,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY г.',
            LLL : 'D MMMM YYYY г., H:mm',
            LLLL : 'dddd, D MMMM YYYY г., H:mm'
        },
        calendar : {
            sameDay: '[Сегодня, в] LT',
            nextDay: '[Завтра, в] LT',
            lastDay: '[Вчера, в] LT',
            nextWeek: function (now) {
                if (now.week() !== this.week()) {
                    switch (this.day()) {
                        case 0:
                            return '[В следующее] dddd, [в] LT';
                        case 1:
                        case 2:
                        case 4:
                            return '[В следующий] dddd, [в] LT';
                        case 3:
                        case 5:
                        case 6:
                            return '[В следующую] dddd, [в] LT';
                    }
                } else {
                    if (this.day() === 2) {
                        return '[Во] dddd, [в] LT';
                    } else {
                        return '[В] dddd, [в] LT';
                    }
                }
            },
            lastWeek: function (now) {
                if (now.week() !== this.week()) {
                    switch (this.day()) {
                        case 0:
                            return '[В прошлое] dddd, [в] LT';
                        case 1:
                        case 2:
                        case 4:
                            return '[В прошлый] dddd, [в] LT';
                        case 3:
                        case 5:
                        case 6:
                            return '[В прошлую] dddd, [в] LT';
                    }
                } else {
                    if (this.day() === 2) {
                        return '[Во] dddd, [в] LT';
                    } else {
                        return '[В] dddd, [в] LT';
                    }
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'через %s',
            past : '%s назад',
            s : 'несколько секунд',
            ss : relativeTimeWithPlural$3,
            m : relativeTimeWithPlural$3,
            mm : relativeTimeWithPlural$3,
            h : 'час',
            hh : relativeTimeWithPlural$3,
            d : 'день',
            dd : relativeTimeWithPlural$3,
            M : 'месяц',
            MM : relativeTimeWithPlural$3,
            y : 'год',
            yy : relativeTimeWithPlural$3
        },
        meridiemParse: /ночи|утра|дня|вечера/i,
        isPM : function (input) {
            return /^(дня|вечера)$/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'ночи';
            } else if (hour < 12) {
                return 'утра';
            } else if (hour < 17) {
                return 'дня';
            } else {
                return 'вечера';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(й|го|я)/,
        ordinal: function (number, period) {
            switch (period) {
                case 'M':
                case 'd':
                case 'DDD':
                    return number + '-й';
                case 'D':
                    return number + '-го';
                case 'w':
                case 'W':
                    return number + '-я';
                default:
                    return number;
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var months$8 = [
        'جنوري',
        'فيبروري',
        'مارچ',
        'اپريل',
        'مئي',
        'جون',
        'جولاءِ',
        'آگسٽ',
        'سيپٽمبر',
        'آڪٽوبر',
        'نومبر',
        'ڊسمبر'
    ];
    var days$1 = [
        'آچر',
        'سومر',
        'اڱارو',
        'اربع',
        'خميس',
        'جمع',
        'ڇنڇر'
    ];

    hooks.defineLocale('sd', {
        months : months$8,
        monthsShort : months$8,
        weekdays : days$1,
        weekdaysShort : days$1,
        weekdaysMin : days$1,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd، D MMMM YYYY HH:mm'
        },
        meridiemParse: /صبح|شام/,
        isPM : function (input) {
            return 'شام' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'صبح';
            }
            return 'شام';
        },
        calendar : {
            sameDay : '[اڄ] LT',
            nextDay : '[سڀاڻي] LT',
            nextWeek : 'dddd [اڳين هفتي تي] LT',
            lastDay : '[ڪالهه] LT',
            lastWeek : '[گزريل هفتي] dddd [تي] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s پوء',
            past : '%s اڳ',
            s : 'چند سيڪنڊ',
            ss : '%d سيڪنڊ',
            m : 'هڪ منٽ',
            mm : '%d منٽ',
            h : 'هڪ ڪلاڪ',
            hh : '%d ڪلاڪ',
            d : 'هڪ ڏينهن',
            dd : '%d ڏينهن',
            M : 'هڪ مهينو',
            MM : '%d مهينا',
            y : 'هڪ سال',
            yy : '%d سال'
        },
        preparse: function (string) {
            return string.replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/,/g, '،');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('se', {
        months : 'ođđajagemánnu_guovvamánnu_njukčamánnu_cuoŋománnu_miessemánnu_geassemánnu_suoidnemánnu_borgemánnu_čakčamánnu_golggotmánnu_skábmamánnu_juovlamánnu'.split('_'),
        monthsShort : 'ođđj_guov_njuk_cuo_mies_geas_suoi_borg_čakč_golg_skáb_juov'.split('_'),
        weekdays : 'sotnabeaivi_vuossárga_maŋŋebárga_gaskavahkku_duorastat_bearjadat_lávvardat'.split('_'),
        weekdaysShort : 'sotn_vuos_maŋ_gask_duor_bear_láv'.split('_'),
        weekdaysMin : 's_v_m_g_d_b_L'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'MMMM D. [b.] YYYY',
            LLL : 'MMMM D. [b.] YYYY [ti.] HH:mm',
            LLLL : 'dddd, MMMM D. [b.] YYYY [ti.] HH:mm'
        },
        calendar : {
            sameDay: '[otne ti] LT',
            nextDay: '[ihttin ti] LT',
            nextWeek: 'dddd [ti] LT',
            lastDay: '[ikte ti] LT',
            lastWeek: '[ovddit] dddd [ti] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : '%s geažes',
            past : 'maŋit %s',
            s : 'moadde sekunddat',
            ss: '%d sekunddat',
            m : 'okta minuhta',
            mm : '%d minuhtat',
            h : 'okta diimmu',
            hh : '%d diimmut',
            d : 'okta beaivi',
            dd : '%d beaivvit',
            M : 'okta mánnu',
            MM : '%d mánut',
            y : 'okta jahki',
            yy : '%d jagit'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    /*jshint -W100*/
    hooks.defineLocale('si', {
        months : 'ජනවාරි_පෙබරවාරි_මාර්තු_අප්‍රේල්_මැයි_ජූනි_ජූලි_අගෝස්තු_සැප්තැම්බර්_ඔක්තෝබර්_නොවැම්බර්_දෙසැම්බර්'.split('_'),
        monthsShort : 'ජන_පෙබ_මාර්_අප්_මැයි_ජූනි_ජූලි_අගෝ_සැප්_ඔක්_නොවැ_දෙසැ'.split('_'),
        weekdays : 'ඉරිදා_සඳුදා_අඟහරුවාදා_බදාදා_බ්‍රහස්පතින්දා_සිකුරාදා_සෙනසුරාදා'.split('_'),
        weekdaysShort : 'ඉරි_සඳු_අඟ_බදා_බ්‍රහ_සිකු_සෙන'.split('_'),
        weekdaysMin : 'ඉ_ස_අ_බ_බ්‍ර_සි_සෙ'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'a h:mm',
            LTS : 'a h:mm:ss',
            L : 'YYYY/MM/DD',
            LL : 'YYYY MMMM D',
            LLL : 'YYYY MMMM D, a h:mm',
            LLLL : 'YYYY MMMM D [වැනි] dddd, a h:mm:ss'
        },
        calendar : {
            sameDay : '[අද] LT[ට]',
            nextDay : '[හෙට] LT[ට]',
            nextWeek : 'dddd LT[ට]',
            lastDay : '[ඊයේ] LT[ට]',
            lastWeek : '[පසුගිය] dddd LT[ට]',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%sකින්',
            past : '%sකට පෙර',
            s : 'තත්පර කිහිපය',
            ss : 'තත්පර %d',
            m : 'මිනිත්තුව',
            mm : 'මිනිත්තු %d',
            h : 'පැය',
            hh : 'පැය %d',
            d : 'දිනය',
            dd : 'දින %d',
            M : 'මාසය',
            MM : 'මාස %d',
            y : 'වසර',
            yy : 'වසර %d'
        },
        dayOfMonthOrdinalParse: /\d{1,2} වැනි/,
        ordinal : function (number) {
            return number + ' වැනි';
        },
        meridiemParse : /පෙර වරු|පස් වරු|පෙ.ව|ප.ව./,
        isPM : function (input) {
            return input === 'ප.ව.' || input === 'පස් වරු';
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'ප.ව.' : 'පස් වරු';
            } else {
                return isLower ? 'පෙ.ව.' : 'පෙර වරු';
            }
        }
    });

    //! moment.js locale configuration

    var months$9 = 'január_február_marec_apríl_máj_jún_júl_august_september_október_november_december'.split('_'),
        monthsShort$6 = 'jan_feb_mar_apr_máj_jún_júl_aug_sep_okt_nov_dec'.split('_');
    function plural$5(n) {
        return (n > 1) && (n < 5);
    }
    function translate$9(number, withoutSuffix, key, isFuture) {
        var result = number + ' ';
        switch (key) {
            case 's':  // a few seconds / in a few seconds / a few seconds ago
                return (withoutSuffix || isFuture) ? 'pár sekúnd' : 'pár sekundami';
            case 'ss': // 9 seconds / in 9 seconds / 9 seconds ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'sekundy' : 'sekúnd');
                } else {
                    return result + 'sekundami';
                }
                break;
            case 'm':  // a minute / in a minute / a minute ago
                return withoutSuffix ? 'minúta' : (isFuture ? 'minútu' : 'minútou');
            case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'minúty' : 'minút');
                } else {
                    return result + 'minútami';
                }
                break;
            case 'h':  // an hour / in an hour / an hour ago
                return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
            case 'hh': // 9 hours / in 9 hours / 9 hours ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'hodiny' : 'hodín');
                } else {
                    return result + 'hodinami';
                }
                break;
            case 'd':  // a day / in a day / a day ago
                return (withoutSuffix || isFuture) ? 'deň' : 'dňom';
            case 'dd': // 9 days / in 9 days / 9 days ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'dni' : 'dní');
                } else {
                    return result + 'dňami';
                }
                break;
            case 'M':  // a month / in a month / a month ago
                return (withoutSuffix || isFuture) ? 'mesiac' : 'mesiacom';
            case 'MM': // 9 months / in 9 months / 9 months ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'mesiace' : 'mesiacov');
                } else {
                    return result + 'mesiacmi';
                }
                break;
            case 'y':  // a year / in a year / a year ago
                return (withoutSuffix || isFuture) ? 'rok' : 'rokom';
            case 'yy': // 9 years / in 9 years / 9 years ago
                if (withoutSuffix || isFuture) {
                    return result + (plural$5(number) ? 'roky' : 'rokov');
                } else {
                    return result + 'rokmi';
                }
                break;
        }
    }

    hooks.defineLocale('sk', {
        months : months$9,
        monthsShort : monthsShort$6,
        weekdays : 'nedeľa_pondelok_utorok_streda_štvrtok_piatok_sobota'.split('_'),
        weekdaysShort : 'ne_po_ut_st_št_pi_so'.split('_'),
        weekdaysMin : 'ne_po_ut_st_št_pi_so'.split('_'),
        longDateFormat : {
            LT: 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd D. MMMM YYYY H:mm'
        },
        calendar : {
            sameDay: '[dnes o] LT',
            nextDay: '[zajtra o] LT',
            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[v nedeľu o] LT';
                    case 1:
                    case 2:
                        return '[v] dddd [o] LT';
                    case 3:
                        return '[v stredu o] LT';
                    case 4:
                        return '[vo štvrtok o] LT';
                    case 5:
                        return '[v piatok o] LT';
                    case 6:
                        return '[v sobotu o] LT';
                }
            },
            lastDay: '[včera o] LT',
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[minulú nedeľu o] LT';
                    case 1:
                    case 2:
                        return '[minulý] dddd [o] LT';
                    case 3:
                        return '[minulú stredu o] LT';
                    case 4:
                    case 5:
                        return '[minulý] dddd [o] LT';
                    case 6:
                        return '[minulú sobotu o] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'za %s',
            past : 'pred %s',
            s : translate$9,
            ss : translate$9,
            m : translate$9,
            mm : translate$9,
            h : translate$9,
            hh : translate$9,
            d : translate$9,
            dd : translate$9,
            M : translate$9,
            MM : translate$9,
            y : translate$9,
            yy : translate$9
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function processRelativeTime$6(number, withoutSuffix, key, isFuture) {
        var result = number + ' ';
        switch (key) {
            case 's':
                return withoutSuffix || isFuture ? 'nekaj sekund' : 'nekaj sekundami';
            case 'ss':
                if (number === 1) {
                    result += withoutSuffix ? 'sekundo' : 'sekundi';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'sekundi' : 'sekundah';
                } else if (number < 5) {
                    result += withoutSuffix || isFuture ? 'sekunde' : 'sekundah';
                } else {
                    result += 'sekund';
                }
                return result;
            case 'm':
                return withoutSuffix ? 'ena minuta' : 'eno minuto';
            case 'mm':
                if (number === 1) {
                    result += withoutSuffix ? 'minuta' : 'minuto';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'minuti' : 'minutama';
                } else if (number < 5) {
                    result += withoutSuffix || isFuture ? 'minute' : 'minutami';
                } else {
                    result += withoutSuffix || isFuture ? 'minut' : 'minutami';
                }
                return result;
            case 'h':
                return withoutSuffix ? 'ena ura' : 'eno uro';
            case 'hh':
                if (number === 1) {
                    result += withoutSuffix ? 'ura' : 'uro';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'uri' : 'urama';
                } else if (number < 5) {
                    result += withoutSuffix || isFuture ? 'ure' : 'urami';
                } else {
                    result += withoutSuffix || isFuture ? 'ur' : 'urami';
                }
                return result;
            case 'd':
                return withoutSuffix || isFuture ? 'en dan' : 'enim dnem';
            case 'dd':
                if (number === 1) {
                    result += withoutSuffix || isFuture ? 'dan' : 'dnem';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'dni' : 'dnevoma';
                } else {
                    result += withoutSuffix || isFuture ? 'dni' : 'dnevi';
                }
                return result;
            case 'M':
                return withoutSuffix || isFuture ? 'en mesec' : 'enim mesecem';
            case 'MM':
                if (number === 1) {
                    result += withoutSuffix || isFuture ? 'mesec' : 'mesecem';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'meseca' : 'mesecema';
                } else if (number < 5) {
                    result += withoutSuffix || isFuture ? 'mesece' : 'meseci';
                } else {
                    result += withoutSuffix || isFuture ? 'mesecev' : 'meseci';
                }
                return result;
            case 'y':
                return withoutSuffix || isFuture ? 'eno leto' : 'enim letom';
            case 'yy':
                if (number === 1) {
                    result += withoutSuffix || isFuture ? 'leto' : 'letom';
                } else if (number === 2) {
                    result += withoutSuffix || isFuture ? 'leti' : 'letoma';
                } else if (number < 5) {
                    result += withoutSuffix || isFuture ? 'leta' : 'leti';
                } else {
                    result += withoutSuffix || isFuture ? 'let' : 'leti';
                }
                return result;
        }
    }

    hooks.defineLocale('sl', {
        months : 'januar_februar_marec_april_maj_junij_julij_avgust_september_oktober_november_december'.split('_'),
        monthsShort : 'jan._feb._mar._apr._maj._jun._jul._avg._sep._okt._nov._dec.'.split('_'),
        monthsParseExact: true,
        weekdays : 'nedelja_ponedeljek_torek_sreda_četrtek_petek_sobota'.split('_'),
        weekdaysShort : 'ned._pon._tor._sre._čet._pet._sob.'.split('_'),
        weekdaysMin : 'ne_po_to_sr_če_pe_so'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM YYYY',
            LLL : 'D. MMMM YYYY H:mm',
            LLLL : 'dddd, D. MMMM YYYY H:mm'
        },
        calendar : {
            sameDay  : '[danes ob] LT',
            nextDay  : '[jutri ob] LT',

            nextWeek : function () {
                switch (this.day()) {
                    case 0:
                        return '[v] [nedeljo] [ob] LT';
                    case 3:
                        return '[v] [sredo] [ob] LT';
                    case 6:
                        return '[v] [soboto] [ob] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[v] dddd [ob] LT';
                }
            },
            lastDay  : '[včeraj ob] LT',
            lastWeek : function () {
                switch (this.day()) {
                    case 0:
                        return '[prejšnjo] [nedeljo] [ob] LT';
                    case 3:
                        return '[prejšnjo] [sredo] [ob] LT';
                    case 6:
                        return '[prejšnjo] [soboto] [ob] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[prejšnji] dddd [ob] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'čez %s',
            past   : 'pred %s',
            s      : processRelativeTime$6,
            ss     : processRelativeTime$6,
            m      : processRelativeTime$6,
            mm     : processRelativeTime$6,
            h      : processRelativeTime$6,
            hh     : processRelativeTime$6,
            d      : processRelativeTime$6,
            dd     : processRelativeTime$6,
            M      : processRelativeTime$6,
            MM     : processRelativeTime$6,
            y      : processRelativeTime$6,
            yy     : processRelativeTime$6
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('sq', {
        months : 'Janar_Shkurt_Mars_Prill_Maj_Qershor_Korrik_Gusht_Shtator_Tetor_Nëntor_Dhjetor'.split('_'),
        monthsShort : 'Jan_Shk_Mar_Pri_Maj_Qer_Kor_Gus_Sht_Tet_Nën_Dhj'.split('_'),
        weekdays : 'E Diel_E Hënë_E Martë_E Mërkurë_E Enjte_E Premte_E Shtunë'.split('_'),
        weekdaysShort : 'Die_Hën_Mar_Mër_Enj_Pre_Sht'.split('_'),
        weekdaysMin : 'D_H_Ma_Më_E_P_Sh'.split('_'),
        weekdaysParseExact : true,
        meridiemParse: /PD|MD/,
        isPM: function (input) {
            return input.charAt(0) === 'M';
        },
        meridiem : function (hours, minutes, isLower) {
            return hours < 12 ? 'PD' : 'MD';
        },
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Sot në] LT',
            nextDay : '[Nesër në] LT',
            nextWeek : 'dddd [në] LT',
            lastDay : '[Dje në] LT',
            lastWeek : 'dddd [e kaluar në] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'në %s',
            past : '%s më parë',
            s : 'disa sekonda',
            ss : '%d sekonda',
            m : 'një minutë',
            mm : '%d minuta',
            h : 'një orë',
            hh : '%d orë',
            d : 'një ditë',
            dd : '%d ditë',
            M : 'një muaj',
            MM : '%d muaj',
            y : 'një vit',
            yy : '%d vite'
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var translator$1 = {
        words: { //Different grammatical cases
            ss: ['секунда', 'секунде', 'секунди'],
            m: ['један минут', 'једне минуте'],
            mm: ['минут', 'минуте', 'минута'],
            h: ['један сат', 'једног сата'],
            hh: ['сат', 'сата', 'сати'],
            dd: ['дан', 'дана', 'дана'],
            MM: ['месец', 'месеца', 'месеци'],
            yy: ['година', 'године', 'година']
        },
        correctGrammaticalCase: function (number, wordKey) {
            return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
        },
        translate: function (number, withoutSuffix, key) {
            var wordKey = translator$1.words[key];
            if (key.length === 1) {
                return withoutSuffix ? wordKey[0] : wordKey[1];
            } else {
                return number + ' ' + translator$1.correctGrammaticalCase(number, wordKey);
            }
        }
    };

    hooks.defineLocale('sr-cyrl', {
        months: 'јануар_фебруар_март_април_мај_јун_јул_август_септембар_октобар_новембар_децембар'.split('_'),
        monthsShort: 'јан._феб._мар._апр._мај_јун_јул_авг._сеп._окт._нов._дец.'.split('_'),
        monthsParseExact: true,
        weekdays: 'недеља_понедељак_уторак_среда_четвртак_петак_субота'.split('_'),
        weekdaysShort: 'нед._пон._уто._сре._чет._пет._суб.'.split('_'),
        weekdaysMin: 'не_по_ут_ср_че_пе_су'.split('_'),
        weekdaysParseExact : true,
        longDateFormat: {
            LT: 'H:mm',
            LTS : 'H:mm:ss',
            L: 'DD.MM.YYYY',
            LL: 'D. MMMM YYYY',
            LLL: 'D. MMMM YYYY H:mm',
            LLLL: 'dddd, D. MMMM YYYY H:mm'
        },
        calendar: {
            sameDay: '[данас у] LT',
            nextDay: '[сутра у] LT',
            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[у] [недељу] [у] LT';
                    case 3:
                        return '[у] [среду] [у] LT';
                    case 6:
                        return '[у] [суботу] [у] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[у] dddd [у] LT';
                }
            },
            lastDay  : '[јуче у] LT',
            lastWeek : function () {
                var lastWeekDays = [
                    '[прошле] [недеље] [у] LT',
                    '[прошлог] [понедељка] [у] LT',
                    '[прошлог] [уторка] [у] LT',
                    '[прошле] [среде] [у] LT',
                    '[прошлог] [четвртка] [у] LT',
                    '[прошлог] [петка] [у] LT',
                    '[прошле] [суботе] [у] LT'
                ];
                return lastWeekDays[this.day()];
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'за %s',
            past   : 'пре %s',
            s      : 'неколико секунди',
            ss     : translator$1.translate,
            m      : translator$1.translate,
            mm     : translator$1.translate,
            h      : translator$1.translate,
            hh     : translator$1.translate,
            d      : 'дан',
            dd     : translator$1.translate,
            M      : 'месец',
            MM     : translator$1.translate,
            y      : 'годину',
            yy     : translator$1.translate
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var translator$2 = {
        words: { //Different grammatical cases
            ss: ['sekunda', 'sekunde', 'sekundi'],
            m: ['jedan minut', 'jedne minute'],
            mm: ['minut', 'minute', 'minuta'],
            h: ['jedan sat', 'jednog sata'],
            hh: ['sat', 'sata', 'sati'],
            dd: ['dan', 'dana', 'dana'],
            MM: ['mesec', 'meseca', 'meseci'],
            yy: ['godina', 'godine', 'godina']
        },
        correctGrammaticalCase: function (number, wordKey) {
            return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
        },
        translate: function (number, withoutSuffix, key) {
            var wordKey = translator$2.words[key];
            if (key.length === 1) {
                return withoutSuffix ? wordKey[0] : wordKey[1];
            } else {
                return number + ' ' + translator$2.correctGrammaticalCase(number, wordKey);
            }
        }
    };

    hooks.defineLocale('sr', {
        months: 'januar_februar_mart_april_maj_jun_jul_avgust_septembar_oktobar_novembar_decembar'.split('_'),
        monthsShort: 'jan._feb._mar._apr._maj_jun_jul_avg._sep._okt._nov._dec.'.split('_'),
        monthsParseExact: true,
        weekdays: 'nedelja_ponedeljak_utorak_sreda_četvrtak_petak_subota'.split('_'),
        weekdaysShort: 'ned._pon._uto._sre._čet._pet._sub.'.split('_'),
        weekdaysMin: 'ne_po_ut_sr_če_pe_su'.split('_'),
        weekdaysParseExact : true,
        longDateFormat: {
            LT: 'H:mm',
            LTS : 'H:mm:ss',
            L: 'DD.MM.YYYY',
            LL: 'D. MMMM YYYY',
            LLL: 'D. MMMM YYYY H:mm',
            LLLL: 'dddd, D. MMMM YYYY H:mm'
        },
        calendar: {
            sameDay: '[danas u] LT',
            nextDay: '[sutra u] LT',
            nextWeek: function () {
                switch (this.day()) {
                    case 0:
                        return '[u] [nedelju] [u] LT';
                    case 3:
                        return '[u] [sredu] [u] LT';
                    case 6:
                        return '[u] [subotu] [u] LT';
                    case 1:
                    case 2:
                    case 4:
                    case 5:
                        return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[juče u] LT',
            lastWeek : function () {
                var lastWeekDays = [
                    '[prošle] [nedelje] [u] LT',
                    '[prošlog] [ponedeljka] [u] LT',
                    '[prošlog] [utorka] [u] LT',
                    '[prošle] [srede] [u] LT',
                    '[prošlog] [četvrtka] [u] LT',
                    '[prošlog] [petka] [u] LT',
                    '[prošle] [subote] [u] LT'
                ];
                return lastWeekDays[this.day()];
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : 'za %s',
            past   : 'pre %s',
            s      : 'nekoliko sekundi',
            ss     : translator$2.translate,
            m      : translator$2.translate,
            mm     : translator$2.translate,
            h      : translator$2.translate,
            hh     : translator$2.translate,
            d      : 'dan',
            dd     : translator$2.translate,
            M      : 'mesec',
            MM     : translator$2.translate,
            y      : 'godinu',
            yy     : translator$2.translate
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('ss', {
        months : "Bhimbidvwane_Indlovana_Indlov'lenkhulu_Mabasa_Inkhwekhweti_Inhlaba_Kholwane_Ingci_Inyoni_Imphala_Lweti_Ingongoni".split('_'),
        monthsShort : 'Bhi_Ina_Inu_Mab_Ink_Inh_Kho_Igc_Iny_Imp_Lwe_Igo'.split('_'),
        weekdays : 'Lisontfo_Umsombuluko_Lesibili_Lesitsatfu_Lesine_Lesihlanu_Umgcibelo'.split('_'),
        weekdaysShort : 'Lis_Umb_Lsb_Les_Lsi_Lsh_Umg'.split('_'),
        weekdaysMin : 'Li_Us_Lb_Lt_Ls_Lh_Ug'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendar : {
            sameDay : '[Namuhla nga] LT',
            nextDay : '[Kusasa nga] LT',
            nextWeek : 'dddd [nga] LT',
            lastDay : '[Itolo nga] LT',
            lastWeek : 'dddd [leliphelile] [nga] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'nga %s',
            past : 'wenteka nga %s',
            s : 'emizuzwana lomcane',
            ss : '%d mzuzwana',
            m : 'umzuzu',
            mm : '%d emizuzu',
            h : 'lihora',
            hh : '%d emahora',
            d : 'lilanga',
            dd : '%d emalanga',
            M : 'inyanga',
            MM : '%d tinyanga',
            y : 'umnyaka',
            yy : '%d iminyaka'
        },
        meridiemParse: /ekuseni|emini|entsambama|ebusuku/,
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'ekuseni';
            } else if (hours < 15) {
                return 'emini';
            } else if (hours < 19) {
                return 'entsambama';
            } else {
                return 'ebusuku';
            }
        },
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'ekuseni') {
                return hour;
            } else if (meridiem === 'emini') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'entsambama' || meridiem === 'ebusuku') {
                if (hour === 0) {
                    return 0;
                }
                return hour + 12;
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}/,
        ordinal : '%d',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('sv', {
        months : 'januari_februari_mars_april_maj_juni_juli_augusti_september_oktober_november_december'.split('_'),
        monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec'.split('_'),
        weekdays : 'söndag_måndag_tisdag_onsdag_torsdag_fredag_lördag'.split('_'),
        weekdaysShort : 'sön_mån_tis_ons_tor_fre_lör'.split('_'),
        weekdaysMin : 'sö_må_ti_on_to_fr_lö'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY-MM-DD',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY [kl.] HH:mm',
            LLLL : 'dddd D MMMM YYYY [kl.] HH:mm',
            lll : 'D MMM YYYY HH:mm',
            llll : 'ddd D MMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Idag] LT',
            nextDay: '[Imorgon] LT',
            lastDay: '[Igår] LT',
            nextWeek: '[På] dddd LT',
            lastWeek: '[I] dddd[s] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'om %s',
            past : 'för %s sedan',
            s : 'några sekunder',
            ss : '%d sekunder',
            m : 'en minut',
            mm : '%d minuter',
            h : 'en timme',
            hh : '%d timmar',
            d : 'en dag',
            dd : '%d dagar',
            M : 'en månad',
            MM : '%d månader',
            y : 'ett år',
            yy : '%d år'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(e|a)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'e' :
                (b === 1) ? 'a' :
                (b === 2) ? 'a' :
                (b === 3) ? 'e' : 'e';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('sw', {
        months : 'Januari_Februari_Machi_Aprili_Mei_Juni_Julai_Agosti_Septemba_Oktoba_Novemba_Desemba'.split('_'),
        monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ago_Sep_Okt_Nov_Des'.split('_'),
        weekdays : 'Jumapili_Jumatatu_Jumanne_Jumatano_Alhamisi_Ijumaa_Jumamosi'.split('_'),
        weekdaysShort : 'Jpl_Jtat_Jnne_Jtan_Alh_Ijm_Jmos'.split('_'),
        weekdaysMin : 'J2_J3_J4_J5_Al_Ij_J1'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[leo saa] LT',
            nextDay : '[kesho saa] LT',
            nextWeek : '[wiki ijayo] dddd [saat] LT',
            lastDay : '[jana] LT',
            lastWeek : '[wiki iliyopita] dddd [saat] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s baadaye',
            past : 'tokea %s',
            s : 'hivi punde',
            ss : 'sekunde %d',
            m : 'dakika moja',
            mm : 'dakika %d',
            h : 'saa limoja',
            hh : 'masaa %d',
            d : 'siku moja',
            dd : 'masiku %d',
            M : 'mwezi mmoja',
            MM : 'miezi %d',
            y : 'mwaka mmoja',
            yy : 'miaka %d'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var symbolMap$f = {
        '1': '௧',
        '2': '௨',
        '3': '௩',
        '4': '௪',
        '5': '௫',
        '6': '௬',
        '7': '௭',
        '8': '௮',
        '9': '௯',
        '0': '௦'
    }, numberMap$e = {
        '௧': '1',
        '௨': '2',
        '௩': '3',
        '௪': '4',
        '௫': '5',
        '௬': '6',
        '௭': '7',
        '௮': '8',
        '௯': '9',
        '௦': '0'
    };

    hooks.defineLocale('ta', {
        months : 'ஜனவரி_பிப்ரவரி_மார்ச்_ஏப்ரல்_மே_ஜூன்_ஜூலை_ஆகஸ்ட்_செப்டெம்பர்_அக்டோபர்_நவம்பர்_டிசம்பர்'.split('_'),
        monthsShort : 'ஜனவரி_பிப்ரவரி_மார்ச்_ஏப்ரல்_மே_ஜூன்_ஜூலை_ஆகஸ்ட்_செப்டெம்பர்_அக்டோபர்_நவம்பர்_டிசம்பர்'.split('_'),
        weekdays : 'ஞாயிற்றுக்கிழமை_திங்கட்கிழமை_செவ்வாய்கிழமை_புதன்கிழமை_வியாழக்கிழமை_வெள்ளிக்கிழமை_சனிக்கிழமை'.split('_'),
        weekdaysShort : 'ஞாயிறு_திங்கள்_செவ்வாய்_புதன்_வியாழன்_வெள்ளி_சனி'.split('_'),
        weekdaysMin : 'ஞா_தி_செ_பு_வி_வெ_ச'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, HH:mm',
            LLLL : 'dddd, D MMMM YYYY, HH:mm'
        },
        calendar : {
            sameDay : '[இன்று] LT',
            nextDay : '[நாளை] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[நேற்று] LT',
            lastWeek : '[கடந்த வாரம்] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s இல்',
            past : '%s முன்',
            s : 'ஒரு சில விநாடிகள்',
            ss : '%d விநாடிகள்',
            m : 'ஒரு நிமிடம்',
            mm : '%d நிமிடங்கள்',
            h : 'ஒரு மணி நேரம்',
            hh : '%d மணி நேரம்',
            d : 'ஒரு நாள்',
            dd : '%d நாட்கள்',
            M : 'ஒரு மாதம்',
            MM : '%d மாதங்கள்',
            y : 'ஒரு வருடம்',
            yy : '%d ஆண்டுகள்'
        },
        dayOfMonthOrdinalParse: /\d{1,2}வது/,
        ordinal : function (number) {
            return number + 'வது';
        },
        preparse: function (string) {
            return string.replace(/[௧௨௩௪௫௬௭௮௯௦]/g, function (match) {
                return numberMap$e[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap$f[match];
            });
        },
        // refer http://ta.wikipedia.org/s/1er1
        meridiemParse: /யாமம்|வைகறை|காலை|நண்பகல்|எற்பாடு|மாலை/,
        meridiem : function (hour, minute, isLower) {
            if (hour < 2) {
                return ' யாமம்';
            } else if (hour < 6) {
                return ' வைகறை';  // வைகறை
            } else if (hour < 10) {
                return ' காலை'; // காலை
            } else if (hour < 14) {
                return ' நண்பகல்'; // நண்பகல்
            } else if (hour < 18) {
                return ' எற்பாடு'; // எற்பாடு
            } else if (hour < 22) {
                return ' மாலை'; // மாலை
            } else {
                return ' யாமம்';
            }
        },
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'யாமம்') {
                return hour < 2 ? hour : hour + 12;
            } else if (meridiem === 'வைகறை' || meridiem === 'காலை') {
                return hour;
            } else if (meridiem === 'நண்பகல்') {
                return hour >= 10 ? hour : hour + 12;
            } else {
                return hour + 12;
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('te', {
        months : 'జనవరి_ఫిబ్రవరి_మార్చి_ఏప్రిల్_మే_జూన్_జులై_ఆగస్టు_సెప్టెంబర్_అక్టోబర్_నవంబర్_డిసెంబర్'.split('_'),
        monthsShort : 'జన._ఫిబ్ర._మార్చి_ఏప్రి._మే_జూన్_జులై_ఆగ._సెప్._అక్టో._నవ._డిసె.'.split('_'),
        monthsParseExact : true,
        weekdays : 'ఆదివారం_సోమవారం_మంగళవారం_బుధవారం_గురువారం_శుక్రవారం_శనివారం'.split('_'),
        weekdaysShort : 'ఆది_సోమ_మంగళ_బుధ_గురు_శుక్ర_శని'.split('_'),
        weekdaysMin : 'ఆ_సో_మం_బు_గు_శు_శ'.split('_'),
        longDateFormat : {
            LT : 'A h:mm',
            LTS : 'A h:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY, A h:mm',
            LLLL : 'dddd, D MMMM YYYY, A h:mm'
        },
        calendar : {
            sameDay : '[నేడు] LT',
            nextDay : '[రేపు] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[నిన్న] LT',
            lastWeek : '[గత] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s లో',
            past : '%s క్రితం',
            s : 'కొన్ని క్షణాలు',
            ss : '%d సెకన్లు',
            m : 'ఒక నిమిషం',
            mm : '%d నిమిషాలు',
            h : 'ఒక గంట',
            hh : '%d గంటలు',
            d : 'ఒక రోజు',
            dd : '%d రోజులు',
            M : 'ఒక నెల',
            MM : '%d నెలలు',
            y : 'ఒక సంవత్సరం',
            yy : '%d సంవత్సరాలు'
        },
        dayOfMonthOrdinalParse : /\d{1,2}వ/,
        ordinal : '%dవ',
        meridiemParse: /రాత్రి|ఉదయం|మధ్యాహ్నం|సాయంత్రం/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'రాత్రి') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'ఉదయం') {
                return hour;
            } else if (meridiem === 'మధ్యాహ్నం') {
                return hour >= 10 ? hour : hour + 12;
            } else if (meridiem === 'సాయంత్రం') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'రాత్రి';
            } else if (hour < 10) {
                return 'ఉదయం';
            } else if (hour < 17) {
                return 'మధ్యాహ్నం';
            } else if (hour < 20) {
                return 'సాయంత్రం';
            } else {
                return 'రాత్రి';
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 6th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('tet', {
        months : 'Janeiru_Fevereiru_Marsu_Abril_Maiu_Juñu_Jullu_Agustu_Setembru_Outubru_Novembru_Dezembru'.split('_'),
        monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
        weekdays : 'Domingu_Segunda_Tersa_Kuarta_Kinta_Sesta_Sabadu'.split('_'),
        weekdaysShort : 'Dom_Seg_Ters_Kua_Kint_Sest_Sab'.split('_'),
        weekdaysMin : 'Do_Seg_Te_Ku_Ki_Ses_Sa'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Ohin iha] LT',
            nextDay: '[Aban iha] LT',
            nextWeek: 'dddd [iha] LT',
            lastDay: '[Horiseik iha] LT',
            lastWeek: 'dddd [semana kotuk] [iha] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'iha %s',
            past : '%s liuba',
            s : 'minutu balun',
            ss : 'minutu %d',
            m : 'minutu ida',
            mm : 'minutu %d',
            h : 'oras ida',
            hh : 'oras %d',
            d : 'loron ida',
            dd : 'loron %d',
            M : 'fulan ida',
            MM : 'fulan %d',
            y : 'tinan ida',
            yy : 'tinan %d'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var suffixes$3 = {
        0: '-ум',
        1: '-ум',
        2: '-юм',
        3: '-юм',
        4: '-ум',
        5: '-ум',
        6: '-ум',
        7: '-ум',
        8: '-ум',
        9: '-ум',
        10: '-ум',
        12: '-ум',
        13: '-ум',
        20: '-ум',
        30: '-юм',
        40: '-ум',
        50: '-ум',
        60: '-ум',
        70: '-ум',
        80: '-ум',
        90: '-ум',
        100: '-ум'
    };

    hooks.defineLocale('tg', {
        months : 'январ_феврал_март_апрел_май_июн_июл_август_сентябр_октябр_ноябр_декабр'.split('_'),
        monthsShort : 'янв_фев_мар_апр_май_июн_июл_авг_сен_окт_ноя_дек'.split('_'),
        weekdays : 'якшанбе_душанбе_сешанбе_чоршанбе_панҷшанбе_ҷумъа_шанбе'.split('_'),
        weekdaysShort : 'яшб_дшб_сшб_чшб_пшб_ҷум_шнб'.split('_'),
        weekdaysMin : 'яш_дш_сш_чш_пш_ҷм_шб'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[Имрӯз соати] LT',
            nextDay : '[Пагоҳ соати] LT',
            lastDay : '[Дирӯз соати] LT',
            nextWeek : 'dddd[и] [ҳафтаи оянда соати] LT',
            lastWeek : 'dddd[и] [ҳафтаи гузашта соати] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'баъди %s',
            past : '%s пеш',
            s : 'якчанд сония',
            m : 'як дақиқа',
            mm : '%d дақиқа',
            h : 'як соат',
            hh : '%d соат',
            d : 'як рӯз',
            dd : '%d рӯз',
            M : 'як моҳ',
            MM : '%d моҳ',
            y : 'як сол',
            yy : '%d сол'
        },
        meridiemParse: /шаб|субҳ|рӯз|бегоҳ/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === 'шаб') {
                return hour < 4 ? hour : hour + 12;
            } else if (meridiem === 'субҳ') {
                return hour;
            } else if (meridiem === 'рӯз') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === 'бегоҳ') {
                return hour + 12;
            }
        },
        meridiem: function (hour, minute, isLower) {
            if (hour < 4) {
                return 'шаб';
            } else if (hour < 11) {
                return 'субҳ';
            } else if (hour < 16) {
                return 'рӯз';
            } else if (hour < 19) {
                return 'бегоҳ';
            } else {
                return 'шаб';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(ум|юм)/,
        ordinal: function (number) {
            var a = number % 10,
                b = number >= 100 ? 100 : null;
            return number + (suffixes$3[number] || suffixes$3[a] || suffixes$3[b]);
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('th', {
        months : 'มกราคม_กุมภาพันธ์_มีนาคม_เมษายน_พฤษภาคม_มิถุนายน_กรกฎาคม_สิงหาคม_กันยายน_ตุลาคม_พฤศจิกายน_ธันวาคม'.split('_'),
        monthsShort : 'ม.ค._ก.พ._มี.ค._เม.ย._พ.ค._มิ.ย._ก.ค._ส.ค._ก.ย._ต.ค._พ.ย._ธ.ค.'.split('_'),
        monthsParseExact: true,
        weekdays : 'อาทิตย์_จันทร์_อังคาร_พุธ_พฤหัสบดี_ศุกร์_เสาร์'.split('_'),
        weekdaysShort : 'อาทิตย์_จันทร์_อังคาร_พุธ_พฤหัส_ศุกร์_เสาร์'.split('_'), // yes, three characters difference
        weekdaysMin : 'อา._จ._อ._พ._พฤ._ศ._ส.'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'H:mm',
            LTS : 'H:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY เวลา H:mm',
            LLLL : 'วันddddที่ D MMMM YYYY เวลา H:mm'
        },
        meridiemParse: /ก่อนเที่ยง|หลังเที่ยง/,
        isPM: function (input) {
            return input === 'หลังเที่ยง';
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'ก่อนเที่ยง';
            } else {
                return 'หลังเที่ยง';
            }
        },
        calendar : {
            sameDay : '[วันนี้ เวลา] LT',
            nextDay : '[พรุ่งนี้ เวลา] LT',
            nextWeek : 'dddd[หน้า เวลา] LT',
            lastDay : '[เมื่อวานนี้ เวลา] LT',
            lastWeek : '[วัน]dddd[ที่แล้ว เวลา] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'อีก %s',
            past : '%sที่แล้ว',
            s : 'ไม่กี่วินาที',
            ss : '%d วินาที',
            m : '1 นาที',
            mm : '%d นาที',
            h : '1 ชั่วโมง',
            hh : '%d ชั่วโมง',
            d : '1 วัน',
            dd : '%d วัน',
            M : '1 เดือน',
            MM : '%d เดือน',
            y : '1 ปี',
            yy : '%d ปี'
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('tl-ph', {
        months : 'Enero_Pebrero_Marso_Abril_Mayo_Hunyo_Hulyo_Agosto_Setyembre_Oktubre_Nobyembre_Disyembre'.split('_'),
        monthsShort : 'Ene_Peb_Mar_Abr_May_Hun_Hul_Ago_Set_Okt_Nob_Dis'.split('_'),
        weekdays : 'Linggo_Lunes_Martes_Miyerkules_Huwebes_Biyernes_Sabado'.split('_'),
        weekdaysShort : 'Lin_Lun_Mar_Miy_Huw_Biy_Sab'.split('_'),
        weekdaysMin : 'Li_Lu_Ma_Mi_Hu_Bi_Sab'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'MM/D/YYYY',
            LL : 'MMMM D, YYYY',
            LLL : 'MMMM D, YYYY HH:mm',
            LLLL : 'dddd, MMMM DD, YYYY HH:mm'
        },
        calendar : {
            sameDay: 'LT [ngayong araw]',
            nextDay: '[Bukas ng] LT',
            nextWeek: 'LT [sa susunod na] dddd',
            lastDay: 'LT [kahapon]',
            lastWeek: 'LT [noong nakaraang] dddd',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'sa loob ng %s',
            past : '%s ang nakalipas',
            s : 'ilang segundo',
            ss : '%d segundo',
            m : 'isang minuto',
            mm : '%d minuto',
            h : 'isang oras',
            hh : '%d oras',
            d : 'isang araw',
            dd : '%d araw',
            M : 'isang buwan',
            MM : '%d buwan',
            y : 'isang taon',
            yy : '%d taon'
        },
        dayOfMonthOrdinalParse: /\d{1,2}/,
        ordinal : function (number) {
            return number;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var numbersNouns = 'pagh_wa’_cha’_wej_loS_vagh_jav_Soch_chorgh_Hut'.split('_');

    function translateFuture(output) {
        var time = output;
        time = (output.indexOf('jaj') !== -1) ?
        time.slice(0, -3) + 'leS' :
        (output.indexOf('jar') !== -1) ?
        time.slice(0, -3) + 'waQ' :
        (output.indexOf('DIS') !== -1) ?
        time.slice(0, -3) + 'nem' :
        time + ' pIq';
        return time;
    }

    function translatePast(output) {
        var time = output;
        time = (output.indexOf('jaj') !== -1) ?
        time.slice(0, -3) + 'Hu’' :
        (output.indexOf('jar') !== -1) ?
        time.slice(0, -3) + 'wen' :
        (output.indexOf('DIS') !== -1) ?
        time.slice(0, -3) + 'ben' :
        time + ' ret';
        return time;
    }

    function translate$a(number, withoutSuffix, string, isFuture) {
        var numberNoun = numberAsNoun(number);
        switch (string) {
            case 'ss':
                return numberNoun + ' lup';
            case 'mm':
                return numberNoun + ' tup';
            case 'hh':
                return numberNoun + ' rep';
            case 'dd':
                return numberNoun + ' jaj';
            case 'MM':
                return numberNoun + ' jar';
            case 'yy':
                return numberNoun + ' DIS';
        }
    }

    function numberAsNoun(number) {
        var hundred = Math.floor((number % 1000) / 100),
        ten = Math.floor((number % 100) / 10),
        one = number % 10,
        word = '';
        if (hundred > 0) {
            word += numbersNouns[hundred] + 'vatlh';
        }
        if (ten > 0) {
            word += ((word !== '') ? ' ' : '') + numbersNouns[ten] + 'maH';
        }
        if (one > 0) {
            word += ((word !== '') ? ' ' : '') + numbersNouns[one];
        }
        return (word === '') ? 'pagh' : word;
    }

    hooks.defineLocale('tlh', {
        months : 'tera’ jar wa’_tera’ jar cha’_tera’ jar wej_tera’ jar loS_tera’ jar vagh_tera’ jar jav_tera’ jar Soch_tera’ jar chorgh_tera’ jar Hut_tera’ jar wa’maH_tera’ jar wa’maH wa’_tera’ jar wa’maH cha’'.split('_'),
        monthsShort : 'jar wa’_jar cha’_jar wej_jar loS_jar vagh_jar jav_jar Soch_jar chorgh_jar Hut_jar wa’maH_jar wa’maH wa’_jar wa’maH cha’'.split('_'),
        monthsParseExact : true,
        weekdays : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
        weekdaysShort : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
        weekdaysMin : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[DaHjaj] LT',
            nextDay: '[wa’leS] LT',
            nextWeek: 'LLL',
            lastDay: '[wa’Hu’] LT',
            lastWeek: 'LLL',
            sameElse: 'L'
        },
        relativeTime : {
            future : translateFuture,
            past : translatePast,
            s : 'puS lup',
            ss : translate$a,
            m : 'wa’ tup',
            mm : translate$a,
            h : 'wa’ rep',
            hh : translate$a,
            d : 'wa’ jaj',
            dd : translate$a,
            M : 'wa’ jar',
            MM : translate$a,
            y : 'wa’ DIS',
            yy : translate$a
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    var suffixes$4 = {
        1: '\'inci',
        5: '\'inci',
        8: '\'inci',
        70: '\'inci',
        80: '\'inci',
        2: '\'nci',
        7: '\'nci',
        20: '\'nci',
        50: '\'nci',
        3: '\'üncü',
        4: '\'üncü',
        100: '\'üncü',
        6: '\'ncı',
        9: '\'uncu',
        10: '\'uncu',
        30: '\'uncu',
        60: '\'ıncı',
        90: '\'ıncı'
    };

    hooks.defineLocale('tr', {
        months : 'Ocak_Şubat_Mart_Nisan_Mayıs_Haziran_Temmuz_Ağustos_Eylül_Ekim_Kasım_Aralık'.split('_'),
        monthsShort : 'Oca_Şub_Mar_Nis_May_Haz_Tem_Ağu_Eyl_Eki_Kas_Ara'.split('_'),
        weekdays : 'Pazar_Pazartesi_Salı_Çarşamba_Perşembe_Cuma_Cumartesi'.split('_'),
        weekdaysShort : 'Paz_Pts_Sal_Çar_Per_Cum_Cts'.split('_'),
        weekdaysMin : 'Pz_Pt_Sa_Ça_Pe_Cu_Ct'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[bugün saat] LT',
            nextDay : '[yarın saat] LT',
            nextWeek : '[gelecek] dddd [saat] LT',
            lastDay : '[dün] LT',
            lastWeek : '[geçen] dddd [saat] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s sonra',
            past : '%s önce',
            s : 'birkaç saniye',
            ss : '%d saniye',
            m : 'bir dakika',
            mm : '%d dakika',
            h : 'bir saat',
            hh : '%d saat',
            d : 'bir gün',
            dd : '%d gün',
            M : 'bir ay',
            MM : '%d ay',
            y : 'bir yıl',
            yy : '%d yıl'
        },
        ordinal: function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'Do':
                case 'DD':
                    return number;
                default:
                    if (number === 0) {  // special case for zero
                        return number + '\'ıncı';
                    }
                    var a = number % 10,
                        b = number % 100 - a,
                        c = number >= 100 ? 100 : null;
                    return number + (suffixes$4[a] || suffixes$4[b] || suffixes$4[c]);
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    // After the year there should be a slash and the amount of years since December 26, 1979 in Roman numerals.
    // This is currently too difficult (maybe even impossible) to add.
    hooks.defineLocale('tzl', {
        months : 'Januar_Fevraglh_Març_Avrïu_Mai_Gün_Julia_Guscht_Setemvar_Listopäts_Noemvar_Zecemvar'.split('_'),
        monthsShort : 'Jan_Fev_Mar_Avr_Mai_Gün_Jul_Gus_Set_Lis_Noe_Zec'.split('_'),
        weekdays : 'Súladi_Lúneçi_Maitzi_Márcuri_Xhúadi_Viénerçi_Sáturi'.split('_'),
        weekdaysShort : 'Súl_Lún_Mai_Már_Xhú_Vié_Sát'.split('_'),
        weekdaysMin : 'Sú_Lú_Ma_Má_Xh_Vi_Sá'.split('_'),
        longDateFormat : {
            LT : 'HH.mm',
            LTS : 'HH.mm.ss',
            L : 'DD.MM.YYYY',
            LL : 'D. MMMM [dallas] YYYY',
            LLL : 'D. MMMM [dallas] YYYY HH.mm',
            LLLL : 'dddd, [li] D. MMMM [dallas] YYYY HH.mm'
        },
        meridiemParse: /d\'o|d\'a/i,
        isPM : function (input) {
            return 'd\'o' === input.toLowerCase();
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'd\'o' : 'D\'O';
            } else {
                return isLower ? 'd\'a' : 'D\'A';
            }
        },
        calendar : {
            sameDay : '[oxhi à] LT',
            nextDay : '[demà à] LT',
            nextWeek : 'dddd [à] LT',
            lastDay : '[ieiri à] LT',
            lastWeek : '[sür el] dddd [lasteu à] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'osprei %s',
            past : 'ja%s',
            s : processRelativeTime$7,
            ss : processRelativeTime$7,
            m : processRelativeTime$7,
            mm : processRelativeTime$7,
            h : processRelativeTime$7,
            hh : processRelativeTime$7,
            d : processRelativeTime$7,
            dd : processRelativeTime$7,
            M : processRelativeTime$7,
            MM : processRelativeTime$7,
            y : processRelativeTime$7,
            yy : processRelativeTime$7
        },
        dayOfMonthOrdinalParse: /\d{1,2}\./,
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    function processRelativeTime$7(number, withoutSuffix, key, isFuture) {
        var format = {
            's': ['viensas secunds', '\'iensas secunds'],
            'ss': [number + ' secunds', '' + number + ' secunds'],
            'm': ['\'n míut', '\'iens míut'],
            'mm': [number + ' míuts', '' + number + ' míuts'],
            'h': ['\'n þora', '\'iensa þora'],
            'hh': [number + ' þoras', '' + number + ' þoras'],
            'd': ['\'n ziua', '\'iensa ziua'],
            'dd': [number + ' ziuas', '' + number + ' ziuas'],
            'M': ['\'n mes', '\'iens mes'],
            'MM': [number + ' mesen', '' + number + ' mesen'],
            'y': ['\'n ar', '\'iens ar'],
            'yy': [number + ' ars', '' + number + ' ars']
        };
        return isFuture ? format[key][0] : (withoutSuffix ? format[key][0] : format[key][1]);
    }

    //! moment.js locale configuration

    hooks.defineLocale('tzm-latn', {
        months : 'innayr_brˤayrˤ_marˤsˤ_ibrir_mayyw_ywnyw_ywlywz_ɣwšt_šwtanbir_ktˤwbrˤ_nwwanbir_dwjnbir'.split('_'),
        monthsShort : 'innayr_brˤayrˤ_marˤsˤ_ibrir_mayyw_ywnyw_ywlywz_ɣwšt_šwtanbir_ktˤwbrˤ_nwwanbir_dwjnbir'.split('_'),
        weekdays : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
        weekdaysShort : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
        weekdaysMin : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[asdkh g] LT',
            nextDay: '[aska g] LT',
            nextWeek: 'dddd [g] LT',
            lastDay: '[assant g] LT',
            lastWeek: 'dddd [g] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'dadkh s yan %s',
            past : 'yan %s',
            s : 'imik',
            ss : '%d imik',
            m : 'minuḍ',
            mm : '%d minuḍ',
            h : 'saɛa',
            hh : '%d tassaɛin',
            d : 'ass',
            dd : '%d ossan',
            M : 'ayowr',
            MM : '%d iyyirn',
            y : 'asgas',
            yy : '%d isgasn'
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('tzm', {
        months : 'ⵉⵏⵏⴰⵢⵔ_ⴱⵕⴰⵢⵕ_ⵎⴰⵕⵚ_ⵉⴱⵔⵉⵔ_ⵎⴰⵢⵢⵓ_ⵢⵓⵏⵢⵓ_ⵢⵓⵍⵢⵓⵣ_ⵖⵓⵛⵜ_ⵛⵓⵜⴰⵏⴱⵉⵔ_ⴽⵟⵓⴱⵕ_ⵏⵓⵡⴰⵏⴱⵉⵔ_ⴷⵓⵊⵏⴱⵉⵔ'.split('_'),
        monthsShort : 'ⵉⵏⵏⴰⵢⵔ_ⴱⵕⴰⵢⵕ_ⵎⴰⵕⵚ_ⵉⴱⵔⵉⵔ_ⵎⴰⵢⵢⵓ_ⵢⵓⵏⵢⵓ_ⵢⵓⵍⵢⵓⵣ_ⵖⵓⵛⵜ_ⵛⵓⵜⴰⵏⴱⵉⵔ_ⴽⵟⵓⴱⵕ_ⵏⵓⵡⴰⵏⴱⵉⵔ_ⴷⵓⵊⵏⴱⵉⵔ'.split('_'),
        weekdays : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
        weekdaysShort : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
        weekdaysMin : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS: 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[ⴰⵙⴷⵅ ⴴ] LT',
            nextDay: '[ⴰⵙⴽⴰ ⴴ] LT',
            nextWeek: 'dddd [ⴴ] LT',
            lastDay: '[ⴰⵚⴰⵏⵜ ⴴ] LT',
            lastWeek: 'dddd [ⴴ] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : 'ⴷⴰⴷⵅ ⵙ ⵢⴰⵏ %s',
            past : 'ⵢⴰⵏ %s',
            s : 'ⵉⵎⵉⴽ',
            ss : '%d ⵉⵎⵉⴽ',
            m : 'ⵎⵉⵏⵓⴺ',
            mm : '%d ⵎⵉⵏⵓⴺ',
            h : 'ⵙⴰⵄⴰ',
            hh : '%d ⵜⴰⵙⵙⴰⵄⵉⵏ',
            d : 'ⴰⵙⵙ',
            dd : '%d oⵙⵙⴰⵏ',
            M : 'ⴰⵢoⵓⵔ',
            MM : '%d ⵉⵢⵢⵉⵔⵏ',
            y : 'ⴰⵙⴳⴰⵙ',
            yy : '%d ⵉⵙⴳⴰⵙⵏ'
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 12th is the first week of the year.
        }
    });

    //! moment.js language configuration

    hooks.defineLocale('ug-cn', {
        months: 'يانۋار_فېۋرال_مارت_ئاپرېل_ماي_ئىيۇن_ئىيۇل_ئاۋغۇست_سېنتەبىر_ئۆكتەبىر_نويابىر_دېكابىر'.split(
            '_'
        ),
        monthsShort: 'يانۋار_فېۋرال_مارت_ئاپرېل_ماي_ئىيۇن_ئىيۇل_ئاۋغۇست_سېنتەبىر_ئۆكتەبىر_نويابىر_دېكابىر'.split(
            '_'
        ),
        weekdays: 'يەكشەنبە_دۈشەنبە_سەيشەنبە_چارشەنبە_پەيشەنبە_جۈمە_شەنبە'.split(
            '_'
        ),
        weekdaysShort: 'يە_دۈ_سە_چا_پە_جۈ_شە'.split('_'),
        weekdaysMin: 'يە_دۈ_سە_چا_پە_جۈ_شە'.split('_'),
        longDateFormat: {
            LT: 'HH:mm',
            LTS: 'HH:mm:ss',
            L: 'YYYY-MM-DD',
            LL: 'YYYY-يىلىM-ئاينىڭD-كۈنى',
            LLL: 'YYYY-يىلىM-ئاينىڭD-كۈنى، HH:mm',
            LLLL: 'dddd، YYYY-يىلىM-ئاينىڭD-كۈنى، HH:mm'
        },
        meridiemParse: /يېرىم كېچە|سەھەر|چۈشتىن بۇرۇن|چۈش|چۈشتىن كېيىن|كەچ/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (
                meridiem === 'يېرىم كېچە' ||
                meridiem === 'سەھەر' ||
                meridiem === 'چۈشتىن بۇرۇن'
            ) {
                return hour;
            } else if (meridiem === 'چۈشتىن كېيىن' || meridiem === 'كەچ') {
                return hour + 12;
            } else {
                return hour >= 11 ? hour : hour + 12;
            }
        },
        meridiem: function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return 'يېرىم كېچە';
            } else if (hm < 900) {
                return 'سەھەر';
            } else if (hm < 1130) {
                return 'چۈشتىن بۇرۇن';
            } else if (hm < 1230) {
                return 'چۈش';
            } else if (hm < 1800) {
                return 'چۈشتىن كېيىن';
            } else {
                return 'كەچ';
            }
        },
        calendar: {
            sameDay: '[بۈگۈن سائەت] LT',
            nextDay: '[ئەتە سائەت] LT',
            nextWeek: '[كېلەركى] dddd [سائەت] LT',
            lastDay: '[تۆنۈگۈن] LT',
            lastWeek: '[ئالدىنقى] dddd [سائەت] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: '%s كېيىن',
            past: '%s بۇرۇن',
            s: 'نەچچە سېكونت',
            ss: '%d سېكونت',
            m: 'بىر مىنۇت',
            mm: '%d مىنۇت',
            h: 'بىر سائەت',
            hh: '%d سائەت',
            d: 'بىر كۈن',
            dd: '%d كۈن',
            M: 'بىر ئاي',
            MM: '%d ئاي',
            y: 'بىر يىل',
            yy: '%d يىل'
        },

        dayOfMonthOrdinalParse: /\d{1,2}(-كۈنى|-ئاي|-ھەپتە)/,
        ordinal: function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'DDD':
                    return number + '-كۈنى';
                case 'w':
                case 'W':
                    return number + '-ھەپتە';
                default:
                    return number;
            }
        },
        preparse: function (string) {
            return string.replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/,/g, '،');
        },
        week: {
            // GB/T 7408-1994《数据元和交换格式·信息交换·日期和时间表示法》与ISO 8601:1988等效
            dow: 1, // Monday is the first day of the week.
            doy: 7 // The week that contains Jan 1st is the first week of the year.
        }
    });

    //! moment.js locale configuration

    function plural$6(word, num) {
        var forms = word.split('_');
        return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
    }
    function relativeTimeWithPlural$4(number, withoutSuffix, key) {
        var format = {
            'ss': withoutSuffix ? 'секунда_секунди_секунд' : 'секунду_секунди_секунд',
            'mm': withoutSuffix ? 'хвилина_хвилини_хвилин' : 'хвилину_хвилини_хвилин',
            'hh': withoutSuffix ? 'година_години_годин' : 'годину_години_годин',
            'dd': 'день_дні_днів',
            'MM': 'місяць_місяці_місяців',
            'yy': 'рік_роки_років'
        };
        if (key === 'm') {
            return withoutSuffix ? 'хвилина' : 'хвилину';
        }
        else if (key === 'h') {
            return withoutSuffix ? 'година' : 'годину';
        }
        else {
            return number + ' ' + plural$6(format[key], +number);
        }
    }
    function weekdaysCaseReplace(m, format) {
        var weekdays = {
            'nominative': 'неділя_понеділок_вівторок_середа_четвер_п’ятниця_субота'.split('_'),
            'accusative': 'неділю_понеділок_вівторок_середу_четвер_п’ятницю_суботу'.split('_'),
            'genitive': 'неділі_понеділка_вівторка_середи_четверга_п’ятниці_суботи'.split('_')
        };

        if (m === true) {
            return weekdays['nominative'].slice(1, 7).concat(weekdays['nominative'].slice(0, 1));
        }
        if (!m) {
            return weekdays['nominative'];
        }

        var nounCase = (/(\[[ВвУу]\]) ?dddd/).test(format) ?
            'accusative' :
            ((/\[?(?:минулої|наступної)? ?\] ?dddd/).test(format) ?
                'genitive' :
                'nominative');
        return weekdays[nounCase][m.day()];
    }
    function processHoursFunction(str) {
        return function () {
            return str + 'о' + (this.hours() === 11 ? 'б' : '') + '] LT';
        };
    }

    hooks.defineLocale('uk', {
        months : {
            'format': 'січня_лютого_березня_квітня_травня_червня_липня_серпня_вересня_жовтня_листопада_грудня'.split('_'),
            'standalone': 'січень_лютий_березень_квітень_травень_червень_липень_серпень_вересень_жовтень_листопад_грудень'.split('_')
        },
        monthsShort : 'січ_лют_бер_квіт_трав_черв_лип_серп_вер_жовт_лист_груд'.split('_'),
        weekdays : weekdaysCaseReplace,
        weekdaysShort : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
        weekdaysMin : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD.MM.YYYY',
            LL : 'D MMMM YYYY р.',
            LLL : 'D MMMM YYYY р., HH:mm',
            LLLL : 'dddd, D MMMM YYYY р., HH:mm'
        },
        calendar : {
            sameDay: processHoursFunction('[Сьогодні '),
            nextDay: processHoursFunction('[Завтра '),
            lastDay: processHoursFunction('[Вчора '),
            nextWeek: processHoursFunction('[У] dddd ['),
            lastWeek: function () {
                switch (this.day()) {
                    case 0:
                    case 3:
                    case 5:
                    case 6:
                        return processHoursFunction('[Минулої] dddd [').call(this);
                    case 1:
                    case 2:
                    case 4:
                        return processHoursFunction('[Минулого] dddd [').call(this);
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : 'за %s',
            past : '%s тому',
            s : 'декілька секунд',
            ss : relativeTimeWithPlural$4,
            m : relativeTimeWithPlural$4,
            mm : relativeTimeWithPlural$4,
            h : 'годину',
            hh : relativeTimeWithPlural$4,
            d : 'день',
            dd : relativeTimeWithPlural$4,
            M : 'місяць',
            MM : relativeTimeWithPlural$4,
            y : 'рік',
            yy : relativeTimeWithPlural$4
        },
        // M. E.: those two are virtually unused but a user might want to implement them for his/her website for some reason
        meridiemParse: /ночі|ранку|дня|вечора/,
        isPM: function (input) {
            return /^(дня|вечора)$/.test(input);
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return 'ночі';
            } else if (hour < 12) {
                return 'ранку';
            } else if (hour < 17) {
                return 'дня';
            } else {
                return 'вечора';
            }
        },
        dayOfMonthOrdinalParse: /\d{1,2}-(й|го)/,
        ordinal: function (number, period) {
            switch (period) {
                case 'M':
                case 'd':
                case 'DDD':
                case 'w':
                case 'W':
                    return number + '-й';
                case 'D':
                    return number + '-го';
                default:
                    return number;
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    var months$a = [
        'جنوری',
        'فروری',
        'مارچ',
        'اپریل',
        'مئی',
        'جون',
        'جولائی',
        'اگست',
        'ستمبر',
        'اکتوبر',
        'نومبر',
        'دسمبر'
    ];
    var days$2 = [
        'اتوار',
        'پیر',
        'منگل',
        'بدھ',
        'جمعرات',
        'جمعہ',
        'ہفتہ'
    ];

    hooks.defineLocale('ur', {
        months : months$a,
        monthsShort : months$a,
        weekdays : days$2,
        weekdaysShort : days$2,
        weekdaysMin : days$2,
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd، D MMMM YYYY HH:mm'
        },
        meridiemParse: /صبح|شام/,
        isPM : function (input) {
            return 'شام' === input;
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return 'صبح';
            }
            return 'شام';
        },
        calendar : {
            sameDay : '[آج بوقت] LT',
            nextDay : '[کل بوقت] LT',
            nextWeek : 'dddd [بوقت] LT',
            lastDay : '[گذشتہ روز بوقت] LT',
            lastWeek : '[گذشتہ] dddd [بوقت] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : '%s بعد',
            past : '%s قبل',
            s : 'چند سیکنڈ',
            ss : '%d سیکنڈ',
            m : 'ایک منٹ',
            mm : '%d منٹ',
            h : 'ایک گھنٹہ',
            hh : '%d گھنٹے',
            d : 'ایک دن',
            dd : '%d دن',
            M : 'ایک ماہ',
            MM : '%d ماہ',
            y : 'ایک سال',
            yy : '%d سال'
        },
        preparse: function (string) {
            return string.replace(/،/g, ',');
        },
        postformat: function (string) {
            return string.replace(/,/g, '،');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('uz-latn', {
        months : 'Yanvar_Fevral_Mart_Aprel_May_Iyun_Iyul_Avgust_Sentabr_Oktabr_Noyabr_Dekabr'.split('_'),
        monthsShort : 'Yan_Fev_Mar_Apr_May_Iyun_Iyul_Avg_Sen_Okt_Noy_Dek'.split('_'),
        weekdays : 'Yakshanba_Dushanba_Seshanba_Chorshanba_Payshanba_Juma_Shanba'.split('_'),
        weekdaysShort : 'Yak_Dush_Sesh_Chor_Pay_Jum_Shan'.split('_'),
        weekdaysMin : 'Ya_Du_Se_Cho_Pa_Ju_Sha'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'D MMMM YYYY, dddd HH:mm'
        },
        calendar : {
            sameDay : '[Bugun soat] LT [da]',
            nextDay : '[Ertaga] LT [da]',
            nextWeek : 'dddd [kuni soat] LT [da]',
            lastDay : '[Kecha soat] LT [da]',
            lastWeek : '[O\'tgan] dddd [kuni soat] LT [da]',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'Yaqin %s ichida',
            past : 'Bir necha %s oldin',
            s : 'soniya',
            ss : '%d soniya',
            m : 'bir daqiqa',
            mm : '%d daqiqa',
            h : 'bir soat',
            hh : '%d soat',
            d : 'bir kun',
            dd : '%d kun',
            M : 'bir oy',
            MM : '%d oy',
            y : 'bir yil',
            yy : '%d yil'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 7th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('uz', {
        months : 'январ_феврал_март_апрел_май_июн_июл_август_сентябр_октябр_ноябр_декабр'.split('_'),
        monthsShort : 'янв_фев_мар_апр_май_июн_июл_авг_сен_окт_ноя_дек'.split('_'),
        weekdays : 'Якшанба_Душанба_Сешанба_Чоршанба_Пайшанба_Жума_Шанба'.split('_'),
        weekdaysShort : 'Якш_Душ_Сеш_Чор_Пай_Жум_Шан'.split('_'),
        weekdaysMin : 'Як_Ду_Се_Чо_Па_Жу_Ша'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'D MMMM YYYY, dddd HH:mm'
        },
        calendar : {
            sameDay : '[Бугун соат] LT [да]',
            nextDay : '[Эртага] LT [да]',
            nextWeek : 'dddd [куни соат] LT [да]',
            lastDay : '[Кеча соат] LT [да]',
            lastWeek : '[Утган] dddd [куни соат] LT [да]',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'Якин %s ичида',
            past : 'Бир неча %s олдин',
            s : 'фурсат',
            ss : '%d фурсат',
            m : 'бир дакика',
            mm : '%d дакика',
            h : 'бир соат',
            hh : '%d соат',
            d : 'бир кун',
            dd : '%d кун',
            M : 'бир ой',
            MM : '%d ой',
            y : 'бир йил',
            yy : '%d йил'
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('vi', {
        months : 'tháng 1_tháng 2_tháng 3_tháng 4_tháng 5_tháng 6_tháng 7_tháng 8_tháng 9_tháng 10_tháng 11_tháng 12'.split('_'),
        monthsShort : 'Th01_Th02_Th03_Th04_Th05_Th06_Th07_Th08_Th09_Th10_Th11_Th12'.split('_'),
        monthsParseExact : true,
        weekdays : 'chủ nhật_thứ hai_thứ ba_thứ tư_thứ năm_thứ sáu_thứ bảy'.split('_'),
        weekdaysShort : 'CN_T2_T3_T4_T5_T6_T7'.split('_'),
        weekdaysMin : 'CN_T2_T3_T4_T5_T6_T7'.split('_'),
        weekdaysParseExact : true,
        meridiemParse: /sa|ch/i,
        isPM : function (input) {
            return /^ch$/i.test(input);
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 12) {
                return isLower ? 'sa' : 'SA';
            } else {
                return isLower ? 'ch' : 'CH';
            }
        },
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM [năm] YYYY',
            LLL : 'D MMMM [năm] YYYY HH:mm',
            LLLL : 'dddd, D MMMM [năm] YYYY HH:mm',
            l : 'DD/M/YYYY',
            ll : 'D MMM YYYY',
            lll : 'D MMM YYYY HH:mm',
            llll : 'ddd, D MMM YYYY HH:mm'
        },
        calendar : {
            sameDay: '[Hôm nay lúc] LT',
            nextDay: '[Ngày mai lúc] LT',
            nextWeek: 'dddd [tuần tới lúc] LT',
            lastDay: '[Hôm qua lúc] LT',
            lastWeek: 'dddd [tuần rồi lúc] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : '%s tới',
            past : '%s trước',
            s : 'vài giây',
            ss : '%d giây' ,
            m : 'một phút',
            mm : '%d phút',
            h : 'một giờ',
            hh : '%d giờ',
            d : 'một ngày',
            dd : '%d ngày',
            M : 'một tháng',
            MM : '%d tháng',
            y : 'một năm',
            yy : '%d năm'
        },
        dayOfMonthOrdinalParse: /\d{1,2}/,
        ordinal : function (number) {
            return number;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('x-pseudo', {
        months : 'J~áñúá~rý_F~ébrú~árý_~Márc~h_Áp~ríl_~Máý_~Júñé~_Júl~ý_Áú~gúst~_Sép~témb~ér_Ó~ctób~ér_Ñ~óvém~bér_~Décé~mbér'.split('_'),
        monthsShort : 'J~áñ_~Féb_~Már_~Ápr_~Máý_~Júñ_~Júl_~Áúg_~Sép_~Óct_~Ñóv_~Déc'.split('_'),
        monthsParseExact : true,
        weekdays : 'S~úñdá~ý_Mó~ñdáý~_Túé~sdáý~_Wéd~ñésd~áý_T~húrs~dáý_~Fríd~áý_S~átúr~dáý'.split('_'),
        weekdaysShort : 'S~úñ_~Móñ_~Túé_~Wéd_~Thú_~Frí_~Sát'.split('_'),
        weekdaysMin : 'S~ú_Mó~_Tú_~Wé_T~h_Fr~_Sá'.split('_'),
        weekdaysParseExact : true,
        longDateFormat : {
            LT : 'HH:mm',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY HH:mm',
            LLLL : 'dddd, D MMMM YYYY HH:mm'
        },
        calendar : {
            sameDay : '[T~ódá~ý át] LT',
            nextDay : '[T~ómó~rró~w át] LT',
            nextWeek : 'dddd [át] LT',
            lastDay : '[Ý~ést~érdá~ý át] LT',
            lastWeek : '[L~ást] dddd [át] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'í~ñ %s',
            past : '%s á~gó',
            s : 'á ~féw ~sécó~ñds',
            ss : '%d s~écóñ~ds',
            m : 'á ~míñ~úté',
            mm : '%d m~íñú~tés',
            h : 'á~ñ hó~úr',
            hh : '%d h~óúrs',
            d : 'á ~dáý',
            dd : '%d d~áýs',
            M : 'á ~móñ~th',
            MM : '%d m~óñt~hs',
            y : 'á ~ýéár',
            yy : '%d ý~éárs'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
        ordinal : function (number) {
            var b = number % 10,
                output = (~~(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('yo', {
        months : 'Sẹ́rẹ́_Èrèlè_Ẹrẹ̀nà_Ìgbé_Èbibi_Òkùdu_Agẹmo_Ògún_Owewe_Ọ̀wàrà_Bélú_Ọ̀pẹ̀̀'.split('_'),
        monthsShort : 'Sẹ́r_Èrl_Ẹrn_Ìgb_Èbi_Òkù_Agẹ_Ògú_Owe_Ọ̀wà_Bél_Ọ̀pẹ̀̀'.split('_'),
        weekdays : 'Àìkú_Ajé_Ìsẹ́gun_Ọjọ́rú_Ọjọ́bọ_Ẹtì_Àbámẹ́ta'.split('_'),
        weekdaysShort : 'Àìk_Ajé_Ìsẹ́_Ọjr_Ọjb_Ẹtì_Àbá'.split('_'),
        weekdaysMin : 'Àì_Aj_Ìs_Ọr_Ọb_Ẹt_Àb'.split('_'),
        longDateFormat : {
            LT : 'h:mm A',
            LTS : 'h:mm:ss A',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY h:mm A',
            LLLL : 'dddd, D MMMM YYYY h:mm A'
        },
        calendar : {
            sameDay : '[Ònì ni] LT',
            nextDay : '[Ọ̀la ni] LT',
            nextWeek : 'dddd [Ọsẹ̀ tón\'bọ] [ni] LT',
            lastDay : '[Àna ni] LT',
            lastWeek : 'dddd [Ọsẹ̀ tólọ́] [ni] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'ní %s',
            past : '%s kọjá',
            s : 'ìsẹjú aayá die',
            ss :'aayá %d',
            m : 'ìsẹjú kan',
            mm : 'ìsẹjú %d',
            h : 'wákati kan',
            hh : 'wákati %d',
            d : 'ọjọ́ kan',
            dd : 'ọjọ́ %d',
            M : 'osù kan',
            MM : 'osù %d',
            y : 'ọdún kan',
            yy : 'ọdún %d'
        },
        dayOfMonthOrdinalParse : /ọjọ́\s\d{1,2}/,
        ordinal : 'ọjọ́ %d',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4 // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('zh-cn', {
        months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
        monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
        weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
        weekdaysShort : '周日_周一_周二_周三_周四_周五_周六'.split('_'),
        weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY/MM/DD',
            LL : 'YYYY年M月D日',
            LLL : 'YYYY年M月D日Ah点mm分',
            LLLL : 'YYYY年M月D日ddddAh点mm分',
            l : 'YYYY/M/D',
            ll : 'YYYY年M月D日',
            lll : 'YYYY年M月D日 HH:mm',
            llll : 'YYYY年M月D日dddd HH:mm'
        },
        meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
        meridiemHour: function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === '凌晨' || meridiem === '早上' ||
                    meridiem === '上午') {
                return hour;
            } else if (meridiem === '下午' || meridiem === '晚上') {
                return hour + 12;
            } else {
                // '中午'
                return hour >= 11 ? hour : hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return '凌晨';
            } else if (hm < 900) {
                return '早上';
            } else if (hm < 1130) {
                return '上午';
            } else if (hm < 1230) {
                return '中午';
            } else if (hm < 1800) {
                return '下午';
            } else {
                return '晚上';
            }
        },
        calendar : {
            sameDay : '[今天]LT',
            nextDay : '[明天]LT',
            nextWeek : '[下]ddddLT',
            lastDay : '[昨天]LT',
            lastWeek : '[上]ddddLT',
            sameElse : 'L'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(日|月|周)/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd':
                case 'D':
                case 'DDD':
                    return number + '日';
                case 'M':
                    return number + '月';
                case 'w':
                case 'W':
                    return number + '周';
                default:
                    return number;
            }
        },
        relativeTime : {
            future : '%s内',
            past : '%s前',
            s : '几秒',
            ss : '%d 秒',
            m : '1 分钟',
            mm : '%d 分钟',
            h : '1 小时',
            hh : '%d 小时',
            d : '1 天',
            dd : '%d 天',
            M : '1 个月',
            MM : '%d 个月',
            y : '1 年',
            yy : '%d 年'
        },
        week : {
            // GB/T 7408-1994《数据元和交换格式·信息交换·日期和时间表示法》与ISO 8601:1988等效
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('zh-hk', {
        months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
        monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
        weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
        weekdaysShort : '週日_週一_週二_週三_週四_週五_週六'.split('_'),
        weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY/MM/DD',
            LL : 'YYYY年M月D日',
            LLL : 'YYYY年M月D日 HH:mm',
            LLLL : 'YYYY年M月D日dddd HH:mm',
            l : 'YYYY/M/D',
            ll : 'YYYY年M月D日',
            lll : 'YYYY年M月D日 HH:mm',
            llll : 'YYYY年M月D日dddd HH:mm'
        },
        meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === '凌晨' || meridiem === '早上' || meridiem === '上午') {
                return hour;
            } else if (meridiem === '中午') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === '下午' || meridiem === '晚上') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return '凌晨';
            } else if (hm < 900) {
                return '早上';
            } else if (hm < 1130) {
                return '上午';
            } else if (hm < 1230) {
                return '中午';
            } else if (hm < 1800) {
                return '下午';
            } else {
                return '晚上';
            }
        },
        calendar : {
            sameDay : '[今天]LT',
            nextDay : '[明天]LT',
            nextWeek : '[下]ddddLT',
            lastDay : '[昨天]LT',
            lastWeek : '[上]ddddLT',
            sameElse : 'L'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(日|月|週)/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd' :
                case 'D' :
                case 'DDD' :
                    return number + '日';
                case 'M' :
                    return number + '月';
                case 'w' :
                case 'W' :
                    return number + '週';
                default :
                    return number;
            }
        },
        relativeTime : {
            future : '%s內',
            past : '%s前',
            s : '幾秒',
            ss : '%d 秒',
            m : '1 分鐘',
            mm : '%d 分鐘',
            h : '1 小時',
            hh : '%d 小時',
            d : '1 天',
            dd : '%d 天',
            M : '1 個月',
            MM : '%d 個月',
            y : '1 年',
            yy : '%d 年'
        }
    });

    //! moment.js locale configuration

    hooks.defineLocale('zh-tw', {
        months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
        monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
        weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
        weekdaysShort : '週日_週一_週二_週三_週四_週五_週六'.split('_'),
        weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            LTS : 'HH:mm:ss',
            L : 'YYYY/MM/DD',
            LL : 'YYYY年M月D日',
            LLL : 'YYYY年M月D日 HH:mm',
            LLLL : 'YYYY年M月D日dddd HH:mm',
            l : 'YYYY/M/D',
            ll : 'YYYY年M月D日',
            lll : 'YYYY年M月D日 HH:mm',
            llll : 'YYYY年M月D日dddd HH:mm'
        },
        meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
        meridiemHour : function (hour, meridiem) {
            if (hour === 12) {
                hour = 0;
            }
            if (meridiem === '凌晨' || meridiem === '早上' || meridiem === '上午') {
                return hour;
            } else if (meridiem === '中午') {
                return hour >= 11 ? hour : hour + 12;
            } else if (meridiem === '下午' || meridiem === '晚上') {
                return hour + 12;
            }
        },
        meridiem : function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return '凌晨';
            } else if (hm < 900) {
                return '早上';
            } else if (hm < 1130) {
                return '上午';
            } else if (hm < 1230) {
                return '中午';
            } else if (hm < 1800) {
                return '下午';
            } else {
                return '晚上';
            }
        },
        calendar : {
            sameDay : '[今天] LT',
            nextDay : '[明天] LT',
            nextWeek : '[下]dddd LT',
            lastDay : '[昨天] LT',
            lastWeek : '[上]dddd LT',
            sameElse : 'L'
        },
        dayOfMonthOrdinalParse: /\d{1,2}(日|月|週)/,
        ordinal : function (number, period) {
            switch (period) {
                case 'd' :
                case 'D' :
                case 'DDD' :
                    return number + '日';
                case 'M' :
                    return number + '月';
                case 'w' :
                case 'W' :
                    return number + '週';
                default :
                    return number;
            }
        },
        relativeTime : {
            future : '%s內',
            past : '%s前',
            s : '幾秒',
            ss : '%d 秒',
            m : '1 分鐘',
            mm : '%d 分鐘',
            h : '1 小時',
            hh : '%d 小時',
            d : '1 天',
            dd : '%d 天',
            M : '1 個月',
            MM : '%d 個月',
            y : '1 年',
            yy : '%d 年'
        }
    });

    hooks.locale('en');

    return hooks;

})));

var process = process || {env: {NODE_ENV: "development"}};
//! moment-timezone.js
//! version : 0.5.25
//! Copyright (c) JS Foundation and other contributors
//! license : MIT
//! github.com/moment/moment-timezone

(function (root, factory) {
	"use strict";

	/*global define*/
	if (typeof module === 'object' && module.exports) {
		module.exports = factory(require('moment')); // Node
	} else if (typeof define === 'function' && define.amd) {
		define(['moment'], factory);                 // AMD
	} else {
		factory(root.moment);                        // Browser
	}
}(this, function (moment) {
	"use strict";

	// Do not load moment-timezone a second time.
	// if (moment.tz !== undefined) {
	// 	logError('Moment Timezone ' + moment.tz.version + ' was already loaded ' + (moment.tz.dataVersion ? 'with data from ' : 'without any data') + moment.tz.dataVersion);
	// 	return moment;
	// }

	var VERSION = "0.5.25",
		zones = {},
		links = {},
		names = {},
		guesses = {},
		cachedGuess;

	if (!moment || typeof moment.version !== 'string') {
		logError('Moment Timezone requires Moment.js. See https://momentjs.com/timezone/docs/#/use-it/browser/');
	}

	var momentVersion = moment.version.split('.'),
		major = +momentVersion[0],
		minor = +momentVersion[1];

	// Moment.js version check
	if (major < 2 || (major === 2 && minor < 6)) {
		logError('Moment Timezone requires Moment.js >= 2.6.0. You are using Moment.js ' + moment.version + '. See momentjs.com');
	}

	/************************************
		Unpacking
	************************************/

	function charCodeToInt(charCode) {
		if (charCode > 96) {
			return charCode - 87;
		} else if (charCode > 64) {
			return charCode - 29;
		}
		return charCode - 48;
	}

	function unpackBase60(string) {
		var i = 0,
			parts = string.split('.'),
			whole = parts[0],
			fractional = parts[1] || '',
			multiplier = 1,
			num,
			out = 0,
			sign = 1;

		// handle negative numbers
		if (string.charCodeAt(0) === 45) {
			i = 1;
			sign = -1;
		}

		// handle digits before the decimal
		for (i; i < whole.length; i++) {
			num = charCodeToInt(whole.charCodeAt(i));
			out = 60 * out + num;
		}

		// handle digits after the decimal
		for (i = 0; i < fractional.length; i++) {
			multiplier = multiplier / 60;
			num = charCodeToInt(fractional.charCodeAt(i));
			out += num * multiplier;
		}

		return out * sign;
	}

	function arrayToInt (array) {
		for (var i = 0; i < array.length; i++) {
			array[i] = unpackBase60(array[i]);
		}
	}

	function intToUntil (array, length) {
		for (var i = 0; i < length; i++) {
			array[i] = Math.round((array[i - 1] || 0) + (array[i] * 60000)); // minutes to milliseconds
		}

		array[length - 1] = Infinity;
	}

	function mapIndices (source, indices) {
		var out = [], i;

		for (i = 0; i < indices.length; i++) {
			out[i] = source[indices[i]];
		}

		return out;
	}

	function unpack (string) {
		var data = string.split('|'),
			offsets = data[2].split(' '),
			indices = data[3].split(''),
			untils  = data[4].split(' ');

		arrayToInt(offsets);
		arrayToInt(indices);
		arrayToInt(untils);

		intToUntil(untils, indices.length);

		return {
			name       : data[0],
			abbrs      : mapIndices(data[1].split(' '), indices),
			offsets    : mapIndices(offsets, indices),
			untils     : untils,
			population : data[5] | 0
		};
	}

	/************************************
		Zone object
	************************************/

	function Zone (packedString) {
		if (packedString) {
			this._set(unpack(packedString));
		}
	}

	Zone.prototype = {
		_set : function (unpacked) {
			this.name       = unpacked.name;
			this.abbrs      = unpacked.abbrs;
			this.untils     = unpacked.untils;
			this.offsets    = unpacked.offsets;
			this.population = unpacked.population;
		},

		_index : function (timestamp) {
			var target = +timestamp,
				untils = this.untils,
				i;

			for (i = 0; i < untils.length; i++) {
				if (target < untils[i]) {
					return i;
				}
			}
		},

		parse : function (timestamp) {
			var target  = +timestamp,
				offsets = this.offsets,
				untils  = this.untils,
				max     = untils.length - 1,
				offset, offsetNext, offsetPrev, i;

			for (i = 0; i < max; i++) {
				offset     = offsets[i];
				offsetNext = offsets[i + 1];
				offsetPrev = offsets[i ? i - 1 : i];

				if (offset < offsetNext && tz.moveAmbiguousForward) {
					offset = offsetNext;
				} else if (offset > offsetPrev && tz.moveInvalidForward) {
					offset = offsetPrev;
				}

				if (target < untils[i] - (offset * 60000)) {
					return offsets[i];
				}
			}

			return offsets[max];
		},

		abbr : function (mom) {
			return this.abbrs[this._index(mom)];
		},

		offset : function (mom) {
			logError("zone.offset has been deprecated in favor of zone.utcOffset");
			return this.offsets[this._index(mom)];
		},

		utcOffset : function (mom) {
			return this.offsets[this._index(mom)];
		}
	};

	/************************************
		Current Timezone
	************************************/

	function OffsetAt(at) {
		var timeString = at.toTimeString();
		var abbr = timeString.match(/\([a-z ]+\)/i);
		if (abbr && abbr[0]) {
			// 17:56:31 GMT-0600 (CST)
			// 17:56:31 GMT-0600 (Central Standard Time)
			abbr = abbr[0].match(/[A-Z]/g);
			abbr = abbr ? abbr.join('') : undefined;
		} else {
			// 17:56:31 CST
			// 17:56:31 GMT+0800 (台北標準時間)
			abbr = timeString.match(/[A-Z]{3,5}/g);
			abbr = abbr ? abbr[0] : undefined;
		}

		if (abbr === 'GMT') {
			abbr = undefined;
		}

		this.at = +at;
		this.abbr = abbr;
		this.offset = at.getTimezoneOffset();
	}

	function ZoneScore(zone) {
		this.zone = zone;
		this.offsetScore = 0;
		this.abbrScore = 0;
	}

	ZoneScore.prototype.scoreOffsetAt = function (offsetAt) {
		this.offsetScore += Math.abs(this.zone.utcOffset(offsetAt.at) - offsetAt.offset);
		if (this.zone.abbr(offsetAt.at).replace(/[^A-Z]/g, '') !== offsetAt.abbr) {
			this.abbrScore++;
		}
	};

	function findChange(low, high) {
		var mid, diff;

		while ((diff = ((high.at - low.at) / 12e4 | 0) * 6e4)) {
			mid = new OffsetAt(new Date(low.at + diff));
			if (mid.offset === low.offset) {
				low = mid;
			} else {
				high = mid;
			}
		}

		return low;
	}

	function userOffsets() {
		var startYear = new Date().getFullYear() - 2,
			last = new OffsetAt(new Date(startYear, 0, 1)),
			offsets = [last],
			change, next, i;

		for (i = 1; i < 48; i++) {
			next = new OffsetAt(new Date(startYear, i, 1));
			if (next.offset !== last.offset) {
				change = findChange(last, next);
				offsets.push(change);
				offsets.push(new OffsetAt(new Date(change.at + 6e4)));
			}
			last = next;
		}

		for (i = 0; i < 4; i++) {
			offsets.push(new OffsetAt(new Date(startYear + i, 0, 1)));
			offsets.push(new OffsetAt(new Date(startYear + i, 6, 1)));
		}

		return offsets;
	}

	function sortZoneScores (a, b) {
		if (a.offsetScore !== b.offsetScore) {
			return a.offsetScore - b.offsetScore;
		}
		if (a.abbrScore !== b.abbrScore) {
			return a.abbrScore - b.abbrScore;
		}
		return b.zone.population - a.zone.population;
	}

	function addToGuesses (name, offsets) {
		var i, offset;
		arrayToInt(offsets);
		for (i = 0; i < offsets.length; i++) {
			offset = offsets[i];
			guesses[offset] = guesses[offset] || {};
			guesses[offset][name] = true;
		}
	}

	function guessesForUserOffsets (offsets) {
		var offsetsLength = offsets.length,
			filteredGuesses = {},
			out = [],
			i, j, guessesOffset;

		for (i = 0; i < offsetsLength; i++) {
			guessesOffset = guesses[offsets[i].offset] || {};
			for (j in guessesOffset) {
				if (guessesOffset.hasOwnProperty(j)) {
					filteredGuesses[j] = true;
				}
			}
		}

		for (i in filteredGuesses) {
			if (filteredGuesses.hasOwnProperty(i)) {
				out.push(names[i]);
			}
		}

		return out;
	}

	function rebuildGuess () {

		// use Intl API when available and returning valid time zone
		try {
			var intlName = Intl.DateTimeFormat().resolvedOptions().timeZone;
			if (intlName && intlName.length > 3) {
				var name = names[normalizeName(intlName)];
				if (name) {
					return name;
				}
				logError("Moment Timezone found " + intlName + " from the Intl api, but did not have that data loaded.");
			}
		} catch (e) {
			// Intl unavailable, fall back to manual guessing.
		}

		var offsets = userOffsets(),
			offsetsLength = offsets.length,
			guesses = guessesForUserOffsets(offsets),
			zoneScores = [],
			zoneScore, i, j;

		for (i = 0; i < guesses.length; i++) {
			zoneScore = new ZoneScore(getZone(guesses[i]), offsetsLength);
			for (j = 0; j < offsetsLength; j++) {
				zoneScore.scoreOffsetAt(offsets[j]);
			}
			zoneScores.push(zoneScore);
		}

		zoneScores.sort(sortZoneScores);

		return zoneScores.length > 0 ? zoneScores[0].zone.name : undefined;
	}

	function guess (ignoreCache) {
		if (!cachedGuess || ignoreCache) {
			cachedGuess = rebuildGuess();
		}
		return cachedGuess;
	}

	/************************************
		Global Methods
	************************************/

	function normalizeName (name) {
		return (name || '').toLowerCase().replace(/\//g, '_');
	}

	function addZone (packed) {
		var i, name, split, normalized;

		if (typeof packed === "string") {
			packed = [packed];
		}

		for (i = 0; i < packed.length; i++) {
			split = packed[i].split('|');
			name = split[0];
			normalized = normalizeName(name);
			zones[normalized] = packed[i];
			names[normalized] = name;
			addToGuesses(normalized, split[2].split(' '));
		}
	}

	function getZone (name, caller) {
		
		name = normalizeName(name);

		var zone = zones[name];
		var link;

		if (zone instanceof Zone) {
			return zone;
		}

		if (typeof zone === 'string') {
			zone = new Zone(zone);
			zones[name] = zone;
			return zone;
		}

		// Pass getZone to prevent recursion more than 1 level deep
		if (links[name] && caller !== getZone && (link = getZone(links[name], getZone))) {
			zone = zones[name] = new Zone();
			zone._set(link);
			zone.name = names[name];
			return zone;
		}

		return null;
	}

	function getNames () {
		var i, out = [];

		for (i in names) {
			if (names.hasOwnProperty(i) && (zones[i] || zones[links[i]]) && names[i]) {
				out.push(names[i]);
			}
		}

		return out.sort();
	}

	function addLink (aliases) {
		var i, alias, normal0, normal1;

		if (typeof aliases === "string") {
			aliases = [aliases];
		}

		for (i = 0; i < aliases.length; i++) {
			alias = aliases[i].split('|');

			normal0 = normalizeName(alias[0]);
			normal1 = normalizeName(alias[1]);

			links[normal0] = normal1;
			names[normal0] = alias[0];

			links[normal1] = normal0;
			names[normal1] = alias[1];
		}
	}

	function loadData (data) {
		addZone(data.zones);
		addLink(data.links);
		tz.dataVersion = data.version;
	}

	function zoneExists (name) {
		if (!zoneExists.didShowError) {
			zoneExists.didShowError = true;
				logError("moment.tz.zoneExists('" + name + "') has been deprecated in favor of !moment.tz.zone('" + name + "')");
		}
		return !!getZone(name);
	}

	function needsOffset (m) {
		var isUnixTimestamp = (m._f === 'X' || m._f === 'x');
		return !!(m._a && (m._tzm === undefined) && !isUnixTimestamp);
	}

	function logError (message) {
		if (typeof console !== 'undefined' && typeof console.error === 'function') {
			console.error(message);
		}
	}

	/************************************
		moment.tz namespace
	************************************/

	function tz (input) {
		var args = Array.prototype.slice.call(arguments, 0, -1),
			name = arguments[arguments.length - 1],
			zone = getZone(name),
			out  = moment.utc.apply(null, args);

		if (zone && !moment.isMoment(input) && needsOffset(out)) {
			out.add(zone.parse(out), 'minutes');
		}

		out.tz(name);

		return out;
	}

	tz.version      = VERSION;
	tz.dataVersion  = '';
	tz._zones       = zones;
	tz._links       = links;
	tz._names       = names;
	tz.add          = addZone;
	tz.link         = addLink;
	tz.load         = loadData;
	tz.zone         = getZone;
	tz.zoneExists   = zoneExists; // deprecated in 0.1.0
	tz.guess        = guess;
	tz.names        = getNames;
	tz.Zone         = Zone;
	tz.unpack       = unpack;
	tz.unpackBase60 = unpackBase60;
	tz.needsOffset  = needsOffset;
	tz.moveInvalidForward   = true;
	tz.moveAmbiguousForward = false;

	/************************************
		Interface with Moment.js
	************************************/

	var fn = moment.fn;

	moment.tz = tz;

	moment.defaultZone = null;

	moment.updateOffset = function (mom, keepTime) {
		var zone = moment.defaultZone,
			offset;

		if (mom._z === undefined) {
			if (zone && needsOffset(mom) && !mom._isUTC) {
				mom._d = moment.utc(mom._a)._d;
				mom.utc().add(zone.parse(mom), 'minutes');
			}
			mom._z = zone;
		}
		if (mom._z) {
			offset = mom._z.utcOffset(mom);
			if (Math.abs(offset) < 16) {
				offset = offset / 60;
			}
			if (mom.utcOffset !== undefined) {
				var z = mom._z;
				mom.utcOffset(-offset, keepTime);
				mom._z = z;
			} else {
				mom.zone(offset, keepTime);
			}
		}
	};

	fn.tz = function (name, keepTime) {
		if (name) {
			if (typeof name !== 'string') {
				throw new Error('Time zone name must be a string, got ' + name + ' [' + typeof name + ']');
			}
			this._z = getZone(name);
			if (this._z) {
				moment.updateOffset(this, keepTime);
			} else {
				logError("Moment Timezone has no data for " + name + ". See http://momentjs.com/timezone/docs/#/data-loading/.");
			}
			return this;
		}
		if (this._z) { return this._z.name; }
	};

	function abbrWrap (old) {
		return function () {
			if (this._z) { return this._z.abbr(this); }
			return old.call(this);
		};
	}

	function resetZoneWrap (old) {
		return function () {
			this._z = null;
			return old.apply(this, arguments);
		};
	}

	function resetZoneWrap2 (old) {
		return function () {
			if (arguments.length > 0) this._z = null;
			return old.apply(this, arguments);
		};
	}

	fn.zoneName  = abbrWrap(fn.zoneName);
	fn.zoneAbbr  = abbrWrap(fn.zoneAbbr);
	fn.utc       = resetZoneWrap(fn.utc);
	fn.local     = resetZoneWrap(fn.local);
	fn.utcOffset = resetZoneWrap2(fn.utcOffset);
	
	moment.tz.setDefault = function(name) {
		if (major < 2 || (major === 2 && minor < 9)) {
			logError('Moment Timezone setDefault() requires Moment.js >= 2.9.0. You are using Moment.js ' + moment.version + '.');
		}
		moment.defaultZone = name ? getZone(name) : null;
		return moment;
	};

	// Cloning a moment should include the _z property.
	var momentProperties = moment.momentProperties;
	if (Object.prototype.toString.call(momentProperties) === '[object Array]') {
		// moment 2.8.1+
		momentProperties.push('_z');
		momentProperties.push('_a');
	} else if (momentProperties) {
		// moment 2.7.0
		momentProperties._z = null;
	}

	loadData({
		"version": "2019a",
		"zones": [
			"Africa/Abidjan|GMT|0|0||48e5",
			"Africa/Nairobi|EAT|-30|0||47e5",
			"Africa/Algiers|CET|-10|0||26e5",
			"Africa/Lagos|WAT|-10|0||17e6",
			"Africa/Maputo|CAT|-20|0||26e5",
			"Africa/Cairo|EET EEST|-20 -30|01010|1M2m0 gL0 e10 mn0|15e6",
			"Africa/Casablanca|+00 +01|0 -10|01010101010101010101010101010101|1LHC0 A00 e00 y00 11A0 uM0 e00 Dc0 11A0 s00 e00 IM0 WM0 mo0 gM0 LA0 WM0 jA0 e00 28M0 e00 2600 e00 28M0 e00 2600 gM0 2600 e00 28M0 e00|32e5",
			"Europe/Paris|CET CEST|-10 -20|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|11e6",
			"Africa/Johannesburg|SAST|-20|0||84e5",
			"Africa/Khartoum|EAT CAT|-30 -20|01|1Usl0|51e5",
			"Africa/Sao_Tome|GMT WAT|0 -10|010|1UQN0 2q00",
			"Africa/Tripoli|EET|-20|0||11e5",
			"Africa/Windhoek|CAT WAT|-20 -10|010101010|1LKo0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0|32e4",
			"America/Adak|HST HDT|a0 90|01010101010101010101010|1Lzo0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|326",
			"America/Anchorage|AKST AKDT|90 80|01010101010101010101010|1Lzn0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|30e4",
			"America/Santo_Domingo|AST|40|0||29e5",
			"America/Fortaleza|-03|30|0||34e5",
			"America/Asuncion|-03 -04|30 40|01010101010101010101010|1LEP0 1ip0 17b0 1ip0 19X0 1fB0 19X0 1fB0 19X0 1ip0 17b0 1ip0 17b0 1ip0 19X0 1fB0 19X0 1fB0 19X0 1fB0 19X0 1ip0|28e5",
			"America/Panama|EST|50|0||15e5",
			"America/Mexico_City|CST CDT|60 50|01010101010101010101010|1LKw0 1lb0 14p0 1lb0 14p0 1nX0 11B0 1nX0 11B0 1nX0 14p0 1lb0 14p0 1lb0 14p0 1nX0 11B0 1nX0 11B0 1nX0 14p0 1lb0|20e6",
			"America/Managua|CST|60|0||22e5",
			"America/La_Paz|-04|40|0||19e5",
			"America/Lima|-05|50|0||11e6",
			"America/Denver|MST MDT|70 60|01010101010101010101010|1Lzl0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|26e5",
			"America/Campo_Grande|-03 -04|30 40|01010101010101010101010|1LqP0 1C10 On0 1zd0 On0 1zd0 On0 1zd0 On0 1HB0 FX0 1HB0 FX0 1HB0 IL0 1HB0 FX0 1HB0 IL0 1EN0 FX0 1HB0|77e4",
			"America/Cancun|CST CDT EST|60 50 50|0102|1LKw0 1lb0 Dd0|63e4",
			"America/Caracas|-0430 -04|4u 40|01|1QMT0|29e5",
			"America/Chicago|CST CDT|60 50|01010101010101010101010|1Lzk0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|92e5",
			"America/Chihuahua|MST MDT|70 60|01010101010101010101010|1LKx0 1lb0 14p0 1lb0 14p0 1nX0 11B0 1nX0 11B0 1nX0 14p0 1lb0 14p0 1lb0 14p0 1nX0 11B0 1nX0 11B0 1nX0 14p0 1lb0|81e4",
			"America/Phoenix|MST|70|0||42e5",
			"America/Los_Angeles|PST PDT|80 70|01010101010101010101010|1Lzm0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|15e6",
			"America/New_York|EST EDT|50 40|01010101010101010101010|1Lzj0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|21e6",
			"America/Fort_Nelson|PST PDT MST|80 70 70|0102|1Lzm0 1zb0 Op0|39e2",
			"America/Halifax|AST ADT|40 30|01010101010101010101010|1Lzi0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|39e4",
			"America/Godthab|-03 -02|30 20|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|17e3",
			"America/Grand_Turk|EST EDT AST|50 40 40|0101210101010101010|1Lzj0 1zb0 Op0 1zb0 5Ip0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|37e2",
			"America/Havana|CST CDT|50 40|01010101010101010101010|1Lzh0 1zc0 Oo0 1zc0 Rc0 1zc0 Oo0 1zc0 Oo0 1zc0 Oo0 1zc0 Oo0 1zc0 Rc0 1zc0 Oo0 1zc0 Oo0 1zc0 Oo0 1zc0|21e5",
			"America/Metlakatla|PST AKST AKDT|80 90 80|012121201212121212121|1PAa0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 uM0 jB0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|14e2",
			"America/Miquelon|-03 -02|30 20|01010101010101010101010|1Lzh0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|61e2",
			"America/Montevideo|-02 -03|20 30|0101|1Lzg0 1o10 11z0|17e5",
			"America/Noronha|-02|20|0||30e2",
			"America/Port-au-Prince|EST EDT|50 40|010101010101010101010|1Lzj0 1zb0 Op0 1zb0 3iN0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|23e5",
			"Antarctica/Palmer|-03 -04|30 40|01010|1LSP0 Rd0 46n0 Ap0|40",
			"America/Santiago|-03 -04|30 40|010101010101010101010|1LSP0 Rd0 46n0 Ap0 1Nb0 Ap0 1Nb0 Ap0 1zb0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0 1qL0 11B0|62e5",
			"America/Sao_Paulo|-02 -03|20 30|01010101010101010101010|1LqO0 1C10 On0 1zd0 On0 1zd0 On0 1zd0 On0 1HB0 FX0 1HB0 FX0 1HB0 IL0 1HB0 FX0 1HB0 IL0 1EN0 FX0 1HB0|20e6",
			"Atlantic/Azores|-01 +00|10 0|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|25e4",
			"America/St_Johns|NST NDT|3u 2u|01010101010101010101010|1Lzhu 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0 Rd0 1zb0 Op0 1zb0 Op0 1zb0 Op0 1zb0|11e4",
			"Antarctica/Casey|+08 +11|-80 -b0|010|1RWg0 3m10|10",
			"Asia/Bangkok|+07|-70|0||15e6",
			"Pacific/Port_Moresby|+10|-a0|0||25e4",
			"Pacific/Guadalcanal|+11|-b0|0||11e4",
			"Asia/Tashkent|+05|-50|0||23e5",
			"Pacific/Auckland|NZDT NZST|-d0 -c0|01010101010101010101010|1LKe0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1cM0 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1io0 1a00|14e5",
			"Asia/Baghdad|+03|-30|0||66e5",
			"Antarctica/Troll|+00 +02|0 -20|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|40",
			"Asia/Dhaka|+06|-60|0||16e6",
			"Asia/Amman|EET EEST|-20 -30|01010101010101010101010|1LGK0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1o00|25e5",
			"Asia/Kamchatka|+12|-c0|0||18e4",
			"Asia/Baku|+04 +05|-40 -50|01010|1LHA0 1o00 11A0 1o00|27e5",
			"Asia/Barnaul|+07 +06|-70 -60|010|1N7v0 3rd0",
			"Asia/Beirut|EET EEST|-20 -30|01010101010101010101010|1LHy0 1nX0 11B0 1nX0 11B0 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0 11B0 1nX0 11B0 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0|22e5",
			"Asia/Kuala_Lumpur|+08|-80|0||71e5",
			"Asia/Kolkata|IST|-5u|0||15e6",
			"Asia/Chita|+10 +08 +09|-a0 -80 -90|012|1N7s0 3re0|33e4",
			"Asia/Ulaanbaatar|+08 +09|-80 -90|01010|1O8G0 1cJ0 1cP0 1cJ0|12e5",
			"Asia/Shanghai|CST|-80|0||23e6",
			"Asia/Colombo|+0530|-5u|0||22e5",
			"Asia/Damascus|EET EEST|-20 -30|01010101010101010101010|1LGK0 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0 11B0 1nX0|26e5",
			"Asia/Dili|+09|-90|0||19e4",
			"Asia/Dubai|+04|-40|0||39e5",
			"Asia/Famagusta|EET EEST +03|-20 -30 -30|0101012010101010101010|1LHB0 1o00 11A0 1o00 11A0 15U0 2Ks0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00",
			"Asia/Gaza|EET EEST|-20 -30|01010101010101010101010|1LGK0 1nX0 1210 1nz0 1220 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0 11B0 1qL0 WN0 1qL0 WN0 1qL0 WN0 1qL0 11B0 1nX0|18e5",
			"Asia/Hong_Kong|HKT|-80|0||73e5",
			"Asia/Hovd|+07 +08|-70 -80|01010|1O8H0 1cJ0 1cP0 1cJ0|81e3",
			"Asia/Irkutsk|+09 +08|-90 -80|01|1N7t0|60e4",
			"Europe/Istanbul|EET EEST +03|-20 -30 -30|0101012|1LI10 1nA0 11A0 1tA0 U00 15w0|13e6",
			"Asia/Jakarta|WIB|-70|0||31e6",
			"Asia/Jayapura|WIT|-90|0||26e4",
			"Asia/Jerusalem|IST IDT|-20 -30|01010101010101010101010|1LGM0 1oL0 10N0 1oL0 10N0 1rz0 W10 1rz0 W10 1rz0 10N0 1oL0 10N0 1oL0 10N0 1rz0 W10 1rz0 W10 1rz0 10N0 1oL0|81e4",
			"Asia/Kabul|+0430|-4u|0||46e5",
			"Asia/Karachi|PKT|-50|0||24e6",
			"Asia/Kathmandu|+0545|-5J|0||12e5",
			"Asia/Yakutsk|+10 +09|-a0 -90|01|1N7s0|28e4",
			"Asia/Krasnoyarsk|+08 +07|-80 -70|01|1N7u0|10e5",
			"Asia/Magadan|+12 +10 +11|-c0 -a0 -b0|012|1N7q0 3Cq0|95e3",
			"Asia/Makassar|WITA|-80|0||15e5",
			"Asia/Manila|PST|-80|0||24e6",
			"Europe/Athens|EET EEST|-20 -30|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|35e5",
			"Asia/Novosibirsk|+07 +06|-70 -60|010|1N7v0 4eN0|15e5",
			"Asia/Omsk|+07 +06|-70 -60|01|1N7v0|12e5",
			"Asia/Pyongyang|KST KST|-90 -8u|010|1P4D0 6BA0|29e5",
			"Asia/Qyzylorda|+06 +05|-60 -50|01|1Xei0|73e4",
			"Asia/Rangoon|+0630|-6u|0||48e5",
			"Asia/Sakhalin|+11 +10|-b0 -a0|010|1N7r0 3rd0|58e4",
			"Asia/Seoul|KST|-90|0||23e6",
			"Asia/Srednekolymsk|+12 +11|-c0 -b0|01|1N7q0|35e2",
			"Asia/Tehran|+0330 +0430|-3u -4u|01010101010101010101010|1LEku 1dz0 1cp0 1dz0 1cp0 1dz0 1cN0 1dz0 1cp0 1dz0 1cp0 1dz0 1cp0 1dz0 1cN0 1dz0 1cp0 1dz0 1cp0 1dz0 1cp0 1dz0|14e6",
			"Asia/Tokyo|JST|-90|0||38e6",
			"Asia/Tomsk|+07 +06|-70 -60|010|1N7v0 3Qp0|10e5",
			"Asia/Vladivostok|+11 +10|-b0 -a0|01|1N7r0|60e4",
			"Asia/Yekaterinburg|+06 +05|-60 -50|01|1N7w0|14e5",
			"Europe/Lisbon|WET WEST|0 -10|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|27e5",
			"Atlantic/Cape_Verde|-01|10|0||50e4",
			"Australia/Sydney|AEDT AEST|-b0 -a0|01010101010101010101010|1LKg0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1fA0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1fA0 1cM0|40e5",
			"Australia/Adelaide|ACDT ACST|-au -9u|01010101010101010101010|1LKgu 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1fA0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1cM0 1fA0 1cM0|11e5",
			"Australia/Brisbane|AEST|-a0|0||20e5",
			"Australia/Darwin|ACST|-9u|0||12e4",
			"Australia/Eucla|+0845|-8J|0||368",
			"Australia/Lord_Howe|+11 +1030|-b0 -au|01010101010101010101010|1LKf0 1cMu 1cLu 1cMu 1cLu 1cMu 1cLu 1cMu 1cLu 1fAu 1cLu 1cMu 1cLu 1cMu 1cLu 1cMu 1cLu 1cMu 1cLu 1cMu 1fzu 1cMu|347",
			"Australia/Perth|AWST|-80|0||18e5",
			"Pacific/Easter|-05 -06|50 60|010101010101010101010|1LSP0 Rd0 46n0 Ap0 1Nb0 Ap0 1Nb0 Ap0 1zb0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0 1nX0 11B0 1qL0 11B0|30e2",
			"Europe/Dublin|GMT IST|0 -10|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|12e5",
			"Etc/GMT-1|+01|-10|0|",
			"Pacific/Fakaofo|+13|-d0|0||483",
			"Pacific/Kiritimati|+14|-e0|0||51e2",
			"Etc/GMT-2|+02|-20|0|",
			"Pacific/Tahiti|-10|a0|0||18e4",
			"Pacific/Niue|-11|b0|0||12e2",
			"Etc/GMT+12|-12|c0|0|",
			"Pacific/Galapagos|-06|60|0||25e3",
			"Etc/GMT+7|-07|70|0|",
			"Pacific/Pitcairn|-08|80|0||56",
			"Pacific/Gambier|-09|90|0||125",
			"Etc/UTC|UTC|0|0|",
			"Europe/Ulyanovsk|+04 +03|-40 -30|010|1N7y0 3rd0|13e5",
			"Europe/London|GMT BST|0 -10|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|10e6",
			"Europe/Chisinau|EET EEST|-20 -30|01010101010101010101010|1LHA0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00|67e4",
			"Europe/Kaliningrad|+03 EET|-30 -20|01|1N7z0|44e4",
			"Europe/Kirov|+04 +03|-40 -30|01|1N7y0|48e4",
			"Europe/Moscow|MSK MSK|-40 -30|01|1N7y0|16e6",
			"Europe/Saratov|+04 +03|-40 -30|010|1N7y0 5810",
			"Europe/Simferopol|EET MSK MSK|-20 -40 -30|012|1LHA0 1nW0|33e4",
			"Europe/Volgograd|+04 +03|-40 -30|010|1N7y0 9Jd0|10e5",
			"Pacific/Honolulu|HST|a0|0||37e4",
			"MET|MET MEST|-10 -20|01010101010101010101010|1LHB0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00 11A0 1o00 11A0 1qM0 WM0 1qM0 WM0 1qM0 11A0 1o00",
			"Pacific/Chatham|+1345 +1245|-dJ -cJ|01010101010101010101010|1LKe0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1cM0 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1io0 1a00|600",
			"Pacific/Apia|+14 +13|-e0 -d0|01010101010101010101010|1LKe0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1cM0 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1fA0 1a00 1io0 1a00|37e3",
			"Pacific/Bougainville|+10 +11|-a0 -b0|01|1NwE0|18e4",
			"Pacific/Fiji|+13 +12|-d0 -c0|01010101010101010101010|1Lfp0 1SN0 uM0 1SM0 uM0 1VA0 s00 1VA0 s00 1VA0 s00 1VA0 uM0 1SM0 uM0 1VA0 s00 1VA0 s00 1VA0 s00 1VA0|88e4",
			"Pacific/Guam|ChST|-a0|0||17e4",
			"Pacific/Marquesas|-0930|9u|0||86e2",
			"Pacific/Pago_Pago|SST|b0|0||37e2",
			"Pacific/Norfolk|+1130 +11|-bu -b0|01|1PoCu|25e4",
			"Pacific/Tongatapu|+13 +14|-d0 -e0|010|1S4d0 s00|75e3"
		],
		"links": [
			"Africa/Abidjan|Africa/Accra",
			"Africa/Abidjan|Africa/Bamako",
			"Africa/Abidjan|Africa/Banjul",
			"Africa/Abidjan|Africa/Bissau",
			"Africa/Abidjan|Africa/Conakry",
			"Africa/Abidjan|Africa/Dakar",
			"Africa/Abidjan|Africa/Freetown",
			"Africa/Abidjan|Africa/Lome",
			"Africa/Abidjan|Africa/Monrovia",
			"Africa/Abidjan|Africa/Nouakchott",
			"Africa/Abidjan|Africa/Ouagadougou",
			"Africa/Abidjan|Africa/Timbuktu",
			"Africa/Abidjan|America/Danmarkshavn",
			"Africa/Abidjan|Atlantic/Reykjavik",
			"Africa/Abidjan|Atlantic/St_Helena",
			"Africa/Abidjan|Etc/GMT",
			"Africa/Abidjan|Etc/GMT+0",
			"Africa/Abidjan|Etc/GMT-0",
			"Africa/Abidjan|Etc/GMT0",
			"Africa/Abidjan|Etc/Greenwich",
			"Africa/Abidjan|GMT",
			"Africa/Abidjan|GMT+0",
			"Africa/Abidjan|GMT-0",
			"Africa/Abidjan|GMT0",
			"Africa/Abidjan|Greenwich",
			"Africa/Abidjan|Iceland",
			"Africa/Algiers|Africa/Tunis",
			"Africa/Cairo|Egypt",
			"Africa/Casablanca|Africa/El_Aaiun",
			"Africa/Johannesburg|Africa/Maseru",
			"Africa/Johannesburg|Africa/Mbabane",
			"Africa/Lagos|Africa/Bangui",
			"Africa/Lagos|Africa/Brazzaville",
			"Africa/Lagos|Africa/Douala",
			"Africa/Lagos|Africa/Kinshasa",
			"Africa/Lagos|Africa/Libreville",
			"Africa/Lagos|Africa/Luanda",
			"Africa/Lagos|Africa/Malabo",
			"Africa/Lagos|Africa/Ndjamena",
			"Africa/Lagos|Africa/Niamey",
			"Africa/Lagos|Africa/Porto-Novo",
			"Africa/Maputo|Africa/Blantyre",
			"Africa/Maputo|Africa/Bujumbura",
			"Africa/Maputo|Africa/Gaborone",
			"Africa/Maputo|Africa/Harare",
			"Africa/Maputo|Africa/Kigali",
			"Africa/Maputo|Africa/Lubumbashi",
			"Africa/Maputo|Africa/Lusaka",
			"Africa/Nairobi|Africa/Addis_Ababa",
			"Africa/Nairobi|Africa/Asmara",
			"Africa/Nairobi|Africa/Asmera",
			"Africa/Nairobi|Africa/Dar_es_Salaam",
			"Africa/Nairobi|Africa/Djibouti",
			"Africa/Nairobi|Africa/Juba",
			"Africa/Nairobi|Africa/Kampala",
			"Africa/Nairobi|Africa/Mogadishu",
			"Africa/Nairobi|Indian/Antananarivo",
			"Africa/Nairobi|Indian/Comoro",
			"Africa/Nairobi|Indian/Mayotte",
			"Africa/Tripoli|Libya",
			"America/Adak|America/Atka",
			"America/Adak|US/Aleutian",
			"America/Anchorage|America/Juneau",
			"America/Anchorage|America/Nome",
			"America/Anchorage|America/Sitka",
			"America/Anchorage|America/Yakutat",
			"America/Anchorage|US/Alaska",
			"America/Campo_Grande|America/Cuiaba",
			"America/Chicago|America/Indiana/Knox",
			"America/Chicago|America/Indiana/Tell_City",
			"America/Chicago|America/Knox_IN",
			"America/Chicago|America/Matamoros",
			"America/Chicago|America/Menominee",
			"America/Chicago|America/North_Dakota/Beulah",
			"America/Chicago|America/North_Dakota/Center",
			"America/Chicago|America/North_Dakota/New_Salem",
			"America/Chicago|America/Rainy_River",
			"America/Chicago|America/Rankin_Inlet",
			"America/Chicago|America/Resolute",
			"America/Chicago|America/Winnipeg",
			"America/Chicago|CST6CDT",
			"America/Chicago|Canada/Central",
			"America/Chicago|US/Central",
			"America/Chicago|US/Indiana-Starke",
			"America/Chihuahua|America/Mazatlan",
			"America/Chihuahua|Mexico/BajaSur",
			"America/Denver|America/Boise",
			"America/Denver|America/Cambridge_Bay",
			"America/Denver|America/Edmonton",
			"America/Denver|America/Inuvik",
			"America/Denver|America/Ojinaga",
			"America/Denver|America/Shiprock",
			"America/Denver|America/Yellowknife",
			"America/Denver|Canada/Mountain",
			"America/Denver|MST7MDT",
			"America/Denver|Navajo",
			"America/Denver|US/Mountain",
			"America/Fortaleza|America/Araguaina",
			"America/Fortaleza|America/Argentina/Buenos_Aires",
			"America/Fortaleza|America/Argentina/Catamarca",
			"America/Fortaleza|America/Argentina/ComodRivadavia",
			"America/Fortaleza|America/Argentina/Cordoba",
			"America/Fortaleza|America/Argentina/Jujuy",
			"America/Fortaleza|America/Argentina/La_Rioja",
			"America/Fortaleza|America/Argentina/Mendoza",
			"America/Fortaleza|America/Argentina/Rio_Gallegos",
			"America/Fortaleza|America/Argentina/Salta",
			"America/Fortaleza|America/Argentina/San_Juan",
			"America/Fortaleza|America/Argentina/San_Luis",
			"America/Fortaleza|America/Argentina/Tucuman",
			"America/Fortaleza|America/Argentina/Ushuaia",
			"America/Fortaleza|America/Bahia",
			"America/Fortaleza|America/Belem",
			"America/Fortaleza|America/Buenos_Aires",
			"America/Fortaleza|America/Catamarca",
			"America/Fortaleza|America/Cayenne",
			"America/Fortaleza|America/Cordoba",
			"America/Fortaleza|America/Jujuy",
			"America/Fortaleza|America/Maceio",
			"America/Fortaleza|America/Mendoza",
			"America/Fortaleza|America/Paramaribo",
			"America/Fortaleza|America/Recife",
			"America/Fortaleza|America/Rosario",
			"America/Fortaleza|America/Santarem",
			"America/Fortaleza|Antarctica/Rothera",
			"America/Fortaleza|Atlantic/Stanley",
			"America/Fortaleza|Etc/GMT+3",
			"America/Halifax|America/Glace_Bay",
			"America/Halifax|America/Goose_Bay",
			"America/Halifax|America/Moncton",
			"America/Halifax|America/Thule",
			"America/Halifax|Atlantic/Bermuda",
			"America/Halifax|Canada/Atlantic",
			"America/Havana|Cuba",
			"America/La_Paz|America/Boa_Vista",
			"America/La_Paz|America/Guyana",
			"America/La_Paz|America/Manaus",
			"America/La_Paz|America/Porto_Velho",
			"America/La_Paz|Brazil/West",
			"America/La_Paz|Etc/GMT+4",
			"America/Lima|America/Bogota",
			"America/Lima|America/Eirunepe",
			"America/Lima|America/Guayaquil",
			"America/Lima|America/Porto_Acre",
			"America/Lima|America/Rio_Branco",
			"America/Lima|Brazil/Acre",
			"America/Lima|Etc/GMT+5",
			"America/Los_Angeles|America/Dawson",
			"America/Los_Angeles|America/Ensenada",
			"America/Los_Angeles|America/Santa_Isabel",
			"America/Los_Angeles|America/Tijuana",
			"America/Los_Angeles|America/Vancouver",
			"America/Los_Angeles|America/Whitehorse",
			"America/Los_Angeles|Canada/Pacific",
			"America/Los_Angeles|Canada/Yukon",
			"America/Los_Angeles|Mexico/BajaNorte",
			"America/Los_Angeles|PST8PDT",
			"America/Los_Angeles|US/Pacific",
			"America/Los_Angeles|US/Pacific-New",
			"America/Managua|America/Belize",
			"America/Managua|America/Costa_Rica",
			"America/Managua|America/El_Salvador",
			"America/Managua|America/Guatemala",
			"America/Managua|America/Regina",
			"America/Managua|America/Swift_Current",
			"America/Managua|America/Tegucigalpa",
			"America/Managua|Canada/Saskatchewan",
			"America/Mexico_City|America/Bahia_Banderas",
			"America/Mexico_City|America/Merida",
			"America/Mexico_City|America/Monterrey",
			"America/Mexico_City|Mexico/General",
			"America/New_York|America/Detroit",
			"America/New_York|America/Fort_Wayne",
			"America/New_York|America/Indiana/Indianapolis",
			"America/New_York|America/Indiana/Marengo",
			"America/New_York|America/Indiana/Petersburg",
			"America/New_York|America/Indiana/Vevay",
			"America/New_York|America/Indiana/Vincennes",
			"America/New_York|America/Indiana/Winamac",
			"America/New_York|America/Indianapolis",
			"America/New_York|America/Iqaluit",
			"America/New_York|America/Kentucky/Louisville",
			"America/New_York|America/Kentucky/Monticello",
			"America/New_York|America/Louisville",
			"America/New_York|America/Montreal",
			"America/New_York|America/Nassau",
			"America/New_York|America/Nipigon",
			"America/New_York|America/Pangnirtung",
			"America/New_York|America/Thunder_Bay",
			"America/New_York|America/Toronto",
			"America/New_York|Canada/Eastern",
			"America/New_York|EST5EDT",
			"America/New_York|US/East-Indiana",
			"America/New_York|US/Eastern",
			"America/New_York|US/Michigan",
			"America/Noronha|Atlantic/South_Georgia",
			"America/Noronha|Brazil/DeNoronha",
			"America/Noronha|Etc/GMT+2",
			"America/Panama|America/Atikokan",
			"America/Panama|America/Cayman",
			"America/Panama|America/Coral_Harbour",
			"America/Panama|America/Jamaica",
			"America/Panama|EST",
			"America/Panama|Jamaica",
			"America/Phoenix|America/Creston",
			"America/Phoenix|America/Dawson_Creek",
			"America/Phoenix|America/Hermosillo",
			"America/Phoenix|MST",
			"America/Phoenix|US/Arizona",
			"America/Santiago|Chile/Continental",
			"America/Santo_Domingo|America/Anguilla",
			"America/Santo_Domingo|America/Antigua",
			"America/Santo_Domingo|America/Aruba",
			"America/Santo_Domingo|America/Barbados",
			"America/Santo_Domingo|America/Blanc-Sablon",
			"America/Santo_Domingo|America/Curacao",
			"America/Santo_Domingo|America/Dominica",
			"America/Santo_Domingo|America/Grenada",
			"America/Santo_Domingo|America/Guadeloupe",
			"America/Santo_Domingo|America/Kralendijk",
			"America/Santo_Domingo|America/Lower_Princes",
			"America/Santo_Domingo|America/Marigot",
			"America/Santo_Domingo|America/Martinique",
			"America/Santo_Domingo|America/Montserrat",
			"America/Santo_Domingo|America/Port_of_Spain",
			"America/Santo_Domingo|America/Puerto_Rico",
			"America/Santo_Domingo|America/St_Barthelemy",
			"America/Santo_Domingo|America/St_Kitts",
			"America/Santo_Domingo|America/St_Lucia",
			"America/Santo_Domingo|America/St_Thomas",
			"America/Santo_Domingo|America/St_Vincent",
			"America/Santo_Domingo|America/Tortola",
			"America/Santo_Domingo|America/Virgin",
			"America/Sao_Paulo|Brazil/East",
			"America/St_Johns|Canada/Newfoundland",
			"Antarctica/Palmer|America/Punta_Arenas",
			"Asia/Baghdad|Antarctica/Syowa",
			"Asia/Baghdad|Asia/Aden",
			"Asia/Baghdad|Asia/Bahrain",
			"Asia/Baghdad|Asia/Kuwait",
			"Asia/Baghdad|Asia/Qatar",
			"Asia/Baghdad|Asia/Riyadh",
			"Asia/Baghdad|Etc/GMT-3",
			"Asia/Baghdad|Europe/Minsk",
			"Asia/Bangkok|Antarctica/Davis",
			"Asia/Bangkok|Asia/Ho_Chi_Minh",
			"Asia/Bangkok|Asia/Novokuznetsk",
			"Asia/Bangkok|Asia/Phnom_Penh",
			"Asia/Bangkok|Asia/Saigon",
			"Asia/Bangkok|Asia/Vientiane",
			"Asia/Bangkok|Etc/GMT-7",
			"Asia/Bangkok|Indian/Christmas",
			"Asia/Dhaka|Antarctica/Vostok",
			"Asia/Dhaka|Asia/Almaty",
			"Asia/Dhaka|Asia/Bishkek",
			"Asia/Dhaka|Asia/Dacca",
			"Asia/Dhaka|Asia/Kashgar",
			"Asia/Dhaka|Asia/Qostanay",
			"Asia/Dhaka|Asia/Thimbu",
			"Asia/Dhaka|Asia/Thimphu",
			"Asia/Dhaka|Asia/Urumqi",
			"Asia/Dhaka|Etc/GMT-6",
			"Asia/Dhaka|Indian/Chagos",
			"Asia/Dili|Etc/GMT-9",
			"Asia/Dili|Pacific/Palau",
			"Asia/Dubai|Asia/Muscat",
			"Asia/Dubai|Asia/Tbilisi",
			"Asia/Dubai|Asia/Yerevan",
			"Asia/Dubai|Etc/GMT-4",
			"Asia/Dubai|Europe/Samara",
			"Asia/Dubai|Indian/Mahe",
			"Asia/Dubai|Indian/Mauritius",
			"Asia/Dubai|Indian/Reunion",
			"Asia/Gaza|Asia/Hebron",
			"Asia/Hong_Kong|Hongkong",
			"Asia/Jakarta|Asia/Pontianak",
			"Asia/Jerusalem|Asia/Tel_Aviv",
			"Asia/Jerusalem|Israel",
			"Asia/Kamchatka|Asia/Anadyr",
			"Asia/Kamchatka|Etc/GMT-12",
			"Asia/Kamchatka|Kwajalein",
			"Asia/Kamchatka|Pacific/Funafuti",
			"Asia/Kamchatka|Pacific/Kwajalein",
			"Asia/Kamchatka|Pacific/Majuro",
			"Asia/Kamchatka|Pacific/Nauru",
			"Asia/Kamchatka|Pacific/Tarawa",
			"Asia/Kamchatka|Pacific/Wake",
			"Asia/Kamchatka|Pacific/Wallis",
			"Asia/Kathmandu|Asia/Katmandu",
			"Asia/Kolkata|Asia/Calcutta",
			"Asia/Kuala_Lumpur|Asia/Brunei",
			"Asia/Kuala_Lumpur|Asia/Kuching",
			"Asia/Kuala_Lumpur|Asia/Singapore",
			"Asia/Kuala_Lumpur|Etc/GMT-8",
			"Asia/Kuala_Lumpur|Singapore",
			"Asia/Makassar|Asia/Ujung_Pandang",
			"Asia/Rangoon|Asia/Yangon",
			"Asia/Rangoon|Indian/Cocos",
			"Asia/Seoul|ROK",
			"Asia/Shanghai|Asia/Chongqing",
			"Asia/Shanghai|Asia/Chungking",
			"Asia/Shanghai|Asia/Harbin",
			"Asia/Shanghai|Asia/Macao",
			"Asia/Shanghai|Asia/Macau",
			"Asia/Shanghai|Asia/Taipei",
			"Asia/Shanghai|PRC",
			"Asia/Shanghai|ROC",
			"Asia/Tashkent|Antarctica/Mawson",
			"Asia/Tashkent|Asia/Aqtau",
			"Asia/Tashkent|Asia/Aqtobe",
			"Asia/Tashkent|Asia/Ashgabat",
			"Asia/Tashkent|Asia/Ashkhabad",
			"Asia/Tashkent|Asia/Atyrau",
			"Asia/Tashkent|Asia/Dushanbe",
			"Asia/Tashkent|Asia/Oral",
			"Asia/Tashkent|Asia/Samarkand",
			"Asia/Tashkent|Etc/GMT-5",
			"Asia/Tashkent|Indian/Kerguelen",
			"Asia/Tashkent|Indian/Maldives",
			"Asia/Tehran|Iran",
			"Asia/Tokyo|Japan",
			"Asia/Ulaanbaatar|Asia/Choibalsan",
			"Asia/Ulaanbaatar|Asia/Ulan_Bator",
			"Asia/Vladivostok|Asia/Ust-Nera",
			"Asia/Yakutsk|Asia/Khandyga",
			"Atlantic/Azores|America/Scoresbysund",
			"Atlantic/Cape_Verde|Etc/GMT+1",
			"Australia/Adelaide|Australia/Broken_Hill",
			"Australia/Adelaide|Australia/South",
			"Australia/Adelaide|Australia/Yancowinna",
			"Australia/Brisbane|Australia/Lindeman",
			"Australia/Brisbane|Australia/Queensland",
			"Australia/Darwin|Australia/North",
			"Australia/Lord_Howe|Australia/LHI",
			"Australia/Perth|Australia/West",
			"Australia/Sydney|Australia/ACT",
			"Australia/Sydney|Australia/Canberra",
			"Australia/Sydney|Australia/Currie",
			"Australia/Sydney|Australia/Hobart",
			"Australia/Sydney|Australia/Melbourne",
			"Australia/Sydney|Australia/NSW",
			"Australia/Sydney|Australia/Tasmania",
			"Australia/Sydney|Australia/Victoria",
			"Etc/UTC|Etc/UCT",
			"Etc/UTC|Etc/Universal",
			"Etc/UTC|Etc/Zulu",
			"Etc/UTC|UCT",
			"Etc/UTC|UTC",
			"Etc/UTC|Universal",
			"Etc/UTC|Zulu",
			"Europe/Athens|Asia/Nicosia",
			"Europe/Athens|EET",
			"Europe/Athens|Europe/Bucharest",
			"Europe/Athens|Europe/Helsinki",
			"Europe/Athens|Europe/Kiev",
			"Europe/Athens|Europe/Mariehamn",
			"Europe/Athens|Europe/Nicosia",
			"Europe/Athens|Europe/Riga",
			"Europe/Athens|Europe/Sofia",
			"Europe/Athens|Europe/Tallinn",
			"Europe/Athens|Europe/Uzhgorod",
			"Europe/Athens|Europe/Vilnius",
			"Europe/Athens|Europe/Zaporozhye",
			"Europe/Chisinau|Europe/Tiraspol",
			"Europe/Dublin|Eire",
			"Europe/Istanbul|Asia/Istanbul",
			"Europe/Istanbul|Turkey",
			"Europe/Lisbon|Atlantic/Canary",
			"Europe/Lisbon|Atlantic/Faeroe",
			"Europe/Lisbon|Atlantic/Faroe",
			"Europe/Lisbon|Atlantic/Madeira",
			"Europe/Lisbon|Portugal",
			"Europe/Lisbon|WET",
			"Europe/London|Europe/Belfast",
			"Europe/London|Europe/Guernsey",
			"Europe/London|Europe/Isle_of_Man",
			"Europe/London|Europe/Jersey",
			"Europe/London|GB",
			"Europe/London|GB-Eire",
			"Europe/Moscow|W-SU",
			"Europe/Paris|Africa/Ceuta",
			"Europe/Paris|Arctic/Longyearbyen",
			"Europe/Paris|Atlantic/Jan_Mayen",
			"Europe/Paris|CET",
			"Europe/Paris|Europe/Amsterdam",
			"Europe/Paris|Europe/Andorra",
			"Europe/Paris|Europe/Belgrade",
			"Europe/Paris|Europe/Berlin",
			"Europe/Paris|Europe/Bratislava",
			"Europe/Paris|Europe/Brussels",
			"Europe/Paris|Europe/Budapest",
			"Europe/Paris|Europe/Busingen",
			"Europe/Paris|Europe/Copenhagen",
			"Europe/Paris|Europe/Gibraltar",
			"Europe/Paris|Europe/Ljubljana",
			"Europe/Paris|Europe/Luxembourg",
			"Europe/Paris|Europe/Madrid",
			"Europe/Paris|Europe/Malta",
			"Europe/Paris|Europe/Monaco",
			"Europe/Paris|Europe/Oslo",
			"Europe/Paris|Europe/Podgorica",
			"Europe/Paris|Europe/Prague",
			"Europe/Paris|Europe/Rome",
			"Europe/Paris|Europe/San_Marino",
			"Europe/Paris|Europe/Sarajevo",
			"Europe/Paris|Europe/Skopje",
			"Europe/Paris|Europe/Stockholm",
			"Europe/Paris|Europe/Tirane",
			"Europe/Paris|Europe/Vaduz",
			"Europe/Paris|Europe/Vatican",
			"Europe/Paris|Europe/Vienna",
			"Europe/Paris|Europe/Warsaw",
			"Europe/Paris|Europe/Zagreb",
			"Europe/Paris|Europe/Zurich",
			"Europe/Paris|Poland",
			"Europe/Ulyanovsk|Europe/Astrakhan",
			"Pacific/Auckland|Antarctica/McMurdo",
			"Pacific/Auckland|Antarctica/South_Pole",
			"Pacific/Auckland|NZ",
			"Pacific/Chatham|NZ-CHAT",
			"Pacific/Easter|Chile/EasterIsland",
			"Pacific/Fakaofo|Etc/GMT-13",
			"Pacific/Fakaofo|Pacific/Enderbury",
			"Pacific/Galapagos|Etc/GMT+6",
			"Pacific/Gambier|Etc/GMT+9",
			"Pacific/Guadalcanal|Antarctica/Macquarie",
			"Pacific/Guadalcanal|Etc/GMT-11",
			"Pacific/Guadalcanal|Pacific/Efate",
			"Pacific/Guadalcanal|Pacific/Kosrae",
			"Pacific/Guadalcanal|Pacific/Noumea",
			"Pacific/Guadalcanal|Pacific/Pohnpei",
			"Pacific/Guadalcanal|Pacific/Ponape",
			"Pacific/Guam|Pacific/Saipan",
			"Pacific/Honolulu|HST",
			"Pacific/Honolulu|Pacific/Johnston",
			"Pacific/Honolulu|US/Hawaii",
			"Pacific/Kiritimati|Etc/GMT-14",
			"Pacific/Niue|Etc/GMT+11",
			"Pacific/Pago_Pago|Pacific/Midway",
			"Pacific/Pago_Pago|Pacific/Samoa",
			"Pacific/Pago_Pago|US/Samoa",
			"Pacific/Pitcairn|Etc/GMT+8",
			"Pacific/Port_Moresby|Antarctica/DumontDUrville",
			"Pacific/Port_Moresby|Etc/GMT-10",
			"Pacific/Port_Moresby|Pacific/Chuuk",
			"Pacific/Port_Moresby|Pacific/Truk",
			"Pacific/Port_Moresby|Pacific/Yap",
			"Pacific/Tahiti|Etc/GMT+10",
			"Pacific/Tahiti|Pacific/Rarotonga"
		]
	});


	return moment;
}));

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require moment-with-locales
//= require moment-timezone-with-data-10-year-range

 var MomentUtil =(function(){
     return {

    duration : function (start, end) {
        return (end?moment(end):moment()).diff(moment(start));
    },

    formatTime : function (text, format) {
        var time = moment.isMoment(text) ? text : moment(text);
        if (text && time.isValid()) {
            return time.format(format);
        } else {
            return '';
        }
    },
     formatTimeUTC: function (text, format) {
         var time = moment.isMoment(text) ? text : moment(text);
         if (text && time.isValid()) {
             return time.utc().format(format);
         } else {
             return '';
         }
     },
    formatTimeSimple : function (text) {
        return MomentUtil.formatTime(text, 'h:mm:ss a');
    },
    formatTimeAtDate : function (text) {
        var time = moment.isMoment(text) ? text : moment(text);
        if (!text || !time.isValid()) {
            return '';
        }
        var now = moment();
        var ms = now.diff(time);
        if(ms < 0 ){
            ms = -ms
        }
        var since = moment.duration(ms);
        if (since.asDays() < 1 && now.month() == time.month()) {
            //same date
            return time.format('h:mm a');
        }
        if (since.asDays() < 1 && now.month() != time.month()) {
            //within a day
            return time.format('h:mm a');
        } else if (since.asWeeks() < 1) {
            return time.format('ddd h:mm a');
        } else if (time.year() != now.year()) {
            return time.format('MMM Do YYYY h a');
        } else {
            return time.format('M/D ha');
        }
    },
         formatTimeAtDateUTC: function (text) {
             "use strict";

             return MomentUtil.formatTimeAtDate(moment.utc(text));
         },
    formatDurationSimple : function (ms) {
        if (ms < 0) {
            return '';
        }
        var duration = moment.duration(ms);
        var y = 0;
        if (duration.asYears() >= 1) {
            y = Math.floor(duration.asYears());
            duration = duration.subtract(y, 'years');
        }
        var d = 0 ;
        if(duration.asDays()>=1.0){
            d = Math.floor(duration.asDays());
            duration = duration.subtract(d,'days');
        }
        var m = duration.minutes();
        var s = duration.seconds();
        return (y>0?y+'y ':'') + (d>0?d+'d ':'') + duration.hours() + '.' + (m < 10 ? '0' + m : m) + ':' + (s < 10 ? '0' + s : s);
    },
    formatDurationHumanize : function (ms) {
        if (ms < 0) {
            return '';
        }
        var duration = moment.duration(ms);
        var valarr = [];
        if (duration.years() > 0) {
            valarr.push(duration.years() + 'y');
        }
        if (duration.months() > 0) {
            valarr.push(duration.months() + 'M');
        }
        if (duration.days() > 0) {
            valarr.push(duration.days() + 'd');
        }
        if (duration.hours() > 0) {
            valarr.push(duration.hours() + 'h');
        }
        if (duration.minutes() > 0) {
            valarr.push(duration.minutes() + 'm');
        }
        if (duration.seconds() > 0) {
            var s = duration.seconds();
            if (duration.milliseconds() > 0) {
                s++;
            }
            valarr.push(s + 's');
        } else if (ms < 1000) {
            valarr.push('0s');
        }
        return valarr.join(' ');
    },
    formatDurationMomentHumanize : function (ms) {
        if (ms < 0) {
            return '';
        }
        var duration = moment.duration(ms);
        return duration.humanize();
    }
};
})();

var process = process || {env: {NODE_ENV: "development"}};
/*!
 * Knockout JavaScript library v3.5.1
 * (c) The Knockout.js team - http://knockoutjs.com/
 * License: MIT (http://www.opensource.org/licenses/mit-license.php)
 */

(function(){
  var DEBUG=true;
  (function(undefined){
      // (0, eval)('this') is a robust way of getting a reference to the global object
      // For details, see http://stackoverflow.com/questions/14119988/return-this-0-evalthis/14120023#14120023
      var window = this || (0, eval)('this'),
          document = window['document'],
          navigator = window['navigator'],
          jQueryInstance = window["jQuery"],
          JSON = window["JSON"];
  
      if (!jQueryInstance && typeof jQuery !== "undefined") {
          jQueryInstance = jQuery;
      }
  (function(factory) {
      // Support three module loading scenarios
      if (typeof define === 'function' && define['amd']) {
          // [1] AMD anonymous module
          define(['exports', 'require'], factory);
      } else if (typeof exports === 'object' && typeof module === 'object') {
          // [2] CommonJS/Node.js
          factory(module['exports'] || exports);  // module.exports is for Node.js
      } else {
          // [3] No module loader (plain <script> tag) - put directly in global namespace
          factory(window['ko'] = {});
      }
  }(function(koExports, amdRequire){
  // Internally, all KO objects are attached to koExports (even the non-exported ones whose names will be minified by the closure compiler).
  // In the future, the following "ko" variable may be made distinct from "koExports" so that private objects are not externally reachable.
  var ko = typeof koExports !== 'undefined' ? koExports : {};
  // Google Closure Compiler helpers (used only to make the minified file smaller)
  ko.exportSymbol = function(koPath, object) {
      var tokens = koPath.split(".");
  
      // In the future, "ko" may become distinct from "koExports" (so that non-exported objects are not reachable)
      // At that point, "target" would be set to: (typeof koExports !== "undefined" ? koExports : ko)
      var target = ko;
  
      for (var i = 0; i < tokens.length - 1; i++)
          target = target[tokens[i]];
      target[tokens[tokens.length - 1]] = object;
  };
  ko.exportProperty = function(owner, publicName, object) {
      owner[publicName] = object;
  };
  ko.version = "3.5.1";
  
  ko.exportSymbol('version', ko.version);
  // For any options that may affect various areas of Knockout and aren't directly associated with data binding.
  ko.options = {
      'deferUpdates': false,
      'useOnlyNativeEvents': false,
      'foreachHidesDestroyed': false
  };
  
  //ko.exportSymbol('options', ko.options);   // 'options' isn't minified
  ko.utils = (function () {
      var hasOwnProperty = Object.prototype.hasOwnProperty;
  
      function objectForEach(obj, action) {
          for (var prop in obj) {
              if (hasOwnProperty.call(obj, prop)) {
                  action(prop, obj[prop]);
              }
          }
      }
  
      function extend(target, source) {
          if (source) {
              for(var prop in source) {
                  if(hasOwnProperty.call(source, prop)) {
                      target[prop] = source[prop];
                  }
              }
          }
          return target;
      }
  
      function setPrototypeOf(obj, proto) {
          obj.__proto__ = proto;
          return obj;
      }
  
      var canSetPrototype = ({ __proto__: [] } instanceof Array);
      var canUseSymbols = !DEBUG && typeof Symbol === 'function';
  
      // Represent the known event types in a compact way, then at runtime transform it into a hash with event name as key (for fast lookup)
      var knownEvents = {}, knownEventTypesByEventName = {};
      var keyEventTypeName = (navigator && /Firefox\/2/i.test(navigator.userAgent)) ? 'KeyboardEvent' : 'UIEvents';
      knownEvents[keyEventTypeName] = ['keyup', 'keydown', 'keypress'];
      knownEvents['MouseEvents'] = ['click', 'dblclick', 'mousedown', 'mouseup', 'mousemove', 'mouseover', 'mouseout', 'mouseenter', 'mouseleave'];
      objectForEach(knownEvents, function(eventType, knownEventsForType) {
          if (knownEventsForType.length) {
              for (var i = 0, j = knownEventsForType.length; i < j; i++)
                  knownEventTypesByEventName[knownEventsForType[i]] = eventType;
          }
      });
      var eventsThatMustBeRegisteredUsingAttachEvent = { 'propertychange': true }; // Workaround for an IE9 issue - https://github.com/SteveSanderson/knockout/issues/406
  
      // Detect IE versions for bug workarounds (uses IE conditionals, not UA string, for robustness)
      // Note that, since IE 10 does not support conditional comments, the following logic only detects IE < 10.
      // Currently this is by design, since IE 10+ behaves correctly when treated as a standard browser.
      // If there is a future need to detect specific versions of IE10+, we will amend this.
      var ieVersion = document && (function() {
          var version = 3, div = document.createElement('div'), iElems = div.getElementsByTagName('i');
  
          // Keep constructing conditional HTML blocks until we hit one that resolves to an empty fragment
          while (
              div.innerHTML = '<!--[if gt IE ' + (++version) + ']><i></i><![endif]-->',
              iElems[0]
          ) {}
          return version > 4 ? version : undefined;
      }());
      var isIe6 = ieVersion === 6,
          isIe7 = ieVersion === 7;
  
      function isClickOnCheckableElement(element, eventType) {
          if ((ko.utils.tagNameLower(element) !== "input") || !element.type) return false;
          if (eventType.toLowerCase() != "click") return false;
          var inputType = element.type;
          return (inputType == "checkbox") || (inputType == "radio");
      }
  
      // For details on the pattern for changing node classes
      // see: https://github.com/knockout/knockout/issues/1597
      var cssClassNameRegex = /\S+/g;
  
      var jQueryEventAttachName;
  
      function toggleDomNodeCssClass(node, classNames, shouldHaveClass) {
          var addOrRemoveFn;
          if (classNames) {
              if (typeof node.classList === 'object') {
                  addOrRemoveFn = node.classList[shouldHaveClass ? 'add' : 'remove'];
                  ko.utils.arrayForEach(classNames.match(cssClassNameRegex), function(className) {
                      addOrRemoveFn.call(node.classList, className);
                  });
              } else if (typeof node.className['baseVal'] === 'string') {
                  // SVG tag .classNames is an SVGAnimatedString instance
                  toggleObjectClassPropertyString(node.className, 'baseVal', classNames, shouldHaveClass);
              } else {
                  // node.className ought to be a string.
                  toggleObjectClassPropertyString(node, 'className', classNames, shouldHaveClass);
              }
          }
      }
  
      function toggleObjectClassPropertyString(obj, prop, classNames, shouldHaveClass) {
          // obj/prop is either a node/'className' or a SVGAnimatedString/'baseVal'.
          var currentClassNames = obj[prop].match(cssClassNameRegex) || [];
          ko.utils.arrayForEach(classNames.match(cssClassNameRegex), function(className) {
              ko.utils.addOrRemoveItem(currentClassNames, className, shouldHaveClass);
          });
          obj[prop] = currentClassNames.join(" ");
      }
  
      return {
          fieldsIncludedWithJsonPost: ['authenticity_token', /^__RequestVerificationToken(_.*)?$/],
  
          arrayForEach: function (array, action, actionOwner) {
              for (var i = 0, j = array.length; i < j; i++) {
                  action.call(actionOwner, array[i], i, array);
              }
          },
  
          arrayIndexOf: typeof Array.prototype.indexOf == "function"
              ? function (array, item) {
                  return Array.prototype.indexOf.call(array, item);
              }
              : function (array, item) {
                  for (var i = 0, j = array.length; i < j; i++) {
                      if (array[i] === item)
                          return i;
                  }
                  return -1;
              },
  
          arrayFirst: function (array, predicate, predicateOwner) {
              for (var i = 0, j = array.length; i < j; i++) {
                  if (predicate.call(predicateOwner, array[i], i, array))
                      return array[i];
              }
              return undefined;
          },
  
          arrayRemoveItem: function (array, itemToRemove) {
              var index = ko.utils.arrayIndexOf(array, itemToRemove);
              if (index > 0) {
                  array.splice(index, 1);
              }
              else if (index === 0) {
                  array.shift();
              }
          },
  
          arrayGetDistinctValues: function (array) {
              var result = [];
              if (array) {
                  ko.utils.arrayForEach(array, function(item) {
                      if (ko.utils.arrayIndexOf(result, item) < 0)
                          result.push(item);
                  });
              }
              return result;
          },
  
          arrayMap: function (array, mapping, mappingOwner) {
              var result = [];
              if (array) {
                  for (var i = 0, j = array.length; i < j; i++)
                      result.push(mapping.call(mappingOwner, array[i], i));
              }
              return result;
          },
  
          arrayFilter: function (array, predicate, predicateOwner) {
              var result = [];
              if (array) {
                  for (var i = 0, j = array.length; i < j; i++)
                      if (predicate.call(predicateOwner, array[i], i))
                          result.push(array[i]);
              }
              return result;
          },
  
          arrayPushAll: function (array, valuesToPush) {
              if (valuesToPush instanceof Array)
                  array.push.apply(array, valuesToPush);
              else
                  for (var i = 0, j = valuesToPush.length; i < j; i++)
                      array.push(valuesToPush[i]);
              return array;
          },
  
          addOrRemoveItem: function(array, value, included) {
              var existingEntryIndex = ko.utils.arrayIndexOf(ko.utils.peekObservable(array), value);
              if (existingEntryIndex < 0) {
                  if (included)
                      array.push(value);
              } else {
                  if (!included)
                      array.splice(existingEntryIndex, 1);
              }
          },
  
          canSetPrototype: canSetPrototype,
  
          extend: extend,
  
          setPrototypeOf: setPrototypeOf,
  
          setPrototypeOfOrExtend: canSetPrototype ? setPrototypeOf : extend,
  
          objectForEach: objectForEach,
  
          objectMap: function(source, mapping, mappingOwner) {
              if (!source)
                  return source;
              var target = {};
              for (var prop in source) {
                  if (hasOwnProperty.call(source, prop)) {
                      target[prop] = mapping.call(mappingOwner, source[prop], prop, source);
                  }
              }
              return target;
          },
  
          emptyDomNode: function (domNode) {
              while (domNode.firstChild) {
                  ko.removeNode(domNode.firstChild);
              }
          },
  
          moveCleanedNodesToContainerElement: function(nodes) {
              // Ensure it's a real array, as we're about to reparent the nodes and
              // we don't want the underlying collection to change while we're doing that.
              var nodesArray = ko.utils.makeArray(nodes);
              var templateDocument = (nodesArray[0] && nodesArray[0].ownerDocument) || document;
  
              var container = templateDocument.createElement('div');
              for (var i = 0, j = nodesArray.length; i < j; i++) {
                  container.appendChild(ko.cleanNode(nodesArray[i]));
              }
              return container;
          },
  
          cloneNodes: function (nodesArray, shouldCleanNodes) {
              for (var i = 0, j = nodesArray.length, newNodesArray = []; i < j; i++) {
                  var clonedNode = nodesArray[i].cloneNode(true);
                  newNodesArray.push(shouldCleanNodes ? ko.cleanNode(clonedNode) : clonedNode);
              }
              return newNodesArray;
          },
  
          setDomNodeChildren: function (domNode, childNodes) {
              ko.utils.emptyDomNode(domNode);
              if (childNodes) {
                  for (var i = 0, j = childNodes.length; i < j; i++)
                      domNode.appendChild(childNodes[i]);
              }
          },
  
          replaceDomNodes: function (nodeToReplaceOrNodeArray, newNodesArray) {
              var nodesToReplaceArray = nodeToReplaceOrNodeArray.nodeType ? [nodeToReplaceOrNodeArray] : nodeToReplaceOrNodeArray;
              if (nodesToReplaceArray.length > 0) {
                  var insertionPoint = nodesToReplaceArray[0];
                  var parent = insertionPoint.parentNode;
                  for (var i = 0, j = newNodesArray.length; i < j; i++)
                      parent.insertBefore(newNodesArray[i], insertionPoint);
                  for (var i = 0, j = nodesToReplaceArray.length; i < j; i++) {
                      ko.removeNode(nodesToReplaceArray[i]);
                  }
              }
          },
  
          fixUpContinuousNodeArray: function(continuousNodeArray, parentNode) {
              // Before acting on a set of nodes that were previously outputted by a template function, we have to reconcile
              // them against what is in the DOM right now. It may be that some of the nodes have already been removed, or that
              // new nodes might have been inserted in the middle, for example by a binding. Also, there may previously have been
              // leading comment nodes (created by rewritten string-based templates) that have since been removed during binding.
              // So, this function translates the old "map" output array into its best guess of the set of current DOM nodes.
              //
              // Rules:
              //   [A] Any leading nodes that have been removed should be ignored
              //       These most likely correspond to memoization nodes that were already removed during binding
              //       See https://github.com/knockout/knockout/pull/440
              //   [B] Any trailing nodes that have been remove should be ignored
              //       This prevents the code here from adding unrelated nodes to the array while processing rule [C]
              //       See https://github.com/knockout/knockout/pull/1903
              //   [C] We want to output a continuous series of nodes. So, ignore any nodes that have already been removed,
              //       and include any nodes that have been inserted among the previous collection
  
              if (continuousNodeArray.length) {
                  // The parent node can be a virtual element; so get the real parent node
                  parentNode = (parentNode.nodeType === 8 && parentNode.parentNode) || parentNode;
  
                  // Rule [A]
                  while (continuousNodeArray.length && continuousNodeArray[0].parentNode !== parentNode)
                      continuousNodeArray.splice(0, 1);
  
                  // Rule [B]
                  while (continuousNodeArray.length > 1 && continuousNodeArray[continuousNodeArray.length - 1].parentNode !== parentNode)
                      continuousNodeArray.length--;
  
                  // Rule [C]
                  if (continuousNodeArray.length > 1) {
                      var current = continuousNodeArray[0], last = continuousNodeArray[continuousNodeArray.length - 1];
                      // Replace with the actual new continuous node set
                      continuousNodeArray.length = 0;
                      while (current !== last) {
                          continuousNodeArray.push(current);
                          current = current.nextSibling;
                      }
                      continuousNodeArray.push(last);
                  }
              }
              return continuousNodeArray;
          },
  
          setOptionNodeSelectionState: function (optionNode, isSelected) {
              // IE6 sometimes throws "unknown error" if you try to write to .selected directly, whereas Firefox struggles with setAttribute. Pick one based on browser.
              if (ieVersion < 7)
                  optionNode.setAttribute("selected", isSelected);
              else
                  optionNode.selected = isSelected;
          },
  
          stringTrim: function (string) {
              return string === null || string === undefined ? '' :
                  string.trim ?
                      string.trim() :
                      string.toString().replace(/^[\s\xa0]+|[\s\xa0]+$/g, '');
          },
  
          stringStartsWith: function (string, startsWith) {
              string = string || "";
              if (startsWith.length > string.length)
                  return false;
              return string.substring(0, startsWith.length) === startsWith;
          },
  
          domNodeIsContainedBy: function (node, containedByNode) {
              if (node === containedByNode)
                  return true;
              if (node.nodeType === 11)
                  return false; // Fixes issue #1162 - can't use node.contains for document fragments on IE8
              if (containedByNode.contains)
                  return containedByNode.contains(node.nodeType !== 1 ? node.parentNode : node);
              if (containedByNode.compareDocumentPosition)
                  return (containedByNode.compareDocumentPosition(node) & 16) == 16;
              while (node && node != containedByNode) {
                  node = node.parentNode;
              }
              return !!node;
          },
  
          domNodeIsAttachedToDocument: function (node) {
              return ko.utils.domNodeIsContainedBy(node, node.ownerDocument.documentElement);
          },
  
          anyDomNodeIsAttachedToDocument: function(nodes) {
              return !!ko.utils.arrayFirst(nodes, ko.utils.domNodeIsAttachedToDocument);
          },
  
          tagNameLower: function(element) {
              // For HTML elements, tagName will always be upper case; for XHTML elements, it'll be lower case.
              // Possible future optimization: If we know it's an element from an XHTML document (not HTML),
              // we don't need to do the .toLowerCase() as it will always be lower case anyway.
              return element && element.tagName && element.tagName.toLowerCase();
          },
  
          catchFunctionErrors: function (delegate) {
              return ko['onError'] ? function () {
                  try {
                      return delegate.apply(this, arguments);
                  } catch (e) {
                      ko['onError'] && ko['onError'](e);
                      throw e;
                  }
              } : delegate;
          },
  
          setTimeout: function (handler, timeout) {
              return setTimeout(ko.utils.catchFunctionErrors(handler), timeout);
          },
  
          deferError: function (error) {
              setTimeout(function () {
                  ko['onError'] && ko['onError'](error);
                  throw error;
              }, 0);
          },
  
          registerEventHandler: function (element, eventType, handler) {
              var wrappedHandler = ko.utils.catchFunctionErrors(handler);
  
              var mustUseAttachEvent = eventsThatMustBeRegisteredUsingAttachEvent[eventType];
              if (!ko.options['useOnlyNativeEvents'] && !mustUseAttachEvent && jQueryInstance) {
                  if (!jQueryEventAttachName) {
                      jQueryEventAttachName = (typeof jQueryInstance(element)['on'] == 'function') ? 'on' : 'bind';
                  }
                  jQueryInstance(element)[jQueryEventAttachName](eventType, wrappedHandler);
              } else if (!mustUseAttachEvent && typeof element.addEventListener == "function")
                  element.addEventListener(eventType, wrappedHandler, false);
              else if (typeof element.attachEvent != "undefined") {
                  var attachEventHandler = function (event) { wrappedHandler.call(element, event); },
                      attachEventName = "on" + eventType;
                  element.attachEvent(attachEventName, attachEventHandler);
  
                  // IE does not dispose attachEvent handlers automatically (unlike with addEventListener)
                  // so to avoid leaks, we have to remove them manually. See bug #856
                  ko.utils.domNodeDisposal.addDisposeCallback(element, function() {
                      element.detachEvent(attachEventName, attachEventHandler);
                  });
              } else
                  throw new Error("Browser doesn't support addEventListener or attachEvent");
          },
  
          triggerEvent: function (element, eventType) {
              if (!(element && element.nodeType))
                  throw new Error("element must be a DOM node when calling triggerEvent");
  
              // For click events on checkboxes and radio buttons, jQuery toggles the element checked state *after* the
              // event handler runs instead of *before*. (This was fixed in 1.9 for checkboxes but not for radio buttons.)
              // IE doesn't change the checked state when you trigger the click event using "fireEvent".
              // In both cases, we'll use the click method instead.
              var useClickWorkaround = isClickOnCheckableElement(element, eventType);
  
              if (!ko.options['useOnlyNativeEvents'] && jQueryInstance && !useClickWorkaround) {
                  jQueryInstance(element)['trigger'](eventType);
              } else if (typeof document.createEvent == "function") {
                  if (typeof element.dispatchEvent == "function") {
                      var eventCategory = knownEventTypesByEventName[eventType] || "HTMLEvents";
                      var event = document.createEvent(eventCategory);
                      event.initEvent(eventType, true, true, window, 0, 0, 0, 0, 0, false, false, false, false, 0, element);
                      element.dispatchEvent(event);
                  }
                  else
                      throw new Error("The supplied element doesn't support dispatchEvent");
              } else if (useClickWorkaround && element.click) {
                  element.click();
              } else if (typeof element.fireEvent != "undefined") {
                  element.fireEvent("on" + eventType);
              } else {
                  throw new Error("Browser doesn't support triggering events");
              }
          },
  
          unwrapObservable: function (value) {
              return ko.isObservable(value) ? value() : value;
          },
  
          peekObservable: function (value) {
              return ko.isObservable(value) ? value.peek() : value;
          },
  
          toggleDomNodeCssClass: toggleDomNodeCssClass,
  
          setTextContent: function(element, textContent) {
              var value = ko.utils.unwrapObservable(textContent);
              if ((value === null) || (value === undefined))
                  value = "";
  
              // We need there to be exactly one child: a text node.
              // If there are no children, more than one, or if it's not a text node,
              // we'll clear everything and create a single text node.
              var innerTextNode = ko.virtualElements.firstChild(element);
              if (!innerTextNode || innerTextNode.nodeType != 3 || ko.virtualElements.nextSibling(innerTextNode)) {
                  ko.virtualElements.setDomNodeChildren(element, [element.ownerDocument.createTextNode(value)]);
              } else {
                  innerTextNode.data = value;
              }
  
              ko.utils.forceRefresh(element);
          },
  
          setElementName: function(element, name) {
              element.name = name;
  
              // Workaround IE 6/7 issue
              // - https://github.com/SteveSanderson/knockout/issues/197
              // - http://www.matts411.com/post/setting_the_name_attribute_in_ie_dom/
              if (ieVersion <= 7) {
                  try {
                      var escapedName = element.name.replace(/[&<>'"]/g, function(r){ return "&#" + r.charCodeAt(0) + ";"; });
                      element.mergeAttributes(document.createElement("<input name='" + escapedName + "'/>"), false);
                  }
                  catch(e) {} // For IE9 with doc mode "IE9 Standards" and browser mode "IE9 Compatibility View"
              }
          },
  
          forceRefresh: function(node) {
              // Workaround for an IE9 rendering bug - https://github.com/SteveSanderson/knockout/issues/209
              if (ieVersion >= 9) {
                  // For text nodes and comment nodes (most likely virtual elements), we will have to refresh the container
                  var elem = node.nodeType == 1 ? node : node.parentNode;
                  if (elem.style)
                      elem.style.zoom = elem.style.zoom;
              }
          },
  
          ensureSelectElementIsRenderedCorrectly: function(selectElement) {
              // Workaround for IE9 rendering bug - it doesn't reliably display all the text in dynamically-added select boxes unless you force it to re-render by updating the width.
              // (See https://github.com/SteveSanderson/knockout/issues/312, http://stackoverflow.com/questions/5908494/select-only-shows-first-char-of-selected-option)
              // Also fixes IE7 and IE8 bug that causes selects to be zero width if enclosed by 'if' or 'with'. (See issue #839)
              if (ieVersion) {
                  var originalWidth = selectElement.style.width;
                  selectElement.style.width = 0;
                  selectElement.style.width = originalWidth;
              }
          },
  
          range: function (min, max) {
              min = ko.utils.unwrapObservable(min);
              max = ko.utils.unwrapObservable(max);
              var result = [];
              for (var i = min; i <= max; i++)
                  result.push(i);
              return result;
          },
  
          makeArray: function(arrayLikeObject) {
              var result = [];
              for (var i = 0, j = arrayLikeObject.length; i < j; i++) {
                  result.push(arrayLikeObject[i]);
              };
              return result;
          },
  
          createSymbolOrString: function(identifier) {
              return canUseSymbols ? Symbol(identifier) : identifier;
          },
  
          isIe6 : isIe6,
          isIe7 : isIe7,
          ieVersion : ieVersion,
  
          getFormFields: function(form, fieldName) {
              var fields = ko.utils.makeArray(form.getElementsByTagName("input")).concat(ko.utils.makeArray(form.getElementsByTagName("textarea")));
              var isMatchingField = (typeof fieldName == 'string')
                  ? function(field) { return field.name === fieldName }
                  : function(field) { return fieldName.test(field.name) }; // Treat fieldName as regex or object containing predicate
              var matches = [];
              for (var i = fields.length - 1; i >= 0; i--) {
                  if (isMatchingField(fields[i]))
                      matches.push(fields[i]);
              };
              return matches;
          },
  
          parseJson: function (jsonString) {
              if (typeof jsonString == "string") {
                  jsonString = ko.utils.stringTrim(jsonString);
                  if (jsonString) {
                      if (JSON && JSON.parse) // Use native parsing where available
                          return JSON.parse(jsonString);
                      return (new Function("return " + jsonString))(); // Fallback on less safe parsing for older browsers
                  }
              }
              return null;
          },
  
          stringifyJson: function (data, replacer, space) {   // replacer and space are optional
              if (!JSON || !JSON.stringify)
                  throw new Error("Cannot find JSON.stringify(). Some browsers (e.g., IE < 8) don't support it natively, but you can overcome this by adding a script reference to json2.js, downloadable from http://www.json.org/json2.js");
              return JSON.stringify(ko.utils.unwrapObservable(data), replacer, space);
          },
  
          postJson: function (urlOrForm, data, options) {
              options = options || {};
              var params = options['params'] || {};
              var includeFields = options['includeFields'] || this.fieldsIncludedWithJsonPost;
              var url = urlOrForm;
  
              // If we were given a form, use its 'action' URL and pick out any requested field values
              if((typeof urlOrForm == 'object') && (ko.utils.tagNameLower(urlOrForm) === "form")) {
                  var originalForm = urlOrForm;
                  url = originalForm.action;
                  for (var i = includeFields.length - 1; i >= 0; i--) {
                      var fields = ko.utils.getFormFields(originalForm, includeFields[i]);
                      for (var j = fields.length - 1; j >= 0; j--)
                          params[fields[j].name] = fields[j].value;
                  }
              }
  
              data = ko.utils.unwrapObservable(data);
              var form = document.createElement("form");
              form.style.display = "none";
              form.action = url;
              form.method = "post";
              for (var key in data) {
                  // Since 'data' this is a model object, we include all properties including those inherited from its prototype
                  var input = document.createElement("input");
                  input.type = "hidden";
                  input.name = key;
                  input.value = ko.utils.stringifyJson(ko.utils.unwrapObservable(data[key]));
                  form.appendChild(input);
              }
              objectForEach(params, function(key, value) {
                  var input = document.createElement("input");
                  input.type = "hidden";
                  input.name = key;
                  input.value = value;
                  form.appendChild(input);
              });
              document.body.appendChild(form);
              options['submitter'] ? options['submitter'](form) : form.submit();
              setTimeout(function () { form.parentNode.removeChild(form); }, 0);
          }
      }
  }());
  
  ko.exportSymbol('utils', ko.utils);
  ko.exportSymbol('utils.arrayForEach', ko.utils.arrayForEach);
  ko.exportSymbol('utils.arrayFirst', ko.utils.arrayFirst);
  ko.exportSymbol('utils.arrayFilter', ko.utils.arrayFilter);
  ko.exportSymbol('utils.arrayGetDistinctValues', ko.utils.arrayGetDistinctValues);
  ko.exportSymbol('utils.arrayIndexOf', ko.utils.arrayIndexOf);
  ko.exportSymbol('utils.arrayMap', ko.utils.arrayMap);
  ko.exportSymbol('utils.arrayPushAll', ko.utils.arrayPushAll);
  ko.exportSymbol('utils.arrayRemoveItem', ko.utils.arrayRemoveItem);
  ko.exportSymbol('utils.cloneNodes', ko.utils.cloneNodes);
  ko.exportSymbol('utils.createSymbolOrString', ko.utils.createSymbolOrString);
  ko.exportSymbol('utils.extend', ko.utils.extend);
  ko.exportSymbol('utils.fieldsIncludedWithJsonPost', ko.utils.fieldsIncludedWithJsonPost);
  ko.exportSymbol('utils.getFormFields', ko.utils.getFormFields);
  ko.exportSymbol('utils.objectMap', ko.utils.objectMap);
  ko.exportSymbol('utils.peekObservable', ko.utils.peekObservable);
  ko.exportSymbol('utils.postJson', ko.utils.postJson);
  ko.exportSymbol('utils.parseJson', ko.utils.parseJson);
  ko.exportSymbol('utils.registerEventHandler', ko.utils.registerEventHandler);
  ko.exportSymbol('utils.stringifyJson', ko.utils.stringifyJson);
  ko.exportSymbol('utils.range', ko.utils.range);
  ko.exportSymbol('utils.toggleDomNodeCssClass', ko.utils.toggleDomNodeCssClass);
  ko.exportSymbol('utils.triggerEvent', ko.utils.triggerEvent);
  ko.exportSymbol('utils.unwrapObservable', ko.utils.unwrapObservable);
  ko.exportSymbol('utils.objectForEach', ko.utils.objectForEach);
  ko.exportSymbol('utils.addOrRemoveItem', ko.utils.addOrRemoveItem);
  ko.exportSymbol('utils.setTextContent', ko.utils.setTextContent);
  ko.exportSymbol('unwrap', ko.utils.unwrapObservable); // Convenient shorthand, because this is used so commonly
  
  if (!Function.prototype['bind']) {
      // Function.prototype.bind is a standard part of ECMAScript 5th Edition (December 2009, http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-262.pdf)
      // In case the browser doesn't implement it natively, provide a JavaScript implementation. This implementation is based on the one in prototype.js
      Function.prototype['bind'] = function (object) {
          var originalFunction = this;
          if (arguments.length === 1) {
              return function () {
                  return originalFunction.apply(object, arguments);
              };
          } else {
              var partialArgs = Array.prototype.slice.call(arguments, 1);
              return function () {
                  var args = partialArgs.slice(0);
                  args.push.apply(args, arguments);
                  return originalFunction.apply(object, args);
              };
          }
      };
  }
  
  ko.utils.domData = new (function () {
      var uniqueId = 0;
      var dataStoreKeyExpandoPropertyName = "__ko__" + (new Date).getTime();
      var dataStore = {};
  
      var getDataForNode, clear;
      if (!ko.utils.ieVersion) {
          // We considered using WeakMap, but it has a problem in IE 11 and Edge that prevents using
          // it cross-window, so instead we just store the data directly on the node.
          // See https://github.com/knockout/knockout/issues/2141
          getDataForNode = function (node, createIfNotFound) {
              var dataForNode = node[dataStoreKeyExpandoPropertyName];
              if (!dataForNode && createIfNotFound) {
                  dataForNode = node[dataStoreKeyExpandoPropertyName] = {};
              }
              return dataForNode;
          };
          clear = function (node) {
              if (node[dataStoreKeyExpandoPropertyName]) {
                  delete node[dataStoreKeyExpandoPropertyName];
                  return true; // Exposing "did clean" flag purely so specs can infer whether things have been cleaned up as intended
              }
              return false;
          };
      } else {
          // Old IE versions have memory issues if you store objects on the node, so we use a
          // separate data storage and link to it from the node using a string key.
          getDataForNode = function (node, createIfNotFound) {
              var dataStoreKey = node[dataStoreKeyExpandoPropertyName];
              var hasExistingDataStore = dataStoreKey && (dataStoreKey !== "null") && dataStore[dataStoreKey];
              if (!hasExistingDataStore) {
                  if (!createIfNotFound)
                      return undefined;
                  dataStoreKey = node[dataStoreKeyExpandoPropertyName] = "ko" + uniqueId++;
                  dataStore[dataStoreKey] = {};
              }
              return dataStore[dataStoreKey];
          };
          clear = function (node) {
              var dataStoreKey = node[dataStoreKeyExpandoPropertyName];
              if (dataStoreKey) {
                  delete dataStore[dataStoreKey];
                  node[dataStoreKeyExpandoPropertyName] = null;
                  return true; // Exposing "did clean" flag purely so specs can infer whether things have been cleaned up as intended
              }
              return false;
          };
      }
  
      return {
          get: function (node, key) {
              var dataForNode = getDataForNode(node, false);
              return dataForNode && dataForNode[key];
          },
          set: function (node, key, value) {
              // Make sure we don't actually create a new domData key if we are actually deleting a value
              var dataForNode = getDataForNode(node, value !== undefined /* createIfNotFound */);
              dataForNode && (dataForNode[key] = value);
          },
          getOrSet: function (node, key, value) {
              var dataForNode = getDataForNode(node, true /* createIfNotFound */);
              return dataForNode[key] || (dataForNode[key] = value);
          },
          clear: clear,
  
          nextKey: function () {
              return (uniqueId++) + dataStoreKeyExpandoPropertyName;
          }
      };
  })();
  
  ko.exportSymbol('utils.domData', ko.utils.domData);
  ko.exportSymbol('utils.domData.clear', ko.utils.domData.clear); // Exporting only so specs can clear up after themselves fully
  
  ko.utils.domNodeDisposal = new (function () {
      var domDataKey = ko.utils.domData.nextKey();
      var cleanableNodeTypes = { 1: true, 8: true, 9: true };       // Element, Comment, Document
      var cleanableNodeTypesWithDescendants = { 1: true, 9: true }; // Element, Document
  
      function getDisposeCallbacksCollection(node, createIfNotFound) {
          var allDisposeCallbacks = ko.utils.domData.get(node, domDataKey);
          if ((allDisposeCallbacks === undefined) && createIfNotFound) {
              allDisposeCallbacks = [];
              ko.utils.domData.set(node, domDataKey, allDisposeCallbacks);
          }
          return allDisposeCallbacks;
      }
      function destroyCallbacksCollection(node) {
          ko.utils.domData.set(node, domDataKey, undefined);
      }
  
      function cleanSingleNode(node) {
          // Run all the dispose callbacks
          var callbacks = getDisposeCallbacksCollection(node, false);
          if (callbacks) {
              callbacks = callbacks.slice(0); // Clone, as the array may be modified during iteration (typically, callbacks will remove themselves)
              for (var i = 0; i < callbacks.length; i++)
                  callbacks[i](node);
          }
  
          // Erase the DOM data
          ko.utils.domData.clear(node);
  
          // Perform cleanup needed by external libraries (currently only jQuery, but can be extended)
          ko.utils.domNodeDisposal["cleanExternalData"](node);
  
          // Clear any immediate-child comment nodes, as these wouldn't have been found by
          // node.getElementsByTagName("*") in cleanNode() (comment nodes aren't elements)
          if (cleanableNodeTypesWithDescendants[node.nodeType]) {
              cleanNodesInList(node.childNodes, true/*onlyComments*/);
          }
      }
  
      function cleanNodesInList(nodeList, onlyComments) {
          var cleanedNodes = [], lastCleanedNode;
          for (var i = 0; i < nodeList.length; i++) {
              if (!onlyComments || nodeList[i].nodeType === 8) {
                  cleanSingleNode(cleanedNodes[cleanedNodes.length] = lastCleanedNode = nodeList[i]);
                  if (nodeList[i] !== lastCleanedNode) {
                      while (i-- && ko.utils.arrayIndexOf(cleanedNodes, nodeList[i]) == -1) {}
                  }
              }
          }
      }
  
      return {
          addDisposeCallback : function(node, callback) {
              if (typeof callback != "function")
                  throw new Error("Callback must be a function");
              getDisposeCallbacksCollection(node, true).push(callback);
          },
  
          removeDisposeCallback : function(node, callback) {
              var callbacksCollection = getDisposeCallbacksCollection(node, false);
              if (callbacksCollection) {
                  ko.utils.arrayRemoveItem(callbacksCollection, callback);
                  if (callbacksCollection.length == 0)
                      destroyCallbacksCollection(node);
              }
          },
  
          cleanNode : function(node) {
              ko.dependencyDetection.ignore(function () {
                  // First clean this node, where applicable
                  if (cleanableNodeTypes[node.nodeType]) {
                      cleanSingleNode(node);
  
                      // ... then its descendants, where applicable
                      if (cleanableNodeTypesWithDescendants[node.nodeType]) {
                          cleanNodesInList(node.getElementsByTagName("*"));
                      }
                  }
              });
  
              return node;
          },
  
          removeNode : function(node) {
              ko.cleanNode(node);
              if (node.parentNode)
                  node.parentNode.removeChild(node);
          },
  
          "cleanExternalData" : function (node) {
              // Special support for jQuery here because it's so commonly used.
              // Many jQuery plugins (including jquery.tmpl) store data using jQuery's equivalent of domData
              // so notify it to tear down any resources associated with the node & descendants here.
              if (jQueryInstance && (typeof jQueryInstance['cleanData'] == "function"))
                  jQueryInstance['cleanData']([node]);
          }
      };
  })();
  ko.cleanNode = ko.utils.domNodeDisposal.cleanNode; // Shorthand name for convenience
  ko.removeNode = ko.utils.domNodeDisposal.removeNode; // Shorthand name for convenience
  ko.exportSymbol('cleanNode', ko.cleanNode);
  ko.exportSymbol('removeNode', ko.removeNode);
  ko.exportSymbol('utils.domNodeDisposal', ko.utils.domNodeDisposal);
  ko.exportSymbol('utils.domNodeDisposal.addDisposeCallback', ko.utils.domNodeDisposal.addDisposeCallback);
  ko.exportSymbol('utils.domNodeDisposal.removeDisposeCallback', ko.utils.domNodeDisposal.removeDisposeCallback);
  (function () {
      var none = [0, "", ""],
          table = [1, "<table>", "</table>"],
          tbody = [2, "<table><tbody>", "</tbody></table>"],
          tr = [3, "<table><tbody><tr>", "</tr></tbody></table>"],
          select = [1, "<select multiple='multiple'>", "</select>"],
          lookup = {
              'thead': table,
              'tbody': table,
              'tfoot': table,
              'tr': tbody,
              'td': tr,
              'th': tr,
              'option': select,
              'optgroup': select
          },
  
          // This is needed for old IE if you're *not* using either jQuery or innerShiv. Doesn't affect other cases.
          mayRequireCreateElementHack = ko.utils.ieVersion <= 8;
  
      function getWrap(tags) {
          var m = tags.match(/^(?:<!--.*?-->\s*?)*?<([a-z]+)[\s>]/);
          return (m && lookup[m[1]]) || none;
      }
  
      function simpleHtmlParse(html, documentContext) {
          documentContext || (documentContext = document);
          var windowContext = documentContext['parentWindow'] || documentContext['defaultView'] || window;
  
          // Based on jQuery's "clean" function, but only accounting for table-related elements.
          // If you have referenced jQuery, this won't be used anyway - KO will use jQuery's "clean" function directly
  
          // Note that there's still an issue in IE < 9 whereby it will discard comment nodes that are the first child of
          // a descendant node. For example: "<div><!-- mycomment -->abc</div>" will get parsed as "<div>abc</div>"
          // This won't affect anyone who has referenced jQuery, and there's always the workaround of inserting a dummy node
          // (possibly a text node) in front of the comment. So, KO does not attempt to workaround this IE issue automatically at present.
  
          // Trim whitespace, otherwise indexOf won't work as expected
          var tags = ko.utils.stringTrim(html).toLowerCase(), div = documentContext.createElement("div"),
              wrap = getWrap(tags),
              depth = wrap[0];
  
          // Go to html and back, then peel off extra wrappers
          // Note that we always prefix with some dummy text, because otherwise, IE<9 will strip out leading comment nodes in descendants. Total madness.
          var markup = "ignored<div>" + wrap[1] + html + wrap[2] + "</div>";
          if (typeof windowContext['innerShiv'] == "function") {
              // Note that innerShiv is deprecated in favour of html5shiv. We should consider adding
              // support for html5shiv (except if no explicit support is needed, e.g., if html5shiv
              // somehow shims the native APIs so it just works anyway)
              div.appendChild(windowContext['innerShiv'](markup));
          } else {
              if (mayRequireCreateElementHack) {
                  // The document.createElement('my-element') trick to enable custom elements in IE6-8
                  // only works if we assign innerHTML on an element associated with that document.
                  documentContext.body.appendChild(div);
              }
  
              div.innerHTML = markup;
  
              if (mayRequireCreateElementHack) {
                  div.parentNode.removeChild(div);
              }
          }
  
          // Move to the right depth
          while (depth--)
              div = div.lastChild;
  
          return ko.utils.makeArray(div.lastChild.childNodes);
      }
  
      function jQueryHtmlParse(html, documentContext) {
          // jQuery's "parseHTML" function was introduced in jQuery 1.8.0 and is a documented public API.
          if (jQueryInstance['parseHTML']) {
              return jQueryInstance['parseHTML'](html, documentContext) || []; // Ensure we always return an array and never null
          } else {
              // For jQuery < 1.8.0, we fall back on the undocumented internal "clean" function.
              var elems = jQueryInstance['clean']([html], documentContext);
  
              // As of jQuery 1.7.1, jQuery parses the HTML by appending it to some dummy parent nodes held in an in-memory document fragment.
              // Unfortunately, it never clears the dummy parent nodes from the document fragment, so it leaks memory over time.
              // Fix this by finding the top-most dummy parent element, and detaching it from its owner fragment.
              if (elems && elems[0]) {
                  // Find the top-most parent element that's a direct child of a document fragment
                  var elem = elems[0];
                  while (elem.parentNode && elem.parentNode.nodeType !== 11 /* i.e., DocumentFragment */)
                      elem = elem.parentNode;
                  // ... then detach it
                  if (elem.parentNode)
                      elem.parentNode.removeChild(elem);
              }
  
              return elems;
          }
      }
  
      ko.utils.parseHtmlFragment = function(html, documentContext) {
          return jQueryInstance ?
              jQueryHtmlParse(html, documentContext) :   // As below, benefit from jQuery's optimisations where possible
              simpleHtmlParse(html, documentContext);  // ... otherwise, this simple logic will do in most common cases.
      };
  
      ko.utils.parseHtmlForTemplateNodes = function(html, documentContext) {
          var nodes = ko.utils.parseHtmlFragment(html, documentContext);
          return (nodes.length && nodes[0].parentElement) || ko.utils.moveCleanedNodesToContainerElement(nodes);
      };
  
      ko.utils.setHtml = function(node, html) {
          ko.utils.emptyDomNode(node);
  
          // There's no legitimate reason to display a stringified observable without unwrapping it, so we'll unwrap it
          html = ko.utils.unwrapObservable(html);
  
          if ((html !== null) && (html !== undefined)) {
              if (typeof html != 'string')
                  html = html.toString();
  
              // jQuery contains a lot of sophisticated code to parse arbitrary HTML fragments,
              // for example <tr> elements which are not normally allowed to exist on their own.
              // If you've referenced jQuery we'll use that rather than duplicating its code.
              if (jQueryInstance) {
                  jQueryInstance(node)['html'](html);
              } else {
                  // ... otherwise, use KO's own parsing logic.
                  var parsedNodes = ko.utils.parseHtmlFragment(html, node.ownerDocument);
                  for (var i = 0; i < parsedNodes.length; i++)
                      node.appendChild(parsedNodes[i]);
              }
          }
      };
  })();
  
  ko.exportSymbol('utils.parseHtmlFragment', ko.utils.parseHtmlFragment);
  ko.exportSymbol('utils.setHtml', ko.utils.setHtml);
  
  ko.memoization = (function () {
      var memos = {};
  
      function randomMax8HexChars() {
          return (((1 + Math.random()) * 0x100000000) | 0).toString(16).substring(1);
      }
      function generateRandomId() {
          return randomMax8HexChars() + randomMax8HexChars();
      }
      function findMemoNodes(rootNode, appendToArray) {
          if (!rootNode)
              return;
          if (rootNode.nodeType == 8) {
              var memoId = ko.memoization.parseMemoText(rootNode.nodeValue);
              if (memoId != null)
                  appendToArray.push({ domNode: rootNode, memoId: memoId });
          } else if (rootNode.nodeType == 1) {
              for (var i = 0, childNodes = rootNode.childNodes, j = childNodes.length; i < j; i++)
                  findMemoNodes(childNodes[i], appendToArray);
          }
      }
  
      return {
          memoize: function (callback) {
              if (typeof callback != "function")
                  throw new Error("You can only pass a function to ko.memoization.memoize()");
              var memoId = generateRandomId();
              memos[memoId] = callback;
              return "<!--[ko_memo:" + memoId + "]-->";
          },
  
          unmemoize: function (memoId, callbackParams) {
              var callback = memos[memoId];
              if (callback === undefined)
                  throw new Error("Couldn't find any memo with ID " + memoId + ". Perhaps it's already been unmemoized.");
              try {
                  callback.apply(null, callbackParams || []);
                  return true;
              }
              finally { delete memos[memoId]; }
          },
  
          unmemoizeDomNodeAndDescendants: function (domNode, extraCallbackParamsArray) {
              var memos = [];
              findMemoNodes(domNode, memos);
              for (var i = 0, j = memos.length; i < j; i++) {
                  var node = memos[i].domNode;
                  var combinedParams = [node];
                  if (extraCallbackParamsArray)
                      ko.utils.arrayPushAll(combinedParams, extraCallbackParamsArray);
                  ko.memoization.unmemoize(memos[i].memoId, combinedParams);
                  node.nodeValue = ""; // Neuter this node so we don't try to unmemoize it again
                  if (node.parentNode)
                      node.parentNode.removeChild(node); // If possible, erase it totally (not always possible - someone else might just hold a reference to it then call unmemoizeDomNodeAndDescendants again)
              }
          },
  
          parseMemoText: function (memoText) {
              var match = memoText.match(/^\[ko_memo\:(.*?)\]$/);
              return match ? match[1] : null;
          }
      };
  })();
  
  ko.exportSymbol('memoization', ko.memoization);
  ko.exportSymbol('memoization.memoize', ko.memoization.memoize);
  ko.exportSymbol('memoization.unmemoize', ko.memoization.unmemoize);
  ko.exportSymbol('memoization.parseMemoText', ko.memoization.parseMemoText);
  ko.exportSymbol('memoization.unmemoizeDomNodeAndDescendants', ko.memoization.unmemoizeDomNodeAndDescendants);
  ko.tasks = (function () {
      var scheduler,
          taskQueue = [],
          taskQueueLength = 0,
          nextHandle = 1,
          nextIndexToProcess = 0;
  
      if (window['MutationObserver']) {
          // Chrome 27+, Firefox 14+, IE 11+, Opera 15+, Safari 6.1+
          // From https://github.com/petkaantonov/bluebird * Copyright (c) 2014 Petka Antonov * License: MIT
          scheduler = (function (callback) {
              var div = document.createElement("div");
              new MutationObserver(callback).observe(div, {attributes: true});
              return function () { div.classList.toggle("foo"); };
          })(scheduledProcess);
      } else if (document && "onreadystatechange" in document.createElement("script")) {
          // IE 6-10
          // From https://github.com/YuzuJS/setImmediate * Copyright (c) 2012 Barnesandnoble.com, llc, Donavon West, and Domenic Denicola * License: MIT
          scheduler = function (callback) {
              var script = document.createElement("script");
              script.onreadystatechange = function () {
                  script.onreadystatechange = null;
                  document.documentElement.removeChild(script);
                  script = null;
                  callback();
              };
              document.documentElement.appendChild(script);
          };
      } else {
          scheduler = function (callback) {
              setTimeout(callback, 0);
          };
      }
  
      function processTasks() {
          if (taskQueueLength) {
              // Each mark represents the end of a logical group of tasks and the number of these groups is
              // limited to prevent unchecked recursion.
              var mark = taskQueueLength, countMarks = 0;
  
              // nextIndexToProcess keeps track of where we are in the queue; processTasks can be called recursively without issue
              for (var task; nextIndexToProcess < taskQueueLength; ) {
                  if (task = taskQueue[nextIndexToProcess++]) {
                      if (nextIndexToProcess > mark) {
                          if (++countMarks >= 5000) {
                              nextIndexToProcess = taskQueueLength;   // skip all tasks remaining in the queue since any of them could be causing the recursion
                              ko.utils.deferError(Error("'Too much recursion' after processing " + countMarks + " task groups."));
                              break;
                          }
                          mark = taskQueueLength;
                      }
                      try {
                          task();
                      } catch (ex) {
                          ko.utils.deferError(ex);
                      }
                  }
              }
          }
      }
  
      function scheduledProcess() {
          processTasks();
  
          // Reset the queue
          nextIndexToProcess = taskQueueLength = taskQueue.length = 0;
      }
  
      function scheduleTaskProcessing() {
          ko.tasks['scheduler'](scheduledProcess);
      }
  
      var tasks = {
          'scheduler': scheduler,     // Allow overriding the scheduler
  
          schedule: function (func) {
              if (!taskQueueLength) {
                  scheduleTaskProcessing();
              }
  
              taskQueue[taskQueueLength++] = func;
              return nextHandle++;
          },
  
          cancel: function (handle) {
              var index = handle - (nextHandle - taskQueueLength);
              if (index >= nextIndexToProcess && index < taskQueueLength) {
                  taskQueue[index] = null;
              }
          },
  
          // For testing only: reset the queue and return the previous queue length
          'resetForTesting': function () {
              var length = taskQueueLength - nextIndexToProcess;
              nextIndexToProcess = taskQueueLength = taskQueue.length = 0;
              return length;
          },
  
          runEarly: processTasks
      };
  
      return tasks;
  })();
  
  ko.exportSymbol('tasks', ko.tasks);
  ko.exportSymbol('tasks.schedule', ko.tasks.schedule);
  //ko.exportSymbol('tasks.cancel', ko.tasks.cancel);  "cancel" isn't minified
  ko.exportSymbol('tasks.runEarly', ko.tasks.runEarly);
  ko.extenders = {
      'throttle': function(target, timeout) {
          // Throttling means two things:
  
          // (1) For dependent observables, we throttle *evaluations* so that, no matter how fast its dependencies
          //     notify updates, the target doesn't re-evaluate (and hence doesn't notify) faster than a certain rate
          target['throttleEvaluation'] = timeout;
  
          // (2) For writable targets (observables, or writable dependent observables), we throttle *writes*
          //     so the target cannot change value synchronously or faster than a certain rate
          var writeTimeoutInstance = null;
          return ko.dependentObservable({
              'read': target,
              'write': function(value) {
                  clearTimeout(writeTimeoutInstance);
                  writeTimeoutInstance = ko.utils.setTimeout(function() {
                      target(value);
                  }, timeout);
              }
          });
      },
  
      'rateLimit': function(target, options) {
          var timeout, method, limitFunction;
  
          if (typeof options == 'number') {
              timeout = options;
          } else {
              timeout = options['timeout'];
              method = options['method'];
          }
  
          // rateLimit supersedes deferred updates
          target._deferUpdates = false;
  
          limitFunction = typeof method == 'function' ? method : method == 'notifyWhenChangesStop' ?  debounce : throttle;
          target.limit(function(callback) {
              return limitFunction(callback, timeout, options);
          });
      },
  
      'deferred': function(target, options) {
          if (options !== true) {
              throw new Error('The \'deferred\' extender only accepts the value \'true\', because it is not supported to turn deferral off once enabled.')
          }
  
          if (!target._deferUpdates) {
              target._deferUpdates = true;
              target.limit(function (callback) {
                  var handle,
                      ignoreUpdates = false;
                  return function () {
                      if (!ignoreUpdates) {
                          ko.tasks.cancel(handle);
                          handle = ko.tasks.schedule(callback);
  
                          try {
                              ignoreUpdates = true;
                              target['notifySubscribers'](undefined, 'dirty');
                          } finally {
                              ignoreUpdates = false;
                          }
                      }
                  };
              });
          }
      },
  
      'notify': function(target, notifyWhen) {
          target["equalityComparer"] = notifyWhen == "always" ?
              null :  // null equalityComparer means to always notify
              valuesArePrimitiveAndEqual;
      }
  };
  
  var primitiveTypes = { 'undefined':1, 'boolean':1, 'number':1, 'string':1 };
  function valuesArePrimitiveAndEqual(a, b) {
      var oldValueIsPrimitive = (a === null) || (typeof(a) in primitiveTypes);
      return oldValueIsPrimitive ? (a === b) : false;
  }
  
  function throttle(callback, timeout) {
      var timeoutInstance;
      return function () {
          if (!timeoutInstance) {
              timeoutInstance = ko.utils.setTimeout(function () {
                  timeoutInstance = undefined;
                  callback();
              }, timeout);
          }
      };
  }
  
  function debounce(callback, timeout) {
      var timeoutInstance;
      return function () {
          clearTimeout(timeoutInstance);
          timeoutInstance = ko.utils.setTimeout(callback, timeout);
      };
  }
  
  function applyExtenders(requestedExtenders) {
      var target = this;
      if (requestedExtenders) {
          ko.utils.objectForEach(requestedExtenders, function(key, value) {
              var extenderHandler = ko.extenders[key];
              if (typeof extenderHandler == 'function') {
                  target = extenderHandler(target, value) || target;
              }
          });
      }
      return target;
  }
  
  ko.exportSymbol('extenders', ko.extenders);
  
  ko.subscription = function (target, callback, disposeCallback) {
      this._target = target;
      this._callback = callback;
      this._disposeCallback = disposeCallback;
      this._isDisposed = false;
      this._node = null;
      this._domNodeDisposalCallback = null;
      ko.exportProperty(this, 'dispose', this.dispose);
      ko.exportProperty(this, 'disposeWhenNodeIsRemoved', this.disposeWhenNodeIsRemoved);
  };
  ko.subscription.prototype.dispose = function () {
      var self = this;
      if (!self._isDisposed) {
          if (self._domNodeDisposalCallback) {
              ko.utils.domNodeDisposal.removeDisposeCallback(self._node, self._domNodeDisposalCallback);
          }
          self._isDisposed = true;
          self._disposeCallback();
  
          self._target = self._callback = self._disposeCallback = self._node = self._domNodeDisposalCallback = null;
      }
  };
  ko.subscription.prototype.disposeWhenNodeIsRemoved = function (node) {
      this._node = node;
      ko.utils.domNodeDisposal.addDisposeCallback(node, this._domNodeDisposalCallback = this.dispose.bind(this));
  };
  
  ko.subscribable = function () {
      ko.utils.setPrototypeOfOrExtend(this, ko_subscribable_fn);
      ko_subscribable_fn.init(this);
  }
  
  var defaultEvent = "change";
  
  // Moved out of "limit" to avoid the extra closure
  function limitNotifySubscribers(value, event) {
      if (!event || event === defaultEvent) {
          this._limitChange(value);
      } else if (event === 'beforeChange') {
          this._limitBeforeChange(value);
      } else {
          this._origNotifySubscribers(value, event);
      }
  }
  
  var ko_subscribable_fn = {
      init: function(instance) {
          instance._subscriptions = { "change": [] };
          instance._versionNumber = 1;
      },
  
      subscribe: function (callback, callbackTarget, event) {
          var self = this;
  
          event = event || defaultEvent;
          var boundCallback = callbackTarget ? callback.bind(callbackTarget) : callback;
  
          var subscription = new ko.subscription(self, boundCallback, function () {
              ko.utils.arrayRemoveItem(self._subscriptions[event], subscription);
              if (self.afterSubscriptionRemove)
                  self.afterSubscriptionRemove(event);
          });
  
          if (self.beforeSubscriptionAdd)
              self.beforeSubscriptionAdd(event);
  
          if (!self._subscriptions[event])
              self._subscriptions[event] = [];
          self._subscriptions[event].push(subscription);
  
          return subscription;
      },
  
      "notifySubscribers": function (valueToNotify, event) {
          event = event || defaultEvent;
          if (event === defaultEvent) {
              this.updateVersion();
          }
          if (this.hasSubscriptionsForEvent(event)) {
              var subs = event === defaultEvent && this._changeSubscriptions || this._subscriptions[event].slice(0);
              try {
                  ko.dependencyDetection.begin(); // Begin suppressing dependency detection (by setting the top frame to undefined)
                  for (var i = 0, subscription; subscription = subs[i]; ++i) {
                      // In case a subscription was disposed during the arrayForEach cycle, check
                      // for isDisposed on each subscription before invoking its callback
                      if (!subscription._isDisposed)
                          subscription._callback(valueToNotify);
                  }
              } finally {
                  ko.dependencyDetection.end(); // End suppressing dependency detection
              }
          }
      },
  
      getVersion: function () {
          return this._versionNumber;
      },
  
      hasChanged: function (versionToCheck) {
          return this.getVersion() !== versionToCheck;
      },
  
      updateVersion: function () {
          ++this._versionNumber;
      },
  
      limit: function(limitFunction) {
          var self = this, selfIsObservable = ko.isObservable(self),
              ignoreBeforeChange, notifyNextChange, previousValue, pendingValue, didUpdate,
              beforeChange = 'beforeChange';
  
          if (!self._origNotifySubscribers) {
              self._origNotifySubscribers = self["notifySubscribers"];
              self["notifySubscribers"] = limitNotifySubscribers;
          }
  
          var finish = limitFunction(function() {
              self._notificationIsPending = false;
  
              // If an observable provided a reference to itself, access it to get the latest value.
              // This allows computed observables to delay calculating their value until needed.
              if (selfIsObservable && pendingValue === self) {
                  pendingValue = self._evalIfChanged ? self._evalIfChanged() : self();
              }
              var shouldNotify = notifyNextChange || (didUpdate && self.isDifferent(previousValue, pendingValue));
  
              didUpdate = notifyNextChange = ignoreBeforeChange = false;
  
              if (shouldNotify) {
                  self._origNotifySubscribers(previousValue = pendingValue);
              }
          });
  
          self._limitChange = function(value, isDirty) {
              if (!isDirty || !self._notificationIsPending) {
                  didUpdate = !isDirty;
              }
              self._changeSubscriptions = self._subscriptions[defaultEvent].slice(0);
              self._notificationIsPending = ignoreBeforeChange = true;
              pendingValue = value;
              finish();
          };
          self._limitBeforeChange = function(value) {
              if (!ignoreBeforeChange) {
                  previousValue = value;
                  self._origNotifySubscribers(value, beforeChange);
              }
          };
          self._recordUpdate = function() {
              didUpdate = true;
          };
          self._notifyNextChangeIfValueIsDifferent = function() {
              if (self.isDifferent(previousValue, self.peek(true /*evaluate*/))) {
                  notifyNextChange = true;
              }
          };
      },
  
      hasSubscriptionsForEvent: function(event) {
          return this._subscriptions[event] && this._subscriptions[event].length;
      },
  
      getSubscriptionsCount: function (event) {
          if (event) {
              return this._subscriptions[event] && this._subscriptions[event].length || 0;
          } else {
              var total = 0;
              ko.utils.objectForEach(this._subscriptions, function(eventName, subscriptions) {
                  if (eventName !== 'dirty')
                      total += subscriptions.length;
              });
              return total;
          }
      },
  
      isDifferent: function(oldValue, newValue) {
          return !this['equalityComparer'] || !this['equalityComparer'](oldValue, newValue);
      },
  
      toString: function() {
        return '[object Object]'
      },
  
      extend: applyExtenders
  };
  
  ko.exportProperty(ko_subscribable_fn, 'init', ko_subscribable_fn.init);
  ko.exportProperty(ko_subscribable_fn, 'subscribe', ko_subscribable_fn.subscribe);
  ko.exportProperty(ko_subscribable_fn, 'extend', ko_subscribable_fn.extend);
  ko.exportProperty(ko_subscribable_fn, 'getSubscriptionsCount', ko_subscribable_fn.getSubscriptionsCount);
  
  // For browsers that support proto assignment, we overwrite the prototype of each
  // observable instance. Since observables are functions, we need Function.prototype
  // to still be in the prototype chain.
  if (ko.utils.canSetPrototype) {
      ko.utils.setPrototypeOf(ko_subscribable_fn, Function.prototype);
  }
  
  ko.subscribable['fn'] = ko_subscribable_fn;
  
  
  ko.isSubscribable = function (instance) {
      return instance != null && typeof instance.subscribe == "function" && typeof instance["notifySubscribers"] == "function";
  };
  
  ko.exportSymbol('subscribable', ko.subscribable);
  ko.exportSymbol('isSubscribable', ko.isSubscribable);
  
  ko.computedContext = ko.dependencyDetection = (function () {
      var outerFrames = [],
          currentFrame,
          lastId = 0;
  
      // Return a unique ID that can be assigned to an observable for dependency tracking.
      // Theoretically, you could eventually overflow the number storage size, resulting
      // in duplicate IDs. But in JavaScript, the largest exact integral value is 2^53
      // or 9,007,199,254,740,992. If you created 1,000,000 IDs per second, it would
      // take over 285 years to reach that number.
      // Reference http://blog.vjeux.com/2010/javascript/javascript-max_int-number-limits.html
      function getId() {
          return ++lastId;
      }
  
      function begin(options) {
          outerFrames.push(currentFrame);
          currentFrame = options;
      }
  
      function end() {
          currentFrame = outerFrames.pop();
      }
  
      return {
          begin: begin,
  
          end: end,
  
          registerDependency: function (subscribable) {
              if (currentFrame) {
                  if (!ko.isSubscribable(subscribable))
                      throw new Error("Only subscribable things can act as dependencies");
                  currentFrame.callback.call(currentFrame.callbackTarget, subscribable, subscribable._id || (subscribable._id = getId()));
              }
          },
  
          ignore: function (callback, callbackTarget, callbackArgs) {
              try {
                  begin();
                  return callback.apply(callbackTarget, callbackArgs || []);
              } finally {
                  end();
              }
          },
  
          getDependenciesCount: function () {
              if (currentFrame)
                  return currentFrame.computed.getDependenciesCount();
          },
  
          getDependencies: function () {
              if (currentFrame)
                  return currentFrame.computed.getDependencies();
          },
  
          isInitial: function() {
              if (currentFrame)
                  return currentFrame.isInitial;
          },
  
          computed: function() {
              if (currentFrame)
                  return currentFrame.computed;
          }
      };
  })();
  
  ko.exportSymbol('computedContext', ko.computedContext);
  ko.exportSymbol('computedContext.getDependenciesCount', ko.computedContext.getDependenciesCount);
  ko.exportSymbol('computedContext.getDependencies', ko.computedContext.getDependencies);
  ko.exportSymbol('computedContext.isInitial', ko.computedContext.isInitial);
  ko.exportSymbol('computedContext.registerDependency', ko.computedContext.registerDependency);
  
  ko.exportSymbol('ignoreDependencies', ko.ignoreDependencies = ko.dependencyDetection.ignore);
  var observableLatestValue = ko.utils.createSymbolOrString('_latestValue');
  
  ko.observable = function (initialValue) {
      function observable() {
          if (arguments.length > 0) {
              // Write
  
              // Ignore writes if the value hasn't changed
              if (observable.isDifferent(observable[observableLatestValue], arguments[0])) {
                  observable.valueWillMutate();
                  observable[observableLatestValue] = arguments[0];
                  observable.valueHasMutated();
              }
              return this; // Permits chained assignments
          }
          else {
              // Read
              ko.dependencyDetection.registerDependency(observable); // The caller only needs to be notified of changes if they did a "read" operation
              return observable[observableLatestValue];
          }
      }
  
      observable[observableLatestValue] = initialValue;
  
      // Inherit from 'subscribable'
      if (!ko.utils.canSetPrototype) {
          // 'subscribable' won't be on the prototype chain unless we put it there directly
          ko.utils.extend(observable, ko.subscribable['fn']);
      }
      ko.subscribable['fn'].init(observable);
  
      // Inherit from 'observable'
      ko.utils.setPrototypeOfOrExtend(observable, observableFn);
  
      if (ko.options['deferUpdates']) {
          ko.extenders['deferred'](observable, true);
      }
  
      return observable;
  }
  
  // Define prototype for observables
  var observableFn = {
      'equalityComparer': valuesArePrimitiveAndEqual,
      peek: function() { return this[observableLatestValue]; },
      valueHasMutated: function () {
          this['notifySubscribers'](this[observableLatestValue], 'spectate');
          this['notifySubscribers'](this[observableLatestValue]);
      },
      valueWillMutate: function () { this['notifySubscribers'](this[observableLatestValue], 'beforeChange'); }
  };
  
  // Note that for browsers that don't support proto assignment, the
  // inheritance chain is created manually in the ko.observable constructor
  if (ko.utils.canSetPrototype) {
      ko.utils.setPrototypeOf(observableFn, ko.subscribable['fn']);
  }
  
  var protoProperty = ko.observable.protoProperty = '__ko_proto__';
  observableFn[protoProperty] = ko.observable;
  
  ko.isObservable = function (instance) {
      var proto = typeof instance == 'function' && instance[protoProperty];
      if (proto && proto !== observableFn[protoProperty] && proto !== ko.computed['fn'][protoProperty]) {
          throw Error("Invalid object that looks like an observable; possibly from another Knockout instance");
      }
      return !!proto;
  };
  
  ko.isWriteableObservable = function (instance) {
      return (typeof instance == 'function' && (
          (instance[protoProperty] === observableFn[protoProperty]) ||  // Observable
          (instance[protoProperty] === ko.computed['fn'][protoProperty] && instance.hasWriteFunction)));   // Writable computed observable
  };
  
  ko.exportSymbol('observable', ko.observable);
  ko.exportSymbol('isObservable', ko.isObservable);
  ko.exportSymbol('isWriteableObservable', ko.isWriteableObservable);
  ko.exportSymbol('isWritableObservable', ko.isWriteableObservable);
  ko.exportSymbol('observable.fn', observableFn);
  ko.exportProperty(observableFn, 'peek', observableFn.peek);
  ko.exportProperty(observableFn, 'valueHasMutated', observableFn.valueHasMutated);
  ko.exportProperty(observableFn, 'valueWillMutate', observableFn.valueWillMutate);
  ko.observableArray = function (initialValues) {
      initialValues = initialValues || [];
  
      if (typeof initialValues != 'object' || !('length' in initialValues))
          throw new Error("The argument passed when initializing an observable array must be an array, or null, or undefined.");
  
      var result = ko.observable(initialValues);
      ko.utils.setPrototypeOfOrExtend(result, ko.observableArray['fn']);
      return result.extend({'trackArrayChanges':true});
  };
  
  ko.observableArray['fn'] = {
      'remove': function (valueOrPredicate) {
          var underlyingArray = this.peek();
          var removedValues = [];
          var predicate = typeof valueOrPredicate == "function" && !ko.isObservable(valueOrPredicate) ? valueOrPredicate : function (value) { return value === valueOrPredicate; };
          for (var i = 0; i < underlyingArray.length; i++) {
              var value = underlyingArray[i];
              if (predicate(value)) {
                  if (removedValues.length === 0) {
                      this.valueWillMutate();
                  }
                  if (underlyingArray[i] !== value) {
                      throw Error("Array modified during remove; cannot remove item");
                  }
                  removedValues.push(value);
                  underlyingArray.splice(i, 1);
                  i--;
              }
          }
          if (removedValues.length) {
              this.valueHasMutated();
          }
          return removedValues;
      },
  
      'removeAll': function (arrayOfValues) {
          // If you passed zero args, we remove everything
          if (arrayOfValues === undefined) {
              var underlyingArray = this.peek();
              var allValues = underlyingArray.slice(0);
              this.valueWillMutate();
              underlyingArray.splice(0, underlyingArray.length);
              this.valueHasMutated();
              return allValues;
          }
          // If you passed an arg, we interpret it as an array of entries to remove
          if (!arrayOfValues)
              return [];
          return this['remove'](function (value) {
              return ko.utils.arrayIndexOf(arrayOfValues, value) >= 0;
          });
      },
  
      'destroy': function (valueOrPredicate) {
          var underlyingArray = this.peek();
          var predicate = typeof valueOrPredicate == "function" && !ko.isObservable(valueOrPredicate) ? valueOrPredicate : function (value) { return value === valueOrPredicate; };
          this.valueWillMutate();
          for (var i = underlyingArray.length - 1; i >= 0; i--) {
              var value = underlyingArray[i];
              if (predicate(value))
                  value["_destroy"] = true;
          }
          this.valueHasMutated();
      },
  
      'destroyAll': function (arrayOfValues) {
          // If you passed zero args, we destroy everything
          if (arrayOfValues === undefined)
              return this['destroy'](function() { return true });
  
          // If you passed an arg, we interpret it as an array of entries to destroy
          if (!arrayOfValues)
              return [];
          return this['destroy'](function (value) {
              return ko.utils.arrayIndexOf(arrayOfValues, value) >= 0;
          });
      },
  
      'indexOf': function (item) {
          var underlyingArray = this();
          return ko.utils.arrayIndexOf(underlyingArray, item);
      },
  
      'replace': function(oldItem, newItem) {
          var index = this['indexOf'](oldItem);
          if (index >= 0) {
              this.valueWillMutate();
              this.peek()[index] = newItem;
              this.valueHasMutated();
          }
      },
  
      'sorted': function (compareFunction) {
          var arrayCopy = this().slice(0);
          return compareFunction ? arrayCopy.sort(compareFunction) : arrayCopy.sort();
      },
  
      'reversed': function () {
          return this().slice(0).reverse();
      }
  };
  
  // Note that for browsers that don't support proto assignment, the
  // inheritance chain is created manually in the ko.observableArray constructor
  if (ko.utils.canSetPrototype) {
      ko.utils.setPrototypeOf(ko.observableArray['fn'], ko.observable['fn']);
  }
  
  // Populate ko.observableArray.fn with read/write functions from native arrays
  // Important: Do not add any additional functions here that may reasonably be used to *read* data from the array
  // because we'll eval them without causing subscriptions, so ko.computed output could end up getting stale
  ko.utils.arrayForEach(["pop", "push", "reverse", "shift", "sort", "splice", "unshift"], function (methodName) {
      ko.observableArray['fn'][methodName] = function () {
          // Use "peek" to avoid creating a subscription in any computed that we're executing in the context of
          // (for consistency with mutating regular observables)
          var underlyingArray = this.peek();
          this.valueWillMutate();
          this.cacheDiffForKnownOperation(underlyingArray, methodName, arguments);
          var methodCallResult = underlyingArray[methodName].apply(underlyingArray, arguments);
          this.valueHasMutated();
          // The native sort and reverse methods return a reference to the array, but it makes more sense to return the observable array instead.
          return methodCallResult === underlyingArray ? this : methodCallResult;
      };
  });
  
  // Populate ko.observableArray.fn with read-only functions from native arrays
  ko.utils.arrayForEach(["slice"], function (methodName) {
      ko.observableArray['fn'][methodName] = function () {
          var underlyingArray = this();
          return underlyingArray[methodName].apply(underlyingArray, arguments);
      };
  });
  
  ko.isObservableArray = function (instance) {
      return ko.isObservable(instance)
          && typeof instance["remove"] == "function"
          && typeof instance["push"] == "function";
  };
  
  ko.exportSymbol('observableArray', ko.observableArray);
  ko.exportSymbol('isObservableArray', ko.isObservableArray);
  var arrayChangeEventName = 'arrayChange';
  ko.extenders['trackArrayChanges'] = function(target, options) {
      // Use the provided options--each call to trackArrayChanges overwrites the previously set options
      target.compareArrayOptions = {};
      if (options && typeof options == "object") {
          ko.utils.extend(target.compareArrayOptions, options);
      }
      target.compareArrayOptions['sparse'] = true;
  
      // Only modify the target observable once
      if (target.cacheDiffForKnownOperation) {
          return;
      }
      var trackingChanges = false,
          cachedDiff = null,
          changeSubscription,
          spectateSubscription,
          pendingChanges = 0,
          previousContents,
          underlyingBeforeSubscriptionAddFunction = target.beforeSubscriptionAdd,
          underlyingAfterSubscriptionRemoveFunction = target.afterSubscriptionRemove;
  
      // Watch "subscribe" calls, and for array change events, ensure change tracking is enabled
      target.beforeSubscriptionAdd = function (event) {
          if (underlyingBeforeSubscriptionAddFunction) {
              underlyingBeforeSubscriptionAddFunction.call(target, event);
          }
          if (event === arrayChangeEventName) {
              trackChanges();
          }
      };
      // Watch "dispose" calls, and for array change events, ensure change tracking is disabled when all are disposed
      target.afterSubscriptionRemove = function (event) {
          if (underlyingAfterSubscriptionRemoveFunction) {
              underlyingAfterSubscriptionRemoveFunction.call(target, event);
          }
          if (event === arrayChangeEventName && !target.hasSubscriptionsForEvent(arrayChangeEventName)) {
              if (changeSubscription) {
                  changeSubscription.dispose();
              }
              if (spectateSubscription) {
                  spectateSubscription.dispose();
              }
              spectateSubscription = changeSubscription = null;
              trackingChanges = false;
              previousContents = undefined;
          }
      };
  
      function trackChanges() {
          if (trackingChanges) {
              // Whenever there's a new subscription and there are pending notifications, make sure all previous
              // subscriptions are notified of the change so that all subscriptions are in sync.
              notifyChanges();
              return;
          }
  
          trackingChanges = true;
  
          // Track how many times the array actually changed value
          spectateSubscription = target.subscribe(function () {
              ++pendingChanges;
          }, null, "spectate");
  
          // Each time the array changes value, capture a clone so that on the next
          // change it's possible to produce a diff
          previousContents = [].concat(target.peek() || []);
          cachedDiff = null;
          changeSubscription = target.subscribe(notifyChanges);
  
          function notifyChanges() {
              if (pendingChanges) {
                  // Make a copy of the current contents and ensure it's an array
                  var currentContents = [].concat(target.peek() || []), changes;
  
                  // Compute the diff and issue notifications, but only if someone is listening
                  if (target.hasSubscriptionsForEvent(arrayChangeEventName)) {
                      changes = getChanges(previousContents, currentContents);
                  }
  
                  // Eliminate references to the old, removed items, so they can be GCed
                  previousContents = currentContents;
                  cachedDiff = null;
                  pendingChanges = 0;
  
                  if (changes && changes.length) {
                      target['notifySubscribers'](changes, arrayChangeEventName);
                  }
              }
          }
      }
  
      function getChanges(previousContents, currentContents) {
          // We try to re-use cached diffs.
          // The scenarios where pendingChanges > 1 are when using rate limiting or deferred updates,
          // which without this check would not be compatible with arrayChange notifications. Normally,
          // notifications are issued immediately so we wouldn't be queueing up more than one.
          if (!cachedDiff || pendingChanges > 1) {
              cachedDiff = ko.utils.compareArrays(previousContents, currentContents, target.compareArrayOptions);
          }
  
          return cachedDiff;
      }
  
      target.cacheDiffForKnownOperation = function(rawArray, operationName, args) {
          // Only run if we're currently tracking changes for this observable array
          // and there aren't any pending deferred notifications.
          if (!trackingChanges || pendingChanges) {
              return;
          }
          var diff = [],
              arrayLength = rawArray.length,
              argsLength = args.length,
              offset = 0;
  
          function pushDiff(status, value, index) {
              return diff[diff.length] = { 'status': status, 'value': value, 'index': index };
          }
          switch (operationName) {
              case 'push':
                  offset = arrayLength;
              case 'unshift':
                  for (var index = 0; index < argsLength; index++) {
                      pushDiff('added', args[index], offset + index);
                  }
                  break;
  
              case 'pop':
                  offset = arrayLength - 1;
              case 'shift':
                  if (arrayLength) {
                      pushDiff('deleted', rawArray[offset], offset);
                  }
                  break;
  
              case 'splice':
                  // Negative start index means 'from end of array'. After that we clamp to [0...arrayLength].
                  // See https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/splice
                  var startIndex = Math.min(Math.max(0, args[0] < 0 ? arrayLength + args[0] : args[0]), arrayLength),
                      endDeleteIndex = argsLength === 1 ? arrayLength : Math.min(startIndex + (args[1] || 0), arrayLength),
                      endAddIndex = startIndex + argsLength - 2,
                      endIndex = Math.max(endDeleteIndex, endAddIndex),
                      additions = [], deletions = [];
                  for (var index = startIndex, argsIndex = 2; index < endIndex; ++index, ++argsIndex) {
                      if (index < endDeleteIndex)
                          deletions.push(pushDiff('deleted', rawArray[index], index));
                      if (index < endAddIndex)
                          additions.push(pushDiff('added', args[argsIndex], index));
                  }
                  ko.utils.findMovesInArrayComparison(deletions, additions);
                  break;
  
              default:
                  return;
          }
          cachedDiff = diff;
      };
  };
  var computedState = ko.utils.createSymbolOrString('_state');
  
  ko.computed = ko.dependentObservable = function (evaluatorFunctionOrOptions, evaluatorFunctionTarget, options) {
      if (typeof evaluatorFunctionOrOptions === "object") {
          // Single-parameter syntax - everything is on this "options" param
          options = evaluatorFunctionOrOptions;
      } else {
          // Multi-parameter syntax - construct the options according to the params passed
          options = options || {};
          if (evaluatorFunctionOrOptions) {
              options["read"] = evaluatorFunctionOrOptions;
          }
      }
      if (typeof options["read"] != "function")
          throw Error("Pass a function that returns the value of the ko.computed");
  
      var writeFunction = options["write"];
      var state = {
          latestValue: undefined,
          isStale: true,
          isDirty: true,
          isBeingEvaluated: false,
          suppressDisposalUntilDisposeWhenReturnsFalse: false,
          isDisposed: false,
          pure: false,
          isSleeping: false,
          readFunction: options["read"],
          evaluatorFunctionTarget: evaluatorFunctionTarget || options["owner"],
          disposeWhenNodeIsRemoved: options["disposeWhenNodeIsRemoved"] || options.disposeWhenNodeIsRemoved || null,
          disposeWhen: options["disposeWhen"] || options.disposeWhen,
          domNodeDisposalCallback: null,
          dependencyTracking: {},
          dependenciesCount: 0,
          evaluationTimeoutInstance: null
      };
  
      function computedObservable() {
          if (arguments.length > 0) {
              if (typeof writeFunction === "function") {
                  // Writing a value
                  writeFunction.apply(state.evaluatorFunctionTarget, arguments);
              } else {
                  throw new Error("Cannot write a value to a ko.computed unless you specify a 'write' option. If you wish to read the current value, don't pass any parameters.");
              }
              return this; // Permits chained assignments
          } else {
              // Reading the value
              if (!state.isDisposed) {
                  ko.dependencyDetection.registerDependency(computedObservable);
              }
              if (state.isDirty || (state.isSleeping && computedObservable.haveDependenciesChanged())) {
                  computedObservable.evaluateImmediate();
              }
              return state.latestValue;
          }
      }
  
      computedObservable[computedState] = state;
      computedObservable.hasWriteFunction = typeof writeFunction === "function";
  
      // Inherit from 'subscribable'
      if (!ko.utils.canSetPrototype) {
          // 'subscribable' won't be on the prototype chain unless we put it there directly
          ko.utils.extend(computedObservable, ko.subscribable['fn']);
      }
      ko.subscribable['fn'].init(computedObservable);
  
      // Inherit from 'computed'
      ko.utils.setPrototypeOfOrExtend(computedObservable, computedFn);
  
      if (options['pure']) {
          state.pure = true;
          state.isSleeping = true;     // Starts off sleeping; will awake on the first subscription
          ko.utils.extend(computedObservable, pureComputedOverrides);
      } else if (options['deferEvaluation']) {
          ko.utils.extend(computedObservable, deferEvaluationOverrides);
      }
  
      if (ko.options['deferUpdates']) {
          ko.extenders['deferred'](computedObservable, true);
      }
  
      if (DEBUG) {
          // #1731 - Aid debugging by exposing the computed's options
          computedObservable["_options"] = options;
      }
  
      if (state.disposeWhenNodeIsRemoved) {
          // Since this computed is associated with a DOM node, and we don't want to dispose the computed
          // until the DOM node is *removed* from the document (as opposed to never having been in the document),
          // we'll prevent disposal until "disposeWhen" first returns false.
          state.suppressDisposalUntilDisposeWhenReturnsFalse = true;
  
          // disposeWhenNodeIsRemoved: true can be used to opt into the "only dispose after first false result"
          // behaviour even if there's no specific node to watch. In that case, clear the option so we don't try
          // to watch for a non-node's disposal. This technique is intended for KO's internal use only and shouldn't
          // be documented or used by application code, as it's likely to change in a future version of KO.
          if (!state.disposeWhenNodeIsRemoved.nodeType) {
              state.disposeWhenNodeIsRemoved = null;
          }
      }
  
      // Evaluate, unless sleeping or deferEvaluation is true
      if (!state.isSleeping && !options['deferEvaluation']) {
          computedObservable.evaluateImmediate();
      }
  
      // Attach a DOM node disposal callback so that the computed will be proactively disposed as soon as the node is
      // removed using ko.removeNode. But skip if isActive is false (there will never be any dependencies to dispose).
      if (state.disposeWhenNodeIsRemoved && computedObservable.isActive()) {
          ko.utils.domNodeDisposal.addDisposeCallback(state.disposeWhenNodeIsRemoved, state.domNodeDisposalCallback = function () {
              computedObservable.dispose();
          });
      }
  
      return computedObservable;
  };
  
  // Utility function that disposes a given dependencyTracking entry
  function computedDisposeDependencyCallback(id, entryToDispose) {
      if (entryToDispose !== null && entryToDispose.dispose) {
          entryToDispose.dispose();
      }
  }
  
  // This function gets called each time a dependency is detected while evaluating a computed.
  // It's factored out as a shared function to avoid creating unnecessary function instances during evaluation.
  function computedBeginDependencyDetectionCallback(subscribable, id) {
      var computedObservable = this.computedObservable,
          state = computedObservable[computedState];
      if (!state.isDisposed) {
          if (this.disposalCount && this.disposalCandidates[id]) {
              // Don't want to dispose this subscription, as it's still being used
              computedObservable.addDependencyTracking(id, subscribable, this.disposalCandidates[id]);
              this.disposalCandidates[id] = null; // No need to actually delete the property - disposalCandidates is a transient object anyway
              --this.disposalCount;
          } else if (!state.dependencyTracking[id]) {
              // Brand new subscription - add it
              computedObservable.addDependencyTracking(id, subscribable, state.isSleeping ? { _target: subscribable } : computedObservable.subscribeToDependency(subscribable));
          }
          // If the observable we've accessed has a pending notification, ensure we get notified of the actual final value (bypass equality checks)
          if (subscribable._notificationIsPending) {
              subscribable._notifyNextChangeIfValueIsDifferent();
          }
      }
  }
  
  var computedFn = {
      "equalityComparer": valuesArePrimitiveAndEqual,
      getDependenciesCount: function () {
          return this[computedState].dependenciesCount;
      },
      getDependencies: function () {
          var dependencyTracking = this[computedState].dependencyTracking, dependentObservables = [];
  
          ko.utils.objectForEach(dependencyTracking, function (id, dependency) {
              dependentObservables[dependency._order] = dependency._target;
          });
  
          return dependentObservables;
      },
      hasAncestorDependency: function (obs) {
          if (!this[computedState].dependenciesCount) {
              return false;
          }
          var dependencies = this.getDependencies();
          if (ko.utils.arrayIndexOf(dependencies, obs) !== -1) {
              return true;
          }
          return !!ko.utils.arrayFirst(dependencies, function (dep) {
              return dep.hasAncestorDependency && dep.hasAncestorDependency(obs);
          });
      },
      addDependencyTracking: function (id, target, trackingObj) {
          if (this[computedState].pure && target === this) {
              throw Error("A 'pure' computed must not be called recursively");
          }
  
          this[computedState].dependencyTracking[id] = trackingObj;
          trackingObj._order = this[computedState].dependenciesCount++;
          trackingObj._version = target.getVersion();
      },
      haveDependenciesChanged: function () {
          var id, dependency, dependencyTracking = this[computedState].dependencyTracking;
          for (id in dependencyTracking) {
              if (Object.prototype.hasOwnProperty.call(dependencyTracking, id)) {
                  dependency = dependencyTracking[id];
                  if ((this._evalDelayed && dependency._target._notificationIsPending) || dependency._target.hasChanged(dependency._version)) {
                      return true;
                  }
              }
          }
      },
      markDirty: function () {
          // Process "dirty" events if we can handle delayed notifications
          if (this._evalDelayed && !this[computedState].isBeingEvaluated) {
              this._evalDelayed(false /*isChange*/);
          }
      },
      isActive: function () {
          var state = this[computedState];
          return state.isDirty || state.dependenciesCount > 0;
      },
      respondToChange: function () {
          // Ignore "change" events if we've already scheduled a delayed notification
          if (!this._notificationIsPending) {
              this.evaluatePossiblyAsync();
          } else if (this[computedState].isDirty) {
              this[computedState].isStale = true;
          }
      },
      subscribeToDependency: function (target) {
          if (target._deferUpdates) {
              var dirtySub = target.subscribe(this.markDirty, this, 'dirty'),
                  changeSub = target.subscribe(this.respondToChange, this);
              return {
                  _target: target,
                  dispose: function () {
                      dirtySub.dispose();
                      changeSub.dispose();
                  }
              };
          } else {
              return target.subscribe(this.evaluatePossiblyAsync, this);
          }
      },
      evaluatePossiblyAsync: function () {
          var computedObservable = this,
              throttleEvaluationTimeout = computedObservable['throttleEvaluation'];
          if (throttleEvaluationTimeout && throttleEvaluationTimeout >= 0) {
              clearTimeout(this[computedState].evaluationTimeoutInstance);
              this[computedState].evaluationTimeoutInstance = ko.utils.setTimeout(function () {
                  computedObservable.evaluateImmediate(true /*notifyChange*/);
              }, throttleEvaluationTimeout);
          } else if (computedObservable._evalDelayed) {
              computedObservable._evalDelayed(true /*isChange*/);
          } else {
              computedObservable.evaluateImmediate(true /*notifyChange*/);
          }
      },
      evaluateImmediate: function (notifyChange) {
          var computedObservable = this,
              state = computedObservable[computedState],
              disposeWhen = state.disposeWhen,
              changed = false;
  
          if (state.isBeingEvaluated) {
              // If the evaluation of a ko.computed causes side effects, it's possible that it will trigger its own re-evaluation.
              // This is not desirable (it's hard for a developer to realise a chain of dependencies might cause this, and they almost
              // certainly didn't intend infinite re-evaluations). So, for predictability, we simply prevent ko.computeds from causing
              // their own re-evaluation. Further discussion at https://github.com/SteveSanderson/knockout/pull/387
              return;
          }
  
          // Do not evaluate (and possibly capture new dependencies) if disposed
          if (state.isDisposed) {
              return;
          }
  
          if (state.disposeWhenNodeIsRemoved && !ko.utils.domNodeIsAttachedToDocument(state.disposeWhenNodeIsRemoved) || disposeWhen && disposeWhen()) {
              // See comment above about suppressDisposalUntilDisposeWhenReturnsFalse
              if (!state.suppressDisposalUntilDisposeWhenReturnsFalse) {
                  computedObservable.dispose();
                  return;
              }
          } else {
              // It just did return false, so we can stop suppressing now
              state.suppressDisposalUntilDisposeWhenReturnsFalse = false;
          }
  
          state.isBeingEvaluated = true;
          try {
              changed = this.evaluateImmediate_CallReadWithDependencyDetection(notifyChange);
          } finally {
              state.isBeingEvaluated = false;
          }
  
          return changed;
      },
      evaluateImmediate_CallReadWithDependencyDetection: function (notifyChange) {
          // This function is really just part of the evaluateImmediate logic. You would never call it from anywhere else.
          // Factoring it out into a separate function means it can be independent of the try/catch block in evaluateImmediate,
          // which contributes to saving about 40% off the CPU overhead of computed evaluation (on V8 at least).
  
          var computedObservable = this,
              state = computedObservable[computedState],
              changed = false;
  
          // Initially, we assume that none of the subscriptions are still being used (i.e., all are candidates for disposal).
          // Then, during evaluation, we cross off any that are in fact still being used.
          var isInitial = state.pure ? undefined : !state.dependenciesCount,   // If we're evaluating when there are no previous dependencies, it must be the first time
              dependencyDetectionContext = {
                  computedObservable: computedObservable,
                  disposalCandidates: state.dependencyTracking,
                  disposalCount: state.dependenciesCount
              };
  
          ko.dependencyDetection.begin({
              callbackTarget: dependencyDetectionContext,
              callback: computedBeginDependencyDetectionCallback,
              computed: computedObservable,
              isInitial: isInitial
          });
  
          state.dependencyTracking = {};
          state.dependenciesCount = 0;
  
          var newValue = this.evaluateImmediate_CallReadThenEndDependencyDetection(state, dependencyDetectionContext);
  
          if (!state.dependenciesCount) {
              computedObservable.dispose();
              changed = true; // When evaluation causes a disposal, make sure all dependent computeds get notified so they'll see the new state
          } else {
              changed = computedObservable.isDifferent(state.latestValue, newValue);
          }
  
          if (changed) {
              if (!state.isSleeping) {
                  computedObservable["notifySubscribers"](state.latestValue, "beforeChange");
              } else {
                  computedObservable.updateVersion();
              }
  
              state.latestValue = newValue;
              if (DEBUG) computedObservable._latestValue = newValue;
  
              computedObservable["notifySubscribers"](state.latestValue, "spectate");
  
              if (!state.isSleeping && notifyChange) {
                  computedObservable["notifySubscribers"](state.latestValue);
              }
              if (computedObservable._recordUpdate) {
                  computedObservable._recordUpdate();
              }
          }
  
          if (isInitial) {
              computedObservable["notifySubscribers"](state.latestValue, "awake");
          }
  
          return changed;
      },
      evaluateImmediate_CallReadThenEndDependencyDetection: function (state, dependencyDetectionContext) {
          // This function is really part of the evaluateImmediate_CallReadWithDependencyDetection logic.
          // You'd never call it from anywhere else. Factoring it out means that evaluateImmediate_CallReadWithDependencyDetection
          // can be independent of try/finally blocks, which contributes to saving about 40% off the CPU
          // overhead of computed evaluation (on V8 at least).
  
          try {
              var readFunction = state.readFunction;
              return state.evaluatorFunctionTarget ? readFunction.call(state.evaluatorFunctionTarget) : readFunction();
          } finally {
              ko.dependencyDetection.end();
  
              // For each subscription no longer being used, remove it from the active subscriptions list and dispose it
              if (dependencyDetectionContext.disposalCount && !state.isSleeping) {
                  ko.utils.objectForEach(dependencyDetectionContext.disposalCandidates, computedDisposeDependencyCallback);
              }
  
              state.isStale = state.isDirty = false;
          }
      },
      peek: function (evaluate) {
          // By default, peek won't re-evaluate, except while the computed is sleeping or to get the initial value when "deferEvaluation" is set.
          // Pass in true to evaluate if needed.
          var state = this[computedState];
          if ((state.isDirty && (evaluate || !state.dependenciesCount)) || (state.isSleeping && this.haveDependenciesChanged())) {
              this.evaluateImmediate();
          }
          return state.latestValue;
      },
      limit: function (limitFunction) {
          // Override the limit function with one that delays evaluation as well
          ko.subscribable['fn'].limit.call(this, limitFunction);
          this._evalIfChanged = function () {
              if (!this[computedState].isSleeping) {
                  if (this[computedState].isStale) {
                      this.evaluateImmediate();
                  } else {
                      this[computedState].isDirty = false;
                  }
              }
              return this[computedState].latestValue;
          };
          this._evalDelayed = function (isChange) {
              this._limitBeforeChange(this[computedState].latestValue);
  
              // Mark as dirty
              this[computedState].isDirty = true;
              if (isChange) {
                  this[computedState].isStale = true;
              }
  
              // Pass the observable to the "limit" code, which will evaluate it when
              // it's time to do the notification.
              this._limitChange(this, !isChange /* isDirty */);
          };
      },
      dispose: function () {
          var state = this[computedState];
          if (!state.isSleeping && state.dependencyTracking) {
              ko.utils.objectForEach(state.dependencyTracking, function (id, dependency) {
                  if (dependency.dispose)
                      dependency.dispose();
              });
          }
          if (state.disposeWhenNodeIsRemoved && state.domNodeDisposalCallback) {
              ko.utils.domNodeDisposal.removeDisposeCallback(state.disposeWhenNodeIsRemoved, state.domNodeDisposalCallback);
          }
          state.dependencyTracking = undefined;
          state.dependenciesCount = 0;
          state.isDisposed = true;
          state.isStale = false;
          state.isDirty = false;
          state.isSleeping = false;
          state.disposeWhenNodeIsRemoved = undefined;
          state.disposeWhen = undefined;
          state.readFunction = undefined;
          if (!this.hasWriteFunction) {
              state.evaluatorFunctionTarget = undefined;
          }
      }
  };
  
  var pureComputedOverrides = {
      beforeSubscriptionAdd: function (event) {
          // If asleep, wake up the computed by subscribing to any dependencies.
          var computedObservable = this,
              state = computedObservable[computedState];
          if (!state.isDisposed && state.isSleeping && event == 'change') {
              state.isSleeping = false;
              if (state.isStale || computedObservable.haveDependenciesChanged()) {
                  state.dependencyTracking = null;
                  state.dependenciesCount = 0;
                  if (computedObservable.evaluateImmediate()) {
                      computedObservable.updateVersion();
                  }
              } else {
                  // First put the dependencies in order
                  var dependenciesOrder = [];
                  ko.utils.objectForEach(state.dependencyTracking, function (id, dependency) {
                      dependenciesOrder[dependency._order] = id;
                  });
                  // Next, subscribe to each one
                  ko.utils.arrayForEach(dependenciesOrder, function (id, order) {
                      var dependency = state.dependencyTracking[id],
                          subscription = computedObservable.subscribeToDependency(dependency._target);
                      subscription._order = order;
                      subscription._version = dependency._version;
                      state.dependencyTracking[id] = subscription;
                  });
                  // Waking dependencies may have triggered effects
                  if (computedObservable.haveDependenciesChanged()) {
                      if (computedObservable.evaluateImmediate()) {
                          computedObservable.updateVersion();
                      }
                  }
              }
  
              if (!state.isDisposed) {     // test since evaluating could trigger disposal
                  computedObservable["notifySubscribers"](state.latestValue, "awake");
              }
          }
      },
      afterSubscriptionRemove: function (event) {
          var state = this[computedState];
          if (!state.isDisposed && event == 'change' && !this.hasSubscriptionsForEvent('change')) {
              ko.utils.objectForEach(state.dependencyTracking, function (id, dependency) {
                  if (dependency.dispose) {
                      state.dependencyTracking[id] = {
                          _target: dependency._target,
                          _order: dependency._order,
                          _version: dependency._version
                      };
                      dependency.dispose();
                  }
              });
              state.isSleeping = true;
              this["notifySubscribers"](undefined, "asleep");
          }
      },
      getVersion: function () {
          // Because a pure computed is not automatically updated while it is sleeping, we can't
          // simply return the version number. Instead, we check if any of the dependencies have
          // changed and conditionally re-evaluate the computed observable.
          var state = this[computedState];
          if (state.isSleeping && (state.isStale || this.haveDependenciesChanged())) {
              this.evaluateImmediate();
          }
          return ko.subscribable['fn'].getVersion.call(this);
      }
  };
  
  var deferEvaluationOverrides = {
      beforeSubscriptionAdd: function (event) {
          // This will force a computed with deferEvaluation to evaluate when the first subscription is registered.
          if (event == 'change' || event == 'beforeChange') {
              this.peek();
          }
      }
  };
  
  // Note that for browsers that don't support proto assignment, the
  // inheritance chain is created manually in the ko.computed constructor
  if (ko.utils.canSetPrototype) {
      ko.utils.setPrototypeOf(computedFn, ko.subscribable['fn']);
  }
  
  // Set the proto values for ko.computed
  var protoProp = ko.observable.protoProperty; // == "__ko_proto__"
  computedFn[protoProp] = ko.computed;
  
  ko.isComputed = function (instance) {
      return (typeof instance == 'function' && instance[protoProp] === computedFn[protoProp]);
  };
  
  ko.isPureComputed = function (instance) {
      return ko.isComputed(instance) && instance[computedState] && instance[computedState].pure;
  };
  
  ko.exportSymbol('computed', ko.computed);
  ko.exportSymbol('dependentObservable', ko.computed);    // export ko.dependentObservable for backwards compatibility (1.x)
  ko.exportSymbol('isComputed', ko.isComputed);
  ko.exportSymbol('isPureComputed', ko.isPureComputed);
  ko.exportSymbol('computed.fn', computedFn);
  ko.exportProperty(computedFn, 'peek', computedFn.peek);
  ko.exportProperty(computedFn, 'dispose', computedFn.dispose);
  ko.exportProperty(computedFn, 'isActive', computedFn.isActive);
  ko.exportProperty(computedFn, 'getDependenciesCount', computedFn.getDependenciesCount);
  ko.exportProperty(computedFn, 'getDependencies', computedFn.getDependencies);
  
  ko.pureComputed = function (evaluatorFunctionOrOptions, evaluatorFunctionTarget) {
      if (typeof evaluatorFunctionOrOptions === 'function') {
          return ko.computed(evaluatorFunctionOrOptions, evaluatorFunctionTarget, {'pure':true});
      } else {
          evaluatorFunctionOrOptions = ko.utils.extend({}, evaluatorFunctionOrOptions);   // make a copy of the parameter object
          evaluatorFunctionOrOptions['pure'] = true;
          return ko.computed(evaluatorFunctionOrOptions, evaluatorFunctionTarget);
      }
  }
  ko.exportSymbol('pureComputed', ko.pureComputed);
  
  (function() {
      var maxNestedObservableDepth = 10; // Escape the (unlikely) pathological case where an observable's current value is itself (or similar reference cycle)
  
      ko.toJS = function(rootObject) {
          if (arguments.length == 0)
              throw new Error("When calling ko.toJS, pass the object you want to convert.");
  
          // We just unwrap everything at every level in the object graph
          return mapJsObjectGraph(rootObject, function(valueToMap) {
              // Loop because an observable's value might in turn be another observable wrapper
              for (var i = 0; ko.isObservable(valueToMap) && (i < maxNestedObservableDepth); i++)
                  valueToMap = valueToMap();
              return valueToMap;
          });
      };
  
      ko.toJSON = function(rootObject, replacer, space) {     // replacer and space are optional
          var plainJavaScriptObject = ko.toJS(rootObject);
          return ko.utils.stringifyJson(plainJavaScriptObject, replacer, space);
      };
  
      function mapJsObjectGraph(rootObject, mapInputCallback, visitedObjects) {
          visitedObjects = visitedObjects || new objectLookup();
  
          rootObject = mapInputCallback(rootObject);
          var canHaveProperties = (typeof rootObject == "object") && (rootObject !== null) && (rootObject !== undefined) && (!(rootObject instanceof RegExp)) && (!(rootObject instanceof Date)) && (!(rootObject instanceof String)) && (!(rootObject instanceof Number)) && (!(rootObject instanceof Boolean));
          if (!canHaveProperties)
              return rootObject;
  
          var outputProperties = rootObject instanceof Array ? [] : {};
          visitedObjects.save(rootObject, outputProperties);
  
          visitPropertiesOrArrayEntries(rootObject, function(indexer) {
              var propertyValue = mapInputCallback(rootObject[indexer]);
  
              switch (typeof propertyValue) {
                  case "boolean":
                  case "number":
                  case "string":
                  case "function":
                      outputProperties[indexer] = propertyValue;
                      break;
                  case "object":
                  case "undefined":
                      var previouslyMappedValue = visitedObjects.get(propertyValue);
                      outputProperties[indexer] = (previouslyMappedValue !== undefined)
                          ? previouslyMappedValue
                          : mapJsObjectGraph(propertyValue, mapInputCallback, visitedObjects);
                      break;
              }
          });
  
          return outputProperties;
      }
  
      function visitPropertiesOrArrayEntries(rootObject, visitorCallback) {
          if (rootObject instanceof Array) {
              for (var i = 0; i < rootObject.length; i++)
                  visitorCallback(i);
  
              // For arrays, also respect toJSON property for custom mappings (fixes #278)
              if (typeof rootObject['toJSON'] == 'function')
                  visitorCallback('toJSON');
          } else {
              for (var propertyName in rootObject) {
                  visitorCallback(propertyName);
              }
          }
      };
  
      function objectLookup() {
          this.keys = [];
          this.values = [];
      };
  
      objectLookup.prototype = {
          constructor: objectLookup,
          save: function(key, value) {
              var existingIndex = ko.utils.arrayIndexOf(this.keys, key);
              if (existingIndex >= 0)
                  this.values[existingIndex] = value;
              else {
                  this.keys.push(key);
                  this.values.push(value);
              }
          },
          get: function(key) {
              var existingIndex = ko.utils.arrayIndexOf(this.keys, key);
              return (existingIndex >= 0) ? this.values[existingIndex] : undefined;
          }
      };
  })();
  
  ko.exportSymbol('toJS', ko.toJS);
  ko.exportSymbol('toJSON', ko.toJSON);
  ko.when = function(predicate, callback, context) {
      function kowhen (resolve) {
          var observable = ko.pureComputed(predicate, context).extend({notify:'always'});
          var subscription = observable.subscribe(function(value) {
              if (value) {
                  subscription.dispose();
                  resolve(value);
              }
          });
          // In case the initial value is true, process it right away
          observable['notifySubscribers'](observable.peek());
  
          return subscription;
      }
      if (typeof Promise === "function" && !callback) {
          return new Promise(kowhen);
      } else {
          return kowhen(callback.bind(context));
      }
  };
  
  ko.exportSymbol('when', ko.when);
  (function () {
      var hasDomDataExpandoProperty = '__ko__hasDomDataOptionValue__';
  
      // Normally, SELECT elements and their OPTIONs can only take value of type 'string' (because the values
      // are stored on DOM attributes). ko.selectExtensions provides a way for SELECTs/OPTIONs to have values
      // that are arbitrary objects. This is very convenient when implementing things like cascading dropdowns.
      ko.selectExtensions = {
          readValue : function(element) {
              switch (ko.utils.tagNameLower(element)) {
                  case 'option':
                      if (element[hasDomDataExpandoProperty] === true)
                          return ko.utils.domData.get(element, ko.bindingHandlers.options.optionValueDomDataKey);
                      return ko.utils.ieVersion <= 7
                          ? (element.getAttributeNode('value') && element.getAttributeNode('value').specified ? element.value : element.text)
                          : element.value;
                  case 'select':
                      return element.selectedIndex >= 0 ? ko.selectExtensions.readValue(element.options[element.selectedIndex]) : undefined;
                  default:
                      return element.value;
              }
          },
  
          writeValue: function(element, value, allowUnset) {
              switch (ko.utils.tagNameLower(element)) {
                  case 'option':
                      if (typeof value === "string") {
                          ko.utils.domData.set(element, ko.bindingHandlers.options.optionValueDomDataKey, undefined);
                          if (hasDomDataExpandoProperty in element) { // IE <= 8 throws errors if you delete non-existent properties from a DOM node
                              delete element[hasDomDataExpandoProperty];
                          }
                          element.value = value;
                      }
                      else {
                          // Store arbitrary object using DomData
                          ko.utils.domData.set(element, ko.bindingHandlers.options.optionValueDomDataKey, value);
                          element[hasDomDataExpandoProperty] = true;
  
                          // Special treatment of numbers is just for backward compatibility. KO 1.2.1 wrote numerical values to element.value.
                          element.value = typeof value === "number" ? value : "";
                      }
                      break;
                  case 'select':
                      if (value === "" || value === null)       // A blank string or null value will select the caption
                          value = undefined;
                      var selection = -1;
                      for (var i = 0, n = element.options.length, optionValue; i < n; ++i) {
                          optionValue = ko.selectExtensions.readValue(element.options[i]);
                          // Include special check to handle selecting a caption with a blank string value
                          if (optionValue == value || (optionValue === "" && value === undefined)) {
                              selection = i;
                              break;
                          }
                      }
                      if (allowUnset || selection >= 0 || (value === undefined && element.size > 1)) {
                          element.selectedIndex = selection;
                          if (ko.utils.ieVersion === 6) {
                              // Workaround for IE6 bug: It won't reliably apply values to SELECT nodes during the same execution thread
                              // right after you've changed the set of OPTION nodes on it. So for that node type, we'll schedule a second thread
                              // to apply the value as well.
                              ko.utils.setTimeout(function () {
                                  element.selectedIndex = selection;
                              }, 0);
                          }
                      }
                      break;
                  default:
                      if ((value === null) || (value === undefined))
                          value = "";
                      element.value = value;
                      break;
              }
          }
      };
  })();
  
  ko.exportSymbol('selectExtensions', ko.selectExtensions);
  ko.exportSymbol('selectExtensions.readValue', ko.selectExtensions.readValue);
  ko.exportSymbol('selectExtensions.writeValue', ko.selectExtensions.writeValue);
  ko.expressionRewriting = (function () {
      var javaScriptReservedWords = ["true", "false", "null", "undefined"];
  
      // Matches something that can be assigned to--either an isolated identifier or something ending with a property accessor
      // This is designed to be simple and avoid false negatives, but could produce false positives (e.g., a+b.c).
      // This also will not properly handle nested brackets (e.g., obj1[obj2['prop']]; see #911).
      var javaScriptAssignmentTarget = /^(?:[$_a-z][$\w]*|(.+)(\.\s*[$_a-z][$\w]*|\[.+\]))$/i;
  
      function getWriteableValue(expression) {
          if (ko.utils.arrayIndexOf(javaScriptReservedWords, expression) >= 0)
              return false;
          var match = expression.match(javaScriptAssignmentTarget);
          return match === null ? false : match[1] ? ('Object(' + match[1] + ')' + match[2]) : expression;
      }
  
      // The following regular expressions will be used to split an object-literal string into tokens
  
      var specials = ',"\'`{}()/:[\\]',    // These characters have special meaning to the parser and must not appear in the middle of a token, except as part of a string.
          // Create the actual regular expression by or-ing the following regex strings. The order is important.
          bindingToken = RegExp([
              // These match strings, either with double quotes, single quotes, or backticks
              '"(?:\\\\.|[^"])*"',
              "'(?:\\\\.|[^'])*'",
              "`(?:\\\\.|[^`])*`",
              // Match C style comments
              "/\\*(?:[^*]|\\*+[^*/])*\\*+/",
              // Match C++ style comments
              "//.*\n",
              // Match a regular expression (text enclosed by slashes), but will also match sets of divisions
              // as a regular expression (this is handled by the parsing loop below).
              '/(?:\\\\.|[^/])+/\w*',
              // Match text (at least two characters) that does not contain any of the above special characters,
              // although some of the special characters are allowed to start it (all but the colon and comma).
              // The text can contain spaces, but leading or trailing spaces are skipped.
              '[^\\s:,/][^' + specials + ']*[^\\s' + specials + ']',
              // Match any non-space character not matched already. This will match colons and commas, since they're
              // not matched by "everyThingElse", but will also match any other single character that wasn't already
              // matched (for example: in "a: 1, b: 2", each of the non-space characters will be matched by oneNotSpace).
              '[^\\s]'
          ].join('|'), 'g'),
  
          // Match end of previous token to determine whether a slash is a division or regex.
          divisionLookBehind = /[\])"'A-Za-z0-9_$]+$/,
          keywordRegexLookBehind = {'in':1,'return':1,'typeof':1};
  
      function parseObjectLiteral(objectLiteralString) {
          // Trim leading and trailing spaces from the string
          var str = ko.utils.stringTrim(objectLiteralString);
  
          // Trim braces '{' surrounding the whole object literal
          if (str.charCodeAt(0) === 123) str = str.slice(1, -1);
  
          // Add a newline to correctly match a C++ style comment at the end of the string and
          // add a comma so that we don't need a separate code block to deal with the last item
          str += "\n,";
  
          // Split into tokens
          var result = [], toks = str.match(bindingToken), key, values = [], depth = 0;
  
          if (toks.length > 1) {
              for (var i = 0, tok; tok = toks[i]; ++i) {
                  var c = tok.charCodeAt(0);
                  // A comma signals the end of a key/value pair if depth is zero
                  if (c === 44) { // ","
                      if (depth <= 0) {
                          result.push((key && values.length) ? {key: key, value: values.join('')} : {'unknown': key || values.join('')});
                          key = depth = 0;
                          values = [];
                          continue;
                      }
                  // Simply skip the colon that separates the name and value
                  } else if (c === 58) { // ":"
                      if (!depth && !key && values.length === 1) {
                          key = values.pop();
                          continue;
                      }
                  // Comments: skip them
                  } else if (c === 47 && tok.length > 1 && (tok.charCodeAt(1) === 47 || tok.charCodeAt(1) === 42)) {  // "//" or "/*"
                      continue;
                  // A set of slashes is initially matched as a regular expression, but could be division
                  } else if (c === 47 && i && tok.length > 1) {  // "/"
                      // Look at the end of the previous token to determine if the slash is actually division
                      var match = toks[i-1].match(divisionLookBehind);
                      if (match && !keywordRegexLookBehind[match[0]]) {
                          // The slash is actually a division punctuator; re-parse the remainder of the string (not including the slash)
                          str = str.substr(str.indexOf(tok) + 1);
                          toks = str.match(bindingToken);
                          i = -1;
                          // Continue with just the slash
                          tok = '/';
                      }
                  // Increment depth for parentheses, braces, and brackets so that interior commas are ignored
                  } else if (c === 40 || c === 123 || c === 91) { // '(', '{', '['
                      ++depth;
                  } else if (c === 41 || c === 125 || c === 93) { // ')', '}', ']'
                      --depth;
                  // The key will be the first token; if it's a string, trim the quotes
                  } else if (!key && !values.length && (c === 34 || c === 39)) { // '"', "'"
                      tok = tok.slice(1, -1);
                  }
                  values.push(tok);
              }
              if (depth > 0) {
                  throw Error("Unbalanced parentheses, braces, or brackets");
              }
          }
          return result;
      }
  
      // Two-way bindings include a write function that allow the handler to update the value even if it's not an observable.
      var twoWayBindings = {};
  
      function preProcessBindings(bindingsStringOrKeyValueArray, bindingOptions) {
          bindingOptions = bindingOptions || {};
  
          function processKeyValue(key, val) {
              var writableVal;
              function callPreprocessHook(obj) {
                  return (obj && obj['preprocess']) ? (val = obj['preprocess'](val, key, processKeyValue)) : true;
              }
              if (!bindingParams) {
                  if (!callPreprocessHook(ko['getBindingHandler'](key)))
                      return;
  
                  if (twoWayBindings[key] && (writableVal = getWriteableValue(val))) {
                      // For two-way bindings, provide a write method in case the value
                      // isn't a writable observable.
                      var writeKey = typeof twoWayBindings[key] == 'string' ? twoWayBindings[key] : key;
                      propertyAccessorResultStrings.push("'" + writeKey + "':function(_z){" + writableVal + "=_z}");
                  }
              }
              // Values are wrapped in a function so that each value can be accessed independently
              if (makeValueAccessors) {
                  val = 'function(){return ' + val + ' }';
              }
              resultStrings.push("'" + key + "':" + val);
          }
  
          var resultStrings = [],
              propertyAccessorResultStrings = [],
              makeValueAccessors = bindingOptions['valueAccessors'],
              bindingParams = bindingOptions['bindingParams'],
              keyValueArray = typeof bindingsStringOrKeyValueArray === "string" ?
                  parseObjectLiteral(bindingsStringOrKeyValueArray) : bindingsStringOrKeyValueArray;
  
          ko.utils.arrayForEach(keyValueArray, function(keyValue) {
              processKeyValue(keyValue.key || keyValue['unknown'], keyValue.value);
          });
  
          if (propertyAccessorResultStrings.length)
              processKeyValue('_ko_property_writers', "{" + propertyAccessorResultStrings.join(",") + " }");
  
          return resultStrings.join(",");
      }
  
      return {
          bindingRewriteValidators: [],
  
          twoWayBindings: twoWayBindings,
  
          parseObjectLiteral: parseObjectLiteral,
  
          preProcessBindings: preProcessBindings,
  
          keyValueArrayContainsKey: function(keyValueArray, key) {
              for (var i = 0; i < keyValueArray.length; i++)
                  if (keyValueArray[i]['key'] == key)
                      return true;
              return false;
          },
  
          // Internal, private KO utility for updating model properties from within bindings
          // property:            If the property being updated is (or might be) an observable, pass it here
          //                      If it turns out to be a writable observable, it will be written to directly
          // allBindings:         An object with a get method to retrieve bindings in the current execution context.
          //                      This will be searched for a '_ko_property_writers' property in case you're writing to a non-observable
          // key:                 The key identifying the property to be written. Example: for { hasFocus: myValue }, write to 'myValue' by specifying the key 'hasFocus'
          // value:               The value to be written
          // checkIfDifferent:    If true, and if the property being written is a writable observable, the value will only be written if
          //                      it is !== existing value on that writable observable
          writeValueToProperty: function(property, allBindings, key, value, checkIfDifferent) {
              if (!property || !ko.isObservable(property)) {
                  var propWriters = allBindings.get('_ko_property_writers');
                  if (propWriters && propWriters[key])
                      propWriters[key](value);
              } else if (ko.isWriteableObservable(property) && (!checkIfDifferent || property.peek() !== value)) {
                  property(value);
              }
          }
      };
  })();
  
  ko.exportSymbol('expressionRewriting', ko.expressionRewriting);
  ko.exportSymbol('expressionRewriting.bindingRewriteValidators', ko.expressionRewriting.bindingRewriteValidators);
  ko.exportSymbol('expressionRewriting.parseObjectLiteral', ko.expressionRewriting.parseObjectLiteral);
  ko.exportSymbol('expressionRewriting.preProcessBindings', ko.expressionRewriting.preProcessBindings);
  
  // Making bindings explicitly declare themselves as "two way" isn't ideal in the long term (it would be better if
  // all bindings could use an official 'property writer' API without needing to declare that they might). However,
  // since this is not, and has never been, a public API (_ko_property_writers was never documented), it's acceptable
  // as an internal implementation detail in the short term.
  // For those developers who rely on _ko_property_writers in their custom bindings, we expose _twoWayBindings as an
  // undocumented feature that makes it relatively easy to upgrade to KO 3.0. However, this is still not an official
  // public API, and we reserve the right to remove it at any time if we create a real public property writers API.
  ko.exportSymbol('expressionRewriting._twoWayBindings', ko.expressionRewriting.twoWayBindings);
  
  // For backward compatibility, define the following aliases. (Previously, these function names were misleading because
  // they referred to JSON specifically, even though they actually work with arbitrary JavaScript object literal expressions.)
  ko.exportSymbol('jsonExpressionRewriting', ko.expressionRewriting);
  ko.exportSymbol('jsonExpressionRewriting.insertPropertyAccessorsIntoJson', ko.expressionRewriting.preProcessBindings);
  (function() {
      // "Virtual elements" is an abstraction on top of the usual DOM API which understands the notion that comment nodes
      // may be used to represent hierarchy (in addition to the DOM's natural hierarchy).
      // If you call the DOM-manipulating functions on ko.virtualElements, you will be able to read and write the state
      // of that virtual hierarchy
      //
      // The point of all this is to support containerless templates (e.g., <!-- ko foreach:someCollection -->blah<!-- /ko -->)
      // without having to scatter special cases all over the binding and templating code.
  
      // IE 9 cannot reliably read the "nodeValue" property of a comment node (see https://github.com/SteveSanderson/knockout/issues/186)
      // but it does give them a nonstandard alternative property called "text" that it can read reliably. Other browsers don't have that property.
      // So, use node.text where available, and node.nodeValue elsewhere
      var commentNodesHaveTextProperty = document && document.createComment("test").text === "<!--test-->";
  
      var startCommentRegex = commentNodesHaveTextProperty ? /^<!--\s*ko(?:\s+([\s\S]+))?\s*-->$/ : /^\s*ko(?:\s+([\s\S]+))?\s*$/;
      var endCommentRegex =   commentNodesHaveTextProperty ? /^<!--\s*\/ko\s*-->$/ : /^\s*\/ko\s*$/;
      var htmlTagsWithOptionallyClosingChildren = { 'ul': true, 'ol': true };
  
      function isStartComment(node) {
          return (node.nodeType == 8) && startCommentRegex.test(commentNodesHaveTextProperty ? node.text : node.nodeValue);
      }
  
      function isEndComment(node) {
          return (node.nodeType == 8) && endCommentRegex.test(commentNodesHaveTextProperty ? node.text : node.nodeValue);
      }
  
      function isUnmatchedEndComment(node) {
          return isEndComment(node) && !(ko.utils.domData.get(node, matchedEndCommentDataKey));
      }
  
      var matchedEndCommentDataKey = "__ko_matchedEndComment__"
  
      function getVirtualChildren(startComment, allowUnbalanced) {
          var currentNode = startComment;
          var depth = 1;
          var children = [];
          while (currentNode = currentNode.nextSibling) {
              if (isEndComment(currentNode)) {
                  ko.utils.domData.set(currentNode, matchedEndCommentDataKey, true);
                  depth--;
                  if (depth === 0)
                      return children;
              }
  
              children.push(currentNode);
  
              if (isStartComment(currentNode))
                  depth++;
          }
          if (!allowUnbalanced)
              throw new Error("Cannot find closing comment tag to match: " + startComment.nodeValue);
          return null;
      }
  
      function getMatchingEndComment(startComment, allowUnbalanced) {
          var allVirtualChildren = getVirtualChildren(startComment, allowUnbalanced);
          if (allVirtualChildren) {
              if (allVirtualChildren.length > 0)
                  return allVirtualChildren[allVirtualChildren.length - 1].nextSibling;
              return startComment.nextSibling;
          } else
              return null; // Must have no matching end comment, and allowUnbalanced is true
      }
  
      function getUnbalancedChildTags(node) {
          // e.g., from <div>OK</div><!-- ko blah --><span>Another</span>, returns: <!-- ko blah --><span>Another</span>
          //       from <div>OK</div><!-- /ko --><!-- /ko -->,             returns: <!-- /ko --><!-- /ko -->
          var childNode = node.firstChild, captureRemaining = null;
          if (childNode) {
              do {
                  if (captureRemaining)                   // We already hit an unbalanced node and are now just scooping up all subsequent nodes
                      captureRemaining.push(childNode);
                  else if (isStartComment(childNode)) {
                      var matchingEndComment = getMatchingEndComment(childNode, /* allowUnbalanced: */ true);
                      if (matchingEndComment)             // It's a balanced tag, so skip immediately to the end of this virtual set
                          childNode = matchingEndComment;
                      else
                          captureRemaining = [childNode]; // It's unbalanced, so start capturing from this point
                  } else if (isEndComment(childNode)) {
                      captureRemaining = [childNode];     // It's unbalanced (if it wasn't, we'd have skipped over it already), so start capturing
                  }
              } while (childNode = childNode.nextSibling);
          }
          return captureRemaining;
      }
  
      ko.virtualElements = {
          allowedBindings: {},
  
          childNodes: function(node) {
              return isStartComment(node) ? getVirtualChildren(node) : node.childNodes;
          },
  
          emptyNode: function(node) {
              if (!isStartComment(node))
                  ko.utils.emptyDomNode(node);
              else {
                  var virtualChildren = ko.virtualElements.childNodes(node);
                  for (var i = 0, j = virtualChildren.length; i < j; i++)
                      ko.removeNode(virtualChildren[i]);
              }
          },
  
          setDomNodeChildren: function(node, childNodes) {
              if (!isStartComment(node))
                  ko.utils.setDomNodeChildren(node, childNodes);
              else {
                  ko.virtualElements.emptyNode(node);
                  var endCommentNode = node.nextSibling; // Must be the next sibling, as we just emptied the children
                  for (var i = 0, j = childNodes.length; i < j; i++)
                      endCommentNode.parentNode.insertBefore(childNodes[i], endCommentNode);
              }
          },
  
          prepend: function(containerNode, nodeToPrepend) {
              var insertBeforeNode;
  
              if (isStartComment(containerNode)) {
                  // Start comments must always have a parent and at least one following sibling (the end comment)
                  insertBeforeNode = containerNode.nextSibling;
                  containerNode = containerNode.parentNode;
              } else {
                  insertBeforeNode = containerNode.firstChild;
              }
  
              if (!insertBeforeNode) {
                  containerNode.appendChild(nodeToPrepend);
              } else if (nodeToPrepend !== insertBeforeNode) {       // IE will sometimes crash if you try to insert a node before itself
                  containerNode.insertBefore(nodeToPrepend, insertBeforeNode);
              }
          },
  
          insertAfter: function(containerNode, nodeToInsert, insertAfterNode) {
              if (!insertAfterNode) {
                  ko.virtualElements.prepend(containerNode, nodeToInsert);
              } else {
                  // Children of start comments must always have a parent and at least one following sibling (the end comment)
                  var insertBeforeNode = insertAfterNode.nextSibling;
  
                  if (isStartComment(containerNode)) {
                      containerNode = containerNode.parentNode;
                  }
  
                  if (!insertBeforeNode) {
                      containerNode.appendChild(nodeToInsert);
                  } else if (nodeToInsert !== insertBeforeNode) {       // IE will sometimes crash if you try to insert a node before itself
                      containerNode.insertBefore(nodeToInsert, insertBeforeNode);
                  }
              }
          },
  
          firstChild: function(node) {
              if (!isStartComment(node)) {
                  if (node.firstChild && isEndComment(node.firstChild)) {
                      throw new Error("Found invalid end comment, as the first child of " + node);
                  }
                  return node.firstChild;
              } else if (!node.nextSibling || isEndComment(node.nextSibling)) {
                  return null;
              } else {
                  return node.nextSibling;
              }
          },
  
          nextSibling: function(node) {
              if (isStartComment(node)) {
                  node = getMatchingEndComment(node);
              }
  
              if (node.nextSibling && isEndComment(node.nextSibling)) {
                  if (isUnmatchedEndComment(node.nextSibling)) {
                      throw Error("Found end comment without a matching opening comment, as child of " + node);
                  } else {
                      return null;
                  }
              } else {
                  return node.nextSibling;
              }
          },
  
          hasBindingValue: isStartComment,
  
          virtualNodeBindingValue: function(node) {
              var regexMatch = (commentNodesHaveTextProperty ? node.text : node.nodeValue).match(startCommentRegex);
              return regexMatch ? regexMatch[1] : null;
          },
  
          normaliseVirtualElementDomStructure: function(elementVerified) {
              // Workaround for https://github.com/SteveSanderson/knockout/issues/155
              // (IE <= 8 or IE 9 quirks mode parses your HTML weirdly, treating closing </li> tags as if they don't exist, thereby moving comment nodes
              // that are direct descendants of <ul> into the preceding <li>)
              if (!htmlTagsWithOptionallyClosingChildren[ko.utils.tagNameLower(elementVerified)])
                  return;
  
              // Scan immediate children to see if they contain unbalanced comment tags. If they do, those comment tags
              // must be intended to appear *after* that child, so move them there.
              var childNode = elementVerified.firstChild;
              if (childNode) {
                  do {
                      if (childNode.nodeType === 1) {
                          var unbalancedTags = getUnbalancedChildTags(childNode);
                          if (unbalancedTags) {
                              // Fix up the DOM by moving the unbalanced tags to where they most likely were intended to be placed - *after* the child
                              var nodeToInsertBefore = childNode.nextSibling;
                              for (var i = 0; i < unbalancedTags.length; i++) {
                                  if (nodeToInsertBefore)
                                      elementVerified.insertBefore(unbalancedTags[i], nodeToInsertBefore);
                                  else
                                      elementVerified.appendChild(unbalancedTags[i]);
                              }
                          }
                      }
                  } while (childNode = childNode.nextSibling);
              }
          }
      };
  })();
  ko.exportSymbol('virtualElements', ko.virtualElements);
  ko.exportSymbol('virtualElements.allowedBindings', ko.virtualElements.allowedBindings);
  ko.exportSymbol('virtualElements.emptyNode', ko.virtualElements.emptyNode);
  //ko.exportSymbol('virtualElements.firstChild', ko.virtualElements.firstChild);     // firstChild is not minified
  ko.exportSymbol('virtualElements.insertAfter', ko.virtualElements.insertAfter);
  //ko.exportSymbol('virtualElements.nextSibling', ko.virtualElements.nextSibling);   // nextSibling is not minified
  ko.exportSymbol('virtualElements.prepend', ko.virtualElements.prepend);
  ko.exportSymbol('virtualElements.setDomNodeChildren', ko.virtualElements.setDomNodeChildren);
  (function() {
      var defaultBindingAttributeName = "data-bind";
  
      ko.bindingProvider = function() {
          this.bindingCache = {};
      };
  
      ko.utils.extend(ko.bindingProvider.prototype, {
          'nodeHasBindings': function(node) {
              switch (node.nodeType) {
                  case 1: // Element
                      return node.getAttribute(defaultBindingAttributeName) != null
                          || ko.components['getComponentNameForNode'](node);
                  case 8: // Comment node
                      return ko.virtualElements.hasBindingValue(node);
                  default: return false;
              }
          },
  
          'getBindings': function(node, bindingContext) {
              var bindingsString = this['getBindingsString'](node, bindingContext),
                  parsedBindings = bindingsString ? this['parseBindingsString'](bindingsString, bindingContext, node) : null;
              return ko.components.addBindingsForCustomElement(parsedBindings, node, bindingContext, /* valueAccessors */ false);
          },
  
          'getBindingAccessors': function(node, bindingContext) {
              var bindingsString = this['getBindingsString'](node, bindingContext),
                  parsedBindings = bindingsString ? this['parseBindingsString'](bindingsString, bindingContext, node, { 'valueAccessors': true }) : null;
              return ko.components.addBindingsForCustomElement(parsedBindings, node, bindingContext, /* valueAccessors */ true);
          },
  
          // The following function is only used internally by this default provider.
          // It's not part of the interface definition for a general binding provider.
          'getBindingsString': function(node, bindingContext) {
              switch (node.nodeType) {
                  case 1: return node.getAttribute(defaultBindingAttributeName);   // Element
                  case 8: return ko.virtualElements.virtualNodeBindingValue(node); // Comment node
                  default: return null;
              }
          },
  
          // The following function is only used internally by this default provider.
          // It's not part of the interface definition for a general binding provider.
          'parseBindingsString': function(bindingsString, bindingContext, node, options) {
              try {
                  var bindingFunction = createBindingsStringEvaluatorViaCache(bindingsString, this.bindingCache, options);
                  return bindingFunction(bindingContext, node);
              } catch (ex) {
                  ex.message = "Unable to parse bindings.\nBindings value: " + bindingsString + "\nMessage: " + ex.message;
                  throw ex;
              }
          }
      });
  
      ko.bindingProvider['instance'] = new ko.bindingProvider();
  
      function createBindingsStringEvaluatorViaCache(bindingsString, cache, options) {
          var cacheKey = bindingsString + (options && options['valueAccessors'] || '');
          return cache[cacheKey]
              || (cache[cacheKey] = createBindingsStringEvaluator(bindingsString, options));
      }
  
      function createBindingsStringEvaluator(bindingsString, options) {
          // Build the source for a function that evaluates "expression"
          // For each scope variable, add an extra level of "with" nesting
          // Example result: with(sc1) { with(sc0) { return (expression) } }
          var rewrittenBindings = ko.expressionRewriting.preProcessBindings(bindingsString, options),
              functionBody = "with($context){with($data||{}){return{" + rewrittenBindings + "}}}";
          return new Function("$context", "$element", functionBody);
      }
  })();
  
  ko.exportSymbol('bindingProvider', ko.bindingProvider);
  (function () {
      // Hide or don't minify context properties, see https://github.com/knockout/knockout/issues/2294
      var contextSubscribable = ko.utils.createSymbolOrString('_subscribable');
      var contextAncestorBindingInfo = ko.utils.createSymbolOrString('_ancestorBindingInfo');
      var contextDataDependency = ko.utils.createSymbolOrString('_dataDependency');
  
      ko.bindingHandlers = {};
  
      // The following element types will not be recursed into during binding.
      var bindingDoesNotRecurseIntoElementTypes = {
          // Don't want bindings that operate on text nodes to mutate <script> and <textarea> contents,
          // because it's unexpected and a potential XSS issue.
          // Also bindings should not operate on <template> elements since this breaks in Internet Explorer
          // and because such elements' contents are always intended to be bound in a different context
          // from where they appear in the document.
          'script': true,
          'textarea': true,
          'template': true
      };
  
      // Use an overridable method for retrieving binding handlers so that plugins may support dynamically created handlers
      ko['getBindingHandler'] = function(bindingKey) {
          return ko.bindingHandlers[bindingKey];
      };
  
      var inheritParentVm = {};
  
      // The ko.bindingContext constructor is only called directly to create the root context. For child
      // contexts, use bindingContext.createChildContext or bindingContext.extend.
      ko.bindingContext = function(dataItemOrAccessor, parentContext, dataItemAlias, extendCallback, options) {
  
          // The binding context object includes static properties for the current, parent, and root view models.
          // If a view model is actually stored in an observable, the corresponding binding context object, and
          // any child contexts, must be updated when the view model is changed.
          function updateContext() {
              // Most of the time, the context will directly get a view model object, but if a function is given,
              // we call the function to retrieve the view model. If the function accesses any observables or returns
              // an observable, the dependency is tracked, and those observables can later cause the binding
              // context to be updated.
              var dataItemOrObservable = isFunc ? realDataItemOrAccessor() : realDataItemOrAccessor,
                  dataItem = ko.utils.unwrapObservable(dataItemOrObservable);
  
              if (parentContext) {
                  // Copy $root and any custom properties from the parent context
                  ko.utils.extend(self, parentContext);
  
                  // Copy Symbol properties
                  if (contextAncestorBindingInfo in parentContext) {
                      self[contextAncestorBindingInfo] = parentContext[contextAncestorBindingInfo];
                  }
              } else {
                  self['$parents'] = [];
                  self['$root'] = dataItem;
  
                  // Export 'ko' in the binding context so it will be available in bindings and templates
                  // even if 'ko' isn't exported as a global, such as when using an AMD loader.
                  // See https://github.com/SteveSanderson/knockout/issues/490
                  self['ko'] = ko;
              }
  
              self[contextSubscribable] = subscribable;
  
              if (shouldInheritData) {
                  dataItem = self['$data'];
              } else {
                  self['$rawData'] = dataItemOrObservable;
                  self['$data'] = dataItem;
              }
  
              if (dataItemAlias)
                  self[dataItemAlias] = dataItem;
  
              // The extendCallback function is provided when creating a child context or extending a context.
              // It handles the specific actions needed to finish setting up the binding context. Actions in this
              // function could also add dependencies to this binding context.
              if (extendCallback)
                  extendCallback(self, parentContext, dataItem);
  
              // When a "parent" context is given and we don't already have a dependency on its context, register a dependency on it.
              // Thus whenever the parent context is updated, this context will also be updated.
              if (parentContext && parentContext[contextSubscribable] && !ko.computedContext.computed().hasAncestorDependency(parentContext[contextSubscribable])) {
                  parentContext[contextSubscribable]();
              }
  
              if (dataDependency) {
                  self[contextDataDependency] = dataDependency;
              }
  
              return self['$data'];
          }
  
          var self = this,
              shouldInheritData = dataItemOrAccessor === inheritParentVm,
              realDataItemOrAccessor = shouldInheritData ? undefined : dataItemOrAccessor,
              isFunc = typeof(realDataItemOrAccessor) == "function" && !ko.isObservable(realDataItemOrAccessor),
              nodes,
              subscribable,
              dataDependency = options && options['dataDependency'];
  
          if (options && options['exportDependencies']) {
              // The "exportDependencies" option means that the calling code will track any dependencies and re-create
              // the binding context when they change.
              updateContext();
          } else {
              subscribable = ko.pureComputed(updateContext);
              subscribable.peek();
  
              // At this point, the binding context has been initialized, and the "subscribable" computed observable is
              // subscribed to any observables that were accessed in the process. If there is nothing to track, the
              // computed will be inactive, and we can safely throw it away. If it's active, the computed is stored in
              // the context object.
              if (subscribable.isActive()) {
                  // Always notify because even if the model ($data) hasn't changed, other context properties might have changed
                  subscribable['equalityComparer'] = null;
              } else {
                  self[contextSubscribable] = undefined;
              }
          }
      }
  
      // Extend the binding context hierarchy with a new view model object. If the parent context is watching
      // any observables, the new child context will automatically get a dependency on the parent context.
      // But this does not mean that the $data value of the child context will also get updated. If the child
      // view model also depends on the parent view model, you must provide a function that returns the correct
      // view model on each update.
      ko.bindingContext.prototype['createChildContext'] = function (dataItemOrAccessor, dataItemAlias, extendCallback, options) {
          if (!options && dataItemAlias && typeof dataItemAlias == "object") {
              options = dataItemAlias;
              dataItemAlias = options['as'];
              extendCallback = options['extend'];
          }
  
          if (dataItemAlias && options && options['noChildContext']) {
              var isFunc = typeof(dataItemOrAccessor) == "function" && !ko.isObservable(dataItemOrAccessor);
              return new ko.bindingContext(inheritParentVm, this, null, function (self) {
                  if (extendCallback)
                      extendCallback(self);
                  self[dataItemAlias] = isFunc ? dataItemOrAccessor() : dataItemOrAccessor;
              }, options);
          }
  
          return new ko.bindingContext(dataItemOrAccessor, this, dataItemAlias, function (self, parentContext) {
              // Extend the context hierarchy by setting the appropriate pointers
              self['$parentContext'] = parentContext;
              self['$parent'] = parentContext['$data'];
              self['$parents'] = (parentContext['$parents'] || []).slice(0);
              self['$parents'].unshift(self['$parent']);
              if (extendCallback)
                  extendCallback(self);
          }, options);
      };
  
      // Extend the binding context with new custom properties. This doesn't change the context hierarchy.
      // Similarly to "child" contexts, provide a function here to make sure that the correct values are set
      // when an observable view model is updated.
      ko.bindingContext.prototype['extend'] = function(properties, options) {
          return new ko.bindingContext(inheritParentVm, this, null, function(self, parentContext) {
              ko.utils.extend(self, typeof(properties) == "function" ? properties(self) : properties);
          }, options);
      };
  
      var boundElementDomDataKey = ko.utils.domData.nextKey();
  
      function asyncContextDispose(node) {
          var bindingInfo = ko.utils.domData.get(node, boundElementDomDataKey),
              asyncContext = bindingInfo && bindingInfo.asyncContext;
          if (asyncContext) {
              bindingInfo.asyncContext = null;
              asyncContext.notifyAncestor();
          }
      }
      function AsyncCompleteContext(node, bindingInfo, ancestorBindingInfo) {
          this.node = node;
          this.bindingInfo = bindingInfo;
          this.asyncDescendants = [];
          this.childrenComplete = false;
  
          if (!bindingInfo.asyncContext) {
              ko.utils.domNodeDisposal.addDisposeCallback(node, asyncContextDispose);
          }
  
          if (ancestorBindingInfo && ancestorBindingInfo.asyncContext) {
              ancestorBindingInfo.asyncContext.asyncDescendants.push(node);
              this.ancestorBindingInfo = ancestorBindingInfo;
          }
      }
      AsyncCompleteContext.prototype.notifyAncestor = function () {
          if (this.ancestorBindingInfo && this.ancestorBindingInfo.asyncContext) {
              this.ancestorBindingInfo.asyncContext.descendantComplete(this.node);
          }
      };
      AsyncCompleteContext.prototype.descendantComplete = function (node) {
          ko.utils.arrayRemoveItem(this.asyncDescendants, node);
          if (!this.asyncDescendants.length && this.childrenComplete) {
              this.completeChildren();
          }
      };
      AsyncCompleteContext.prototype.completeChildren = function () {
          this.childrenComplete = true;
          if (this.bindingInfo.asyncContext && !this.asyncDescendants.length) {
              this.bindingInfo.asyncContext = null;
              ko.utils.domNodeDisposal.removeDisposeCallback(this.node, asyncContextDispose);
              ko.bindingEvent.notify(this.node, ko.bindingEvent.descendantsComplete);
              this.notifyAncestor();
          }
      };
  
      ko.bindingEvent = {
          childrenComplete: "childrenComplete",
          descendantsComplete : "descendantsComplete",
  
          subscribe: function (node, event, callback, context, options) {
              var bindingInfo = ko.utils.domData.getOrSet(node, boundElementDomDataKey, {});
              if (!bindingInfo.eventSubscribable) {
                  bindingInfo.eventSubscribable = new ko.subscribable;
              }
              if (options && options['notifyImmediately'] && bindingInfo.notifiedEvents[event]) {
                  ko.dependencyDetection.ignore(callback, context, [node]);
              }
              return bindingInfo.eventSubscribable.subscribe(callback, context, event);
          },
  
          notify: function (node, event) {
              var bindingInfo = ko.utils.domData.get(node, boundElementDomDataKey);
              if (bindingInfo) {
                  bindingInfo.notifiedEvents[event] = true;
                  if (bindingInfo.eventSubscribable) {
                      bindingInfo.eventSubscribable['notifySubscribers'](node, event);
                  }
                  if (event == ko.bindingEvent.childrenComplete) {
                      if (bindingInfo.asyncContext) {
                          bindingInfo.asyncContext.completeChildren();
                      } else if (bindingInfo.asyncContext === undefined && bindingInfo.eventSubscribable && bindingInfo.eventSubscribable.hasSubscriptionsForEvent(ko.bindingEvent.descendantsComplete)) {
                          // It's currently an error to register a descendantsComplete handler for a node that was never registered as completing asynchronously.
                          // That's because without the asyncContext, we don't have a way to know that all descendants have completed.
                          throw new Error("descendantsComplete event not supported for bindings on this node");
                      }
                  }
              }
          },
  
          startPossiblyAsyncContentBinding: function (node, bindingContext) {
              var bindingInfo = ko.utils.domData.getOrSet(node, boundElementDomDataKey, {});
  
              if (!bindingInfo.asyncContext) {
                  bindingInfo.asyncContext = new AsyncCompleteContext(node, bindingInfo, bindingContext[contextAncestorBindingInfo]);
              }
  
              // If the provided context was already extended with this node's binding info, just return the extended context
              if (bindingContext[contextAncestorBindingInfo] == bindingInfo) {
                  return bindingContext;
              }
  
              return bindingContext['extend'](function (ctx) {
                  ctx[contextAncestorBindingInfo] = bindingInfo;
              });
          }
      };
  
      // Returns the valueAccessor function for a binding value
      function makeValueAccessor(value) {
          return function() {
              return value;
          };
      }
  
      // Returns the value of a valueAccessor function
      function evaluateValueAccessor(valueAccessor) {
          return valueAccessor();
      }
  
      // Given a function that returns bindings, create and return a new object that contains
      // binding value-accessors functions. Each accessor function calls the original function
      // so that it always gets the latest value and all dependencies are captured. This is used
      // by ko.applyBindingsToNode and getBindingsAndMakeAccessors.
      function makeAccessorsFromFunction(callback) {
          return ko.utils.objectMap(ko.dependencyDetection.ignore(callback), function(value, key) {
              return function() {
                  return callback()[key];
              };
          });
      }
  
      // Given a bindings function or object, create and return a new object that contains
      // binding value-accessors functions. This is used by ko.applyBindingsToNode.
      function makeBindingAccessors(bindings, context, node) {
          if (typeof bindings === 'function') {
              return makeAccessorsFromFunction(bindings.bind(null, context, node));
          } else {
              return ko.utils.objectMap(bindings, makeValueAccessor);
          }
      }
  
      // This function is used if the binding provider doesn't include a getBindingAccessors function.
      // It must be called with 'this' set to the provider instance.
      function getBindingsAndMakeAccessors(node, context) {
          return makeAccessorsFromFunction(this['getBindings'].bind(this, node, context));
      }
  
      function validateThatBindingIsAllowedForVirtualElements(bindingName) {
          var validator = ko.virtualElements.allowedBindings[bindingName];
          if (!validator)
              throw new Error("The binding '" + bindingName + "' cannot be used with virtual elements")
      }
  
      function applyBindingsToDescendantsInternal(bindingContext, elementOrVirtualElement) {
          var nextInQueue = ko.virtualElements.firstChild(elementOrVirtualElement);
  
          if (nextInQueue) {
              var currentChild,
                  provider = ko.bindingProvider['instance'],
                  preprocessNode = provider['preprocessNode'];
  
              // Preprocessing allows a binding provider to mutate a node before bindings are applied to it. For example it's
              // possible to insert new siblings after it, and/or replace the node with a different one. This can be used to
              // implement custom binding syntaxes, such as {{ value }} for string interpolation, or custom element types that
              // trigger insertion of <template> contents at that point in the document.
              if (preprocessNode) {
                  while (currentChild = nextInQueue) {
                      nextInQueue = ko.virtualElements.nextSibling(currentChild);
                      preprocessNode.call(provider, currentChild);
                  }
                  // Reset nextInQueue for the next loop
                  nextInQueue = ko.virtualElements.firstChild(elementOrVirtualElement);
              }
  
              while (currentChild = nextInQueue) {
                  // Keep a record of the next child *before* applying bindings, in case the binding removes the current child from its position
                  nextInQueue = ko.virtualElements.nextSibling(currentChild);
                  applyBindingsToNodeAndDescendantsInternal(bindingContext, currentChild);
              }
          }
          ko.bindingEvent.notify(elementOrVirtualElement, ko.bindingEvent.childrenComplete);
      }
  
      function applyBindingsToNodeAndDescendantsInternal(bindingContext, nodeVerified) {
          var bindingContextForDescendants = bindingContext;
  
          var isElement = (nodeVerified.nodeType === 1);
          if (isElement) // Workaround IE <= 8 HTML parsing weirdness
              ko.virtualElements.normaliseVirtualElementDomStructure(nodeVerified);
  
          // Perf optimisation: Apply bindings only if...
          // (1) We need to store the binding info for the node (all element nodes)
          // (2) It might have bindings (e.g., it has a data-bind attribute, or it's a marker for a containerless template)
          var shouldApplyBindings = isElement || ko.bindingProvider['instance']['nodeHasBindings'](nodeVerified);
          if (shouldApplyBindings)
              bindingContextForDescendants = applyBindingsToNodeInternal(nodeVerified, null, bindingContext)['bindingContextForDescendants'];
  
          if (bindingContextForDescendants && !bindingDoesNotRecurseIntoElementTypes[ko.utils.tagNameLower(nodeVerified)]) {
              applyBindingsToDescendantsInternal(bindingContextForDescendants, nodeVerified);
          }
      }
  
      function topologicalSortBindings(bindings) {
          // Depth-first sort
          var result = [],                // The list of key/handler pairs that we will return
              bindingsConsidered = {},    // A temporary record of which bindings are already in 'result'
              cyclicDependencyStack = []; // Keeps track of a depth-search so that, if there's a cycle, we know which bindings caused it
          ko.utils.objectForEach(bindings, function pushBinding(bindingKey) {
              if (!bindingsConsidered[bindingKey]) {
                  var binding = ko['getBindingHandler'](bindingKey);
                  if (binding) {
                      // First add dependencies (if any) of the current binding
                      if (binding['after']) {
                          cyclicDependencyStack.push(bindingKey);
                          ko.utils.arrayForEach(binding['after'], function(bindingDependencyKey) {
                              if (bindings[bindingDependencyKey]) {
                                  if (ko.utils.arrayIndexOf(cyclicDependencyStack, bindingDependencyKey) !== -1) {
                                      throw Error("Cannot combine the following bindings, because they have a cyclic dependency: " + cyclicDependencyStack.join(", "));
                                  } else {
                                      pushBinding(bindingDependencyKey);
                                  }
                              }
                          });
                          cyclicDependencyStack.length--;
                      }
                      // Next add the current binding
                      result.push({ key: bindingKey, handler: binding });
                  }
                  bindingsConsidered[bindingKey] = true;
              }
          });
  
          return result;
      }
  
      function applyBindingsToNodeInternal(node, sourceBindings, bindingContext) {
          var bindingInfo = ko.utils.domData.getOrSet(node, boundElementDomDataKey, {});
  
          // Prevent multiple applyBindings calls for the same node, except when a binding value is specified
          var alreadyBound = bindingInfo.alreadyBound;
          if (!sourceBindings) {
              if (alreadyBound) {
                  throw Error("You cannot apply bindings multiple times to the same element.");
              }
              bindingInfo.alreadyBound = true;
          }
          if (!alreadyBound) {
              bindingInfo.context = bindingContext;
          }
          if (!bindingInfo.notifiedEvents) {
              bindingInfo.notifiedEvents = {};
          }
  
          // Use bindings if given, otherwise fall back on asking the bindings provider to give us some bindings
          var bindings;
          if (sourceBindings && typeof sourceBindings !== 'function') {
              bindings = sourceBindings;
          } else {
              var provider = ko.bindingProvider['instance'],
                  getBindings = provider['getBindingAccessors'] || getBindingsAndMakeAccessors;
  
              // Get the binding from the provider within a computed observable so that we can update the bindings whenever
              // the binding context is updated or if the binding provider accesses observables.
              var bindingsUpdater = ko.dependentObservable(
                  function() {
                      bindings = sourceBindings ? sourceBindings(bindingContext, node) : getBindings.call(provider, node, bindingContext);
                      // Register a dependency on the binding context to support observable view models.
                      if (bindings) {
                          if (bindingContext[contextSubscribable]) {
                              bindingContext[contextSubscribable]();
                          }
                          if (bindingContext[contextDataDependency]) {
                              bindingContext[contextDataDependency]();
                          }
                      }
                      return bindings;
                  },
                  null, { disposeWhenNodeIsRemoved: node }
              );
  
              if (!bindings || !bindingsUpdater.isActive())
                  bindingsUpdater = null;
          }
  
          var contextToExtend = bindingContext;
          var bindingHandlerThatControlsDescendantBindings;
          if (bindings) {
              // Return the value accessor for a given binding. When bindings are static (won't be updated because of a binding
              // context update), just return the value accessor from the binding. Otherwise, return a function that always gets
              // the latest binding value and registers a dependency on the binding updater.
              var getValueAccessor = bindingsUpdater
                  ? function(bindingKey) {
                      return function() {
                          return evaluateValueAccessor(bindingsUpdater()[bindingKey]);
                      };
                  } : function(bindingKey) {
                      return bindings[bindingKey];
                  };
  
              // Use of allBindings as a function is maintained for backwards compatibility, but its use is deprecated
              function allBindings() {
                  return ko.utils.objectMap(bindingsUpdater ? bindingsUpdater() : bindings, evaluateValueAccessor);
              }
              // The following is the 3.x allBindings API
              allBindings['get'] = function(key) {
                  return bindings[key] && evaluateValueAccessor(getValueAccessor(key));
              };
              allBindings['has'] = function(key) {
                  return key in bindings;
              };
  
              if (ko.bindingEvent.childrenComplete in bindings) {
                  ko.bindingEvent.subscribe(node, ko.bindingEvent.childrenComplete, function () {
                      var callback = evaluateValueAccessor(bindings[ko.bindingEvent.childrenComplete]);
                      if (callback) {
                          var nodes = ko.virtualElements.childNodes(node);
                          if (nodes.length) {
                              callback(nodes, ko.dataFor(nodes[0]));
                          }
                      }
                  });
              }
  
              if (ko.bindingEvent.descendantsComplete in bindings) {
                  contextToExtend = ko.bindingEvent.startPossiblyAsyncContentBinding(node, bindingContext);
                  ko.bindingEvent.subscribe(node, ko.bindingEvent.descendantsComplete, function () {
                      var callback = evaluateValueAccessor(bindings[ko.bindingEvent.descendantsComplete]);
                      if (callback && ko.virtualElements.firstChild(node)) {
                          callback(node);
                      }
                  });
              }
  
              // First put the bindings into the right order
              var orderedBindings = topologicalSortBindings(bindings);
  
              // Go through the sorted bindings, calling init and update for each
              ko.utils.arrayForEach(orderedBindings, function(bindingKeyAndHandler) {
                  // Note that topologicalSortBindings has already filtered out any nonexistent binding handlers,
                  // so bindingKeyAndHandler.handler will always be nonnull.
                  var handlerInitFn = bindingKeyAndHandler.handler["init"],
                      handlerUpdateFn = bindingKeyAndHandler.handler["update"],
                      bindingKey = bindingKeyAndHandler.key;
  
                  if (node.nodeType === 8) {
                      validateThatBindingIsAllowedForVirtualElements(bindingKey);
                  }
  
                  try {
                      // Run init, ignoring any dependencies
                      if (typeof handlerInitFn == "function") {
                          ko.dependencyDetection.ignore(function() {
                              var initResult = handlerInitFn(node, getValueAccessor(bindingKey), allBindings, contextToExtend['$data'], contextToExtend);
  
                              // If this binding handler claims to control descendant bindings, make a note of this
                              if (initResult && initResult['controlsDescendantBindings']) {
                                  if (bindingHandlerThatControlsDescendantBindings !== undefined)
                                      throw new Error("Multiple bindings (" + bindingHandlerThatControlsDescendantBindings + " and " + bindingKey + ") are trying to control descendant bindings of the same element. You cannot use these bindings together on the same element.");
                                  bindingHandlerThatControlsDescendantBindings = bindingKey;
                              }
                          });
                      }
  
                      // Run update in its own computed wrapper
                      if (typeof handlerUpdateFn == "function") {
                          ko.dependentObservable(
                              function() {
                                  handlerUpdateFn(node, getValueAccessor(bindingKey), allBindings, contextToExtend['$data'], contextToExtend);
                              },
                              null,
                              { disposeWhenNodeIsRemoved: node }
                          );
                      }
                  } catch (ex) {
                      ex.message = "Unable to process binding \"" + bindingKey + ": " + bindings[bindingKey] + "\"\nMessage: " + ex.message;
                      throw ex;
                  }
              });
          }
  
          var shouldBindDescendants = bindingHandlerThatControlsDescendantBindings === undefined;
          return {
              'shouldBindDescendants': shouldBindDescendants,
              'bindingContextForDescendants': shouldBindDescendants && contextToExtend
          };
      };
  
      ko.storedBindingContextForNode = function (node) {
          var bindingInfo = ko.utils.domData.get(node, boundElementDomDataKey);
          return bindingInfo && bindingInfo.context;
      }
  
      function getBindingContext(viewModelOrBindingContext, extendContextCallback) {
          return viewModelOrBindingContext && (viewModelOrBindingContext instanceof ko.bindingContext)
              ? viewModelOrBindingContext
              : new ko.bindingContext(viewModelOrBindingContext, undefined, undefined, extendContextCallback);
      }
  
      ko.applyBindingAccessorsToNode = function (node, bindings, viewModelOrBindingContext) {
          if (node.nodeType === 1) // If it's an element, workaround IE <= 8 HTML parsing weirdness
              ko.virtualElements.normaliseVirtualElementDomStructure(node);
          return applyBindingsToNodeInternal(node, bindings, getBindingContext(viewModelOrBindingContext));
      };
  
      ko.applyBindingsToNode = function (node, bindings, viewModelOrBindingContext) {
          var context = getBindingContext(viewModelOrBindingContext);
          return ko.applyBindingAccessorsToNode(node, makeBindingAccessors(bindings, context, node), context);
      };
  
      ko.applyBindingsToDescendants = function(viewModelOrBindingContext, rootNode) {
          if (rootNode.nodeType === 1 || rootNode.nodeType === 8)
              applyBindingsToDescendantsInternal(getBindingContext(viewModelOrBindingContext), rootNode);
      };
  
      ko.applyBindings = function (viewModelOrBindingContext, rootNode, extendContextCallback) {
          // If jQuery is loaded after Knockout, we won't initially have access to it. So save it here.
          if (!jQueryInstance && window['jQuery']) {
              jQueryInstance = window['jQuery'];
          }
  
          if (arguments.length < 2) {
              rootNode = document.body;
              if (!rootNode) {
                  throw Error("ko.applyBindings: could not find document.body; has the document been loaded?");
              }
          } else if (!rootNode || (rootNode.nodeType !== 1 && rootNode.nodeType !== 8)) {
              throw Error("ko.applyBindings: first parameter should be your view model; second parameter should be a DOM node");
          }
  
          applyBindingsToNodeAndDescendantsInternal(getBindingContext(viewModelOrBindingContext, extendContextCallback), rootNode);
      };
  
      // Retrieving binding context from arbitrary nodes
      ko.contextFor = function(node) {
          // We can only do something meaningful for elements and comment nodes (in particular, not text nodes, as IE can't store domdata for them)
          if (node && (node.nodeType === 1 || node.nodeType === 8)) {
              return ko.storedBindingContextForNode(node);
          }
          return undefined;
      };
      ko.dataFor = function(node) {
          var context = ko.contextFor(node);
          return context ? context['$data'] : undefined;
      };
  
      ko.exportSymbol('bindingHandlers', ko.bindingHandlers);
      ko.exportSymbol('bindingEvent', ko.bindingEvent);
      ko.exportSymbol('bindingEvent.subscribe', ko.bindingEvent.subscribe);
      ko.exportSymbol('bindingEvent.startPossiblyAsyncContentBinding', ko.bindingEvent.startPossiblyAsyncContentBinding);
      ko.exportSymbol('applyBindings', ko.applyBindings);
      ko.exportSymbol('applyBindingsToDescendants', ko.applyBindingsToDescendants);
      ko.exportSymbol('applyBindingAccessorsToNode', ko.applyBindingAccessorsToNode);
      ko.exportSymbol('applyBindingsToNode', ko.applyBindingsToNode);
      ko.exportSymbol('contextFor', ko.contextFor);
      ko.exportSymbol('dataFor', ko.dataFor);
  })();
  (function(undefined) {
      var loadingSubscribablesCache = {}, // Tracks component loads that are currently in flight
          loadedDefinitionsCache = {};    // Tracks component loads that have already completed
  
      ko.components = {
          get: function(componentName, callback) {
              var cachedDefinition = getObjectOwnProperty(loadedDefinitionsCache, componentName);
              if (cachedDefinition) {
                  // It's already loaded and cached. Reuse the same definition object.
                  // Note that for API consistency, even cache hits complete asynchronously by default.
                  // You can bypass this by putting synchronous:true on your component config.
                  if (cachedDefinition.isSynchronousComponent) {
                      ko.dependencyDetection.ignore(function() { // See comment in loaderRegistryBehaviors.js for reasoning
                          callback(cachedDefinition.definition);
                      });
                  } else {
                      ko.tasks.schedule(function() { callback(cachedDefinition.definition); });
                  }
              } else {
                  // Join the loading process that is already underway, or start a new one.
                  loadComponentAndNotify(componentName, callback);
              }
          },
  
          clearCachedDefinition: function(componentName) {
              delete loadedDefinitionsCache[componentName];
          },
  
          _getFirstResultFromLoaders: getFirstResultFromLoaders
      };
  
      function getObjectOwnProperty(obj, propName) {
          return Object.prototype.hasOwnProperty.call(obj, propName) ? obj[propName] : undefined;
      }
  
      function loadComponentAndNotify(componentName, callback) {
          var subscribable = getObjectOwnProperty(loadingSubscribablesCache, componentName),
              completedAsync;
          if (!subscribable) {
              // It's not started loading yet. Start loading, and when it's done, move it to loadedDefinitionsCache.
              subscribable = loadingSubscribablesCache[componentName] = new ko.subscribable();
              subscribable.subscribe(callback);
  
              beginLoadingComponent(componentName, function(definition, config) {
                  var isSynchronousComponent = !!(config && config['synchronous']);
                  loadedDefinitionsCache[componentName] = { definition: definition, isSynchronousComponent: isSynchronousComponent };
                  delete loadingSubscribablesCache[componentName];
  
                  // For API consistency, all loads complete asynchronously. However we want to avoid
                  // adding an extra task schedule if it's unnecessary (i.e., the completion is already
                  // async).
                  //
                  // You can bypass the 'always asynchronous' feature by putting the synchronous:true
                  // flag on your component configuration when you register it.
                  if (completedAsync || isSynchronousComponent) {
                      // Note that notifySubscribers ignores any dependencies read within the callback.
                      // See comment in loaderRegistryBehaviors.js for reasoning
                      subscribable['notifySubscribers'](definition);
                  } else {
                      ko.tasks.schedule(function() {
                          subscribable['notifySubscribers'](definition);
                      });
                  }
              });
              completedAsync = true;
          } else {
              subscribable.subscribe(callback);
          }
      }
  
      function beginLoadingComponent(componentName, callback) {
          getFirstResultFromLoaders('getConfig', [componentName], function(config) {
              if (config) {
                  // We have a config, so now load its definition
                  getFirstResultFromLoaders('loadComponent', [componentName, config], function(definition) {
                      callback(definition, config);
                  });
              } else {
                  // The component has no config - it's unknown to all the loaders.
                  // Note that this is not an error (e.g., a module loading error) - that would abort the
                  // process and this callback would not run. For this callback to run, all loaders must
                  // have confirmed they don't know about this component.
                  callback(null, null);
              }
          });
      }
  
      function getFirstResultFromLoaders(methodName, argsExceptCallback, callback, candidateLoaders) {
          // On the first call in the stack, start with the full set of loaders
          if (!candidateLoaders) {
              candidateLoaders = ko.components['loaders'].slice(0); // Use a copy, because we'll be mutating this array
          }
  
          // Try the next candidate
          var currentCandidateLoader = candidateLoaders.shift();
          if (currentCandidateLoader) {
              var methodInstance = currentCandidateLoader[methodName];
              if (methodInstance) {
                  var wasAborted = false,
                      synchronousReturnValue = methodInstance.apply(currentCandidateLoader, argsExceptCallback.concat(function(result) {
                          if (wasAborted) {
                              callback(null);
                          } else if (result !== null) {
                              // This candidate returned a value. Use it.
                              callback(result);
                          } else {
                              // Try the next candidate
                              getFirstResultFromLoaders(methodName, argsExceptCallback, callback, candidateLoaders);
                          }
                      }));
  
                  // Currently, loaders may not return anything synchronously. This leaves open the possibility
                  // that we'll extend the API to support synchronous return values in the future. It won't be
                  // a breaking change, because currently no loader is allowed to return anything except undefined.
                  if (synchronousReturnValue !== undefined) {
                      wasAborted = true;
  
                      // Method to suppress exceptions will remain undocumented. This is only to keep
                      // KO's specs running tidily, since we can observe the loading got aborted without
                      // having exceptions cluttering up the console too.
                      if (!currentCandidateLoader['suppressLoaderExceptions']) {
                          throw new Error('Component loaders must supply values by invoking the callback, not by returning values synchronously.');
                      }
                  }
              } else {
                  // This candidate doesn't have the relevant handler. Synchronously move on to the next one.
                  getFirstResultFromLoaders(methodName, argsExceptCallback, callback, candidateLoaders);
              }
          } else {
              // No candidates returned a value
              callback(null);
          }
      }
  
      // Reference the loaders via string name so it's possible for developers
      // to replace the whole array by assigning to ko.components.loaders
      ko.components['loaders'] = [];
  
      ko.exportSymbol('components', ko.components);
      ko.exportSymbol('components.get', ko.components.get);
      ko.exportSymbol('components.clearCachedDefinition', ko.components.clearCachedDefinition);
  })();
  (function(undefined) {
  
      // The default loader is responsible for two things:
      // 1. Maintaining the default in-memory registry of component configuration objects
      //    (i.e., the thing you're writing to when you call ko.components.register(someName, ...))
      // 2. Answering requests for components by fetching configuration objects
      //    from that default in-memory registry and resolving them into standard
      //    component definition objects (of the form { createViewModel: ..., template: ... })
      // Custom loaders may override either of these facilities, i.e.,
      // 1. To supply configuration objects from some other source (e.g., conventions)
      // 2. Or, to resolve configuration objects by loading viewmodels/templates via arbitrary logic.
  
      var defaultConfigRegistry = {};
  
      ko.components.register = function(componentName, config) {
          if (!config) {
              throw new Error('Invalid configuration for ' + componentName);
          }
  
          if (ko.components.isRegistered(componentName)) {
              throw new Error('Component ' + componentName + ' is already registered');
          }
  
          defaultConfigRegistry[componentName] = config;
      };
  
      ko.components.isRegistered = function(componentName) {
          return Object.prototype.hasOwnProperty.call(defaultConfigRegistry, componentName);
      };
  
      ko.components.unregister = function(componentName) {
          delete defaultConfigRegistry[componentName];
          ko.components.clearCachedDefinition(componentName);
      };
  
      ko.components.defaultLoader = {
          'getConfig': function(componentName, callback) {
              var result = ko.components.isRegistered(componentName)
                  ? defaultConfigRegistry[componentName]
                  : null;
              callback(result);
          },
  
          'loadComponent': function(componentName, config, callback) {
              var errorCallback = makeErrorCallback(componentName);
              possiblyGetConfigFromAmd(errorCallback, config, function(loadedConfig) {
                  resolveConfig(componentName, errorCallback, loadedConfig, callback);
              });
          },
  
          'loadTemplate': function(componentName, templateConfig, callback) {
              resolveTemplate(makeErrorCallback(componentName), templateConfig, callback);
          },
  
          'loadViewModel': function(componentName, viewModelConfig, callback) {
              resolveViewModel(makeErrorCallback(componentName), viewModelConfig, callback);
          }
      };
  
      var createViewModelKey = 'createViewModel';
  
      // Takes a config object of the form { template: ..., viewModel: ... }, and asynchronously convert it
      // into the standard component definition format:
      //    { template: <ArrayOfDomNodes>, createViewModel: function(params, componentInfo) { ... } }.
      // Since both template and viewModel may need to be resolved asynchronously, both tasks are performed
      // in parallel, and the results joined when both are ready. We don't depend on any promises infrastructure,
      // so this is implemented manually below.
      function resolveConfig(componentName, errorCallback, config, callback) {
          var result = {},
              makeCallBackWhenZero = 2,
              tryIssueCallback = function() {
                  if (--makeCallBackWhenZero === 0) {
                      callback(result);
                  }
              },
              templateConfig = config['template'],
              viewModelConfig = config['viewModel'];
  
          if (templateConfig) {
              possiblyGetConfigFromAmd(errorCallback, templateConfig, function(loadedConfig) {
                  ko.components._getFirstResultFromLoaders('loadTemplate', [componentName, loadedConfig], function(resolvedTemplate) {
                      result['template'] = resolvedTemplate;
                      tryIssueCallback();
                  });
              });
          } else {
              tryIssueCallback();
          }
  
          if (viewModelConfig) {
              possiblyGetConfigFromAmd(errorCallback, viewModelConfig, function(loadedConfig) {
                  ko.components._getFirstResultFromLoaders('loadViewModel', [componentName, loadedConfig], function(resolvedViewModel) {
                      result[createViewModelKey] = resolvedViewModel;
                      tryIssueCallback();
                  });
              });
          } else {
              tryIssueCallback();
          }
      }
  
      function resolveTemplate(errorCallback, templateConfig, callback) {
          if (typeof templateConfig === 'string') {
              // Markup - parse it
              callback(ko.utils.parseHtmlFragment(templateConfig));
          } else if (templateConfig instanceof Array) {
              // Assume already an array of DOM nodes - pass through unchanged
              callback(templateConfig);
          } else if (isDocumentFragment(templateConfig)) {
              // Document fragment - use its child nodes
              callback(ko.utils.makeArray(templateConfig.childNodes));
          } else if (templateConfig['element']) {
              var element = templateConfig['element'];
              if (isDomElement(element)) {
                  // Element instance - copy its child nodes
                  callback(cloneNodesFromTemplateSourceElement(element));
              } else if (typeof element === 'string') {
                  // Element ID - find it, then copy its child nodes
                  var elemInstance = document.getElementById(element);
                  if (elemInstance) {
                      callback(cloneNodesFromTemplateSourceElement(elemInstance));
                  } else {
                      errorCallback('Cannot find element with ID ' + element);
                  }
              } else {
                  errorCallback('Unknown element type: ' + element);
              }
          } else {
              errorCallback('Unknown template value: ' + templateConfig);
          }
      }
  
      function resolveViewModel(errorCallback, viewModelConfig, callback) {
          if (typeof viewModelConfig === 'function') {
              // Constructor - convert to standard factory function format
              // By design, this does *not* supply componentInfo to the constructor, as the intent is that
              // componentInfo contains non-viewmodel data (e.g., the component's element) that should only
              // be used in factory functions, not viewmodel constructors.
              callback(function (params /*, componentInfo */) {
                  return new viewModelConfig(params);
              });
          } else if (typeof viewModelConfig[createViewModelKey] === 'function') {
              // Already a factory function - use it as-is
              callback(viewModelConfig[createViewModelKey]);
          } else if ('instance' in viewModelConfig) {
              // Fixed object instance - promote to createViewModel format for API consistency
              var fixedInstance = viewModelConfig['instance'];
              callback(function (params, componentInfo) {
                  return fixedInstance;
              });
          } else if ('viewModel' in viewModelConfig) {
              // Resolved AMD module whose value is of the form { viewModel: ... }
              resolveViewModel(errorCallback, viewModelConfig['viewModel'], callback);
          } else {
              errorCallback('Unknown viewModel value: ' + viewModelConfig);
          }
      }
  
      function cloneNodesFromTemplateSourceElement(elemInstance) {
          switch (ko.utils.tagNameLower(elemInstance)) {
              case 'script':
                  return ko.utils.parseHtmlFragment(elemInstance.text);
              case 'textarea':
                  return ko.utils.parseHtmlFragment(elemInstance.value);
              case 'template':
                  // For browsers with proper <template> element support (i.e., where the .content property
                  // gives a document fragment), use that document fragment.
                  if (isDocumentFragment(elemInstance.content)) {
                      return ko.utils.cloneNodes(elemInstance.content.childNodes);
                  }
          }
  
          // Regular elements such as <div>, and <template> elements on old browsers that don't really
          // understand <template> and just treat it as a regular container
          return ko.utils.cloneNodes(elemInstance.childNodes);
      }
  
      function isDomElement(obj) {
          if (window['HTMLElement']) {
              return obj instanceof HTMLElement;
          } else {
              return obj && obj.tagName && obj.nodeType === 1;
          }
      }
  
      function isDocumentFragment(obj) {
          if (window['DocumentFragment']) {
              return obj instanceof DocumentFragment;
          } else {
              return obj && obj.nodeType === 11;
          }
      }
  
      function possiblyGetConfigFromAmd(errorCallback, config, callback) {
          if (typeof config['require'] === 'string') {
              // The config is the value of an AMD module
              if (amdRequire || window['require']) {
                  (amdRequire || window['require'])([config['require']], function (module) {
                      if (module && typeof module === 'object' && module.__esModule && module.default) {
                          module = module.default;
                      }
                      callback(module);
                  });
              } else {
                  errorCallback('Uses require, but no AMD loader is present');
              }
          } else {
              callback(config);
          }
      }
  
      function makeErrorCallback(componentName) {
          return function (message) {
              throw new Error('Component \'' + componentName + '\': ' + message);
          };
      }
  
      ko.exportSymbol('components.register', ko.components.register);
      ko.exportSymbol('components.isRegistered', ko.components.isRegistered);
      ko.exportSymbol('components.unregister', ko.components.unregister);
  
      // Expose the default loader so that developers can directly ask it for configuration
      // or to resolve configuration
      ko.exportSymbol('components.defaultLoader', ko.components.defaultLoader);
  
      // By default, the default loader is the only registered component loader
      ko.components['loaders'].push(ko.components.defaultLoader);
  
      // Privately expose the underlying config registry for use in old-IE shim
      ko.components._allRegisteredComponents = defaultConfigRegistry;
  })();
  (function (undefined) {
      // Overridable API for determining which component name applies to a given node. By overriding this,
      // you can for example map specific tagNames to components that are not preregistered.
      ko.components['getComponentNameForNode'] = function(node) {
          var tagNameLower = ko.utils.tagNameLower(node);
          if (ko.components.isRegistered(tagNameLower)) {
              // Try to determine that this node can be considered a *custom* element; see https://github.com/knockout/knockout/issues/1603
              if (tagNameLower.indexOf('-') != -1 || ('' + node) == "[object HTMLUnknownElement]" || (ko.utils.ieVersion <= 8 && node.tagName === tagNameLower)) {
                  return tagNameLower;
              }
          }
      };
  
      ko.components.addBindingsForCustomElement = function(allBindings, node, bindingContext, valueAccessors) {
          // Determine if it's really a custom element matching a component
          if (node.nodeType === 1) {
              var componentName = ko.components['getComponentNameForNode'](node);
              if (componentName) {
                  // It does represent a component, so add a component binding for it
                  allBindings = allBindings || {};
  
                  if (allBindings['component']) {
                      // Avoid silently overwriting some other 'component' binding that may already be on the element
                      throw new Error('Cannot use the "component" binding on a custom element matching a component');
                  }
  
                  var componentBindingValue = { 'name': componentName, 'params': getComponentParamsFromCustomElement(node, bindingContext) };
  
                  allBindings['component'] = valueAccessors
                      ? function() { return componentBindingValue; }
                      : componentBindingValue;
              }
          }
  
          return allBindings;
      }
  
      var nativeBindingProviderInstance = new ko.bindingProvider();
  
      function getComponentParamsFromCustomElement(elem, bindingContext) {
          var paramsAttribute = elem.getAttribute('params');
  
          if (paramsAttribute) {
              var params = nativeBindingProviderInstance['parseBindingsString'](paramsAttribute, bindingContext, elem, { 'valueAccessors': true, 'bindingParams': true }),
                  rawParamComputedValues = ko.utils.objectMap(params, function(paramValue, paramName) {
                      return ko.computed(paramValue, null, { disposeWhenNodeIsRemoved: elem });
                  }),
                  result = ko.utils.objectMap(rawParamComputedValues, function(paramValueComputed, paramName) {
                      var paramValue = paramValueComputed.peek();
                      // Does the evaluation of the parameter value unwrap any observables?
                      if (!paramValueComputed.isActive()) {
                          // No it doesn't, so there's no need for any computed wrapper. Just pass through the supplied value directly.
                          // Example: "someVal: firstName, age: 123" (whether or not firstName is an observable/computed)
                          return paramValue;
                      } else {
                          // Yes it does. Supply a computed property that unwraps both the outer (binding expression)
                          // level of observability, and any inner (resulting model value) level of observability.
                          // This means the component doesn't have to worry about multiple unwrapping. If the value is a
                          // writable observable, the computed will also be writable and pass the value on to the observable.
                          return ko.computed({
                              'read': function() {
                                  return ko.utils.unwrapObservable(paramValueComputed());
                              },
                              'write': ko.isWriteableObservable(paramValue) && function(value) {
                                  paramValueComputed()(value);
                              },
                              disposeWhenNodeIsRemoved: elem
                          });
                      }
                  });
  
              // Give access to the raw computeds, as long as that wouldn't overwrite any custom param also called '$raw'
              // This is in case the developer wants to react to outer (binding) observability separately from inner
              // (model value) observability, or in case the model value observable has subobservables.
              if (!Object.prototype.hasOwnProperty.call(result, '$raw')) {
                  result['$raw'] = rawParamComputedValues;
              }
  
              return result;
          } else {
              // For consistency, absence of a "params" attribute is treated the same as the presence of
              // any empty one. Otherwise component viewmodels need special code to check whether or not
              // 'params' or 'params.$raw' is null/undefined before reading subproperties, which is annoying.
              return { '$raw': {} };
          }
      }
  
      // --------------------------------------------------------------------------------
      // Compatibility code for older (pre-HTML5) IE browsers
  
      if (ko.utils.ieVersion < 9) {
          // Whenever you preregister a component, enable it as a custom element in the current document
          ko.components['register'] = (function(originalFunction) {
              return function(componentName) {
                  document.createElement(componentName); // Allows IE<9 to parse markup containing the custom element
                  return originalFunction.apply(this, arguments);
              }
          })(ko.components['register']);
  
          // Whenever you create a document fragment, enable all preregistered component names as custom elements
          // This is needed to make innerShiv/jQuery HTML parsing correctly handle the custom elements
          document.createDocumentFragment = (function(originalFunction) {
              return function() {
                  var newDocFrag = originalFunction(),
                      allComponents = ko.components._allRegisteredComponents;
                  for (var componentName in allComponents) {
                      if (Object.prototype.hasOwnProperty.call(allComponents, componentName)) {
                          newDocFrag.createElement(componentName);
                      }
                  }
                  return newDocFrag;
              };
          })(document.createDocumentFragment);
      }
  })();(function(undefined) {
      var componentLoadingOperationUniqueId = 0;
  
      ko.bindingHandlers['component'] = {
          'init': function(element, valueAccessor, ignored1, ignored2, bindingContext) {
              var currentViewModel,
                  currentLoadingOperationId,
                  afterRenderSub,
                  disposeAssociatedComponentViewModel = function () {
                      var currentViewModelDispose = currentViewModel && currentViewModel['dispose'];
                      if (typeof currentViewModelDispose === 'function') {
                          currentViewModelDispose.call(currentViewModel);
                      }
                      if (afterRenderSub) {
                          afterRenderSub.dispose();
                      }
                      afterRenderSub = null;
                      currentViewModel = null;
                      // Any in-flight loading operation is no longer relevant, so make sure we ignore its completion
                      currentLoadingOperationId = null;
                  },
                  originalChildNodes = ko.utils.makeArray(ko.virtualElements.childNodes(element));
  
              ko.virtualElements.emptyNode(element);
              ko.utils.domNodeDisposal.addDisposeCallback(element, disposeAssociatedComponentViewModel);
  
              ko.computed(function () {
                  var value = ko.utils.unwrapObservable(valueAccessor()),
                      componentName, componentParams;
  
                  if (typeof value === 'string') {
                      componentName = value;
                  } else {
                      componentName = ko.utils.unwrapObservable(value['name']);
                      componentParams = ko.utils.unwrapObservable(value['params']);
                  }
  
                  if (!componentName) {
                      throw new Error('No component name specified');
                  }
  
                  var asyncContext = ko.bindingEvent.startPossiblyAsyncContentBinding(element, bindingContext);
  
                  var loadingOperationId = currentLoadingOperationId = ++componentLoadingOperationUniqueId;
                  ko.components.get(componentName, function(componentDefinition) {
                      // If this is not the current load operation for this element, ignore it.
                      if (currentLoadingOperationId !== loadingOperationId) {
                          return;
                      }
  
                      // Clean up previous state
                      disposeAssociatedComponentViewModel();
  
                      // Instantiate and bind new component. Implicitly this cleans any old DOM nodes.
                      if (!componentDefinition) {
                          throw new Error('Unknown component \'' + componentName + '\'');
                      }
                      cloneTemplateIntoElement(componentName, componentDefinition, element);
  
                      var componentInfo = {
                          'element': element,
                          'templateNodes': originalChildNodes
                      };
  
                      var componentViewModel = createViewModel(componentDefinition, componentParams, componentInfo),
                          childBindingContext = asyncContext['createChildContext'](componentViewModel, {
                              'extend': function(ctx) {
                                  ctx['$component'] = componentViewModel;
                                  ctx['$componentTemplateNodes'] = originalChildNodes;
                              }
                          });
  
                      if (componentViewModel && componentViewModel['koDescendantsComplete']) {
                          afterRenderSub = ko.bindingEvent.subscribe(element, ko.bindingEvent.descendantsComplete, componentViewModel['koDescendantsComplete'], componentViewModel);
                      }
  
                      currentViewModel = componentViewModel;
                      ko.applyBindingsToDescendants(childBindingContext, element);
                  });
              }, null, { disposeWhenNodeIsRemoved: element });
  
              return { 'controlsDescendantBindings': true };
          }
      };
  
      ko.virtualElements.allowedBindings['component'] = true;
  
      function cloneTemplateIntoElement(componentName, componentDefinition, element) {
          var template = componentDefinition['template'];
          if (!template) {
              throw new Error('Component \'' + componentName + '\' has no template');
          }
  
          var clonedNodesArray = ko.utils.cloneNodes(template);
          ko.virtualElements.setDomNodeChildren(element, clonedNodesArray);
      }
  
      function createViewModel(componentDefinition, componentParams, componentInfo) {
          var componentViewModelFactory = componentDefinition['createViewModel'];
          return componentViewModelFactory
              ? componentViewModelFactory.call(componentDefinition, componentParams, componentInfo)
              : componentParams; // Template-only component
      }
  
  })();
  var attrHtmlToJavaScriptMap = { 'class': 'className', 'for': 'htmlFor' };
  ko.bindingHandlers['attr'] = {
      'update': function(element, valueAccessor, allBindings) {
          var value = ko.utils.unwrapObservable(valueAccessor()) || {};
          ko.utils.objectForEach(value, function(attrName, attrValue) {
              attrValue = ko.utils.unwrapObservable(attrValue);
  
              // Find the namespace of this attribute, if any.
              var prefixLen = attrName.indexOf(':');
              var namespace = "lookupNamespaceURI" in element && prefixLen > 0 && element.lookupNamespaceURI(attrName.substr(0, prefixLen));
  
              // To cover cases like "attr: { checked:someProp }", we want to remove the attribute entirely
              // when someProp is a "no value"-like value (strictly null, false, or undefined)
              // (because the absence of the "checked" attr is how to mark an element as not checked, etc.)
              var toRemove = (attrValue === false) || (attrValue === null) || (attrValue === undefined);
              if (toRemove) {
                  namespace ? element.removeAttributeNS(namespace, attrName) : element.removeAttribute(attrName);
              } else {
                  attrValue = attrValue.toString();
              }
  
              // In IE <= 7 and IE8 Quirks Mode, you have to use the JavaScript property name instead of the
              // HTML attribute name for certain attributes. IE8 Standards Mode supports the correct behavior,
              // but instead of figuring out the mode, we'll just set the attribute through the JavaScript
              // property for IE <= 8.
              if (ko.utils.ieVersion <= 8 && attrName in attrHtmlToJavaScriptMap) {
                  attrName = attrHtmlToJavaScriptMap[attrName];
                  if (toRemove)
                      element.removeAttribute(attrName);
                  else
                      element[attrName] = attrValue;
              } else if (!toRemove) {
                  namespace ? element.setAttributeNS(namespace, attrName, attrValue) : element.setAttribute(attrName, attrValue);
              }
  
              // Treat "name" specially - although you can think of it as an attribute, it also needs
              // special handling on older versions of IE (https://github.com/SteveSanderson/knockout/pull/333)
              // Deliberately being case-sensitive here because XHTML would regard "Name" as a different thing
              // entirely, and there's no strong reason to allow for such casing in HTML.
              if (attrName === "name") {
                  ko.utils.setElementName(element, toRemove ? "" : attrValue);
              }
          });
      }
  };
  (function() {
  
  ko.bindingHandlers['checked'] = {
      'after': ['value', 'attr'],
      'init': function (element, valueAccessor, allBindings) {
          var checkedValue = ko.pureComputed(function() {
              // Treat "value" like "checkedValue" when it is included with "checked" binding
              if (allBindings['has']('checkedValue')) {
                  return ko.utils.unwrapObservable(allBindings.get('checkedValue'));
              } else if (useElementValue) {
                  if (allBindings['has']('value')) {
                      return ko.utils.unwrapObservable(allBindings.get('value'));
                  } else {
                      return element.value;
                  }
              }
          });
  
          function updateModel() {
              // This updates the model value from the view value.
              // It runs in response to DOM events (click) and changes in checkedValue.
              var isChecked = element.checked,
                  elemValue = checkedValue();
  
              // When we're first setting up this computed, don't change any model state.
              if (ko.computedContext.isInitial()) {
                  return;
              }
  
              // We can ignore unchecked radio buttons, because some other radio
              // button will be checked, and that one can take care of updating state.
              // Also ignore value changes to an already unchecked checkbox.
              if (!isChecked && (isRadio || ko.computedContext.getDependenciesCount())) {
                  return;
              }
  
              var modelValue = ko.dependencyDetection.ignore(valueAccessor);
              if (valueIsArray) {
                  var writableValue = rawValueIsNonArrayObservable ? modelValue.peek() : modelValue,
                      saveOldValue = oldElemValue;
                  oldElemValue = elemValue;
  
                  if (saveOldValue !== elemValue) {
                      // When we're responding to the checkedValue changing, and the element is
                      // currently checked, replace the old elem value with the new elem value
                      // in the model array.
                      if (isChecked) {
                          ko.utils.addOrRemoveItem(writableValue, elemValue, true);
                          ko.utils.addOrRemoveItem(writableValue, saveOldValue, false);
                      }
                  } else {
                      // When we're responding to the user having checked/unchecked a checkbox,
                      // add/remove the element value to the model array.
                      ko.utils.addOrRemoveItem(writableValue, elemValue, isChecked);
                  }
  
                  if (rawValueIsNonArrayObservable && ko.isWriteableObservable(modelValue)) {
                      modelValue(writableValue);
                  }
              } else {
                  if (isCheckbox) {
                      if (elemValue === undefined) {
                          elemValue = isChecked;
                      } else if (!isChecked) {
                          elemValue = undefined;
                      }
                  }
                  ko.expressionRewriting.writeValueToProperty(modelValue, allBindings, 'checked', elemValue, true);
              }
          };
  
          function updateView() {
              // This updates the view value from the model value.
              // It runs in response to changes in the bound (checked) value.
              var modelValue = ko.utils.unwrapObservable(valueAccessor()),
                  elemValue = checkedValue();
  
              if (valueIsArray) {
                  // When a checkbox is bound to an array, being checked represents its value being present in that array
                  element.checked = ko.utils.arrayIndexOf(modelValue, elemValue) >= 0;
                  oldElemValue = elemValue;
              } else if (isCheckbox && elemValue === undefined) {
                  // When a checkbox is bound to any other value (not an array) and "checkedValue" is not defined,
                  // being checked represents the value being trueish
                  element.checked = !!modelValue;
              } else {
                  // Otherwise, being checked means that the checkbox or radio button's value corresponds to the model value
                  element.checked = (checkedValue() === modelValue);
              }
          };
  
          var isCheckbox = element.type == "checkbox",
              isRadio = element.type == "radio";
  
          // Only bind to check boxes and radio buttons
          if (!isCheckbox && !isRadio) {
              return;
          }
  
          var rawValue = valueAccessor(),
              valueIsArray = isCheckbox && (ko.utils.unwrapObservable(rawValue) instanceof Array),
              rawValueIsNonArrayObservable = !(valueIsArray && rawValue.push && rawValue.splice),
              useElementValue = isRadio || valueIsArray,
              oldElemValue = valueIsArray ? checkedValue() : undefined;
  
          // IE 6 won't allow radio buttons to be selected unless they have a name
          if (isRadio && !element.name)
              ko.bindingHandlers['uniqueName']['init'](element, function() { return true });
  
          // Set up two computeds to update the binding:
  
          // The first responds to changes in the checkedValue value and to element clicks
          ko.computed(updateModel, null, { disposeWhenNodeIsRemoved: element });
          ko.utils.registerEventHandler(element, "click", updateModel);
  
          // The second responds to changes in the model value (the one associated with the checked binding)
          ko.computed(updateView, null, { disposeWhenNodeIsRemoved: element });
  
          rawValue = undefined;
      }
  };
  ko.expressionRewriting.twoWayBindings['checked'] = true;
  
  ko.bindingHandlers['checkedValue'] = {
      'update': function (element, valueAccessor) {
          element.value = ko.utils.unwrapObservable(valueAccessor());
      }
  };
  
  })();var classesWrittenByBindingKey = '__ko__cssValue';
  ko.bindingHandlers['class'] = {
      'update': function (element, valueAccessor) {
          var value = ko.utils.stringTrim(ko.utils.unwrapObservable(valueAccessor()));
          ko.utils.toggleDomNodeCssClass(element, element[classesWrittenByBindingKey], false);
          element[classesWrittenByBindingKey] = value;
          ko.utils.toggleDomNodeCssClass(element, value, true);
      }
  };
  
  ko.bindingHandlers['css'] = {
      'update': function (element, valueAccessor) {
          var value = ko.utils.unwrapObservable(valueAccessor());
          if (value !== null && typeof value == "object") {
              ko.utils.objectForEach(value, function(className, shouldHaveClass) {
                  shouldHaveClass = ko.utils.unwrapObservable(shouldHaveClass);
                  ko.utils.toggleDomNodeCssClass(element, className, shouldHaveClass);
              });
          } else {
              ko.bindingHandlers['class']['update'](element, valueAccessor);
          }
      }
  };
  ko.bindingHandlers['enable'] = {
      'update': function (element, valueAccessor) {
          var value = ko.utils.unwrapObservable(valueAccessor());
          if (value && element.disabled)
              element.removeAttribute("disabled");
          else if ((!value) && (!element.disabled))
              element.disabled = true;
      }
  };
  
  ko.bindingHandlers['disable'] = {
      'update': function (element, valueAccessor) {
          ko.bindingHandlers['enable']['update'](element, function() { return !ko.utils.unwrapObservable(valueAccessor()) });
      }
  };
  // For certain common events (currently just 'click'), allow a simplified data-binding syntax
  // e.g. click:handler instead of the usual full-length event:{click:handler}
  function makeEventHandlerShortcut(eventName) {
      ko.bindingHandlers[eventName] = {
          'init': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
              var newValueAccessor = function () {
                  var result = {};
                  result[eventName] = valueAccessor();
                  return result;
              };
              return ko.bindingHandlers['event']['init'].call(this, element, newValueAccessor, allBindings, viewModel, bindingContext);
          }
      }
  }
  
  ko.bindingHandlers['event'] = {
      'init' : function (element, valueAccessor, allBindings, viewModel, bindingContext) {
          var eventsToHandle = valueAccessor() || {};
          ko.utils.objectForEach(eventsToHandle, function(eventName) {
              if (typeof eventName == "string") {
                  ko.utils.registerEventHandler(element, eventName, function (event) {
                      var handlerReturnValue;
                      var handlerFunction = valueAccessor()[eventName];
                      if (!handlerFunction)
                          return;
  
                      try {
                          // Take all the event args, and prefix with the viewmodel
                          var argsForHandler = ko.utils.makeArray(arguments);
                          viewModel = bindingContext['$data'];
                          argsForHandler.unshift(viewModel);
                          handlerReturnValue = handlerFunction.apply(viewModel, argsForHandler);
                      } finally {
                          if (handlerReturnValue !== true) { // Normally we want to prevent default action. Developer can override this be explicitly returning true.
                              if (event.preventDefault)
                                  event.preventDefault();
                              else
                                  event.returnValue = false;
                          }
                      }
  
                      var bubble = allBindings.get(eventName + 'Bubble') !== false;
                      if (!bubble) {
                          event.cancelBubble = true;
                          if (event.stopPropagation)
                              event.stopPropagation();
                      }
                  });
              }
          });
      }
  };
  // "foreach: someExpression" is equivalent to "template: { foreach: someExpression }"
  // "foreach: { data: someExpression, afterAdd: myfn }" is equivalent to "template: { foreach: someExpression, afterAdd: myfn }"
  ko.bindingHandlers['foreach'] = {
      makeTemplateValueAccessor: function(valueAccessor) {
          return function() {
              var modelValue = valueAccessor(),
                  unwrappedValue = ko.utils.peekObservable(modelValue);    // Unwrap without setting a dependency here
  
              // If unwrappedValue is the array, pass in the wrapped value on its own
              // The value will be unwrapped and tracked within the template binding
              // (See https://github.com/SteveSanderson/knockout/issues/523)
              if ((!unwrappedValue) || typeof unwrappedValue.length == "number")
                  return { 'foreach': modelValue, 'templateEngine': ko.nativeTemplateEngine.instance };
  
              // If unwrappedValue.data is the array, preserve all relevant options and unwrap again value so we get updates
              ko.utils.unwrapObservable(modelValue);
              return {
                  'foreach': unwrappedValue['data'],
                  'as': unwrappedValue['as'],
                  'noChildContext': unwrappedValue['noChildContext'],
                  'includeDestroyed': unwrappedValue['includeDestroyed'],
                  'afterAdd': unwrappedValue['afterAdd'],
                  'beforeRemove': unwrappedValue['beforeRemove'],
                  'afterRender': unwrappedValue['afterRender'],
                  'beforeMove': unwrappedValue['beforeMove'],
                  'afterMove': unwrappedValue['afterMove'],
                  'templateEngine': ko.nativeTemplateEngine.instance
              };
          };
      },
      'init': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
          return ko.bindingHandlers['template']['init'](element, ko.bindingHandlers['foreach'].makeTemplateValueAccessor(valueAccessor));
      },
      'update': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
          return ko.bindingHandlers['template']['update'](element, ko.bindingHandlers['foreach'].makeTemplateValueAccessor(valueAccessor), allBindings, viewModel, bindingContext);
      }
  };
  ko.expressionRewriting.bindingRewriteValidators['foreach'] = false; // Can't rewrite control flow bindings
  ko.virtualElements.allowedBindings['foreach'] = true;
  var hasfocusUpdatingProperty = '__ko_hasfocusUpdating';
  var hasfocusLastValue = '__ko_hasfocusLastValue';
  ko.bindingHandlers['hasfocus'] = {
      'init': function(element, valueAccessor, allBindings) {
          var handleElementFocusChange = function(isFocused) {
              // Where possible, ignore which event was raised and determine focus state using activeElement,
              // as this avoids phantom focus/blur events raised when changing tabs in modern browsers.
              // However, not all KO-targeted browsers (Firefox 2) support activeElement. For those browsers,
              // prevent a loss of focus when changing tabs/windows by setting a flag that prevents hasfocus
              // from calling 'blur()' on the element when it loses focus.
              // Discussion at https://github.com/SteveSanderson/knockout/pull/352
              element[hasfocusUpdatingProperty] = true;
              var ownerDoc = element.ownerDocument;
              if ("activeElement" in ownerDoc) {
                  var active;
                  try {
                      active = ownerDoc.activeElement;
                  } catch(e) {
                      // IE9 throws if you access activeElement during page load (see issue #703)
                      active = ownerDoc.body;
                  }
                  isFocused = (active === element);
              }
              var modelValue = valueAccessor();
              ko.expressionRewriting.writeValueToProperty(modelValue, allBindings, 'hasfocus', isFocused, true);
  
              //cache the latest value, so we can avoid unnecessarily calling focus/blur in the update function
              element[hasfocusLastValue] = isFocused;
              element[hasfocusUpdatingProperty] = false;
          };
          var handleElementFocusIn = handleElementFocusChange.bind(null, true);
          var handleElementFocusOut = handleElementFocusChange.bind(null, false);
  
          ko.utils.registerEventHandler(element, "focus", handleElementFocusIn);
          ko.utils.registerEventHandler(element, "focusin", handleElementFocusIn); // For IE
          ko.utils.registerEventHandler(element, "blur",  handleElementFocusOut);
          ko.utils.registerEventHandler(element, "focusout",  handleElementFocusOut); // For IE
  
          // Assume element is not focused (prevents "blur" being called initially)
          element[hasfocusLastValue] = false;
      },
      'update': function(element, valueAccessor) {
          var value = !!ko.utils.unwrapObservable(valueAccessor());
  
          if (!element[hasfocusUpdatingProperty] && element[hasfocusLastValue] !== value) {
              value ? element.focus() : element.blur();
  
              // In IE, the blur method doesn't always cause the element to lose focus (for example, if the window is not in focus).
              // Setting focus to the body element does seem to be reliable in IE, but should only be used if we know that the current
              // element was focused already.
              if (!value && element[hasfocusLastValue]) {
                  element.ownerDocument.body.focus();
              }
  
              // For IE, which doesn't reliably fire "focus" or "blur" events synchronously
              ko.dependencyDetection.ignore(ko.utils.triggerEvent, null, [element, value ? "focusin" : "focusout"]);
          }
      }
  };
  ko.expressionRewriting.twoWayBindings['hasfocus'] = true;
  
  ko.bindingHandlers['hasFocus'] = ko.bindingHandlers['hasfocus']; // Make "hasFocus" an alias
  ko.expressionRewriting.twoWayBindings['hasFocus'] = 'hasfocus';
  ko.bindingHandlers['html'] = {
      'init': function() {
          // Prevent binding on the dynamically-injected HTML (as developers are unlikely to expect that, and it has security implications)
          return { 'controlsDescendantBindings': true };
      },
      'update': function (element, valueAccessor) {
          // setHtml will unwrap the value if needed
          ko.utils.setHtml(element, valueAccessor());
      }
  };
  (function () {
  
  // Makes a binding like with or if
  function makeWithIfBinding(bindingKey, isWith, isNot) {
      ko.bindingHandlers[bindingKey] = {
          'init': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
              var didDisplayOnLastUpdate, savedNodes, contextOptions = {}, completeOnRender, needAsyncContext, renderOnEveryChange;
  
              if (isWith) {
                  var as = allBindings.get('as'), noChildContext = allBindings.get('noChildContext');
                  renderOnEveryChange = !(as && noChildContext);
                  contextOptions = { 'as': as, 'noChildContext': noChildContext, 'exportDependencies': renderOnEveryChange };
              }
  
              completeOnRender = allBindings.get("completeOn") == "render";
              needAsyncContext = completeOnRender || allBindings['has'](ko.bindingEvent.descendantsComplete);
  
              ko.computed(function() {
                  var value = ko.utils.unwrapObservable(valueAccessor()),
                      shouldDisplay = !isNot !== !value, // equivalent to isNot ? !value : !!value,
                      isInitial = !savedNodes,
                      childContext;
  
                  if (!renderOnEveryChange && shouldDisplay === didDisplayOnLastUpdate) {
                      return;
                  }
  
                  if (needAsyncContext) {
                      bindingContext = ko.bindingEvent.startPossiblyAsyncContentBinding(element, bindingContext);
                  }
  
                  if (shouldDisplay) {
                      if (!isWith || renderOnEveryChange) {
                          contextOptions['dataDependency'] = ko.computedContext.computed();
                      }
  
                      if (isWith) {
                          childContext = bindingContext['createChildContext'](typeof value == "function" ? value : valueAccessor, contextOptions);
                      } else if (ko.computedContext.getDependenciesCount()) {
                          childContext = bindingContext['extend'](null, contextOptions);
                      } else {
                          childContext = bindingContext;
                      }
                  }
  
                  // Save a copy of the inner nodes on the initial update, but only if we have dependencies.
                  if (isInitial && ko.computedContext.getDependenciesCount()) {
                      savedNodes = ko.utils.cloneNodes(ko.virtualElements.childNodes(element), true /* shouldCleanNodes */);
                  }
  
                  if (shouldDisplay) {
                      if (!isInitial) {
                          ko.virtualElements.setDomNodeChildren(element, ko.utils.cloneNodes(savedNodes));
                      }
  
                      ko.applyBindingsToDescendants(childContext, element);
                  } else {
                      ko.virtualElements.emptyNode(element);
  
                      if (!completeOnRender) {
                          ko.bindingEvent.notify(element, ko.bindingEvent.childrenComplete);
                      }
                  }
  
                  didDisplayOnLastUpdate = shouldDisplay;
  
              }, null, { disposeWhenNodeIsRemoved: element });
  
              return { 'controlsDescendantBindings': true };
          }
      };
      ko.expressionRewriting.bindingRewriteValidators[bindingKey] = false; // Can't rewrite control flow bindings
      ko.virtualElements.allowedBindings[bindingKey] = true;
  }
  
  // Construct the actual binding handlers
  makeWithIfBinding('if');
  makeWithIfBinding('ifnot', false /* isWith */, true /* isNot */);
  makeWithIfBinding('with', true /* isWith */);
  
  })();ko.bindingHandlers['let'] = {
      'init': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
          // Make a modified binding context, with extra properties, and apply it to descendant elements
          var innerContext = bindingContext['extend'](valueAccessor);
          ko.applyBindingsToDescendants(innerContext, element);
  
          return { 'controlsDescendantBindings': true };
      }
  };
  ko.virtualElements.allowedBindings['let'] = true;
  var captionPlaceholder = {};
  ko.bindingHandlers['options'] = {
      'init': function(element) {
          if (ko.utils.tagNameLower(element) !== "select")
              throw new Error("options binding applies only to SELECT elements");
  
          // Remove all existing <option>s.
          while (element.length > 0) {
              element.remove(0);
          }
  
          // Ensures that the binding processor doesn't try to bind the options
          return { 'controlsDescendantBindings': true };
      },
      'update': function (element, valueAccessor, allBindings) {
          function selectedOptions() {
              return ko.utils.arrayFilter(element.options, function (node) { return node.selected; });
          }
  
          var selectWasPreviouslyEmpty = element.length == 0,
              multiple = element.multiple,
              previousScrollTop = (!selectWasPreviouslyEmpty && multiple) ? element.scrollTop : null,
              unwrappedArray = ko.utils.unwrapObservable(valueAccessor()),
              valueAllowUnset = allBindings.get('valueAllowUnset') && allBindings['has']('value'),
              includeDestroyed = allBindings.get('optionsIncludeDestroyed'),
              arrayToDomNodeChildrenOptions = {},
              captionValue,
              filteredArray,
              previousSelectedValues = [];
  
          if (!valueAllowUnset) {
              if (multiple) {
                  previousSelectedValues = ko.utils.arrayMap(selectedOptions(), ko.selectExtensions.readValue);
              } else if (element.selectedIndex >= 0) {
                  previousSelectedValues.push(ko.selectExtensions.readValue(element.options[element.selectedIndex]));
              }
          }
  
          if (unwrappedArray) {
              if (typeof unwrappedArray.length == "undefined") // Coerce single value into array
                  unwrappedArray = [unwrappedArray];
  
              // Filter out any entries marked as destroyed
              filteredArray = ko.utils.arrayFilter(unwrappedArray, function(item) {
                  return includeDestroyed || item === undefined || item === null || !ko.utils.unwrapObservable(item['_destroy']);
              });
  
              // If caption is included, add it to the array
              if (allBindings['has']('optionsCaption')) {
                  captionValue = ko.utils.unwrapObservable(allBindings.get('optionsCaption'));
                  // If caption value is null or undefined, don't show a caption
                  if (captionValue !== null && captionValue !== undefined) {
                      filteredArray.unshift(captionPlaceholder);
                  }
              }
          } else {
              // If a falsy value is provided (e.g. null), we'll simply empty the select element
          }
  
          function applyToObject(object, predicate, defaultValue) {
              var predicateType = typeof predicate;
              if (predicateType == "function")    // Given a function; run it against the data value
                  return predicate(object);
              else if (predicateType == "string") // Given a string; treat it as a property name on the data value
                  return object[predicate];
              else                                // Given no optionsText arg; use the data value itself
                  return defaultValue;
          }
  
          // The following functions can run at two different times:
          // The first is when the whole array is being updated directly from this binding handler.
          // The second is when an observable value for a specific array entry is updated.
          // oldOptions will be empty in the first case, but will be filled with the previously generated option in the second.
          var itemUpdate = false;
          function optionForArrayItem(arrayEntry, index, oldOptions) {
              if (oldOptions.length) {
                  previousSelectedValues = !valueAllowUnset && oldOptions[0].selected ? [ ko.selectExtensions.readValue(oldOptions[0]) ] : [];
                  itemUpdate = true;
              }
              var option = element.ownerDocument.createElement("option");
              if (arrayEntry === captionPlaceholder) {
                  ko.utils.setTextContent(option, allBindings.get('optionsCaption'));
                  ko.selectExtensions.writeValue(option, undefined);
              } else {
                  // Apply a value to the option element
                  var optionValue = applyToObject(arrayEntry, allBindings.get('optionsValue'), arrayEntry);
                  ko.selectExtensions.writeValue(option, ko.utils.unwrapObservable(optionValue));
  
                  // Apply some text to the option element
                  var optionText = applyToObject(arrayEntry, allBindings.get('optionsText'), optionValue);
                  ko.utils.setTextContent(option, optionText);
              }
              return [option];
          }
  
          // By using a beforeRemove callback, we delay the removal until after new items are added. This fixes a selection
          // problem in IE<=8 and Firefox. See https://github.com/knockout/knockout/issues/1208
          arrayToDomNodeChildrenOptions['beforeRemove'] =
              function (option) {
                  element.removeChild(option);
              };
  
          function setSelectionCallback(arrayEntry, newOptions) {
              if (itemUpdate && valueAllowUnset) {
                  // The model value is authoritative, so make sure its value is the one selected
                  ko.bindingEvent.notify(element, ko.bindingEvent.childrenComplete);
              } else if (previousSelectedValues.length) {
                  // IE6 doesn't like us to assign selection to OPTION nodes before they're added to the document.
                  // That's why we first added them without selection. Now it's time to set the selection.
                  var isSelected = ko.utils.arrayIndexOf(previousSelectedValues, ko.selectExtensions.readValue(newOptions[0])) >= 0;
                  ko.utils.setOptionNodeSelectionState(newOptions[0], isSelected);
  
                  // If this option was changed from being selected during a single-item update, notify the change
                  if (itemUpdate && !isSelected) {
                      ko.dependencyDetection.ignore(ko.utils.triggerEvent, null, [element, "change"]);
                  }
              }
          }
  
          var callback = setSelectionCallback;
          if (allBindings['has']('optionsAfterRender') && typeof allBindings.get('optionsAfterRender') == "function") {
              callback = function(arrayEntry, newOptions) {
                  setSelectionCallback(arrayEntry, newOptions);
                  ko.dependencyDetection.ignore(allBindings.get('optionsAfterRender'), null, [newOptions[0], arrayEntry !== captionPlaceholder ? arrayEntry : undefined]);
              }
          }
  
          ko.utils.setDomNodeChildrenFromArrayMapping(element, filteredArray, optionForArrayItem, arrayToDomNodeChildrenOptions, callback);
  
          if (!valueAllowUnset) {
              // Determine if the selection has changed as a result of updating the options list
              var selectionChanged;
              if (multiple) {
                  // For a multiple-select box, compare the new selection count to the previous one
                  // But if nothing was selected before, the selection can't have changed
                  selectionChanged = previousSelectedValues.length && selectedOptions().length < previousSelectedValues.length;
              } else {
                  // For a single-select box, compare the current value to the previous value
                  // But if nothing was selected before or nothing is selected now, just look for a change in selection
                  selectionChanged = (previousSelectedValues.length && element.selectedIndex >= 0)
                      ? (ko.selectExtensions.readValue(element.options[element.selectedIndex]) !== previousSelectedValues[0])
                      : (previousSelectedValues.length || element.selectedIndex >= 0);
              }
  
              // Ensure consistency between model value and selected option.
              // If the dropdown was changed so that selection is no longer the same,
              // notify the value or selectedOptions binding.
              if (selectionChanged) {
                  ko.dependencyDetection.ignore(ko.utils.triggerEvent, null, [element, "change"]);
              }
          }
  
          if (valueAllowUnset || ko.computedContext.isInitial()) {
              ko.bindingEvent.notify(element, ko.bindingEvent.childrenComplete);
          }
  
          // Workaround for IE bug
          ko.utils.ensureSelectElementIsRenderedCorrectly(element);
  
          if (previousScrollTop && Math.abs(previousScrollTop - element.scrollTop) > 20)
              element.scrollTop = previousScrollTop;
      }
  };
  ko.bindingHandlers['options'].optionValueDomDataKey = ko.utils.domData.nextKey();
  ko.bindingHandlers['selectedOptions'] = {
      'init': function (element, valueAccessor, allBindings) {
          function updateFromView() {
              var value = valueAccessor(), valueToWrite = [];
              ko.utils.arrayForEach(element.getElementsByTagName("option"), function(node) {
                  if (node.selected)
                      valueToWrite.push(ko.selectExtensions.readValue(node));
              });
              ko.expressionRewriting.writeValueToProperty(value, allBindings, 'selectedOptions', valueToWrite);
          }
  
          function updateFromModel() {
              var newValue = ko.utils.unwrapObservable(valueAccessor()),
                  previousScrollTop = element.scrollTop;
  
              if (newValue && typeof newValue.length == "number") {
                  ko.utils.arrayForEach(element.getElementsByTagName("option"), function(node) {
                      var isSelected = ko.utils.arrayIndexOf(newValue, ko.selectExtensions.readValue(node)) >= 0;
                      if (node.selected != isSelected) {      // This check prevents flashing of the select element in IE
                          ko.utils.setOptionNodeSelectionState(node, isSelected);
                      }
                  });
              }
  
              element.scrollTop = previousScrollTop;
          }
  
          if (ko.utils.tagNameLower(element) != "select") {
              throw new Error("selectedOptions binding applies only to SELECT elements");
          }
  
          var updateFromModelComputed;
          ko.bindingEvent.subscribe(element, ko.bindingEvent.childrenComplete, function () {
              if (!updateFromModelComputed) {
                  ko.utils.registerEventHandler(element, "change", updateFromView);
                  updateFromModelComputed = ko.computed(updateFromModel, null, { disposeWhenNodeIsRemoved: element });
              } else {
                  updateFromView();
              }
          }, null, { 'notifyImmediately': true });
      },
      'update': function() {} // Keep for backwards compatibility with code that may have wrapped binding
  };
  ko.expressionRewriting.twoWayBindings['selectedOptions'] = true;
  ko.bindingHandlers['style'] = {
      'update': function (element, valueAccessor) {
          var value = ko.utils.unwrapObservable(valueAccessor() || {});
          ko.utils.objectForEach(value, function(styleName, styleValue) {
              styleValue = ko.utils.unwrapObservable(styleValue);
  
              if (styleValue === null || styleValue === undefined || styleValue === false) {
                  // Empty string removes the value, whereas null/undefined have no effect
                  styleValue = "";
              }
  
              if (jQueryInstance) {
                  jQueryInstance(element)['css'](styleName, styleValue);
              } else if (/^--/.test(styleName)) {
                  // Is styleName a custom CSS property?
                  element.style.setProperty(styleName, styleValue);
              } else {
                  styleName = styleName.replace(/-(\w)/g, function (all, letter) {
                      return letter.toUpperCase();
                  });
  
                  var previousStyle = element.style[styleName];
                  element.style[styleName] = styleValue;
  
                  if (styleValue !== previousStyle && element.style[styleName] == previousStyle && !isNaN(styleValue)) {
                      element.style[styleName] = styleValue + "px";
                  }
              }
          });
      }
  };
  ko.bindingHandlers['submit'] = {
      'init': function (element, valueAccessor, allBindings, viewModel, bindingContext) {
          if (typeof valueAccessor() != "function")
              throw new Error("The value for a submit binding must be a function");
          ko.utils.registerEventHandler(element, "submit", function (event) {
              var handlerReturnValue;
              var value = valueAccessor();
              try { handlerReturnValue = value.call(bindingContext['$data'], element); }
              finally {
                  if (handlerReturnValue !== true) { // Normally we want to prevent default action. Developer can override this be explicitly returning true.
                      if (event.preventDefault)
                          event.preventDefault();
                      else
                          event.returnValue = false;
                  }
              }
          });
      }
  };
  ko.bindingHandlers['text'] = {
      'init': function() {
          // Prevent binding on the dynamically-injected text node (as developers are unlikely to expect that, and it has security implications).
          // It should also make things faster, as we no longer have to consider whether the text node might be bindable.
          return { 'controlsDescendantBindings': true };
      },
      'update': function (element, valueAccessor) {
          ko.utils.setTextContent(element, valueAccessor());
      }
  };
  ko.virtualElements.allowedBindings['text'] = true;
  (function () {
  
  if (window && window.navigator) {
      var parseVersion = function (matches) {
          if (matches) {
              return parseFloat(matches[1]);
          }
      };
  
      // Detect various browser versions because some old versions don't fully support the 'input' event
      var userAgent = window.navigator.userAgent,
          operaVersion, chromeVersion, safariVersion, firefoxVersion, ieVersion, edgeVersion;
  
      (operaVersion = window.opera && window.opera.version && parseInt(window.opera.version()))
          || (edgeVersion = parseVersion(userAgent.match(/Edge\/([^ ]+)$/)))
          || (chromeVersion = parseVersion(userAgent.match(/Chrome\/([^ ]+)/)))
          || (safariVersion = parseVersion(userAgent.match(/Version\/([^ ]+) Safari/)))
          || (firefoxVersion = parseVersion(userAgent.match(/Firefox\/([^ ]+)/)))
          || (ieVersion = ko.utils.ieVersion || parseVersion(userAgent.match(/MSIE ([^ ]+)/)))      // Detects up to IE 10
          || (ieVersion = parseVersion(userAgent.match(/rv:([^ )]+)/)));      // Detects IE 11
  }
  
  // IE 8 and 9 have bugs that prevent the normal events from firing when the value changes.
  // But it does fire the 'selectionchange' event on many of those, presumably because the
  // cursor is moving and that counts as the selection changing. The 'selectionchange' event is
  // fired at the document level only and doesn't directly indicate which element changed. We
  // set up just one event handler for the document and use 'activeElement' to determine which
  // element was changed.
  if (ieVersion >= 8 && ieVersion < 10) {
      var selectionChangeRegisteredName = ko.utils.domData.nextKey(),
          selectionChangeHandlerName = ko.utils.domData.nextKey();
      var selectionChangeHandler = function(event) {
          var target = this.activeElement,
              handler = target && ko.utils.domData.get(target, selectionChangeHandlerName);
          if (handler) {
              handler(event);
          }
      };
      var registerForSelectionChangeEvent = function (element, handler) {
          var ownerDoc = element.ownerDocument;
          if (!ko.utils.domData.get(ownerDoc, selectionChangeRegisteredName)) {
              ko.utils.domData.set(ownerDoc, selectionChangeRegisteredName, true);
              ko.utils.registerEventHandler(ownerDoc, 'selectionchange', selectionChangeHandler);
          }
          ko.utils.domData.set(element, selectionChangeHandlerName, handler);
      };
  }
  
  ko.bindingHandlers['textInput'] = {
      'init': function (element, valueAccessor, allBindings) {
  
          var previousElementValue = element.value,
              timeoutHandle,
              elementValueBeforeEvent;
  
          var updateModel = function (event) {
              clearTimeout(timeoutHandle);
              elementValueBeforeEvent = timeoutHandle = undefined;
  
              var elementValue = element.value;
              if (previousElementValue !== elementValue) {
                  // Provide a way for tests to know exactly which event was processed
                  if (DEBUG && event) element['_ko_textInputProcessedEvent'] = event.type;
                  previousElementValue = elementValue;
                  ko.expressionRewriting.writeValueToProperty(valueAccessor(), allBindings, 'textInput', elementValue);
              }
          };
  
          var deferUpdateModel = function (event) {
              if (!timeoutHandle) {
                  // The elementValueBeforeEvent variable is set *only* during the brief gap between an
                  // event firing and the updateModel function running. This allows us to ignore model
                  // updates that are from the previous state of the element, usually due to techniques
                  // such as rateLimit. Such updates, if not ignored, can cause keystrokes to be lost.
                  elementValueBeforeEvent = element.value;
                  var handler = DEBUG ? updateModel.bind(element, {type: event.type}) : updateModel;
                  timeoutHandle = ko.utils.setTimeout(handler, 4);
              }
          };
  
          // IE9 will mess up the DOM if you handle events synchronously which results in DOM changes (such as other bindings);
          // so we'll make sure all updates are asynchronous
          var ieUpdateModel = ko.utils.ieVersion == 9 ? deferUpdateModel : updateModel,
              ourUpdate = false;
  
          var updateView = function () {
              var modelValue = ko.utils.unwrapObservable(valueAccessor());
  
              if (modelValue === null || modelValue === undefined) {
                  modelValue = '';
              }
  
              if (elementValueBeforeEvent !== undefined && modelValue === elementValueBeforeEvent) {
                  ko.utils.setTimeout(updateView, 4);
                  return;
              }
  
              // Update the element only if the element and model are different. On some browsers, updating the value
              // will move the cursor to the end of the input, which would be bad while the user is typing.
              if (element.value !== modelValue) {
                  ourUpdate = true;  // Make sure we ignore events (propertychange) that result from updating the value
                  element.value = modelValue;
                  ourUpdate = false;
                  previousElementValue = element.value; // In case the browser changes the value (see #2281)
              }
          };
  
          var onEvent = function (event, handler) {
              ko.utils.registerEventHandler(element, event, handler);
          };
  
          if (DEBUG && ko.bindingHandlers['textInput']['_forceUpdateOn']) {
              // Provide a way for tests to specify exactly which events are bound
              ko.utils.arrayForEach(ko.bindingHandlers['textInput']['_forceUpdateOn'], function(eventName) {
                  if (eventName.slice(0,5) == 'after') {
                      onEvent(eventName.slice(5), deferUpdateModel);
                  } else {
                      onEvent(eventName, updateModel);
                  }
              });
          } else {
              if (ieVersion) {
                  // All versions (including 11) of Internet Explorer have a bug that they don't generate an input or propertychange event when ESC is pressed
                  onEvent('keypress', updateModel);
              }
              if (ieVersion < 11) {
                  // Internet Explorer <= 8 doesn't support the 'input' event, but does include 'propertychange' that fires whenever
                  // any property of an element changes. Unlike 'input', it also fires if a property is changed from JavaScript code,
                  // but that's an acceptable compromise for this binding. IE 9 and 10 support 'input', but since they don't always
                  // fire it when using autocomplete, we'll use 'propertychange' for them also.
                  onEvent('propertychange', function(event) {
                      if (!ourUpdate && event.propertyName === 'value') {
                          ieUpdateModel(event);
                      }
                  });
              }
              if (ieVersion == 8) {
                  // IE 8 has a bug where it fails to fire 'propertychange' on the first update following a value change from
                  // JavaScript code. It also doesn't fire if you clear the entire value. To fix this, we bind to the following
                  // events too.
                  onEvent('keyup', updateModel);      // A single keystoke
                  onEvent('keydown', updateModel);    // The first character when a key is held down
              }
              if (registerForSelectionChangeEvent) {
                  // Internet Explorer 9 doesn't fire the 'input' event when deleting text, including using
                  // the backspace, delete, or ctrl-x keys, clicking the 'x' to clear the input, dragging text
                  // out of the field, and cutting or deleting text using the context menu. 'selectionchange'
                  // can detect all of those except dragging text out of the field, for which we use 'dragend'.
                  // These are also needed in IE8 because of the bug described above.
                  registerForSelectionChangeEvent(element, ieUpdateModel);  // 'selectionchange' covers cut, paste, drop, delete, etc.
                  onEvent('dragend', deferUpdateModel);
              }
  
              if (!ieVersion || ieVersion >= 9) {
                  // All other supported browsers support the 'input' event, which fires whenever the content of the element is changed
                  // through the user interface.
                  onEvent('input', ieUpdateModel);
              }
  
              if (safariVersion < 5 && ko.utils.tagNameLower(element) === "textarea") {
                  // Safari <5 doesn't fire the 'input' event for <textarea> elements (it does fire 'textInput'
                  // but only when typing). So we'll just catch as much as we can with keydown, cut, and paste.
                  onEvent('keydown', deferUpdateModel);
                  onEvent('paste', deferUpdateModel);
                  onEvent('cut', deferUpdateModel);
              } else if (operaVersion < 11) {
                  // Opera 10 doesn't always fire the 'input' event for cut, paste, undo & drop operations.
                  // We can try to catch some of those using 'keydown'.
                  onEvent('keydown', deferUpdateModel);
              } else if (firefoxVersion < 4.0) {
                  // Firefox <= 3.6 doesn't fire the 'input' event when text is filled in through autocomplete
                  onEvent('DOMAutoComplete', updateModel);
  
                  // Firefox <=3.5 doesn't fire the 'input' event when text is dropped into the input.
                  onEvent('dragdrop', updateModel);       // <3.5
                  onEvent('drop', updateModel);           // 3.5
              } else if (edgeVersion && element.type === "number") {
                  // Microsoft Edge doesn't fire 'input' or 'change' events for number inputs when
                  // the value is changed via the up / down arrow keys
                  onEvent('keydown', deferUpdateModel);
              }
          }
  
          // Bind to the change event so that we can catch programmatic updates of the value that fire this event.
          onEvent('change', updateModel);
  
          // To deal with browsers that don't notify any kind of event for some changes (IE, Safari, etc.)
          onEvent('blur', updateModel);
  
          ko.computed(updateView, null, { disposeWhenNodeIsRemoved: element });
      }
  };
  ko.expressionRewriting.twoWayBindings['textInput'] = true;
  
  // textinput is an alias for textInput
  ko.bindingHandlers['textinput'] = {
      // preprocess is the only way to set up a full alias
      'preprocess': function (value, name, addBinding) {
          addBinding('textInput', value);
      }
  };
  
  })();ko.bindingHandlers['uniqueName'] = {
      'init': function (element, valueAccessor) {
          if (valueAccessor()) {
              var name = "ko_unique_" + (++ko.bindingHandlers['uniqueName'].currentIndex);
              ko.utils.setElementName(element, name);
          }
      }
  };
  ko.bindingHandlers['uniqueName'].currentIndex = 0;
  ko.bindingHandlers['using'] = {
      'init': function(element, valueAccessor, allBindings, viewModel, bindingContext) {
          var options;
  
          if (allBindings['has']('as')) {
              options = { 'as': allBindings.get('as'), 'noChildContext': allBindings.get('noChildContext') };
          }
  
          var innerContext = bindingContext['createChildContext'](valueAccessor, options);
          ko.applyBindingsToDescendants(innerContext, element);
  
          return { 'controlsDescendantBindings': true };
      }
  };
  ko.virtualElements.allowedBindings['using'] = true;
  ko.bindingHandlers['value'] = {
      'init': function (element, valueAccessor, allBindings) {
          var tagName = ko.utils.tagNameLower(element),
              isInputElement = tagName == "input";
  
          // If the value binding is placed on a radio/checkbox, then just pass through to checkedValue and quit
          if (isInputElement && (element.type == "checkbox" || element.type == "radio")) {
              ko.applyBindingAccessorsToNode(element, { 'checkedValue': valueAccessor });
              return;
          }
  
          var eventsToCatch = [];
          var requestedEventsToCatch = allBindings.get("valueUpdate");
          var propertyChangedFired = false;
          var elementValueBeforeEvent = null;
  
          if (requestedEventsToCatch) {
              // Allow both individual event names, and arrays of event names
              if (typeof requestedEventsToCatch == "string") {
                  eventsToCatch = [requestedEventsToCatch];
              } else {
                  eventsToCatch = ko.utils.arrayGetDistinctValues(requestedEventsToCatch);
              }
              ko.utils.arrayRemoveItem(eventsToCatch, "change");  // We'll subscribe to "change" events later
          }
  
          var valueUpdateHandler = function() {
              elementValueBeforeEvent = null;
              propertyChangedFired = false;
              var modelValue = valueAccessor();
              var elementValue = ko.selectExtensions.readValue(element);
              ko.expressionRewriting.writeValueToProperty(modelValue, allBindings, 'value', elementValue);
          }
  
          // Workaround for https://github.com/SteveSanderson/knockout/issues/122
          // IE doesn't fire "change" events on textboxes if the user selects a value from its autocomplete list
          var ieAutoCompleteHackNeeded = ko.utils.ieVersion && isInputElement && element.type == "text"
                                         && element.autocomplete != "off" && (!element.form || element.form.autocomplete != "off");
          if (ieAutoCompleteHackNeeded && ko.utils.arrayIndexOf(eventsToCatch, "propertychange") == -1) {
              ko.utils.registerEventHandler(element, "propertychange", function () { propertyChangedFired = true });
              ko.utils.registerEventHandler(element, "focus", function () { propertyChangedFired = false });
              ko.utils.registerEventHandler(element, "blur", function() {
                  if (propertyChangedFired) {
                      valueUpdateHandler();
                  }
              });
          }
  
          ko.utils.arrayForEach(eventsToCatch, function(eventName) {
              // The syntax "after<eventname>" means "run the handler asynchronously after the event"
              // This is useful, for example, to catch "keydown" events after the browser has updated the control
              // (otherwise, ko.selectExtensions.readValue(this) will receive the control's value *before* the key event)
              var handler = valueUpdateHandler;
              if (ko.utils.stringStartsWith(eventName, "after")) {
                  handler = function() {
                      // The elementValueBeforeEvent variable is non-null *only* during the brief gap between
                      // a keyX event firing and the valueUpdateHandler running, which is scheduled to happen
                      // at the earliest asynchronous opportunity. We store this temporary information so that
                      // if, between keyX and valueUpdateHandler, the underlying model value changes separately,
                      // we can overwrite that model value change with the value the user just typed. Otherwise,
                      // techniques like rateLimit can trigger model changes at critical moments that will
                      // override the user's inputs, causing keystrokes to be lost.
                      elementValueBeforeEvent = ko.selectExtensions.readValue(element);
                      ko.utils.setTimeout(valueUpdateHandler, 0);
                  };
                  eventName = eventName.substring("after".length);
              }
              ko.utils.registerEventHandler(element, eventName, handler);
          });
  
          var updateFromModel;
  
          if (isInputElement && element.type == "file") {
              // For file input elements, can only write the empty string
              updateFromModel = function () {
                  var newValue = ko.utils.unwrapObservable(valueAccessor());
                  if (newValue === null || newValue === undefined || newValue === "") {
                      element.value = "";
                  } else {
                      ko.dependencyDetection.ignore(valueUpdateHandler);  // reset the model to match the element
                  }
              }
          } else {
              updateFromModel = function () {
                  var newValue = ko.utils.unwrapObservable(valueAccessor());
                  var elementValue = ko.selectExtensions.readValue(element);
  
                  if (elementValueBeforeEvent !== null && newValue === elementValueBeforeEvent) {
                      ko.utils.setTimeout(updateFromModel, 0);
                      return;
                  }
  
                  var valueHasChanged = newValue !== elementValue;
  
                  if (valueHasChanged || elementValue === undefined) {
                      if (tagName === "select") {
                          var allowUnset = allBindings.get('valueAllowUnset');
                          ko.selectExtensions.writeValue(element, newValue, allowUnset);
                          if (!allowUnset && newValue !== ko.selectExtensions.readValue(element)) {
                              // If you try to set a model value that can't be represented in an already-populated dropdown, reject that change,
                              // because you're not allowed to have a model value that disagrees with a visible UI selection.
                              ko.dependencyDetection.ignore(valueUpdateHandler);
                          }
                      } else {
                          ko.selectExtensions.writeValue(element, newValue);
                      }
                  }
              };
          }
  
          if (tagName === "select") {
              var updateFromModelComputed;
              ko.bindingEvent.subscribe(element, ko.bindingEvent.childrenComplete, function () {
                  if (!updateFromModelComputed) {
                      ko.utils.registerEventHandler(element, "change", valueUpdateHandler);
                      updateFromModelComputed = ko.computed(updateFromModel, null, { disposeWhenNodeIsRemoved: element });
                  } else if (allBindings.get('valueAllowUnset')) {
                      updateFromModel();
                  } else {
                      valueUpdateHandler();
                  }
              }, null, { 'notifyImmediately': true });
          } else {
              ko.utils.registerEventHandler(element, "change", valueUpdateHandler);
              ko.computed(updateFromModel, null, { disposeWhenNodeIsRemoved: element });
          }
      },
      'update': function() {} // Keep for backwards compatibility with code that may have wrapped value binding
  };
  ko.expressionRewriting.twoWayBindings['value'] = true;
  ko.bindingHandlers['visible'] = {
      'update': function (element, valueAccessor) {
          var value = ko.utils.unwrapObservable(valueAccessor());
          var isCurrentlyVisible = !(element.style.display == "none");
          if (value && !isCurrentlyVisible)
              element.style.display = "";
          else if ((!value) && isCurrentlyVisible)
              element.style.display = "none";
      }
  };
  
  ko.bindingHandlers['hidden'] = {
      'update': function (element, valueAccessor) {
          ko.bindingHandlers['visible']['update'](element, function() { return !ko.utils.unwrapObservable(valueAccessor()) });
      }
  };
  // 'click' is just a shorthand for the usual full-length event:{click:handler}
  makeEventHandlerShortcut('click');
  // If you want to make a custom template engine,
  //
  // [1] Inherit from this class (like ko.nativeTemplateEngine does)
  // [2] Override 'renderTemplateSource', supplying a function with this signature:
  //
  //        function (templateSource, bindingContext, options) {
  //            // - templateSource.text() is the text of the template you should render
  //            // - bindingContext.$data is the data you should pass into the template
  //            //   - you might also want to make bindingContext.$parent, bindingContext.$parents,
  //            //     and bindingContext.$root available in the template too
  //            // - options gives you access to any other properties set on "data-bind: { template: options }"
  //            // - templateDocument is the document object of the template
  //            //
  //            // Return value: an array of DOM nodes
  //        }
  //
  // [3] Override 'createJavaScriptEvaluatorBlock', supplying a function with this signature:
  //
  //        function (script) {
  //            // Return value: Whatever syntax means "Evaluate the JavaScript statement 'script' and output the result"
  //            //               For example, the jquery.tmpl template engine converts 'someScript' to '${ someScript }'
  //        }
  //
  //     This is only necessary if you want to allow data-bind attributes to reference arbitrary template variables.
  //     If you don't want to allow that, you can set the property 'allowTemplateRewriting' to false (like ko.nativeTemplateEngine does)
  //     and then you don't need to override 'createJavaScriptEvaluatorBlock'.
  
  ko.templateEngine = function () { };
  
  ko.templateEngine.prototype['renderTemplateSource'] = function (templateSource, bindingContext, options, templateDocument) {
      throw new Error("Override renderTemplateSource");
  };
  
  ko.templateEngine.prototype['createJavaScriptEvaluatorBlock'] = function (script) {
      throw new Error("Override createJavaScriptEvaluatorBlock");
  };
  
  ko.templateEngine.prototype['makeTemplateSource'] = function(template, templateDocument) {
      // Named template
      if (typeof template == "string") {
          templateDocument = templateDocument || document;
          var elem = templateDocument.getElementById(template);
          if (!elem)
              throw new Error("Cannot find template with ID " + template);
          return new ko.templateSources.domElement(elem);
      } else if ((template.nodeType == 1) || (template.nodeType == 8)) {
          // Anonymous template
          return new ko.templateSources.anonymousTemplate(template);
      } else
          throw new Error("Unknown template type: " + template);
  };
  
  ko.templateEngine.prototype['renderTemplate'] = function (template, bindingContext, options, templateDocument) {
      var templateSource = this['makeTemplateSource'](template, templateDocument);
      return this['renderTemplateSource'](templateSource, bindingContext, options, templateDocument);
  };
  
  ko.templateEngine.prototype['isTemplateRewritten'] = function (template, templateDocument) {
      // Skip rewriting if requested
      if (this['allowTemplateRewriting'] === false)
          return true;
      return this['makeTemplateSource'](template, templateDocument)['data']("isRewritten");
  };
  
  ko.templateEngine.prototype['rewriteTemplate'] = function (template, rewriterCallback, templateDocument) {
      var templateSource = this['makeTemplateSource'](template, templateDocument);
      var rewritten = rewriterCallback(templateSource['text']());
      templateSource['text'](rewritten);
      templateSource['data']("isRewritten", true);
  };
  
  ko.exportSymbol('templateEngine', ko.templateEngine);
  
  ko.templateRewriting = (function () {
      var memoizeDataBindingAttributeSyntaxRegex = /(<([a-z]+\d*)(?:\s+(?!data-bind\s*=\s*)[a-z0-9\-]+(?:=(?:\"[^\"]*\"|\'[^\']*\'|[^>]*))?)*\s+)data-bind\s*=\s*(["'])([\s\S]*?)\3/gi;
      var memoizeVirtualContainerBindingSyntaxRegex = /<!--\s*ko\b\s*([\s\S]*?)\s*-->/g;
  
      function validateDataBindValuesForRewriting(keyValueArray) {
          var allValidators = ko.expressionRewriting.bindingRewriteValidators;
          for (var i = 0; i < keyValueArray.length; i++) {
              var key = keyValueArray[i]['key'];
              if (Object.prototype.hasOwnProperty.call(allValidators, key)) {
                  var validator = allValidators[key];
  
                  if (typeof validator === "function") {
                      var possibleErrorMessage = validator(keyValueArray[i]['value']);
                      if (possibleErrorMessage)
                          throw new Error(possibleErrorMessage);
                  } else if (!validator) {
                      throw new Error("This template engine does not support the '" + key + "' binding within its templates");
                  }
              }
          }
      }
  
      function constructMemoizedTagReplacement(dataBindAttributeValue, tagToRetain, nodeName, templateEngine) {
          var dataBindKeyValueArray = ko.expressionRewriting.parseObjectLiteral(dataBindAttributeValue);
          validateDataBindValuesForRewriting(dataBindKeyValueArray);
          var rewrittenDataBindAttributeValue = ko.expressionRewriting.preProcessBindings(dataBindKeyValueArray, {'valueAccessors':true});
  
          // For no obvious reason, Opera fails to evaluate rewrittenDataBindAttributeValue unless it's wrapped in an additional
          // anonymous function, even though Opera's built-in debugger can evaluate it anyway. No other browser requires this
          // extra indirection.
          var applyBindingsToNextSiblingScript =
              "ko.__tr_ambtns(function($context,$element){return(function(){return{ " + rewrittenDataBindAttributeValue + " } })()},'" + nodeName.toLowerCase() + "')";
          return templateEngine['createJavaScriptEvaluatorBlock'](applyBindingsToNextSiblingScript) + tagToRetain;
      }
  
      return {
          ensureTemplateIsRewritten: function (template, templateEngine, templateDocument) {
              if (!templateEngine['isTemplateRewritten'](template, templateDocument))
                  templateEngine['rewriteTemplate'](template, function (htmlString) {
                      return ko.templateRewriting.memoizeBindingAttributeSyntax(htmlString, templateEngine);
                  }, templateDocument);
          },
  
          memoizeBindingAttributeSyntax: function (htmlString, templateEngine) {
              return htmlString.replace(memoizeDataBindingAttributeSyntaxRegex, function () {
                  return constructMemoizedTagReplacement(/* dataBindAttributeValue: */ arguments[4], /* tagToRetain: */ arguments[1], /* nodeName: */ arguments[2], templateEngine);
              }).replace(memoizeVirtualContainerBindingSyntaxRegex, function() {
                  return constructMemoizedTagReplacement(/* dataBindAttributeValue: */ arguments[1], /* tagToRetain: */ "<!-- ko -->", /* nodeName: */ "#comment", templateEngine);
              });
          },
  
          applyMemoizedBindingsToNextSibling: function (bindings, nodeName) {
              return ko.memoization.memoize(function (domNode, bindingContext) {
                  var nodeToBind = domNode.nextSibling;
                  if (nodeToBind && nodeToBind.nodeName.toLowerCase() === nodeName) {
                      ko.applyBindingAccessorsToNode(nodeToBind, bindings, bindingContext);
                  }
              });
          }
      }
  })();
  
  
  // Exported only because it has to be referenced by string lookup from within rewritten template
  ko.exportSymbol('__tr_ambtns', ko.templateRewriting.applyMemoizedBindingsToNextSibling);
  (function() {
      // A template source represents a read/write way of accessing a template. This is to eliminate the need for template loading/saving
      // logic to be duplicated in every template engine (and means they can all work with anonymous templates, etc.)
      //
      // Two are provided by default:
      //  1. ko.templateSources.domElement       - reads/writes the text content of an arbitrary DOM element
      //  2. ko.templateSources.anonymousElement - uses ko.utils.domData to read/write text *associated* with the DOM element, but
      //                                           without reading/writing the actual element text content, since it will be overwritten
      //                                           with the rendered template output.
      // You can implement your own template source if you want to fetch/store templates somewhere other than in DOM elements.
      // Template sources need to have the following functions:
      //   text() 			- returns the template text from your storage location
      //   text(value)		- writes the supplied template text to your storage location
      //   data(key)			- reads values stored using data(key, value) - see below
      //   data(key, value)	- associates "value" with this template and the key "key". Is used to store information like "isRewritten".
      //
      // Optionally, template sources can also have the following functions:
      //   nodes()            - returns a DOM element containing the nodes of this template, where available
      //   nodes(value)       - writes the given DOM element to your storage location
      // If a DOM element is available for a given template source, template engines are encouraged to use it in preference over text()
      // for improved speed. However, all templateSources must supply text() even if they don't supply nodes().
      //
      // Once you've implemented a templateSource, make your template engine use it by subclassing whatever template engine you were
      // using and overriding "makeTemplateSource" to return an instance of your custom template source.
  
      ko.templateSources = {};
  
      // ---- ko.templateSources.domElement -----
  
      // template types
      var templateScript = 1,
          templateTextArea = 2,
          templateTemplate = 3,
          templateElement = 4;
  
      ko.templateSources.domElement = function(element) {
          this.domElement = element;
  
          if (element) {
              var tagNameLower = ko.utils.tagNameLower(element);
              this.templateType =
                  tagNameLower === "script" ? templateScript :
                  tagNameLower === "textarea" ? templateTextArea :
                      // For browsers with proper <template> element support, where the .content property gives a document fragment
                  tagNameLower == "template" && element.content && element.content.nodeType === 11 ? templateTemplate :
                  templateElement;
          }
      }
  
      ko.templateSources.domElement.prototype['text'] = function(/* valueToWrite */) {
          var elemContentsProperty = this.templateType === templateScript ? "text"
                                   : this.templateType === templateTextArea ? "value"
                                   : "innerHTML";
  
          if (arguments.length == 0) {
              return this.domElement[elemContentsProperty];
          } else {
              var valueToWrite = arguments[0];
              if (elemContentsProperty === "innerHTML")
                  ko.utils.setHtml(this.domElement, valueToWrite);
              else
                  this.domElement[elemContentsProperty] = valueToWrite;
          }
      };
  
      var dataDomDataPrefix = ko.utils.domData.nextKey() + "_";
      ko.templateSources.domElement.prototype['data'] = function(key /*, valueToWrite */) {
          if (arguments.length === 1) {
              return ko.utils.domData.get(this.domElement, dataDomDataPrefix + key);
          } else {
              ko.utils.domData.set(this.domElement, dataDomDataPrefix + key, arguments[1]);
          }
      };
  
      var templatesDomDataKey = ko.utils.domData.nextKey();
      function getTemplateDomData(element) {
          return ko.utils.domData.get(element, templatesDomDataKey) || {};
      }
      function setTemplateDomData(element, data) {
          ko.utils.domData.set(element, templatesDomDataKey, data);
      }
  
      ko.templateSources.domElement.prototype['nodes'] = function(/* valueToWrite */) {
          var element = this.domElement;
          if (arguments.length == 0) {
              var templateData = getTemplateDomData(element),
                  nodes = templateData.containerData || (
                      this.templateType === templateTemplate ? element.content :
                      this.templateType === templateElement ? element :
                      undefined);
              if (!nodes || templateData.alwaysCheckText) {
                  // If the template is associated with an element that stores the template as text,
                  // parse and cache the nodes whenever there's new text content available. This allows
                  // the user to update the template content by updating the text of template node.
                  var text = this['text']();
                  if (text && text !== templateData.textData) {
                      nodes = ko.utils.parseHtmlForTemplateNodes(text, element.ownerDocument);
                      setTemplateDomData(element, {containerData: nodes, textData: text, alwaysCheckText: true});
                  }
              }
              return nodes;
          } else {
              var valueToWrite = arguments[0];
              if (this.templateType !== undefined) {
                  this['text']("");   // clear the text from the node
              }
              setTemplateDomData(element, {containerData: valueToWrite});
          }
      };
  
      // ---- ko.templateSources.anonymousTemplate -----
      // Anonymous templates are normally saved/retrieved as DOM nodes through "nodes".
      // For compatibility, you can also read "text"; it will be serialized from the nodes on demand.
      // Writing to "text" is still supported, but then the template data will not be available as DOM nodes.
  
      ko.templateSources.anonymousTemplate = function(element) {
          this.domElement = element;
      }
      ko.templateSources.anonymousTemplate.prototype = new ko.templateSources.domElement();
      ko.templateSources.anonymousTemplate.prototype.constructor = ko.templateSources.anonymousTemplate;
      ko.templateSources.anonymousTemplate.prototype['text'] = function(/* valueToWrite */) {
          if (arguments.length == 0) {
              var templateData = getTemplateDomData(this.domElement);
              if (templateData.textData === undefined && templateData.containerData)
                  templateData.textData = templateData.containerData.innerHTML;
              return templateData.textData;
          } else {
              var valueToWrite = arguments[0];
              setTemplateDomData(this.domElement, {textData: valueToWrite});
          }
      };
  
      ko.exportSymbol('templateSources', ko.templateSources);
      ko.exportSymbol('templateSources.domElement', ko.templateSources.domElement);
      ko.exportSymbol('templateSources.anonymousTemplate', ko.templateSources.anonymousTemplate);
  })();
  (function () {
      var _templateEngine;
      ko.setTemplateEngine = function (templateEngine) {
          if ((templateEngine != undefined) && !(templateEngine instanceof ko.templateEngine))
              throw new Error("templateEngine must inherit from ko.templateEngine");
          _templateEngine = templateEngine;
      }
  
      function invokeForEachNodeInContinuousRange(firstNode, lastNode, action) {
          var node, nextInQueue = firstNode, firstOutOfRangeNode = ko.virtualElements.nextSibling(lastNode);
          while (nextInQueue && ((node = nextInQueue) !== firstOutOfRangeNode)) {
              nextInQueue = ko.virtualElements.nextSibling(node);
              action(node, nextInQueue);
          }
      }
  
      function activateBindingsOnContinuousNodeArray(continuousNodeArray, bindingContext) {
          // To be used on any nodes that have been rendered by a template and have been inserted into some parent element
          // Walks through continuousNodeArray (which *must* be continuous, i.e., an uninterrupted sequence of sibling nodes, because
          // the algorithm for walking them relies on this), and for each top-level item in the virtual-element sense,
          // (1) Does a regular "applyBindings" to associate bindingContext with this node and to activate any non-memoized bindings
          // (2) Unmemoizes any memos in the DOM subtree (e.g., to activate bindings that had been memoized during template rewriting)
  
          if (continuousNodeArray.length) {
              var firstNode = continuousNodeArray[0],
                  lastNode = continuousNodeArray[continuousNodeArray.length - 1],
                  parentNode = firstNode.parentNode,
                  provider = ko.bindingProvider['instance'],
                  preprocessNode = provider['preprocessNode'];
  
              if (preprocessNode) {
                  invokeForEachNodeInContinuousRange(firstNode, lastNode, function(node, nextNodeInRange) {
                      var nodePreviousSibling = node.previousSibling;
                      var newNodes = preprocessNode.call(provider, node);
                      if (newNodes) {
                          if (node === firstNode)
                              firstNode = newNodes[0] || nextNodeInRange;
                          if (node === lastNode)
                              lastNode = newNodes[newNodes.length - 1] || nodePreviousSibling;
                      }
                  });
  
                  // Because preprocessNode can change the nodes, including the first and last nodes, update continuousNodeArray to match.
                  // We need the full set, including inner nodes, because the unmemoize step might remove the first node (and so the real
                  // first node needs to be in the array).
                  continuousNodeArray.length = 0;
                  if (!firstNode) { // preprocessNode might have removed all the nodes, in which case there's nothing left to do
                      return;
                  }
                  if (firstNode === lastNode) {
                      continuousNodeArray.push(firstNode);
                  } else {
                      continuousNodeArray.push(firstNode, lastNode);
                      ko.utils.fixUpContinuousNodeArray(continuousNodeArray, parentNode);
                  }
              }
  
              // Need to applyBindings *before* unmemoziation, because unmemoization might introduce extra nodes (that we don't want to re-bind)
              // whereas a regular applyBindings won't introduce new memoized nodes
              invokeForEachNodeInContinuousRange(firstNode, lastNode, function(node) {
                  if (node.nodeType === 1 || node.nodeType === 8)
                      ko.applyBindings(bindingContext, node);
              });
              invokeForEachNodeInContinuousRange(firstNode, lastNode, function(node) {
                  if (node.nodeType === 1 || node.nodeType === 8)
                      ko.memoization.unmemoizeDomNodeAndDescendants(node, [bindingContext]);
              });
  
              // Make sure any changes done by applyBindings or unmemoize are reflected in the array
              ko.utils.fixUpContinuousNodeArray(continuousNodeArray, parentNode);
          }
      }
  
      function getFirstNodeFromPossibleArray(nodeOrNodeArray) {
          return nodeOrNodeArray.nodeType ? nodeOrNodeArray
                                          : nodeOrNodeArray.length > 0 ? nodeOrNodeArray[0]
                                          : null;
      }
  
      function executeTemplate(targetNodeOrNodeArray, renderMode, template, bindingContext, options) {
          options = options || {};
          var firstTargetNode = targetNodeOrNodeArray && getFirstNodeFromPossibleArray(targetNodeOrNodeArray);
          var templateDocument = (firstTargetNode || template || {}).ownerDocument;
          var templateEngineToUse = (options['templateEngine'] || _templateEngine);
          ko.templateRewriting.ensureTemplateIsRewritten(template, templateEngineToUse, templateDocument);
          var renderedNodesArray = templateEngineToUse['renderTemplate'](template, bindingContext, options, templateDocument);
  
          // Loosely check result is an array of DOM nodes
          if ((typeof renderedNodesArray.length != "number") || (renderedNodesArray.length > 0 && typeof renderedNodesArray[0].nodeType != "number"))
              throw new Error("Template engine must return an array of DOM nodes");
  
          var haveAddedNodesToParent = false;
          switch (renderMode) {
              case "replaceChildren":
                  ko.virtualElements.setDomNodeChildren(targetNodeOrNodeArray, renderedNodesArray);
                  haveAddedNodesToParent = true;
                  break;
              case "replaceNode":
                  ko.utils.replaceDomNodes(targetNodeOrNodeArray, renderedNodesArray);
                  haveAddedNodesToParent = true;
                  break;
              case "ignoreTargetNode": break;
              default:
                  throw new Error("Unknown renderMode: " + renderMode);
          }
  
          if (haveAddedNodesToParent) {
              activateBindingsOnContinuousNodeArray(renderedNodesArray, bindingContext);
              if (options['afterRender']) {
                  ko.dependencyDetection.ignore(options['afterRender'], null, [renderedNodesArray, bindingContext[options['as'] || '$data']]);
              }
              if (renderMode == "replaceChildren") {
                  ko.bindingEvent.notify(targetNodeOrNodeArray, ko.bindingEvent.childrenComplete);
              }
          }
  
          return renderedNodesArray;
      }
  
      function resolveTemplateName(template, data, context) {
          // The template can be specified as:
          if (ko.isObservable(template)) {
              // 1. An observable, with string value
              return template();
          } else if (typeof template === 'function') {
              // 2. A function of (data, context) returning a string
              return template(data, context);
          } else {
              // 3. A string
              return template;
          }
      }
  
      ko.renderTemplate = function (template, dataOrBindingContext, options, targetNodeOrNodeArray, renderMode) {
          options = options || {};
          if ((options['templateEngine'] || _templateEngine) == undefined)
              throw new Error("Set a template engine before calling renderTemplate");
          renderMode = renderMode || "replaceChildren";
  
          if (targetNodeOrNodeArray) {
              var firstTargetNode = getFirstNodeFromPossibleArray(targetNodeOrNodeArray);
  
              var whenToDispose = function () { return (!firstTargetNode) || !ko.utils.domNodeIsAttachedToDocument(firstTargetNode); }; // Passive disposal (on next evaluation)
              var activelyDisposeWhenNodeIsRemoved = (firstTargetNode && renderMode == "replaceNode") ? firstTargetNode.parentNode : firstTargetNode;
  
              return ko.dependentObservable( // So the DOM is automatically updated when any dependency changes
                  function () {
                      // Ensure we've got a proper binding context to work with
                      var bindingContext = (dataOrBindingContext && (dataOrBindingContext instanceof ko.bindingContext))
                          ? dataOrBindingContext
                          : new ko.bindingContext(dataOrBindingContext, null, null, null, { "exportDependencies": true });
  
                      var templateName = resolveTemplateName(template, bindingContext['$data'], bindingContext),
                          renderedNodesArray = executeTemplate(targetNodeOrNodeArray, renderMode, templateName, bindingContext, options);
  
                      if (renderMode == "replaceNode") {
                          targetNodeOrNodeArray = renderedNodesArray;
                          firstTargetNode = getFirstNodeFromPossibleArray(targetNodeOrNodeArray);
                      }
                  },
                  null,
                  { disposeWhen: whenToDispose, disposeWhenNodeIsRemoved: activelyDisposeWhenNodeIsRemoved }
              );
          } else {
              // We don't yet have a DOM node to evaluate, so use a memo and render the template later when there is a DOM node
              return ko.memoization.memoize(function (domNode) {
                  ko.renderTemplate(template, dataOrBindingContext, options, domNode, "replaceNode");
              });
          }
      };
  
      ko.renderTemplateForEach = function (template, arrayOrObservableArray, options, targetNode, parentBindingContext) {
          // Since setDomNodeChildrenFromArrayMapping always calls executeTemplateForArrayItem and then
          // activateBindingsCallback for added items, we can store the binding context in the former to use in the latter.
          var arrayItemContext, asName = options['as'];
  
          // This will be called by setDomNodeChildrenFromArrayMapping to get the nodes to add to targetNode
          var executeTemplateForArrayItem = function (arrayValue, index) {
              // Support selecting template as a function of the data being rendered
              arrayItemContext = parentBindingContext['createChildContext'](arrayValue, {
                  'as': asName,
                  'noChildContext': options['noChildContext'],
                  'extend': function(context) {
                      context['$index'] = index;
                      if (asName) {
                          context[asName + "Index"] = index;
                      }
                  }
              });
  
              var templateName = resolveTemplateName(template, arrayValue, arrayItemContext);
              return executeTemplate(targetNode, "ignoreTargetNode", templateName, arrayItemContext, options);
          };
  
          // This will be called whenever setDomNodeChildrenFromArrayMapping has added nodes to targetNode
          var activateBindingsCallback = function(arrayValue, addedNodesArray, index) {
              activateBindingsOnContinuousNodeArray(addedNodesArray, arrayItemContext);
              if (options['afterRender'])
                  options['afterRender'](addedNodesArray, arrayValue);
  
              // release the "cache" variable, so that it can be collected by
              // the GC when its value isn't used from within the bindings anymore.
              arrayItemContext = null;
          };
  
          var setDomNodeChildrenFromArrayMapping = function (newArray, changeList) {
              // Call setDomNodeChildrenFromArrayMapping, ignoring any observables unwrapped within (most likely from a callback function).
              // If the array items are observables, though, they will be unwrapped in executeTemplateForArrayItem and managed within setDomNodeChildrenFromArrayMapping.
              ko.dependencyDetection.ignore(ko.utils.setDomNodeChildrenFromArrayMapping, null, [targetNode, newArray, executeTemplateForArrayItem, options, activateBindingsCallback, changeList]);
              ko.bindingEvent.notify(targetNode, ko.bindingEvent.childrenComplete);
          };
  
          var shouldHideDestroyed = (options['includeDestroyed'] === false) || (ko.options['foreachHidesDestroyed'] && !options['includeDestroyed']);
  
          if (!shouldHideDestroyed && !options['beforeRemove'] && ko.isObservableArray(arrayOrObservableArray)) {
              setDomNodeChildrenFromArrayMapping(arrayOrObservableArray.peek());
  
              var subscription = arrayOrObservableArray.subscribe(function (changeList) {
                  setDomNodeChildrenFromArrayMapping(arrayOrObservableArray(), changeList);
              }, null, "arrayChange");
              subscription.disposeWhenNodeIsRemoved(targetNode);
  
              return subscription;
          } else {
              return ko.dependentObservable(function () {
                  var unwrappedArray = ko.utils.unwrapObservable(arrayOrObservableArray) || [];
                  if (typeof unwrappedArray.length == "undefined") // Coerce single value into array
                      unwrappedArray = [unwrappedArray];
  
                  if (shouldHideDestroyed) {
                      // Filter out any entries marked as destroyed
                      unwrappedArray = ko.utils.arrayFilter(unwrappedArray, function(item) {
                          return item === undefined || item === null || !ko.utils.unwrapObservable(item['_destroy']);
                      });
                  }
                  setDomNodeChildrenFromArrayMapping(unwrappedArray);
  
              }, null, { disposeWhenNodeIsRemoved: targetNode });
          }
      };
  
      var templateComputedDomDataKey = ko.utils.domData.nextKey();
      function disposeOldComputedAndStoreNewOne(element, newComputed) {
          var oldComputed = ko.utils.domData.get(element, templateComputedDomDataKey);
          if (oldComputed && (typeof(oldComputed.dispose) == 'function'))
              oldComputed.dispose();
          ko.utils.domData.set(element, templateComputedDomDataKey, (newComputed && (!newComputed.isActive || newComputed.isActive())) ? newComputed : undefined);
      }
  
      var cleanContainerDomDataKey = ko.utils.domData.nextKey();
      ko.bindingHandlers['template'] = {
          'init': function(element, valueAccessor) {
              // Support anonymous templates
              var bindingValue = ko.utils.unwrapObservable(valueAccessor());
              if (typeof bindingValue == "string" || 'name' in bindingValue) {
                  // It's a named template - clear the element
                  ko.virtualElements.emptyNode(element);
              } else if ('nodes' in bindingValue) {
                  // We've been given an array of DOM nodes. Save them as the template source.
                  // There is no known use case for the node array being an observable array (if the output
                  // varies, put that behavior *into* your template - that's what templates are for), and
                  // the implementation would be a mess, so assert that it's not observable.
                  var nodes = bindingValue['nodes'] || [];
                  if (ko.isObservable(nodes)) {
                      throw new Error('The "nodes" option must be a plain, non-observable array.');
                  }
  
                  // If the nodes are already attached to a KO-generated container, we reuse that container without moving the
                  // elements to a new one (we check only the first node, as the nodes are always moved together)
                  var container = nodes[0] && nodes[0].parentNode;
                  if (!container || !ko.utils.domData.get(container, cleanContainerDomDataKey)) {
                      container = ko.utils.moveCleanedNodesToContainerElement(nodes);
                      ko.utils.domData.set(container, cleanContainerDomDataKey, true);
                  }
  
                  new ko.templateSources.anonymousTemplate(element)['nodes'](container);
              } else {
                  // It's an anonymous template - store the element contents, then clear the element
                  var templateNodes = ko.virtualElements.childNodes(element);
                  if (templateNodes.length > 0) {
                      var container = ko.utils.moveCleanedNodesToContainerElement(templateNodes); // This also removes the nodes from their current parent
                      new ko.templateSources.anonymousTemplate(element)['nodes'](container);
                  } else {
                      throw new Error("Anonymous template defined, but no template content was provided");
                  }
              }
              return { 'controlsDescendantBindings': true };
          },
          'update': function (element, valueAccessor, allBindings, viewModel, bindingContext) {
              var value = valueAccessor(),
                  options = ko.utils.unwrapObservable(value),
                  shouldDisplay = true,
                  templateComputed = null,
                  template;
  
              if (typeof options == "string") {
                  template = value;
                  options = {};
              } else {
                  template = 'name' in options ? options['name'] : element;
  
                  // Support "if"/"ifnot" conditions
                  if ('if' in options)
                      shouldDisplay = ko.utils.unwrapObservable(options['if']);
                  if (shouldDisplay && 'ifnot' in options)
                      shouldDisplay = !ko.utils.unwrapObservable(options['ifnot']);
  
                  // Don't show anything if an empty name is given (see #2446)
                  if (shouldDisplay && !template) {
                      shouldDisplay = false;
                  }
              }
  
              if ('foreach' in options) {
                  // Render once for each data point (treating data set as empty if shouldDisplay==false)
                  var dataArray = (shouldDisplay && options['foreach']) || [];
                  templateComputed = ko.renderTemplateForEach(template, dataArray, options, element, bindingContext);
              } else if (!shouldDisplay) {
                  ko.virtualElements.emptyNode(element);
              } else {
                  // Render once for this single data point (or use the viewModel if no data was provided)
                  var innerBindingContext = bindingContext;
                  if ('data' in options) {
                      innerBindingContext = bindingContext['createChildContext'](options['data'], {
                          'as': options['as'],
                          'noChildContext': options['noChildContext'],
                          'exportDependencies': true
                      });
                  }
                  templateComputed = ko.renderTemplate(template, innerBindingContext, options, element);
              }
  
              // It only makes sense to have a single template computed per element (otherwise which one should have its output displayed?)
              disposeOldComputedAndStoreNewOne(element, templateComputed);
          }
      };
  
      // Anonymous templates can't be rewritten. Give a nice error message if you try to do it.
      ko.expressionRewriting.bindingRewriteValidators['template'] = function(bindingValue) {
          var parsedBindingValue = ko.expressionRewriting.parseObjectLiteral(bindingValue);
  
          if ((parsedBindingValue.length == 1) && parsedBindingValue[0]['unknown'])
              return null; // It looks like a string literal, not an object literal, so treat it as a named template (which is allowed for rewriting)
  
          if (ko.expressionRewriting.keyValueArrayContainsKey(parsedBindingValue, "name"))
              return null; // Named templates can be rewritten, so return "no error"
          return "This template engine does not support anonymous templates nested within its templates";
      };
  
      ko.virtualElements.allowedBindings['template'] = true;
  })();
  
  ko.exportSymbol('setTemplateEngine', ko.setTemplateEngine);
  ko.exportSymbol('renderTemplate', ko.renderTemplate);
  // Go through the items that have been added and deleted and try to find matches between them.
  ko.utils.findMovesInArrayComparison = function (left, right, limitFailedCompares) {
      if (left.length && right.length) {
          var failedCompares, l, r, leftItem, rightItem;
          for (failedCompares = l = 0; (!limitFailedCompares || failedCompares < limitFailedCompares) && (leftItem = left[l]); ++l) {
              for (r = 0; rightItem = right[r]; ++r) {
                  if (leftItem['value'] === rightItem['value']) {
                      leftItem['moved'] = rightItem['index'];
                      rightItem['moved'] = leftItem['index'];
                      right.splice(r, 1);         // This item is marked as moved; so remove it from right list
                      failedCompares = r = 0;     // Reset failed compares count because we're checking for consecutive failures
                      break;
                  }
              }
              failedCompares += r;
          }
      }
  };
  
  ko.utils.compareArrays = (function () {
      var statusNotInOld = 'added', statusNotInNew = 'deleted';
  
      // Simple calculation based on Levenshtein distance.
      function compareArrays(oldArray, newArray, options) {
          // For backward compatibility, if the third arg is actually a bool, interpret
          // it as the old parameter 'dontLimitMoves'. Newer code should use { dontLimitMoves: true }.
          options = (typeof options === 'boolean') ? { 'dontLimitMoves': options } : (options || {});
          oldArray = oldArray || [];
          newArray = newArray || [];
  
          if (oldArray.length < newArray.length)
              return compareSmallArrayToBigArray(oldArray, newArray, statusNotInOld, statusNotInNew, options);
          else
              return compareSmallArrayToBigArray(newArray, oldArray, statusNotInNew, statusNotInOld, options);
      }
  
      function compareSmallArrayToBigArray(smlArray, bigArray, statusNotInSml, statusNotInBig, options) {
          var myMin = Math.min,
              myMax = Math.max,
              editDistanceMatrix = [],
              smlIndex, smlIndexMax = smlArray.length,
              bigIndex, bigIndexMax = bigArray.length,
              compareRange = (bigIndexMax - smlIndexMax) || 1,
              maxDistance = smlIndexMax + bigIndexMax + 1,
              thisRow, lastRow,
              bigIndexMaxForRow, bigIndexMinForRow;
  
          for (smlIndex = 0; smlIndex <= smlIndexMax; smlIndex++) {
              lastRow = thisRow;
              editDistanceMatrix.push(thisRow = []);
              bigIndexMaxForRow = myMin(bigIndexMax, smlIndex + compareRange);
              bigIndexMinForRow = myMax(0, smlIndex - 1);
              for (bigIndex = bigIndexMinForRow; bigIndex <= bigIndexMaxForRow; bigIndex++) {
                  if (!bigIndex)
                      thisRow[bigIndex] = smlIndex + 1;
                  else if (!smlIndex)  // Top row - transform empty array into new array via additions
                      thisRow[bigIndex] = bigIndex + 1;
                  else if (smlArray[smlIndex - 1] === bigArray[bigIndex - 1])
                      thisRow[bigIndex] = lastRow[bigIndex - 1];                  // copy value (no edit)
                  else {
                      var northDistance = lastRow[bigIndex] || maxDistance;       // not in big (deletion)
                      var westDistance = thisRow[bigIndex - 1] || maxDistance;    // not in small (addition)
                      thisRow[bigIndex] = myMin(northDistance, westDistance) + 1;
                  }
              }
          }
  
          var editScript = [], meMinusOne, notInSml = [], notInBig = [];
          for (smlIndex = smlIndexMax, bigIndex = bigIndexMax; smlIndex || bigIndex;) {
              meMinusOne = editDistanceMatrix[smlIndex][bigIndex] - 1;
              if (bigIndex && meMinusOne === editDistanceMatrix[smlIndex][bigIndex-1]) {
                  notInSml.push(editScript[editScript.length] = {     // added
                      'status': statusNotInSml,
                      'value': bigArray[--bigIndex],
                      'index': bigIndex });
              } else if (smlIndex && meMinusOne === editDistanceMatrix[smlIndex - 1][bigIndex]) {
                  notInBig.push(editScript[editScript.length] = {     // deleted
                      'status': statusNotInBig,
                      'value': smlArray[--smlIndex],
                      'index': smlIndex });
              } else {
                  --bigIndex;
                  --smlIndex;
                  if (!options['sparse']) {
                      editScript.push({
                          'status': "retained",
                          'value': bigArray[bigIndex] });
                  }
              }
          }
  
          // Set a limit on the number of consecutive non-matching comparisons; having it a multiple of
          // smlIndexMax keeps the time complexity of this algorithm linear.
          ko.utils.findMovesInArrayComparison(notInBig, notInSml, !options['dontLimitMoves'] && smlIndexMax * 10);
  
          return editScript.reverse();
      }
  
      return compareArrays;
  })();
  
  ko.exportSymbol('utils.compareArrays', ko.utils.compareArrays);
  (function () {
      // Objective:
      // * Given an input array, a container DOM node, and a function from array elements to arrays of DOM nodes,
      //   map the array elements to arrays of DOM nodes, concatenate together all these arrays, and use them to populate the container DOM node
      // * Next time we're given the same combination of things (with the array possibly having mutated), update the container DOM node
      //   so that its children is again the concatenation of the mappings of the array elements, but don't re-map any array elements that we
      //   previously mapped - retain those nodes, and just insert/delete other ones
  
      // "callbackAfterAddingNodes" will be invoked after any "mapping"-generated nodes are inserted into the container node
      // You can use this, for example, to activate bindings on those nodes.
  
      function mapNodeAndRefreshWhenChanged(containerNode, mapping, valueToMap, callbackAfterAddingNodes, index) {
          // Map this array value inside a dependentObservable so we re-map when any dependency changes
          var mappedNodes = [];
          var dependentObservable = ko.dependentObservable(function() {
              var newMappedNodes = mapping(valueToMap, index, ko.utils.fixUpContinuousNodeArray(mappedNodes, containerNode)) || [];
  
              // On subsequent evaluations, just replace the previously-inserted DOM nodes
              if (mappedNodes.length > 0) {
                  ko.utils.replaceDomNodes(mappedNodes, newMappedNodes);
                  if (callbackAfterAddingNodes)
                      ko.dependencyDetection.ignore(callbackAfterAddingNodes, null, [valueToMap, newMappedNodes, index]);
              }
  
              // Replace the contents of the mappedNodes array, thereby updating the record
              // of which nodes would be deleted if valueToMap was itself later removed
              mappedNodes.length = 0;
              ko.utils.arrayPushAll(mappedNodes, newMappedNodes);
          }, null, { disposeWhenNodeIsRemoved: containerNode, disposeWhen: function() { return !ko.utils.anyDomNodeIsAttachedToDocument(mappedNodes); } });
          return { mappedNodes : mappedNodes, dependentObservable : (dependentObservable.isActive() ? dependentObservable : undefined) };
      }
  
      var lastMappingResultDomDataKey = ko.utils.domData.nextKey(),
          deletedItemDummyValue = ko.utils.domData.nextKey();
  
      ko.utils.setDomNodeChildrenFromArrayMapping = function (domNode, array, mapping, options, callbackAfterAddingNodes, editScript) {
          array = array || [];
          if (typeof array.length == "undefined") // Coerce single value into array
              array = [array];
  
          options = options || {};
          var lastMappingResult = ko.utils.domData.get(domNode, lastMappingResultDomDataKey);
          var isFirstExecution = !lastMappingResult;
  
          // Build the new mapping result
          var newMappingResult = [];
          var lastMappingResultIndex = 0;
          var currentArrayIndex = 0;
  
          var nodesToDelete = [];
          var itemsToMoveFirstIndexes = [];
          var itemsForBeforeRemoveCallbacks = [];
          var itemsForMoveCallbacks = [];
          var itemsForAfterAddCallbacks = [];
          var mapData;
          var countWaitingForRemove = 0;
  
          function itemAdded(value) {
              mapData = { arrayEntry: value, indexObservable: ko.observable(currentArrayIndex++) };
              newMappingResult.push(mapData);
              if (!isFirstExecution) {
                  itemsForAfterAddCallbacks.push(mapData);
              }
          }
  
          function itemMovedOrRetained(oldPosition) {
              mapData = lastMappingResult[oldPosition];
              if (currentArrayIndex !== mapData.indexObservable.peek())
                  itemsForMoveCallbacks.push(mapData);
              // Since updating the index might change the nodes, do so before calling fixUpContinuousNodeArray
              mapData.indexObservable(currentArrayIndex++);
              ko.utils.fixUpContinuousNodeArray(mapData.mappedNodes, domNode);
              newMappingResult.push(mapData);
          }
  
          function callCallback(callback, items) {
              if (callback) {
                  for (var i = 0, n = items.length; i < n; i++) {
                      ko.utils.arrayForEach(items[i].mappedNodes, function(node) {
                          callback(node, i, items[i].arrayEntry);
                      });
                  }
              }
          }
  
          if (isFirstExecution) {
              ko.utils.arrayForEach(array, itemAdded);
          } else {
              if (!editScript || (lastMappingResult && lastMappingResult['_countWaitingForRemove'])) {
                  // Compare the provided array against the previous one
                  var lastArray = ko.utils.arrayMap(lastMappingResult, function (x) { return x.arrayEntry; }),
                      compareOptions = {
                          'dontLimitMoves': options['dontLimitMoves'],
                          'sparse': true
                      };
                  editScript = ko.utils.compareArrays(lastArray, array, compareOptions);
              }
  
              for (var i = 0, editScriptItem, movedIndex, itemIndex; editScriptItem = editScript[i]; i++) {
                  movedIndex = editScriptItem['moved'];
                  itemIndex = editScriptItem['index'];
                  switch (editScriptItem['status']) {
                      case "deleted":
                          while (lastMappingResultIndex < itemIndex) {
                              itemMovedOrRetained(lastMappingResultIndex++);
                          }
                          if (movedIndex === undefined) {
                              mapData = lastMappingResult[lastMappingResultIndex];
  
                              // Stop tracking changes to the mapping for these nodes
                              if (mapData.dependentObservable) {
                                  mapData.dependentObservable.dispose();
                                  mapData.dependentObservable = undefined;
                              }
  
                              // Queue these nodes for later removal
                              if (ko.utils.fixUpContinuousNodeArray(mapData.mappedNodes, domNode).length) {
                                  if (options['beforeRemove']) {
                                      newMappingResult.push(mapData);
                                      countWaitingForRemove++;
                                      if (mapData.arrayEntry === deletedItemDummyValue) {
                                          mapData = null;
                                      } else {
                                          itemsForBeforeRemoveCallbacks.push(mapData);
                                      }
                                  }
                                  if (mapData) {
                                      nodesToDelete.push.apply(nodesToDelete, mapData.mappedNodes);
                                  }
                              }
                          }
                          lastMappingResultIndex++;
                          break;
  
                      case "added":
                          while (currentArrayIndex < itemIndex) {
                              itemMovedOrRetained(lastMappingResultIndex++);
                          }
                          if (movedIndex !== undefined) {
                              itemsToMoveFirstIndexes.push(newMappingResult.length);
                              itemMovedOrRetained(movedIndex);
                          } else {
                              itemAdded(editScriptItem['value']);
                          }
                          break;
                  }
              }
  
              while (currentArrayIndex < array.length) {
                  itemMovedOrRetained(lastMappingResultIndex++);
              }
  
              // Record that the current view may still contain deleted items
              // because it means we won't be able to use a provided editScript.
              newMappingResult['_countWaitingForRemove'] = countWaitingForRemove;
          }
  
          // Store a copy of the array items we just considered so we can difference it next time
          ko.utils.domData.set(domNode, lastMappingResultDomDataKey, newMappingResult);
  
          // Call beforeMove first before any changes have been made to the DOM
          callCallback(options['beforeMove'], itemsForMoveCallbacks);
  
          // Next remove nodes for deleted items (or just clean if there's a beforeRemove callback)
          ko.utils.arrayForEach(nodesToDelete, options['beforeRemove'] ? ko.cleanNode : ko.removeNode);
  
          var i, j, lastNode, nodeToInsert, mappedNodes, activeElement;
  
          // Since most browsers remove the focus from an element when it's moved to another location,
          // save the focused element and try to restore it later.
          try {
              activeElement = domNode.ownerDocument.activeElement;
          } catch(e) {
              // IE9 throws if you access activeElement during page load (see issue #703)
          }
  
          // Try to reduce overall moved nodes by first moving the ones that were marked as moved by the edit script
          if (itemsToMoveFirstIndexes.length) {
              while ((i = itemsToMoveFirstIndexes.shift()) != undefined) {
                  mapData = newMappingResult[i];
                  for (lastNode = undefined; i; ) {
                      if ((mappedNodes = newMappingResult[--i].mappedNodes) && mappedNodes.length) {
                          lastNode = mappedNodes[mappedNodes.length-1];
                          break;
                      }
                  }
                  for (j = 0; nodeToInsert = mapData.mappedNodes[j]; lastNode = nodeToInsert, j++) {
                      ko.virtualElements.insertAfter(domNode, nodeToInsert, lastNode);
                  }
              }
          }
  
          // Next add/reorder the remaining items (will include deleted items if there's a beforeRemove callback)
          for (i = 0; mapData = newMappingResult[i]; i++) {
              // Get nodes for newly added items
              if (!mapData.mappedNodes)
                  ko.utils.extend(mapData, mapNodeAndRefreshWhenChanged(domNode, mapping, mapData.arrayEntry, callbackAfterAddingNodes, mapData.indexObservable));
  
              // Put nodes in the right place if they aren't there already
              for (j = 0; nodeToInsert = mapData.mappedNodes[j]; lastNode = nodeToInsert, j++) {
                  ko.virtualElements.insertAfter(domNode, nodeToInsert, lastNode);
              }
  
              // Run the callbacks for newly added nodes (for example, to apply bindings, etc.)
              if (!mapData.initialized && callbackAfterAddingNodes) {
                  callbackAfterAddingNodes(mapData.arrayEntry, mapData.mappedNodes, mapData.indexObservable);
                  mapData.initialized = true;
                  lastNode = mapData.mappedNodes[mapData.mappedNodes.length - 1];     // get the last node again since it may have been changed by a preprocessor
              }
          }
  
          // Restore the focused element if it had lost focus
          if (activeElement && domNode.ownerDocument.activeElement != activeElement) {
              activeElement.focus();
          }
  
          // If there's a beforeRemove callback, call it after reordering.
          // Note that we assume that the beforeRemove callback will usually be used to remove the nodes using
          // some sort of animation, which is why we first reorder the nodes that will be removed. If the
          // callback instead removes the nodes right away, it would be more efficient to skip reordering them.
          // Perhaps we'll make that change in the future if this scenario becomes more common.
          callCallback(options['beforeRemove'], itemsForBeforeRemoveCallbacks);
  
          // Replace the stored values of deleted items with a dummy value. This provides two benefits: it marks this item
          // as already "removed" so we won't call beforeRemove for it again, and it ensures that the item won't match up
          // with an actual item in the array and appear as "retained" or "moved".
          for (i = 0; i < itemsForBeforeRemoveCallbacks.length; ++i) {
              itemsForBeforeRemoveCallbacks[i].arrayEntry = deletedItemDummyValue;
          }
  
          // Finally call afterMove and afterAdd callbacks
          callCallback(options['afterMove'], itemsForMoveCallbacks);
          callCallback(options['afterAdd'], itemsForAfterAddCallbacks);
      }
  })();
  
  ko.exportSymbol('utils.setDomNodeChildrenFromArrayMapping', ko.utils.setDomNodeChildrenFromArrayMapping);
  ko.nativeTemplateEngine = function () {
      this['allowTemplateRewriting'] = false;
  }
  
  ko.nativeTemplateEngine.prototype = new ko.templateEngine();
  ko.nativeTemplateEngine.prototype.constructor = ko.nativeTemplateEngine;
  ko.nativeTemplateEngine.prototype['renderTemplateSource'] = function (templateSource, bindingContext, options, templateDocument) {
      var useNodesIfAvailable = !(ko.utils.ieVersion < 9), // IE<9 cloneNode doesn't work properly
          templateNodesFunc = useNodesIfAvailable ? templateSource['nodes'] : null,
          templateNodes = templateNodesFunc ? templateSource['nodes']() : null;
  
      if (templateNodes) {
          return ko.utils.makeArray(templateNodes.cloneNode(true).childNodes);
      } else {
          var templateText = templateSource['text']();
          return ko.utils.parseHtmlFragment(templateText, templateDocument);
      }
  };
  
  ko.nativeTemplateEngine.instance = new ko.nativeTemplateEngine();
  ko.setTemplateEngine(ko.nativeTemplateEngine.instance);
  
  ko.exportSymbol('nativeTemplateEngine', ko.nativeTemplateEngine);
  (function() {
      ko.jqueryTmplTemplateEngine = function () {
          // Detect which version of jquery-tmpl you're using. Unfortunately jquery-tmpl
          // doesn't expose a version number, so we have to infer it.
          // Note that as of Knockout 1.3, we only support jQuery.tmpl 1.0.0pre and later,
          // which KO internally refers to as version "2", so older versions are no longer detected.
          var jQueryTmplVersion = this.jQueryTmplVersion = (function() {
              if (!jQueryInstance || !(jQueryInstance['tmpl']))
                  return 0;
              // Since it exposes no official version number, we use our own numbering system. To be updated as jquery-tmpl evolves.
              try {
                  if (jQueryInstance['tmpl']['tag']['tmpl']['open'].toString().indexOf('__') >= 0) {
                      // Since 1.0.0pre, custom tags should append markup to an array called "__"
                      return 2; // Final version of jquery.tmpl
                  }
              } catch(ex) { /* Apparently not the version we were looking for */ }
  
              return 1; // Any older version that we don't support
          })();
  
          function ensureHasReferencedJQueryTemplates() {
              if (jQueryTmplVersion < 2)
                  throw new Error("Your version of jQuery.tmpl is too old. Please upgrade to jQuery.tmpl 1.0.0pre or later.");
          }
  
          function executeTemplate(compiledTemplate, data, jQueryTemplateOptions) {
              return jQueryInstance['tmpl'](compiledTemplate, data, jQueryTemplateOptions);
          }
  
          this['renderTemplateSource'] = function(templateSource, bindingContext, options, templateDocument) {
              templateDocument = templateDocument || document;
              options = options || {};
              ensureHasReferencedJQueryTemplates();
  
              // Ensure we have stored a precompiled version of this template (don't want to reparse on every render)
              var precompiled = templateSource['data']('precompiled');
              if (!precompiled) {
                  var templateText = templateSource['text']() || "";
                  // Wrap in "with($whatever.koBindingContext) { ... }"
                  templateText = "{{ko_with $item.koBindingContext}}" + templateText + "{{/ko_with}}";
  
                  precompiled = jQueryInstance['template'](null, templateText);
                  templateSource['data']('precompiled', precompiled);
              }
  
              var data = [bindingContext['$data']]; // Prewrap the data in an array to stop jquery.tmpl from trying to unwrap any arrays
              var jQueryTemplateOptions = jQueryInstance['extend']({ 'koBindingContext': bindingContext }, options['templateOptions']);
  
              var resultNodes = executeTemplate(precompiled, data, jQueryTemplateOptions);
              resultNodes['appendTo'](templateDocument.createElement("div")); // Using "appendTo" forces jQuery/jQuery.tmpl to perform necessary cleanup work
  
              jQueryInstance['fragments'] = {}; // Clear jQuery's fragment cache to avoid a memory leak after a large number of template renders
              return resultNodes;
          };
  
          this['createJavaScriptEvaluatorBlock'] = function(script) {
              return "{{ko_code ((function() { return " + script + " })()) }}";
          };
  
          this['addTemplate'] = function(templateName, templateMarkup) {
              document.write("<script type='text/html' id='" + templateName + "'>" + templateMarkup + "<" + "/script>");
          };
  
          if (jQueryTmplVersion > 0) {
              jQueryInstance['tmpl']['tag']['ko_code'] = {
                  open: "__.push($1 || '');"
              };
              jQueryInstance['tmpl']['tag']['ko_with'] = {
                  open: "with($1) {",
                  close: "} "
              };
          }
      };
  
      ko.jqueryTmplTemplateEngine.prototype = new ko.templateEngine();
      ko.jqueryTmplTemplateEngine.prototype.constructor = ko.jqueryTmplTemplateEngine;
  
      // Use this one by default *only if jquery.tmpl is referenced*
      var jqueryTmplTemplateEngineInstance = new ko.jqueryTmplTemplateEngine();
      if (jqueryTmplTemplateEngineInstance.jQueryTmplVersion > 0)
          ko.setTemplateEngine(jqueryTmplTemplateEngineInstance);
  
      ko.exportSymbol('jqueryTmplTemplateEngine', ko.jqueryTmplTemplateEngine);
  })();
  }));
  }());
  })();
var process = process || {env: {NODE_ENV: "development"}};
/*!
 * Knockout Mapping plugin v2.6.0
 * (c) 2013 Steven Sanderson, Roy Jacobs - http://knockoutjs.com/
 * License: MIT (http://www.opensource.org/licenses/mit-license.php)
 */
!function(e){"use strict";if("function"==typeof require&&"object"==typeof exports&&"object"==typeof module)e(require("knockout"),exports);else if("function"==typeof define&&define.amd)define(["knockout","exports"],e);else{if("undefined"==typeof ko)throw new Error("Knockout is required, please ensure it is loaded before loading this mapping plug-in");e(ko,ko.mapping={})}}(function(e,r){"use strict";function t(){for(var e,r,t,n=arguments,a=n.length,i={},o=[];a--;)for(t=n[a],e=t.length;e--;)r=t[e],i[r]||(i[r]=1,o.push(r));return o}function n(e,a){var i;for(var o in a)if(a.hasOwnProperty(o)&&a[o])if(i=r.getType(e[o]),o&&e[o]&&"array"!==i&&"string"!==i)n(e[o],a[o]);else{var u="array"===r.getType(e[o])&&"array"===r.getType(a[o]);u?e[o]=t(e[o],a[o]):e[o]=a[o]}}function a(e,r){var t={};return n(t,e),n(t,r),t}function i(e,r){for(var t=a({},e),n=x.length-1;n>=0;n--){var i=x[n];t[i]&&(t[""]instanceof Object||(t[""]={}),t[""][i]=t[i],delete t[i])}return r&&(t.ignore=o(r.ignore,t.ignore),t.include=o(r.include,t.include),t.copy=o(r.copy,t.copy),t.observe=o(r.observe,t.observe)),t.ignore=o(t.ignore,E.ignore),t.include=o(t.include,E.include),t.copy=o(t.copy,E.copy),t.observe=o(t.observe,E.observe),t.mappedProperties=t.mappedProperties||{},t.copiedProperties=t.copiedProperties||{},t}function o(t,n){return void 0===t?t=[]:"array"!==r.getType(t)&&(t=[t]),void 0===n?n=[]:"array"!==r.getType(n)&&(n=[n]),e.utils.arrayGetDistinctValues(t.concat(n))}function u(r,t){var n=e.dependentObservable;e.dependentObservable=function(t,n,a){a=a||{},t&&"object"==typeof t&&(a=t);var i=a.deferEvaluation,o=a.pure,u=!1,s=function(t){var n=e.dependentObservable;e.dependentObservable=k;var a=e.isWriteableObservable(t);e.dependentObservable=n;var i=k({read:function(){return u||(e.utils.arrayRemoveItem(r,t),u=!0),t.apply(t,arguments)},write:a&&function(e){return t(e)},deferEvaluation:!0});return i.__DO=t,i};a.deferEvaluation=!0;var p=k(t,n,a);return i||o||(p=s(p),r.push(p)),p},e.dependentObservable.fn=k.fn,e.computed=e.dependentObservable;var a=t();return e.dependentObservable=n,e.computed=e.dependentObservable,a}function s(t,n,i,o,l,b,g){var O="array"===r.getType(e.utils.unwrapObservable(n));if(b=b||"",r.isMapped(t)){var k=e.utils.unwrapObservable(t)[m];i=a(k,i)}var T={data:n,parent:g||l},x=function(){return i[o]&&i[o].create instanceof Function},I=function(r){return u(h,function(){return e.utils.unwrapObservable(l)instanceof Array?i[o].create({data:r||T.data,parent:T.parent,skip:j}):i[o].create({data:r||T.data,parent:T.parent})})},E=function(){return i[o]&&i[o].update instanceof Function},P=function(r,t){var n={data:t||T.data,parent:T.parent,target:e.utils.unwrapObservable(r)};return e.isWriteableObservable(r)&&(n.observable=r),i[o].update(n)},J=w.get(n);if(J)return J;if(o=o||"",O){var _=[],W=!1,D=function(e){return e};i[o]&&i[o].key&&(D=i[o].key,W=!0),e.isObservable(t)||(t=e.observableArray([]),t.mappedRemove=function(e){var r="function"==typeof e?e:function(r){return r===D(e)};return t.remove(function(e){return r(D(e))})},t.mappedRemoveAll=function(r){var n=c(r,D);return t.remove(function(r){return-1!==e.utils.arrayIndexOf(n,D(r))})},t.mappedDestroy=function(e){var r="function"==typeof e?e:function(r){return r===D(e)};return t.destroy(function(e){return r(D(e))})},t.mappedDestroyAll=function(r){var n=c(r,D);return t.destroy(function(r){return-1!==e.utils.arrayIndexOf(n,D(r))})},t.mappedIndexOf=function(r){var n=c(t(),D),a=D(r);return e.utils.arrayIndexOf(n,a)},t.mappedGet=function(e){return t()[t.mappedIndexOf(e)]},t.mappedCreate=function(r){if(-1!==t.mappedIndexOf(r))throw new Error("There already is an object with the key that you specified.");var n=x()?I(r):r;if(E()){var a=P(n,r);e.isWriteableObservable(n)?n(a):n=a}return t.push(n),n});var S=c(e.utils.unwrapObservable(t),D).sort(),A=c(n,D);W&&A.sort();var N,M,C,q=e.utils.compareArrays(S,A),F={},R=e.utils.unwrapObservable(n),$={},G=!0;for(N=0,M=R.length;M>N;N++){if(C=D(R[N]),void 0===C||C instanceof Object){G=!1;break}$[C]=R[N]}var K,V,z=[],B=0;for(N=0,M=q.length;M>N;N++){C=q[N];var H,L=b+"["+y(N)+"]";switch(C.status){case"added":K=G?$[C.value]:f(e.utils.unwrapObservable(n),C.value,D),H=s(void 0,K,i,o,t,L,l),x()||(H=e.utils.unwrapObservable(H)),V=p(e.utils.unwrapObservable(n),K,F),H===j?B++:z[V-B]=H,F[V]=!0;break;case"retained":K=G?$[C.value]:f(e.utils.unwrapObservable(n),C.value,D),H=f(t,C.value,D),s(H,K,i,o,t,L,l),V=p(e.utils.unwrapObservable(n),K,F),z[V]=H,F[V]=!0;break;case"deleted":H=f(t,C.value,D)}_.push({event:C.status,item:H})}t(z),i[o]&&i[o].arrayChanged&&e.utils.arrayForEach(_,function(e){i[o].arrayChanged(e.event,e.item)})}else if(d(n)){if(t=e.utils.unwrapObservable(t),!t){if(x()){var Q=I();return E()&&(Q=P(Q)),Q}if(E())return P();t={}}if(E()&&(t=P(t)),w.save(n,t),E())return t;v(n,function(a){var o=b.length?b+"."+y(a):y(a);if(-1===e.utils.arrayIndexOf(i.ignore,o)){if(-1!==e.utils.arrayIndexOf(i.copy,o))return void(t[a]=n[a]);if("object"!=typeof n[a]&&"array"!==r.getType(n[a])&&i.observe.length>0&&-1===e.utils.arrayIndexOf(i.observe,o))return t[a]=n[a],void(i.copiedProperties[o]=!0);var u=w.get(n[a]),p=s(t[a],n[a],i,a,t,o,t),l=u||p;if(i.observe.length>0&&-1===e.utils.arrayIndexOf(i.observe,o))return t[a]=e.utils.unwrapObservable(l),void(i.copiedProperties[o]=!0);e.isWriteableObservable(t[a])?(l=e.utils.unwrapObservable(l),t[a]()!==l&&t[a](l)):(l=void 0===t[a]?l:e.utils.unwrapObservable(l),t[a]=l),i.mappedProperties[o]=!0}})}else switch(r.getType(n)){case"function":E()?e.isWriteableObservable(n)?(n(P(n)),t=n):t=P(n):t=n;break;default:if(e.isWriteableObservable(t)){var U;return E()?(U=P(t),t(U),U):(U=e.utils.unwrapObservable(n),t(U),U)}var X=x()||E();if(t=x()?I():e.observable(e.utils.unwrapObservable(n)),E()&&t(P(t)),X)return t}return t}function p(e,r,t){for(var n=0,a=e.length;a>n;n++)if(t[n]!==!0&&e[n]===r)return n;return null}function l(t,n){var a;return n&&(a=n(t)),"undefined"===r.getType(a)&&(a=t),e.utils.unwrapObservable(a)}function f(r,t,n){r=e.utils.unwrapObservable(r);for(var a=0,i=r.length;i>a;a++){var o=r[a];if(l(o,n)===t)return o}throw new Error("When calling ko.update*, the key '"+t+"' was not found!")}function c(r,t){return e.utils.arrayMap(e.utils.unwrapObservable(r),function(e){return t?l(e,t):e})}function v(e,t){if("array"===r.getType(e))for(var n=0;n<e.length;n++)t(n);else for(var a in e)e.hasOwnProperty(a)&&t(a)}function d(e){if(null===e)return!1;var t=r.getType(e);return"object"===t||"array"===t}function b(e,t,n){var a=e||"";return"array"===r.getType(t)?e&&(a+="["+y(n)+"]"):(e&&(a+="."),a+=y(n)),a}function y(e){var r=(""+e).replace(/~/g,"~~").replace(/\[/g,"~[").replace(/]/g,"~]").replace(/\./g,"~.");return r}function g(){var r=[],t=[];this.save=function(n,a){var i=e.utils.arrayIndexOf(r,n);i>=0?t[i]=a:(r.push(n),t.push(a))},this.get=function(n){var a=e.utils.arrayIndexOf(r,n),i=a>=0?t[a]:void 0;return i}}function O(){var e={},r=function(r){var t;try{t=r}catch(n){t="$$$"}var a=e[t];return e.hasOwnProperty(t)||(a=new g,e[t]=a),a};this.save=function(e,t){r(e).save(e,t)},this.get=function(e){return r(e).get(e)}}e.mapping=r;var h,w,m="__ko_mapping__",k=e.dependentObservable,T=0,x=["create","update","key","arrayChanged"],j={},I={include:["_destroy"],ignore:[],copy:[],observe:[]},E=I;r.isMapped=function(r){var t=e.utils.unwrapObservable(r);return t&&t[m]},r.fromJS=function(e){if(0===arguments.length)throw new Error("When calling ko.fromJS, pass the object you want to convert.");try{T||(h=[],w=new O),T++;var r,t;2===arguments.length&&(arguments[1][m]?t=arguments[1]:r=arguments[1]),3===arguments.length&&(r=arguments[1],t=arguments[2]),t&&(r=a(r,t[m])),r=i(r);var n=s(t,e,r);if(t&&(n=t),!--T)for(;h.length;){var o=h.pop();o&&(o(),o.__DO.throttleEvaluation=o.throttleEvaluation)}return n[m]=a(n[m],r),n}catch(u){throw T=0,u}},r.fromJSON=function(t){var n=Array.prototype.slice.call(arguments,0);return n[0]=e.utils.parseJson(t),r.fromJS.apply(this,n)},r.toJS=function(t,n){if(E||r.resetDefaultOptions(),0===arguments.length)throw new Error("When calling ko.mapping.toJS, pass the object you want to convert.");if("array"!==r.getType(E.ignore))throw new Error("ko.mapping.defaultOptions().ignore should be an array.");if("array"!==r.getType(E.include))throw new Error("ko.mapping.defaultOptions().include should be an array.");if("array"!==r.getType(E.copy))throw new Error("ko.mapping.defaultOptions().copy should be an array.");return n=i(n,t[m]),r.visitModel(t,function(r){return e.utils.unwrapObservable(r)},n)},r.toJSON=function(t,n,a,i){var o=r.toJS(t,n);return e.utils.stringifyJson(o,a,i)},r.defaultOptions=function(){return arguments.length>0?void(E=arguments[0]):E},r.resetDefaultOptions=function(){E={include:I.include.slice(0),ignore:I.ignore.slice(0),copy:I.copy.slice(0),observe:I.observe.slice(0)}},r.getType=function(e){if(e&&"object"==typeof e){if(e.constructor===Date)return"date";if(e.constructor===Array)return"array"}return typeof e},r.visitModel=function(t,n,a){a=a||{},a.visitedObjects=a.visitedObjects||new O;var o,u=e.utils.unwrapObservable(t);if(!d(u))return n(t,a.parentName);a=i(a,u[m]),n(t,a.parentName),o="array"===r.getType(u)?[]:{},a.visitedObjects.save(t,o);var s=a.parentName;return v(u,function(t){var i=y(t);if(!a.ignore||-1===e.utils.arrayIndexOf(a.ignore,i)){var p=u[t];if(a.parentName=b(s,u,t),-1===e.utils.arrayIndexOf(a.copy,i)&&-1===e.utils.arrayIndexOf(a.include,i)){var l=u[m];if(l){var f=l.mappedProperties;if(f&&!f[i]){var c=l.copiedProperties;if(c&&!c[i]&&"array"!==r.getType(u))return}}}switch(r.getType(e.utils.unwrapObservable(p))){case"object":case"array":case"undefined":var v=a.visitedObjects.get(p);o[t]="undefined"!==r.getType(v)?v:r.visitModel(p,n,a);break;default:o[t]=n(p,a.parentName)}}}),o}});
//# sourceMappingURL=knockout.mapping.min.js.map

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
/*
 source: http://stackoverflow.com/questions/14838135/how-to-use-knockout-to-iterate-over-an-object-not-array
 */
/**
 *
 * binding: foreachprop:
 * can also define "unsorted:" binding to not sort keys.
 * <pre><code>
 *      &lt;span data-bind="foreachprop: execution().jobArguments"&gt;
 &lt;span data-bind="text: key"&gt;&lt;/span&gt;:
 &lt;span data-bind="text: value" class="optvalue"&gt;&lt;/span&gt;
 &lt;/span&gt;
 </code></pre>
 */
ko.bindingHandlers.foreachprop = {
    transformObject: function (obj,unsorted) {
        var properties = [];
        for (var key in obj) {
            if (obj.hasOwnProperty(key)) {
                properties.push({ key: key, value: obj[key] });
            }
        }
        if(!unsorted) {
            properties.sort(function (a, b) {
                return a.key.localeCompare(b.key);
            });
        }
        return ko.observableArray(properties);
    },
    init: function (element, valueAccessor, allBindingsAccessor, viewModel, bindingContext) {
        var unsorted = allBindingsAccessor.get('unsorted');
        var value = ko.utils.unwrapObservable(valueAccessor()),
            properties = ko.bindingHandlers.foreachprop.transformObject(value,unsorted);
        var childBindingContext = bindingContext.createChildContext(
            bindingContext.$rawData,
            null // Optionally, pass a string here as an alias for the data item in descendant contexts
        );
        jQuery(element).data('childBindingContext', childBindingContext);
        ko.applyBindingsToNode(element, { foreach: properties }, childBindingContext);
        return { controlsDescendantBindings: true };
    },
    update:function(element, valueAccessor, allBindingsAccessor, viewModel, bindingContext){
        "use strict";
        var unsorted = allBindingsAccessor.get('unsorted');
        var value = ko.utils.unwrapObservable(valueAccessor()),
            properties = ko.bindingHandlers.foreachprop.transformObject(value,unsorted);
        var childBindingContext=jQuery(element).data('childBindingContext');
        ko.bindingHandlers['foreach'].update(element, properties, allBindingsAccessor, viewModel, childBindingContext);
        return { controlsDescendantBindings: true };

    }
};

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Manages cascading reloads of remote option values based on a dependency graph
 */
function RemoteOptionController(data) {
    "use strict";
    var self = this;
    self.loader = data.loader;
    /**
     * container for array of Options, keyed by option name
     */
    self.options = {};
    /**
     * container for array of dependent option names, keyed by option name
     */
    self.dependents = {};
    /**
     * container for array of depdency option names, keyed by option name
     */
    self.dependencies = {};
    /**
     * list of option names
     */
    self.names = [];

    /**
     * container of observer subscriptions, keyed by option name
     */
    self.observers = {};

    /**
     * indicates cyclic dependencies
     */
    self.cyclic = false;

    function emptyValue(val) {
        return (!val || val === '');
    }
    /**
     * Setup dependencies using the loaded options
     * @param joboptions
     */
    self.setupOptions = function (joboptions) {
        ko.utils.arrayForEach(joboptions.options(), function (opt) {
            self.addOption(opt);
            if (opt.hasRemote()) {
                opt.setReloadCallback(self.reloadOptionIfRequirementsMet);
            }
        });
    };
    /**
     * register an option with parameters used for Ajax reload of the field (used to call _loadRemoteOptionValues
     * function)
     * @param opt Option object
     */
    self.addOption = function (opt) {
        self.options[opt.name()] = opt;
        self.names.push(opt.name());
    };

    /**
     * Non remote-values option
     * @param name
     */
    self.addLocalOption = function (name) {
        self.names.push(name);
    };

    /**
     * reload the values for an option by name (calls _loadRemoteOptionValues)
     * @param name
     */
    self.loadRemoteOptionValues = function (name) {
        //stop observing option name if doing so
        // self.stopObserving(name);
        var option = self.options[name];
        self.loader.loadRemoteOptionValues(option, self.options).then(function (data) {
            option.loadRemote(data);
        });
    };


    /**
     * define dependent option names for an option
     * @param name
     * @param depsArr
     */
    self.addOptionDeps = function (name, depsArr) {
        self.dependents[name] = depsArr;
    };

    /**
     * define dependency option names for an option
     * @param name
     * @param depsArr
     */
    self.addOptionDependencies = function (name, depsArr) {
        self.dependencies[name] = depsArr;
    };

    /**
     * reload the option values for an option if all required dependencies are set
     * @param name
     */
    self.reloadOptionIfRequirementsMet = function (name) {
        var skip = false;

        // reload iff: all of its required dependencies have a value
        var missing = [];
        if(self.dependencies[name]) {
            for (var j = 0; j < self.dependencies[name].length; j++) {
                var dependencyName = self.dependencies[name][j];
                var option = self.options[dependencyName];
                if (!option.value() && option.required()) {
                    skip = true;
                    missing.push(dependencyName);
                }
            }
        }
        if (!skip) {
            self.loadRemoteOptionValues(name);
        } else if (self.options[name]) {
            self.options[name].remoteError({
                message: message("options.remote.dependency.missing.required", [name, missing.join(", ")])
            });
        }
    };

    /**
     * notify that a value changed for an option by name, will reload dependents if any
     * @param name
     * @param value
     */
    self.optionValueChanged = function (name, value) {
        //trigger reload
        if (self.dependents[name] && !self.cyclic) {
            for (var i = 0; i < self.dependents[name].length; i++) {
                var dependentName = self.dependents[name][i];
                self.reloadOptionIfRequirementsMet(dependentName);
            }
        }
    };
    /**
     * True if the option should force dependent options to load at start
     * @param name
     * @returns {*}
     */
    self.shouldAutoReload = function (name) {
        var option = self.options[name];
        if (option.hasRemote()) {
            return !!(!self.cyclic && (self.dependents[name] || !emptyValue(option.value())));
        } else {
            return !!((self.dependents[name] && option.enforced()) || !emptyValue(option.value()));
        }
    };
    /**
     * True if The option should load remote values at start
     * @param name
     * @returns {*|boolean}
     */
    self.shouldLoadOnStart = function (name) {
        return self.options[name].hasRemote() && (!self.dependencies[name] || self.cyclic);
    };

    /**
     * Auto reload dependent options if necessary
     * @param name
     * @returns {boolean}
     */
    self.doOptionAutoReload = function (name) {
        if (self.shouldAutoReload(name)) {
            //trigger change immediately
            var value = self.options[name].value();
            self.optionValueChanged(name, value);
            return true;
        }
        return false;
    };

    /**
     * load remote option dataset from json data
     * @param data
     */
    self.loadData = function (data) {
        if(!data){
            return;
        }
        if (data['optionsDependenciesCyclic']) {
            self.cyclic = data['optionsDependenciesCyclic'];
        }
        for (var opt in data.options) {
            var params = data.options[opt];
            if (params['optionDependencies']) {
                self.addOptionDependencies(opt, params['optionDependencies']);
            }
            if (params['optionDeps']) {
                self.addOptionDeps(opt, params['optionDeps']);
            }

        }
    };

    self.unsubscribeAll = function () {
        "use strict";
        for(var p in self.observers){
            self.observers[p].dispose();
        }
        self.observers={};
    };
    /**
     * starts observing changes for option field by name
     * @param name
     */
    self.observeChangesFor = function (name) {
        // this.stopObserving(name);
        //observe field value change and trigger reloads
        self.observers[name] = self.options[name].value.subscribe(function (newval) {
            self.optionValueChanged(name, newval);
        });
    };
    /**
     * begin by loading on start values, reloading autoreload values, and then observe
     * changes.
     */
    self.begin = function () {
        for (var i = 0; i < self.names.length; i++) {
            var name = self.names[i];
            if (self.shouldLoadOnStart(name)) {
                self.loadRemoteOptionValues(name);
            }
            self.doOptionAutoReload(self.names[i]);
        }

        for (var i = 0; i < self.names.length; i++) {
            self.observeChangesFor(self.names[i]);
        }
    };

}

function RemoteOptionLoader(data) {
    "use strict";
    var self = this;
    self.url = data.url;
    self.fieldPrefix = data.fieldPrefix;
    self.id = data.id;
    //load remote values
    self.loadRemoteOptionValues = function (opt, options) {
        opt.loading(true);
        var params = {option: opt.name(), selectedvalue: opt.value(), id: self.id};
        //
        if (null != options) {
            for (var xopt in options) {
                if (xopt != opt.name()) {
                    params[self.fieldPrefix + xopt] = options[xopt].value();
                }
            }
        }
        return jQuery.ajax({
            method: 'GET',
            type: 'json',
            url: _genUrl(self.url, params),
            success: function (data, status, jqxhr) {
                //show loading spinner at least for a little while
                setTimeout(function () {
                    opt.loading(false)
                },200);
            },
            error: function (jqxhr, status, message) {
                setTimeout(function () {
                    opt.loading(false)
                },200);
                opt.remoteError({error: "ERROR loading result from Rundeck server: " + status + ": " + message});
            }
        });
    }
}

var process = process || {env: {NODE_ENV: "development"}};

/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/**
 * hook to initialize custom bootstrap popover mechanism
 * &ltdiv data-bind="bootstrapPopover: true, bootstrapPopoverContentRef: '#elemid' &gt;
 * or
 * &ltdiv data-bind="bootstrapPopover: true, bootstrapPopoverContentFor: '#elemid' &gt;
 */
ko.bindingHandlers.bootstrapPopover = {
    init: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
        if(allBindings.get('bootstrapPopoverContentRef')){
            const doHideTooltip = ko.unwrap(allBindings.get('bootstrapPopoverHideTooltip'))||false
            const opts = ko.unwrap(allBindings.get('bootstrapPopoverOptions')) || {}
            _initPopoverContentRef(null, jQuery.extend(
                opts,
                {
                    element: element,
                    contentRef: ko.unwrap(allBindings.get('bootstrapPopoverContentRef')),
                    onShown: function () {
                        _initPopoverMousedownCatch('body', '._mousedown_popup_allowed', function (e) {
                            jQuery(element).popover("hide")
                        })
                        if(doHideTooltip){
                            jQuery(element).tooltip("hide")
                        }
                    }
                }
                )
            )
            ko.utils.domNodeDisposal.addDisposeCallback(element, function () {
                jQuery(element).popover("destroy")
            })
        }else if (allBindings.get('bootstrapPopoverContentFor')) {
            _initPopoverContentFor(null,{element:element,target:ko.unwrap(allBindings.get('bootstrapPopoverContentFor'))});
        }
    }
};

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/**
 * Initializes bootstrap tooltip on the dom element. Usage: &lt;div data-bind="bootstrapTooltip: true" title="blah" &gt;
 * tip: if the title of the element is bound to an observable, pass the same one as the binding, like
 * &lt;div data-bind="bootstrapTooltip: mytooltipObservable" title="blah" &gt;, to trigger updates when it changes.
 * @type {{init: ko.bindingHandlers.bootstrapTooltip.init}}
 */
ko.bindingHandlers.bootstrapTooltip = {
    init: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
        "use strict";
        jQuery(element).tooltip({});

        ko.utils.domNodeDisposal.addDisposeCallback(element, function() {
            jQuery(element).tooltip("destroy");
        });
    },
    update: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
        "use strict";
        var val = valueAccessor();
        if(ko.isObservable(val)){
            val = ko.unwrap(val);
            jQuery(element).tooltip('destroy');
            jQuery(element).data('original-title',null);
            jQuery(element).tooltip({});
        }
    }
};

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
//= require ko/handler-bootstrapPopover
//= require ko/handler-bootstrapTooltip


var process = process || {env: {NODE_ENV: "development"}};

/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

jQuery(function () {
    ko.bindingHandlers.datetimepicker = {
        init: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
            var opts = {useCurrent: true, sideBySide: true};
            var val = valueAccessor();
            if (ko.isObservable(val)) {
                val = ko.unwrap(val);
            }
            if (typeof(val) === 'string') {
                opts.defaultDate = val;
            }
            var dateFormat = allBindings.get('dateFormat');
            if (ko.isObservable(dateFormat)) {
                dateFormat = ko.unwrap(dateFormat);
            }
            if (typeof(dateFormat) === 'string') {
                opts.format = dateFormat;
            }
            if (opts.defaultDate) {
                try {
                    var m = moment(opts.defaultDate, opts.format, true);
                    if (!m.isValid()) {
                        opts.defaultDate = null;
                    } else {
                        //use a moment obj, due to datetimepicker bug https://github.com/Eonasdan/bootstrap-datetimepicker/issues/1704
                        opts.defaultDate = m;
                    }
                } catch (e) {
                    opts.defaultDate = null;
                }
            }
            // var locale = ko.unwrap(allBindings.get('locale'))
            //locale requires moment locale
            jQuery(element).datetimepicker(opts);

            ko.utils.domNodeDisposal.addDisposeCallback(element, function () {
                var picker = jQuery(element).data("DateTimePicker");
                if (picker) {
                    picker.destroy();
                }
            });
            //when a user changes the date, update the view model
            ko.utils.registerEventHandler(element, "dp.change", function (event) {
                var value = valueAccessor();
                if (ko.isObservable(value)) {
                    if (event.date != null && !(event.date instanceof Date)) {
                        value(moment(event.date).format(dateFormat));
                    } else {
                        value(event.date);
                    }
                }
            });
        },
        update: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
            "use strict";
            var widget = jQuery(element).data("DateTimePicker");
            //when the view model is updated, update the widget
            if (widget) {
                var koDate = ko.utils.unwrapObservable(valueAccessor());
                widget.date(koDate);
            }
        }
    };
});
var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
//= require ko/handler-datetimepicker
var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
/*
 */
ko.bindingHandlers.executeOnEnter = {
    init: function (element, valueAccessor, allBindings, viewModel, bindingContext) {
        var handler = allBindings.get('executeOnEnter');
        jQuery(element).on('keypress',function (event) {
            var keyCode = (event.which ? event.which : event.keyCode);
            if (keyCode === 13) {
                handler.call(bindingContext.$data,event);
                return false;
            }
            return true;
        });
    }
};

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function NodeFilterLinkParams(params){
    var self=this;
    self.count=params.count||'';
    self.linkicon=params.linkicon||'';
    self.classnames=params.classnames||'';
    self.filterval=params.filterval||'';
    self.filterkey=params.filterkey||'';
    self.filter=params.filter||'';
    self.linktext=params.linktext||'';
    self.textcss=params.textcss||'';
    self.suffix=params.suffix||'';
    self.tag=params.tag;
    self.title=params.title||'';
    self.filterData=function($root){
        if(self.filter){
            return {
                'data-node-filter':ko.unwrap(self.filter),
                href:$root.linkForFilterString(ko.unwrap(self.filter))
            }
        }else{
            return {
                'data-node-filter': $root.escapeFilter(self.filterkey + ':')+' ' + $root.escapeFilter(ko.unwrap(self.filterval)) ,
                href: $root.linkForFilterParams(self.filterkey,ko.unwrap(self.filterval))
            };
        }
    };
    self.attributes=function($root){
        var data= self.filterData($root);
        if(self.tag){
            data['data-node-tag']=ko.unwrap(self.tag);
        }
        if(self.title){
            data['title']=ko.unwrap(self.title);
        }
        return data;
    };
    self.viewtext=function(){
        return (self.linktext? ko.unwrap(self.linktext) : self.filter?ko.unwrap(self.filter):ko.unwrap(self.filterval))+ko.unwrap(self.suffix);
    };
}
// console.log('NodeFilterLinkParams', NodeFilterLinkParams);
ko.components.register('node-filter-link', {
    viewModel:NodeFilterLinkParams,
    template: '<a  class="nodefilterlink link-quiet"  href="#"  data-bind="attr: attributes($root),  css: classnames"  > \
    <span data-bind="if: linkicon"><i data-bind="css: linkicon"></i></span>\
    <span data-bind="if: !linkicon"><span data-bind="text: viewtext()" style="text-transform: none;"></span></span>\
    <span data-bind="if: count">(<span data-bind="text: count"></span>)</span>\
    </a>'
});

ko.components.register('node-exclude-filter-link', {
    viewModel:NodeFilterLinkParams,
    template: '<a  class="nodeexcludefilterlink link-quiet"  href="#"  data-bind="attr: attributes($root),  css: classnames"  > \
    <span data-bind="if: linkicon"><i data-bind="css: linkicon"></i></span>\
    <span data-bind="if: !linkicon"><span data-bind="text: viewtext()" style="text-transform: none;"></span></span> \
    <span data-bind="if: count">(<span data-bind="text: count"></span>)</span> \
    </a>'
});

var process = process || {env: {NODE_ENV: "development"}};

/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*
 * Useful for i18n messages; replace the text content of an element, by substituting values into placeholders.
 * Placeholders are in the form '{0}','{1}', etc.  The "messageTemplate" binding value can be a single value, which
 * will be used for {0}, or it can be an object with a 'value' property, possibly observable, containing an array
 * for the replacement values.
 * If a binding "messageTemplatePluralize: true" is set, then the template text is treated as a singular and a plural
 * version of the same text, separated by "|" character.  If the first bound data value is "1", then singular form
 * is used, otherwise the plural form is used.
 *
 */
ko.bindingHandlers.messageTemplate = {
    init: function (element, valueAccessor, allBindings, viewModel, bindingContext) {

        var text=jQuery(element).text();
        jQuery(element).data('ko-message-template',text);
        return { 'controlsDescendantBindings': true };
    },
    update:function(element, valueAccessor, allBindings, viewModel, bindingContext){
        var pluralize=allBindings.get('messageTemplatePluralize');
        var data=ko.utils.unwrapObservable(valueAccessor());
        var template=jQuery(element).data('ko-message-template');

        var text = messageTemplate(template,data,pluralize);
        ko.utils.setTextContent(element, text);
    }
};

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2018 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
//= require ko/handler-messageTemplate
var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2019 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Makes an alias `css2:` to allow combining static and dynamic css classes in one element
 */
ko.bindingHandlers.css2 = ko.bindingHandlers.css

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//= require vendor/knockout.min
//= require vendor/knockout-mapping
//= require knockout-onenter
//= require knockout-foreachprop
//= require knockout-node-filter-link
//= require ko/binding-popover
//= require ko/binding-message-template
//= require ko/binding-css2

var NODE_FILTER_ALL='.*';
function NodeSummary(data){
    var self=this;
    self.error=ko.observable();
    self.tags=ko.observableArray();
    self.defaultFilter=ko.observable();
    self.totalCount=ko.observable(0);
    self.baseUrl=data.baseUrl?data.baseUrl:'';
    self.filterToDelete=ko.observable();
    
    self.reload=function(){
      jQuery.ajax({
          url:_genUrl(appLinks.frameworkNodeSummaryAjax),
          type:'GET',

          error: function (jqxhr, status, err) {
              if (jqxhr.responseJSON && jqxhr.responseJSON.message) {
                  self.error(jqxhr.responseJSON.message)
              } else if (jqxhr.status === 403) {
                  self.error('Not authorized')
              } else {
                  console.log('Nodes Summary: request failed for nodeSummaryAjax: ' + err, jqxhr)
                  self.error('Nodes Summary: request failed: ' + err)
              }
          }
      }).done(function(data){
          ko.mapping.fromJS(data,{},self);
      });
    };
    self.linkForTagFilter=function(tag){
        return _genUrl(self.baseUrl,{filter: 'tags:'+tag.tag()});
    };
    /**
     * Generate URL for the NodeFilters object
     * @param nodefilters
     * @returns {*}
     */
    self.linkForNodeFilters=function(nodefilters){
        return _genUrl(self.baseUrl,nodefilters.getPageParams());
    };

    if(data) {
        ko.mapping.fromJS(data, {}, self);
    }
}
var CSSColors='aliceblue antiquewhite aqua aquamarine azure beige bisque black blanchedalmond blue blueviolet brown burlywood cadetblue chartreuse chocolate coral cornflowerblue cornsilk crimson cyan darkblue darkcyan darkgoldenrod darkgray darkgreen darkkhaki darkmagenta darkolivegreen darkorange darkorchid darkred darksalmon darkseagreen darkslateblue darkslategray darkturquoise darkviolet deeppink deepskyblue dimgray dodgerblue firebrick floralwhite forestgreen fuchsia gainsboro ghostwhite gold goldenrod gray green greenyellow honeydew hotpink indianred indigo ivory khaki lavender lavenderblush lawngreen lemonchiffon lightblue lightcoral lightcyan lightgoldenrodyellow lightgreen lightgrey lightpink lightsalmon lightseagreen lightskyblue lightslategray lightsteelblue lightyellow lime limegreen linen magenta maroon mediumaquamarine mediumblue mediumorchid mediumpurple mediumseagreen mediumslateblue mediumspringgreen mediumturquoise mediumvioletred midnightblue mintcream mistyrose moccasin navajowhite navy oldlace olive olivedrab orange orangered orchid palegoldenrod palegreen paleturquoise palevioletred papayawhip peachpuff peru pink plum powderblue purple red rosybrown royalblue saddlebrown salmon sandybrown seagreen seashell sienna silver skyblue slateblue slategray snow springgreen steelblue tan teal thistle tomato turquoise violet wheat white whitesmoke yellow yellowgreen'.split(' ');
function NodeEntry(data){
    var self=this;

    if(data) {
        ko.mapping.fromJS(data, {}, self);
    }
}
function TagSummary(data){
    var self=this;
    self.tag=ko.observable(data.tag);
    self.value=ko.observable(data.value);
    if(data){
        ko.mapping.fromJS(data, {}, self);
    }
}
/**
 * The set of node results
 * @param data
 * @constructor
 */
function NodeSet(data) {
    var self = this;
    self.nodes=ko.observableArray([]);
    self.tagsummary=ko.observableArray([]);

    var mapping = {

        'nodes': {
            //nb: don't use a key: function because in some cases
            //we need a new dom element every time we redraw nodes,
            //and a key function will cause re-use of the element

            create: function (options) {
                return new NodeEntry(options.data);
            }
        }
        ,
        'tagsummary': {
            key: function (data) {
                return ko.utils.unwrapObservable(data.tag);
            },
            create: function (options) {
                return new TagSummary(options.data);
            }
        }
    };

    self.glyphiconCss=function(name){
        if(name.match(/^glyphicon-[a-z-]+$/)){
            return 'glyphicon '+name;
        }else if(name.match(/^fa-[a-z-]+$/)){
            return 'fas '+name;
        }else if(name.match(/^fab-[a-z-]+$/)){
            return 'fab fa-'+name.substring(4);
        }
        return '';
    };
    self.glyphiconBadges=function(attributes){
        var badges=[];
        if(attributes['ui:badges']){
            var found=attributes['ui:badges']().split(/,\s*/g);
            for(var i=0;i<found.length;i++){
                if(found[i].match(/^glyphicon-[a-z-]+$/)){
                    badges.push(found[i]);
                }else if(found[i].match(/^fa-[a-z-]+$/)){
                    badges.push(found[i]);
                }else if(found[i].match(/^fab-[a-z-]+$/)){
                    badges.push(found[i]);
                }
            }
        }

        return badges;
    };
    self.hasOsData = function (attributes) {
        return ['osName', 'osFamily', 'osVersion', 'osArch'].findIndex((val) => attributes[val]) >= 0
    }
    self.isAnsiFg=function(str){
        return str!=null && typeof(str)=='string' && str.match(/^ansi-fg-(light-)?(black|green|red|yellow|blue|magenta|cyan|white)$/);
    };
    self.isStyleFg=function(str){
        return str!=null && typeof(str)=='string' && str.match(/^#[0-9a-fA-F]{3,6}$/) || CSSColors.indexOf(str)>=0;
    };
    self.isAnsiBg=function(str){
        return str!=null && typeof(str)=='string' && str.match(/^ansi-bg-(black|green|red|yellow|blue|magenta|cyan|white|default)$/);
    };
    self.isStyleBg=function(str){
        return str!=null && typeof(str)=='string' && str.match(/^#[0-9a-fA-F]{3,6}$/)|| CSSColors.indexOf(str)>=0;
    };
    self.iconFgCss=function(attrs,attrName){
        var uiIconColor = attrs[attrName]?attrs[attrName]():null;
        var uiColor = attrs['ui:color']?attrs['ui:color']():null;
        if(self.isAnsiFg(uiIconColor)){
            return uiIconColor;
        }else if(self.isAnsiFg(uiColor)){
            return uiColor;
        }
        return null;
    };
    self.iconBgCss=function(attrs,attrName){
        var uiIconBgcolor = attrs['ui:icon:bgcolor']?attrs['ui:icon:bgcolor']():null;
        var uiBgcolor = attrs['ui:bgcolor']?attrs['ui:bgcolor']():null;
        if(self.isAnsiBg(uiIconBgcolor)){
            return uiIconBgcolor;
        }else if(self.isAnsiBg(uiBgcolor)){
            return uiBgcolor;
        }
        return null;
    };
    self.statusIconCss=function(attrs){
        var classnames=[];
        var fgColor= self.iconFgCss(attrs,'ui:status:color');
        if(fgColor){
            classnames.push(fgColor);
        }
        var bgColor = self.iconBgCss(attrs,'ui:status:bgcolor');
        if(bgColor){
            classnames.push(bgColor);
        }
        return classnames.join(' ');
    };
    self.iconCss=function(attrs){
        var classnames=[];
        var fgColor= self.iconFgCss(attrs,'ui:icon:color');
        if(fgColor){
            classnames.push(fgColor);
        }
        var bgColor = self.iconBgCss(attrs,'ui:icon:bgcolor');
        if(bgColor){
            classnames.push(bgColor);
        }
        return classnames.join(' ');
    };
    self.nodeFgCss=function(attrs) {
        var uiColor = attrs['ui:color'] ? attrs['ui:color']() : null;
        if (self.isAnsiFg(uiColor)) {
            return uiColor;
        }
        return null
    };
    self.nodeBgCss=function(attrs) {
        var uiBgcolor = attrs['ui:bgcolor']?attrs['ui:bgcolor']():null;
        if(self.isAnsiBg(uiBgcolor)){
            return uiBgcolor;
        }
        return null
    };
    self.nodeCss = function (attrs, other) {
        var classnames=[];
        var uiColor = self.nodeFgCss(attrs);
        if(uiColor){
            classnames.push(uiColor);
        }
        var uiBgcolor = self.nodeBgCss(attrs);
        if(uiBgcolor){
            classnames.push(uiBgcolor);
        }
        const cnames = classnames.join(' ')
        if (other) {
            return jQuery.extend({}, other, {[cnames]: true})
        }
        return cnames
    };
    self.iconStyle=function(attrs){
        var styles={};
        if(!self.iconFgCss(attrs,'ui:icon:color')) {
            var uiIconColor = attrs['ui:icon:color']?attrs['ui:icon:color']():null;
            var uiColor = attrs['ui:color']?attrs['ui:color']():null;
            if (self.isStyleFg(uiIconColor)){
                styles['color']=uiIconColor;
            }else if(self.isStyleFg(uiColor)){
                styles['color']=uiColor;
            }
        }
        if(!self.iconBgCss(attrs,'ui:icon:bgcolor')) {
            var uiIconBgcolor = attrs['ui:icon:bgcolor']?attrs['ui:icon:bgcolor']():null;
            var uiBgcolor = attrs['ui:bgcolor']?attrs['ui:bgcolor']():null;
            if (self.isStyleBg(uiIconBgcolor)){
                styles['background-color']=uiIconBgcolor;
            }else if(self.isStyleBg(uiBgcolor)){
                styles['background-color']=uiBgcolor;
            }
        }
        return styles;
    };
    self.statusIconStyle=function(attrs){
        var styles={};
        if(!self.iconFgCss(attrs,'ui:status:color')) {
            var uiIconColor = attrs['ui:status:color']?attrs['ui:status:color']():null;
            var uiColor = attrs['ui:color']?attrs['ui:color']():null;
            if (self.isStyleFg(uiIconColor)){
                styles['color']=uiIconColor;
            }else if(self.isStyleFg(uiColor)){
                styles['color']=uiColor;
            }
        }
        if(!self.iconBgCss(attrs,'ui:status:bgcolor')) {
            var uiIconBgcolor = attrs['ui:status:bgcolor']?attrs['ui:status:bgcolor']():null;
            var uiBgcolor = attrs['ui:bgcolor']?attrs['ui:bgcolor']():null;
            if (self.isStyleBg(uiIconBgcolor)){
                styles['background-color']=uiIconBgcolor;
            }else if(self.isStyleBg(uiBgcolor)){
                styles['background-color']=uiBgcolor;
            }
        }
        return styles;
    };
    self.nodeStyle=function(attrs){

        var styles={};
        var uiColor = attrs['ui:color']?attrs['ui:color']():null;
        if(!self.nodeFgCss(attrs) && self.isStyleFg(uiColor)){
            styles['color']=uiColor;
        }
        var uiBgcolor = attrs['ui:bgcolor']?attrs['ui:bgcolor']():null;
        if(!self.nodeBgCss(attrs) && self.isStyleBg(uiBgcolor)){
            styles['background-color']=uiBgcolor;
        }
        return styles;
    };
    self.startsWith = function (a, b) {
        return ( a.length >= (b.length)) && a.substring(0, b.length) == b;
    };

    self.attributesInNamespace=function(attrs,ns){
        var result=[];
        for(var e in attrs){
            if(self.startsWith(e, ns+':') && attrs[e]()){
                result.push({name:e,value:attrs[e](),shortname: e.substring(ns.length+1)});
            }
        }
        result.sort(function(a,b){return a.shortname.localeCompare(b.shortname);});
        return result;
    };
    self.attributeNamespaceRegex=/^(.+?):.+$/;
    self.attributeNamespaceNames=function(attrs){
        var namespaces=[];
        for(var e in attrs){
            var found=e.match(self.attributeNamespaceRegex);
            if(found && found.length>1){
                if(namespaces.indexOf(found[1])<0){
                    namespaces.push(found[1]);
                }
            }
        }
        namespaces.sort();
        return namespaces;
    };
    self.attributeNamespaces=function(attrs){
        var index={};
        var names=[];
        for(var e in attrs){
            var found=e.match(self.attributeNamespaceRegex);
            if(found && found.length>1){
                if(!index[found[1]]){
                    index[found[1]]=[];
                    names.push(found[1]);
                }
                index[found[1]].push({
                    name:e,
                    value:ko.utils.unwrapObservable(attrs[e]),
                    shortname: e.substring(found[1].length+1)
                });
            }
        }
        names.sort();

        var results=[];
        for(var i=0;i<names.length;i++){
            var values=index[names[i]];
            values.sort(function(a,b){return a.shortname.localeCompare(b.shortname);});
            results.push({ns:names[i],values:values});
        }

        return results;
    };
    self.OsAttributeNames='nodename hostname username description tags osFamily osName osVersion osArch'.split(' ');
    /**
     * Return an object with only attributes for display, excluding ui: namespace, and osAttrs
     *
     * @param attrs
     */
    self.displayAttributes = function (attrs) {
        var result = {};
        for(var e in attrs){
            if(e.indexOf(':')<0 && self.OsAttributeNames.indexOf(e)<0){
                result[e]=attrs[e];
            }
        }
        return result;
    };
    /**
     * Replace context variables in the string for the node details.
     * @param attrs
     * @param str
     */
    self.expandNodeAttributes=function(attrs,str){
        return str.replace(/\$\{node\.([a-zA-Z-.]+)\}/g,function(match, g1, offset, string){
            if(attrs[g1]){
                return attrs[g1]()||'';
            }else{
                return string;
            }
        });
    };
    self.convertTagSummary=function(obj){
        var arr=[];
        for(var e in obj){
            arr.push({tag:e,value:obj[e]});
        }
        return arr;
    };
    self.loadJS=function(data){
        ko.mapping.fromJS({
            nodes:data.nodes,
            tagsummary:self.convertTagSummary(data.tagsummary)
        },mapping,self);
    };
    if(data){
        self.loadJS(data);
    }
}
function NodeFilters(baseRunUrl, baseSaveJobUrl, baseNodesPageUrl, data) {
    var self = this;
    self.baseRunUrl = baseRunUrl;
    self.baseSaveJobUrl = baseSaveJobUrl;
    self.baseNodesPageUrl = baseNodesPageUrl;
    self.filter = ko.observable(data.filter);
    self.filterExclude = ko.observable(data.filterExclude);
    self.filterAll = ko.observable(data.filterAll);
    self.nodeExcludePrecedence = ko.observable(null== data.nodeExcludePrecedence || data.nodeExcludePrecedence?'true':'false');
    self.nodefilterLinkId=data.nodefilterLinkId;
    self.total = ko.observable(0);
    self.allcount = ko.observable(0);
    self.loading=ko.observable(false);
    self.error=ko.observable(null);
    self.project=ko.observable(data.project);
    self.page=ko.observable(data.page?data.page:0);
    self.pagingMax=ko.observable(data.pagingMax?data.pagingMax:20);
    self.paging=ko.observable(data.paging != null ? (data.paging ? true : false) : false);
    self.maxShown=ko.observable(data.maxShown);
    self.view=ko.observable(data.view);
    self.emptyMode=ko.observable(data.emptyMode?data.emptyMode:'localnode');
    self.emptyMessage=ko.observable(data.emptyMessage?data.emptyMessage:'No match');
    self.hideAll=ko.observable(data.hideAll!=null?(data.hideAll?true:false):false);
    self.nodeSummary=ko.observable(data.nodeSummary?data.nodeSummary:null);
    if (self.nodeSummary()) {
        self.nodeSummary().error.subscribe(self.error)
    }
    self.nodeSet=ko.observable(new NodeSet());
    self.truncated=ko.observable(false);
    self.loaded=ko.observable(false);
    self.excludeFilterUncheck = ko.observable(data.excludeFilterUncheck?'true':'false');
    self.nodeFiltersVisible = ko.observable(data.nodeFiltersVisible || true)

    /**
     *
     * can be subscribed to for browser history updating, will be changed to 'true'
     * when a browse event should be recorded, the listener should set it to false.
     */
    self.browse=ko.observable(false);
    /**
     * Names of node attributes to show as columns in table results
     */
    self.colkeys=ko.observableArray([]);

    /**
     * Display value of page number, indexed starting at 1,
     */
    this.pageDisplay = ko.computed({
        read: function () {
            return self.page()+1;
        },
        write: function (value) {
            self.page(value-1);
            self.browse(true);
            self.updateMatchedNodes();
        },
        owner: self
    });
    /**
     * Names of node attributes to show as columns removing ui: namespace
     */
    self.filterColumns=ko.pureComputed(function(){
       return ko.utils.arrayFilter(self.colkeys(),function(el){
           return !self.nodeSet().startsWith(el,'ui:');
       });
    });

    self.useDefaultColumns=ko.pureComputed(function(){
        return self.filterColumns().length < 1
    });

    /**
     * Total column count for table view
     */
    self.totalColumnsCount=ko.pureComputed(function(){
       return self.useDefaultColumns()? 5 : 2 + self.filterColumns().length;
    });

    self.isFilterAll=ko.pureComputed(function(){
        return self.filter() === NODE_FILTER_ALL;
    });
    self.isFilterExcludeAll=ko.pureComputed(function(){
        return self.filterExclude() === NODE_FILTER_ALL;
    });
    self.filterNameDisplay=ko.pureComputed(function(){
       return self.isFilterAll()?'All Nodes':'';
    });
    self.filterExcludeNameDisplay=ko.pureComputed(function(){
        return self.isFilterExcludeAll()?'All Nodes':'';
    });
    self.pageRemaining=ko.computed(function(){
        if(self.total()<=0 || self.page()<0){
            return 0;
        }
        return self.allcount()-(self.page()+1)*self.pagingMax();
    });
    self.hasMoreNodes=ko.computed(function(){
        return self.pageRemaining()>0;
    });
    self.hasMultiplePages=ko.computed(function(){
        return self.pageRemaining()>self.pagingMax();
    });
    self.hasPaging=ko.computed(function(){
        return self.paging() && self.total()>self.pagingMax();
    });
    /**
     * highest number of pages
     */
    self.maxPages = ko.pureComputed(function(){
        return self.paging() && Math.ceil(self.total()/self.pagingMax());
    });
    self.pageNumbersSkipped=ko.computed(function(){
        var arr=[];
        var cur = self.page();
        var maxPages = self.maxPages();
        var buffer=3;
        for(var i=0;i< maxPages;i++){
            if(i>=(cur-buffer) && i<=(cur+buffer) || i<buffer || i>=(maxPages-buffer)) {
                arr.push(i);
            }else if(i==(cur-buffer-1) || i==(cur+buffer+1)) {
                arr.push('..');
            }
        }
        return arr;
    });
    self.nodesTitle = ko.computed(function () {
        return self.allcount() == 1 ?
            data.nodesTitleSingular || 'Node' :
            data.nodesTitlePlural || 'Nodes';
    });
    self.filterWithoutAll = ko.computed({
        read: function () {
            if (self.filterAll() && self.filter() == NODE_FILTER_ALL && self.hideAll()) {
                return '';
            }
            return self.filter();
        },
        write: function (value) {
            self.filter(value);
        },
        owner: this
    });
    self.filterExcludeWithoutAll = ko.computed({
        read: function () {
            if (self.filterExclude() == NODE_FILTER_ALL && self.hideAll()) {
                return '';
            }
            return self.filterExclude();
        },
        write: function (value) {
            self.filterExclude(value);
        },
        owner: this
    });
    self.filterIsSet=ko.pureComputed(function(){
        return !!self.filterWithoutAll();
    });
    self.filter.subscribe(function (newValue) {
        if (newValue == '' && self.hideAll()) {
            self.filterAll(true);
        }else if(newValue==NODE_FILTER_ALL){
            self.filterAll(true);
        }
    });
    self.filter.subscribe(function (newValue) {
        if (newValue == '' && self.emptyMode() == 'blank') {
            self.clear();
        }
    });
    self.filterExclude.subscribe(function (newValue) {
        if (newValue == '' && self.hideAll()) {
            self.filterAll(true);
        }else if(newValue==NODE_FILTER_ALL){
            self.filterAll(true);
        }
    });
    self.filterExclude.subscribe(function (newValue) {
        if (newValue == '' && self.emptyMode() == 'blank') {
            self.clear();
        }
    });
    self.nodeExcludePrecedence.subscribe(function(newValue){
        self.updateMatchedNodes();
    });
    self.excludeFilterUncheck.subscribe(function(newValue){
        self.updateMatchedNodes();
    });
    self.loading.subscribe(function(newValue){
        if(!newValue){
            self.loaded(true);
        }
    });
    self.hasNodes = ko.computed(function () {
        return 0 != self.allcount();
    });
    self.runCommand = function () {
        document.location = _genUrl(self.baseRunUrl, {
            filter: self.filter()
        });
    };
    self.saveJob = function () {
        document.location = _genUrl(self.baseSaveJobUrl, {
            filter: self.filter()
        });
    };
    self.nodesPageViewUrl=ko.computed(function () {
        return _genUrl(self.baseNodesPageUrl, {
            filter: self.filter()
        });
    })
    self.nodesPageView=function(){
        document.location = _genUrl(self.baseNodesPageUrl, {
            filter: self.filter()
        });
    };
    self.clear=function(){
        self.page(0);
        self.total(0);
        self.allcount(-1);
    };
    /**
     * clear filters and results count
     */
    self.reset=function(){
        self.clear();
        self.filterAll(false);
        self.filterWithoutAll(null);
    };
    /**
     * Use a specific filter string and update
     * @param filter the filter string
     */
    self.useFilterString=function(filter){
        self.filterAll(false);
        self.filterWithoutAll(filter);
        self.clear();
        self.updateMatchedNodes();
    };
    /**
     * Generate state object for the current filter
     * @returns {{filter: (*|string), filterAll: (*|string)}}
     */
    self.getPageParams=function(){
        return {
            filter: self.filter()||'',
            filterAll: self.filterAll()||'',
            page:self.page()||0,
            max:self.pagingMax()||20
        };
    };
    /**
     * generate URL for the current filters
     */
    self.getPageUrl=function(){
        return self.nodeSummary().linkForNodeFilters(self);
    };
    /**
     * Generate filter string for params
     * @param params
     */
    self.escapeFilter=function(str){

        if(str != '' && !(str.match(/[ '"\r\n]/))){
            return str==null?'':str;
        }
        var outstr='"';
        if(null!=str) {
            outstr += str.replace(/"/g, '""');
        }
        outstr+='"';
        return outstr;
    };
    /**
     * Generate filter string for params
     * @param params
     */
    self.paramsAsfilter=function(params){
        var comps=[];
        for (var e in params) {
            comps.push(self.escapeFilter(e + ":"));
            comps.push(self.escapeFilter(params[e]));
        }
        return comps.join(" ");
    };
    self.linkForFilterString=function(filter){
        return _genUrl(self.baseNodesPageUrl,{filter:filter});
    };
    /**
     * Generate URL for some filter params
     * @param params
     * @returns {*}
     */
    self.linkForFilterParams=function(params,extra){
        var filter;
        if(typeof(params) == 'object'){
            filter=self.paramsAsfilter(params);
        }else if(typeof(params)=='string' && typeof(extra)=='string'){
            var nparams={};
            nparams[params]=extra;
            filter=self.paramsAsfilter(nparams);
        }
        return self.linkForFilterString(filter);
    };
    self.triggerNodeRemoteEdit=function(node){
        doRemoteEdit(node.nodename(),self.project(),self.nodeSet().expandNodeAttributes(node.attributes,node.attributes['remoteUrl']()))
    };
    /**
     * Update to match state parameters
     * @param data
     */
    self.setPageParams=function(data){
        self.filterAll(data.filterAll);
        self.filter(data.filter);
        self.clear();
        self.page(data.page||0);
        self.pagingMax(data.max||20);
        self.updateMatchedNodes();
    };
    self.selectNodeFilterLink=function(link,isappend){
        var filterString = jQuery(link).data('node-filter');
        var filterTag = jQuery(link).data('node-tag');
        if(filterString && filterString.indexOf("&")>=0){
            filterString = html_unescape(filterString);
        }
        var filterAll = jQuery(link).data('node-filter-all');
        self.selectNodeFilter({filter:filterString,tag:filterTag,filterAll:filterAll},isappend)
    },
    self.selectNodeFilter=function({filter: filterString, filterAll, tag: filterTag}, isappend){
        const oldfilter = self.filter()
        let v = oldfilter ? oldfilter.indexOf('tags: ') : -1
        if(isappend && filterTag && v>=0){
            var first=oldfilter.substring(0, v);
            var rest=oldfilter.substring(v + 6);
            var last='';
            while(rest.indexOf(" ")==0){
                rest = rest.substring(1);
            }
            v = rest.indexOf(" ");
            if(v>0){
                last = rest.substring(v);
                rest = rest.substring(0,v);
            }
            filterString = first + 'tags: ' + rest + '+' + filterTag + last ;
        }else if (filterString && oldfilter && !filterAll && oldfilter !== NODE_FILTER_ALL && isappend) {
            filterString = oldfilter + ' ' + filterString;
        } else if (filterAll) {
            filterString = NODE_FILTER_ALL;
        }
        self.filterAll(filterAll);
        self.filter(filterString);
        self.clear();
        self.updateMatchedNodes();
    };
    self.selectNodeFilterExcludeLink=function(link,isappend){

        var oldfilter = self.filterExclude();
        var filterString = jQuery(link).data('node-filter');
        var filterTag = jQuery(link).data('node-tag');
        if(filterString && filterString.indexOf("&")>=0){
            filterString = html_unescape(filterString);
        }
        var filterAll = jQuery(link).data('node-filter-all');
        var v=oldfilter?oldfilter.indexOf('tags: '):-1;
        if(isappend && filterTag && v>=0){
            var first=oldfilter.substring(0, v);
            var rest=oldfilter.substring(v + 6);
            var last='';
            while(rest.indexOf(" ")==0){
                rest = rest.substring(1);
            }
            v = rest.indexOf(" ");
            if(v>0){
                last = rest.substring(v);
                rest = rest.substring(0,v);
            }
            filterString = first + 'tags: ' + rest + '+' + filterTag + last ;
        }else if (filterString && oldfilter && !filterAll && oldfilter != NODE_FILTER_ALL) {
            filterString = oldfilter + ' ' + filterString;
        } else if (filterAll) {
            filterString = NODE_FILTER_ALL;
        }

        self.filterAll(filterAll);
        self.filterExclude(filterString);
        self.clear();
        self.updateMatchedNodes();

    };
    self.updateNodesNextPage=function(){
        if(!self.page()){
            self.page(0);
        }
        self.page(self.page()+1);
        self.updateMatchedNodes();
    };
    self.browseNodesPage=function(newpage){
        if(self.page()==newpage){
            return;
        }
        if (!newpage || newpage >= self.maxPages() || newpage < 0) {
            newpage=0;
        }
        self.pageDisplay(newpage+1);
    };
    self.browseNodesPageNext=function(){
        if(self.page()+ 1<self.maxPages()) {
            self.browseNodesPage(self.page() + 1);
        }
    };
    self.browseNodesPagePrev=function(){
        if(self.page()>0) {
            self.browseNodesPage(self.page() - 1);
        }
    };
    self.browseNodesPageUrl=function(page){
        var pageParams = self.getPageParams();
        if (page >= 0 && page < self.maxPages()) {
            pageParams.page = page;
        }else{
            pageParams.page=0;
        }
        return _genUrl(self.baseNodesPageUrl, pageParams);
    };
    self.browseNodesPagePrevUrl=function(){
        return self.browseNodesPageUrl(self.page()-1);
    };
    self.browseNodesPageNextUrl=function(){
        return self.browseNodesPageUrl(self.page()+1);
    };
    self.updateNodesRemainingPages=function(){
        self.page(-1);
        self.updateMatchedNodes();
    };
    self.loadJS=function(data){
        ko.mapping.fromJS(data,{},self);
    };
    self.pagingMax.subscribe(function (){
        self.updateMatchedNodes();
    });
    self.newFilterText=function(){
        self.page(0);
        self.updateMatchedNodes();
    };
    self.updateMatchedNodes= function () {
        if(!self.filter() && self.emptyMode()=='blank'){
            return;
        }
        var project=self.project();
        if (!project) {
            return;
        }
        var filterdata =  self.filter()?{filter: self.filter()}:{};
        var filterExcludedata = self.filterExclude()?{filterExclude: self.filterExclude()}:{};
        var excludeFilterUncheck =  self.excludeFilterUncheck()
        var page = self.page();
        var view = self.view() ? self.view() : 'table';
        var basedata = {view: view, declarenone: true, fullresults: true, expanddetail: true, inlinepaging: false, nodefilterLinkId: self.nodefilterLinkId, excludeFilterUncheck: self.excludeFilterUncheck()};
        if(self.paging()){
            basedata.page = page;
            basedata.max = self.pagingMax();
            basedata.inlinepaging=true;
            if (page != 0) {
                basedata.view= 'tableContent';
            }
        }
        if(self.maxShown()){
            basedata.maxShown=self.maxShown();
        }
        var params = jQuery.extend(basedata, filterdata, filterExcludedata);
        if(self.emptyMode()=='localnode' && !self.filter()){
            params.localNodeOnly = 'true';
        }else if(self.emptyMode()=='blank' && !self.filter()){
            self.clear();
            return;
        }
        var exclude=self.nodeExcludePrecedence();
        if (typeof(exclude) == 'string' && exclude === "false"
            || typeof(exclude) == 'boolean' && !exclude) {
            params.nodeExcludePrecedence = "false";
        } else {
            params.nodeExcludePrecedence = "true";
        }
        self.loading(true);
        var requrl=_genUrl(appLinks.frameworkNodesQueryAjax, params);
        jQuery.ajax({
            type:'GET',
            dataType:'json',
            url:requrl,

            error:function(jqxhr,status,err){
                self.loading(false);

                if (jqxhr.responseJSON && jqxhr.responseJSON.message) {
                    self.error(jqxhr.responseJSON.message)
                } else if (jqxhr.status === 403) {
                    self.error('Not authorized')
                } else {
                    console.log('Nodes Query: request failed: ' + err, jqxhr)
                    self.error('Nodes Query: request failed: ' + err)
                }
            },
            success:function(data,status,jqxhr){
                self.loading(false);
                self.nodeSet().loadJS(
                    {
                        nodes:data.allnodes,
                        tagsummary:data.tagsummary
                    });
                self.loadJS({
                    allcount:data.allcount,
                    total:data.total,
                    truncated:data.truncated,
                    colkeys:data.colkeys

                });

            }
        });
    };
}

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var ExecutionOptions = {
    multiVarCheckboxChangeHandler : function(optname,evt) {
        var check = Event.element(evt);
        if ("" == $(check).value && !$(check).checked) {
            //remove it
            var div = $(check.parentNode);
            var parent=$(div).parentNode;
            jQuery(div).fadeTo('fast',0,function(){
                $(div).parentNode.removeChild(div);
            });
        }
    },
    multiVarCheckboxChangeWarningHandler: function(optname,evt) {
        var check = Event.element(evt);
        //show warning text if all checkboxes unchecked
        var parent = $(check).up('div.optionmultiarea');
        var test = true;
        $(parent).trigger('select',"input[type='checkbox']").each(function(e) {
            if ($(e).checked) {
                test = false;
            }
        });
        if (test) {
            $$('#' + optname + '_state span.reqwarning').each(Element.show);
        }else{
            $$('#' + optname + '_state span.reqwarning').each(Element.hide);
        }
    },
     multiVarInputChangeHandler : function(check, evt) {
        var input = Event.element(evt);
        $(check).value = $(input).value;
    },
     multiVarInputKeydownHandler : function(check, evt) {
        var input = Event.element(evt);
        if (noenter(evt)) {
            $(check).value = $(input).value;
            return true;
        } else {
            return false;
        }
    },
    addMultivarValue: function(name, inputarea, value,handler,placement) {
        var div = new Element('div');
        div.addClassName('optionvaluemulti');
        div.setStyle({'opacity':'0'});

        var divwrap = new Element('div');
        divwrap.addClassName('varinput');

        var inpu = new Element('input');
        inpu.setAttribute("type", "checkbox");
        inpu.setAttribute("name", "extra.option." + name);
        inpu.setAttribute("checked", "true");
        inpu.setAttribute("value", null != value ? value : "");
        Event.observe(inpu, 'change', ExecutionOptions.multiVarCheckboxChangeWarningHandler.curry(name));
        Event.observe(inpu, 'change', ExecutionOptions.multiVarCheckboxChangeHandler.curry(name));
        if(handler){
            handler(name, inpu);
        }

        var inpu2 = new Element('input');
        inpu2.setAttribute("type", "text");
        inpu2.setAttribute("name", "_temp");
        inpu2.setAttribute("placeholder", "Enter value");
        if (null != value) {
            inpu2.setAttribute("value", value);
        }
        Event.observe(inpu2, 'change', ExecutionOptions.multiVarInputChangeHandler.curry(inpu));
        Event.observe(inpu2, 'keydown', function(evt){
            var wasNotEnter=ExecutionOptions.multiVarInputKeydownHandler(inpu,evt);
            if(!wasNotEnter && jQuery(inpu2).val()){
                ExecutionOptions.addMultivarValue(name,inputarea,null,handler);
            }
        });
        if (handler) {
            handler(name,inpu2);
        }

        $(divwrap).appendChild(inpu);
        $(divwrap).appendChild(inpu2);
        $(div).appendChild(divwrap);
        var place=placement||'top';
        var doInsert = {};
        doInsert[place]=div;
        $(inputarea).insert(doInsert);
        $$('#' + name + '_state span.reqwarning').each(Element.hide);
        Try.these(
            function() {
                jQuery(div).fadeTo('fast',1);
            },
            function() {
                $(div).show();
                $(div).highlight();
            }
            );

        $(inpu2).trigger('focus');
    }
};
var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * Selectable value with name/value pair
 * @param data
 * @constructor
 */
function OptionVal(data) {
    "use strict";

    var self = this;
    self.label = ko.observable(data.label || null);
    self.value = ko.observable(data.value || null);
    self.selected = ko.observable(data.selected ? true : false);
    self.editable = ko.observable(data.editable ? true : false);
    self.multival = ko.observable(data.multival ? true : false);
    self.resultValue = ko.computed(function () {
        var sel = self.selected();
        var val = self.value();
        if (sel && val) {
            return val;
        }
        return sel || val ? "" : null;
    });
}
var _option_uid=0;
function Option(data, holder) {
    "use strict";

    var self = this;
    self.parent = holder;
    self.remoteLoadCallback = null;
    self.name = ko.observable(data.name);
    self.label = ko.observable(data.label);
    self.uid = ko.observable(data.uid||(++_option_uid+'_opt'));
    self.description = ko.observable(data.description);
    self.descriptionHtml = ko.observable(data.descriptionHtml);
    self.loading = ko.observable(false);
    self.required = ko.observable(data.required ? true : false);
    self.hidden = ko.observable(data.hidden ? true : false);
    self.enforced = ko.observable(data.enforced ? true : false);
    self.isDate = ko.observable(data.isDate ? true : false);
    self.dateFormat = ko.observable(data.dateFormat);
    /**
     * Type: used for file upload
     */
    self.optionType = ko.observable(data.optionType);
    self.fieldName = ko.observable(data.fieldName);
    self.hasError = ko.observable(data.hasError);
    self.hasRemote = ko.observable(data.hasRemote);
    self.fieldName = ko.observable(data.fieldName);
    self.fieldId = ko.observable(data.fieldId);
    self.fieldLabelId = ko.observable(data.fieldLabelId);
    self.optionDepsMet = ko.observable(data.optionDepsMet);
    self.secureInput = ko.observable(data.secureInput);
    self.multivalued = ko.observable(data.multivalued);
    self.multivalueAllSelected = ko.observable(data.multivalueAllSelected ? true : false);
    self.delimiter = ko.observable(data.delimiter);
    self.value = ko.observable(data.value);
    self.initvalue = ko.observable(data.value);
    self.useinit = ko.observable(data.value ? true : false);
    /**
     * static list of values to choose from
     */
    self.values = ko.observableArray(data.values);
    self.valuesFromPlugin = ko.observableArray(data.valuesFromPlugin);
    self.defaultValue = ko.observable(data.defaultValue);
    /**
     * list of values already selected
     */
    self.selectedMultiValues = ko.observableArray(data.selectedMultiValues);
    /**
     * list of values chosen as default for multivalued
     */
    self.defaultMultiValues = ko.observableArray(data.defaultMultiValues);
    /**
     * list of all multivalue strings to choose from
     */
    self.multiValueList = ko.observableArray(data.multiValueList);

    function emptyValue(val) {
        return (!val || val === '');
    }

    self.setReloadCallback = function (func) {
        self.remoteLoadCallback = func;
    };

    self.reloadRemoteValues = function () {
        if (self.hasRemote() && self.remoteLoadCallback) {
            self.remoteLoadCallback(self.name());
        } else {
            return true;
        }
    };

    //set up multivaluelist if default/selected values
    self.evalMultivalueChange = function () {
        if (self.multiValueList().length > 0) {
            //construct value string from selected multivalue options
            var str = '';
            var strs = [];

            var selected = ko.utils.arrayFilter(self.multiValueList(), function (val) {
                return val.selected() && val.value();
            });
            ko.utils.arrayForEach(selected, function (val) {
                strs.push(val.value());
            });
            self.value(strs.join(self.delimiter()));
            self.selectedMultiValues(strs);
        }
    };
    self.createMultivalueEntry = function (obj) {
        var optionVal = new OptionVal(obj);
        optionVal.resultValue.subscribe(function (newval) {
            if (newval == null) {
                //remove from parent
                self.multiValueList.remove(optionVal);
            } else {
                self.evalMultivalueChange();
            }
        });
        return optionVal;
    };
    self.loadedRemoteValues = ko.observable(false);
    self.remoteValues = ko.observableArray([]);
    if (self.multivalued()) {

        var testselected = function (val) {
            if (self.selectedMultiValues() && self.selectedMultiValues().length > 0) {
                if (self.multivalueAllSelected()) {
                    return true;
                } else {
                    return ko.utils.arrayIndexOf(self.selectedMultiValues(), val) >= 0;
                }
            } else if (self.defaultMultiValues() && self.defaultMultiValues().length > 0) {
                return ko.utils.arrayIndexOf(self.defaultMultiValues(), val) >= 0;
            } else if (self.value()) {
                return self.value() == val;
            } else if (self.defaultValue()) {
                return self.defaultValue() == val;
            } else if (self.multivalueAllSelected()) {
                return true;
            }
            return false;
        };
        if(self.selectedMultiValues().length<1 && self.defaultMultiValues().length>0){
            //automatically select the default values
            self.selectedMultiValues(self.defaultMultiValues());
        }
        var addedExtras = false;
        var addExtraSelected = function (selected) {
            if (!self.enforced() && selected) {
                //add any selectedMultiValues that are not in values list

                ko.utils.arrayForEach(selected, function (val) {
                    if (self.values() != null && ko.utils.arrayIndexOf(self.values(), val) >= 0) {
                        return;
                    }

                    var found = ko.utils.arrayFirst(self.multiValueList(), function (oval) {
                        return oval.value() == val;
                    });
                    if (found) {
                        return;
                    }
                    self.multiValueList.unshift(self.createMultivalueEntry({
                        label: val,
                        value: val,
                        selected: !addedExtras,
                        editable: true,
                        multival: true
                    }));
                });
                addedExtras = true;
            }
        };


        if (self.hasRemote()) {
            //when remote values are loaded, set the multivalue entries with them
            self.remoteValues.subscribe(function (newval) {
                var new_values = [];

                ko.utils.arrayForEach(newval, function (val) {
                    new_values.push(self.createMultivalueEntry({
                        label: val.label(),
                        value: val.value(),
                        selected: testselected(val.value()) || val.selected(),
                        editable: false,
                        multival: true
                    }));
                });
                self.multiValueList(new_values);
            });
        } else {
            addExtraSelected(self.selectedMultiValues());

            if (self.values() != null && self.values().length>0) {
                ko.utils.arrayForEach(self.values(), function (val) {
                    var selected = testselected(val);
                    self.multiValueList.push(self.createMultivalueEntry({
                        label: val,
                        value: val,
                        selected: selected,
                        editable: false,
                        multival: true
                    }));
                });
            } else if (self.valuesFromPlugin() != null && self.valuesFromPlugin().length>0) {
                ko.utils.arrayForEach(self.valuesFromPlugin(), function (val) {
                    var selected = testselected(val.value);
                    self.multiValueList.push(self.createMultivalueEntry({
                        label: val.name,
                        value: val.value,
                        selected: selected,
                        editable: false,
                        multival: true
                    }));
                });
            }
        }
        self.multiValueList.subscribe(self.evalMultivalueChange);
    } else if (self.enforced() && self.values().length == 1 && emptyValue(self.value())) {
        //auto-set the value to only allowed value
        self.value(self.defaultValue() || self.values()[0]);
    }else if (!self.enforced() && emptyValue(self.value()) && !emptyValue(self.defaultValue())) {
        //auto-set the value to only allowed value
        self.value(self.defaultValue());
    }
    self.remoteError = ko.observable();

    self.selectedOptionValue = ko.observable(self.value());
    self.defaultStoragePath = ko.observable(data.defaultStoragePath);
    self.dateFormatErr = ko.computed(function () {
        if (!self.isDate() || !self.value() || !self.dateFormat()) {
            return false;
        }
        try {
            var m = moment(self.value(), self.dateFormat(), true);
            return !m.isValid();
        } catch (e) {
            return true;
        }
    });
    self.truncateDefaultValue = ko.computed(function () {
        var val = self.defaultValue();
        if (!val || val.length < 50) return val;
        return val.substring(0, 50);
    });
    self.setDefault = function () {
        self.value(self.defaultValue());
    };
    self.hasSingleEnforcedValue = ko.computed(function () {
        return self.enforced()
            && self.values() != null
            && self.values().length == 1;
    });
    self.singleEnforcedValue = ko.computed(function () {
        return self.hasSingleEnforcedValue() ? self.values()[0] : null;
    });
    self.hasValue = ko.computed(function () {
        return self.value();
    });
    self.hasValues = ko.computed(function () {
        var values = self.values();
        return values != null && values.length > 0;
    });
    self.hasPluginValues = ko.computed(function () {
        var pluginvalues = self.valuesFromPlugin();
        return pluginvalues != null && pluginvalues.length > 0;
    });
    self.hasExtended = ko.computed(function () {
        return !self.secureInput()
            && (
                self.hasValues()
                || self.multivalued()
                || self.hasRemote() && self.remoteValues().length > 0
                || self.hasPluginValues()
            );
    });
    self.hasTextfield = ko.computed(function () {
        return !self.isMultilineType() && ( !self.enforced()
            && (
                !self.multivalued()
                || (self.hasError() && !self.hasExtended())
            )
            || self.secureInput());
    });
    self.showDefaultButton = ko.computed(function () {
        return !self.enforced()
            && !self.multivalued()
            && !self.secureInput()
            && self.defaultValue()
            && !(
                self.values() != null
                && self.values().indexOf(self.defaultValue()) >= 0
            )
            && self.value() != self.defaultValue();
    });

    self.isFileType=ko.computed(function () {
        return self.optionType() == 'file';
    });

    self.isMultilineType = ko.computed(function () {
        return self.optionType() === 'multiline' && self.parent && self.parent.features() && self.parent.features().multilineJobOptions!==null && self.parent.features().multilineJobOptions() === true;
    });

    /**
     * Return the array of option objects to use for displaying the Select input for this option
     */
    self.selectOptions = ko.computed(function () {
        var arr = [];
        if (!self.enforced() && !self.multivalued()) {
            arr.push(new OptionVal({label: message('option.select.choose.text'), value: ''}));
        }
        var remotevalues = self.remoteValues();
        var localvalues = self.values();
        var pluginvalues = self.valuesFromPlugin();

        if (self.hasRemote() && remotevalues != null) {
            ko.utils.arrayForEach(remotevalues, function (val) {
                arr.push(val);
            });
        } else if (self.hasValues()) {
            ko.utils.arrayForEach(localvalues, function (val) {
                arr.push(new OptionVal({label: val, value: val}));
            });
        } else if (self.hasPluginValues()) {
            ko.utils.arrayForEach(pluginvalues, function (val) {
                arr.push(new OptionVal({label: val.name, value: val.value}));
            });
        }
        return arr;
    });
    self.newMultivalueEntry = function () {
        var arr = self.multiValueList;
        arr.unshift(self.createMultivalueEntry({
            label: '_new',
            value: '',
            selected: true,
            editable: true,
            multival: true
        }));
    };
    self.multivalueFieldKeydown = function (obj,evt) {
        var enterKey = !noenter(evt);
        if (enterKey) {
            self.newMultivalueEntry()
        }
        return !enterKey;
    };

    /**
     * When select box chooses an option value, set the value()
     */
    self.selectedOptionValue.subscribe(function (newval) {
        if (newval && typeof(newval) == 'object' && typeof(newval.value) == 'function' && newval.value()) {
            self.value(newval.value());
        } else if (typeof(newval) == 'string') {
            self.value(newval);
        }
    });

    self.loadRemoteValues = function (values, selvalue) {
        self.loadedRemoteValues(false);
        self.remoteError(null);
        var tvalues = [];
        var tmultivalues = [];
        if (self.useinit() && self.initvalue() && !self.multivalued()) {
            tvalues[1] = (self.initvalue());
            self.useinit(false);
        } else if (self.useinit() && self.multivalued() && self.selectedMultiValues().length > 0) {
            tmultivalues[1] = self.selectedMultiValues();
            self.useinit(false);
        }
        if (selvalue && tvalues.indexOf(selvalue) < 0 && !self.multivalued()) {
            tvalues[0] = selvalue;
        } else if (self.multivalued() && selvalue) {
            tmultivalues[0] = selvalue.split(self.delimiter());
        }
        var rvalues = [];
        var tselected = -1;
        var remoteselectedArr = [];
        ko.utils.arrayForEach(values, function (val) {
            var optval;
            if (typeof(val) === 'object') {
                if (val.selected) {
                    remoteselectedArr.push(val.value)
                }
                optval = new OptionVal({label: val.name, value: val.value, selected: val.selected});
            } else if (typeof(val) === 'string') {
                optval = new OptionVal({label: val, value: val});
            }
            if (optval) {
                rvalues.push(optval);
                if (tvalues.length > 0 && tvalues.indexOf(optval.value()) > tselected) {
                    tselected = tvalues.indexOf(optval.value());
                }
            }
        });

        //choose value to select, by preference:
        //1: init value(s)
        //2: remote "selected" value(s)
        //3: input "selected" value(s)

        if (!self.multivalued()) {
            var touse = tselected === 1 ?
                tvalues[tselected] :
                (
                    (remoteselectedArr.length > 0 && remoteselectedArr[0]) ||
                    (tselected >= 0 ? tvalues[tselected] : null)
                );

            if (touse) {
                //choose correct value
                self.selectedOptionValue(touse);
            }
        } else if (self.multivalued()) {
            var touse =
                tmultivalues[1] ||
                (
                    (remoteselectedArr.length > 0 && remoteselectedArr) ||
                    (tmultivalues[0])
                );

            if (touse && touse.length > 0) {
                //choose correct value
                self.selectedMultiValues(touse);
            }
        }

        //triggers refresh of "selectOptions" populating select box
        self.remoteValues(rvalues);
        self.loadedRemoteValues(true);
    };
    /**
     * Option values data loaded from remote JSON request
     * @param data
     */
    self.loadRemote = function (data) {
        if (data.err && data.err.message) {
            var err = data.err;
            if (err) {
                err.url = data.srcUrl;
            }
            self.remoteError(err);
            self.remoteValues([]);
        } else if (data.values) {
            self.loadRemoteValues(data.values, data.selectedvalue);
        }
    };
    self.animateRemove = function (div) {
        jQuery(div).show().slideUp('fast', 0, function () {
            jQuery(div).remove();
        });
    };
    self.animateAdd = function (div) {
        jQuery(div).hide().slideDown('fast',function(){
            jQuery(div).find('input[type=text]').trigger('focus');
        });
    };
}
function JobOptions(data) {
    "use strict";
    var self = this;
    self.options = ko.observableArray();
    self.features = ko.observable({});
    self.remoteoptions = null;
    self.mapping = {
        options: {
            key: function (data) {
                return ko.utils.unwrapObservable(data.name);
            },
            create: function (options) {
                return new Option(options.data, self);
            }
        }
    };

    ko.mapping.fromJS(data, self.mapping, self);
}

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2019 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


function JobRunFormOptions (data) {
    var self = this
    self.nodeFilter = data.nodeFilter
    self.follow = ko.observable(typeof (data.follow) === 'undefined' || data.follow)
    self.debug = ko.observable(data.debug === true || data.debug === 'true')
    self.loglevel = ko.pureComputed(function () {
        return self.debug() ? 'DEBUG' : 'NORMAL'
    })

    self.canOverrideFilter = ko.observable(data.canOverrideFilter)
    self.changeTargetNodes = ko.observable(data.changeTargetNodes)
    self.nodeOverride = ko.observable(data.nodeOverride || 'cherrypick')
    self.hasSelectedNodes = ko.observable(data.hasSelectedNodes)
    self.hasDynamicFilter = ko.observable(data.hasDynamicFilter)
    self.selectedNodes = ko.observableArray(data.selectedNodes || [])
    self.allNodes = data.allNodes || []
    self.groups = data.groups
    self.grouptags = data.grouptags

    self.selectAllNodes = function () {
        self.selectedNodes([].concat(self.allNodes))
    }
    self.selectNoNodes = function () {
        self.selectedNodes([])
    }

    self.deselectNodes = function (nodes) {
        nodes.forEach((e) => {
            while (self.selectedNodes.indexOf(e) >= 0) {
                self.selectedNodes.splice(self.selectedNodes.indexOf(e), 1)
            }
        })
    }
    self.groupSelectAll = function (dom) {
        let group = jQuery(dom).data('group')
        if (!group || !self.groups || !self.groups[group]) {
            return
        }
        self.selectedNodes(
            self.selectedNodes().concat(
                self.groups[group].filter(function (e) {
                    return self.selectedNodes.indexOf(e) < 0
                })
            )
        )
    }
    self.groupSelectNone = function (dom) {
        let group = jQuery(dom).data('group')
        if (!group || !self.groups || !self.groups[group]) {
            return
        }
        self.deselectNodes(self.groups[group])

    }
    self.isNodeCherrypick = ko.pureComputed(function () {
        return self.nodeOverride() === 'cherrypick'
    })
    self.isCherrypickVisible = ko.pureComputed(function () {
        let cherrypick = self.isNodeCherrypick()
        let changeTargetNodes = self.changeTargetNodes()
        return changeTargetNodes && cherrypick
    })
    self.isNodeFilterVisible = ko.pureComputed(function () {
        let cherrypick = self.isNodeCherrypick()
        let changeTargetNodes = self.changeTargetNodes()
        return changeTargetNodes && !cherrypick
    })


}

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2019 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


"use strict"

/**
 * bind via knockout, e.g. data-ko-bind="varName" or data-ko-bind="TypeName"
 */
function initKoBind (sel, mapping,debug) {
    let doc = jQuery('body')
    if (sel) {
        doc = jQuery(sel)
    }
    if(debug){
        console.log(`${debug}: start binding for ${!sel?'body':sel}:`, mapping,doc)
    }
    doc.find('[data-ko-bind]').each(function (i, el) {
        let jqel = jQuery(el)
        const controller = jqel.data('koBind')
        if(mapping[controller] && jqel.data('koBoundController')){
            //already bound, skip
            if(debug){
                console.log(`${debug}: warning: ko-bind for ${controller} : another is already bound:`, el, jqel.data('koBoundController'))
            }
            return
        }
        const data = jqel.data()
        if (controller.match(/^[A-Z]/)) {
            const ctrl = eval(controller)
            if (typeof (ctrl) !== 'function') {
                return
            }
            //create new instance
            let obj = new ctrl(data)
            if (data['koBindVar']) {
                window[data['koBindVar']] = obj
            }
            ko.applyBindings(obj, el)
            jqel.data('koBoundController',obj)
            if(debug){
                console.log(`${debug}: (bind): ko-bind for new ${controller} `, el, obj)
            }
        } else if (
            controller.match(/^[a-z]/) &&
            mapping &&
            typeof (mapping[controller]) === 'object') {
            let ctrl = mapping && mapping[controller]
            ko.applyBindings(ctrl, el)

            jqel.data('koBoundController',ctrl)
            if(debug){
                console.log(`${debug}: (bind): ko-bind for mapped: ${controller}`, el, ctrl)
            }
        } else if(debug) {
            console.log(`${debug}: warning: ko-bind for ${controller} : controller not found.`, el)
        }
    })
}

var process = process || {env: {NODE_ENV: "development"}};
/*
 * Copyright 2019 Rundeck, Inc. (http://rundeck.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */




//= require momentutil
//= require vendor/knockout.min
//= require vendor/knockout-mapping
//= require knockout-foreachprop
//= require menu/job-remote-options
//= require ko/binding-popover
//= require ko/binding-datetimepicker
//= require nodeFiltersKO
//= require executionOptions
//= require menu/joboptions
//= require scheduledExecution/jobRunFormOptionsKO

//= require koBind

/*
 Manifest for "scheduledExecution/show.gsp" page
 */
//TODO: refactor scheduledExecution/show.gsp to move javascript here

function initJobNodeFilters(filterParams){
    var pageParams = loadJsonData('pageParams');
    var nodeSummary = new NodeSummary({baseUrl:appLinks.frameworkNodes});
    return new NodeFilters(
        appLinks.frameworkAdhoc,
        appLinks.scheduledExecutionCreate,
        appLinks.frameworkNodes,
        jQuery.extend(filterParams,{
            project: pageParams.project,
            paging:false,
            emptyMode: 'localnode',
            maxShown:50,
            nodesTitleSingular:message('Node'),
            nodesTitlePlural:message('Node.plural'),
            nodeSummary:nodeSummary
        }));
}

