//go:build cp

// Cp is a utility to copy [assert] for use without introducing a direct dependency.
//
// Usage:
//  1. Add //go:generate go run -tags=cp go-simpler.org/assert/cmd/cp@<version> [-dir=<working-directory>]
//  2. Run go generate ./...
package main

import (
	"context"
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"os"
	"os/exec"
	"os/signal"
	"path/filepath"
	"strings"

	"go-simpler.org/assert"
)

func main() {
	ctx, cancel := signal.NotifyContext(context.Background(), os.Interrupt)
	defer cancel()

	if err := run(ctx); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func run(ctx context.Context) error {
	var workDir string

	fs := flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	fs.StringVar(&workDir, "dir", ".", "the working directory")
	if err := fs.Parse(os.Args[1:]); err != nil {
		return fmt.Errorf("parsing flags: %w", err)
	}

	moduleName, err := readModuleName(ctx)
	if err != nil {
		return err
	}

	importPath := filepath.Join(moduleName, "assert")
	if workDir != "." {
		importPath = filepath.Join(moduleName, workDir, "assert")
	}
	assert.AliasFile = strings.Replace(assert.AliasFile, "go-simpler.org/assert", importPath, 1)

	dirPath := filepath.Join(workDir, "assert", "EF")
	if err := os.MkdirAll(dirPath, 0755); err != nil && !errors.Is(err, os.ErrExist) {
		return err
	}

	mainPath := filepath.Join(workDir, "assert", "assert.go")
	if err := writeFile(mainPath, assert.MainFile); err != nil {
		return err
	}

	aliasPath := filepath.Join(workDir, "assert", "EF", "alias.go")
	if err := writeFile(aliasPath, assert.AliasFile); err != nil {
		return err
	}

	return nil
}

func readModuleName(ctx context.Context) (string, error) {
	args := [...]string{"go", "mod", "edit", "-json"}

	out, err := exec.CommandContext(ctx, args[0], args[1:]...).Output()
	if err != nil {
		var exitErr *exec.ExitError
		if errors.As(err, &exitErr) {
			return "", fmt.Errorf("running %q: %w: %s", strings.Join(args[:], " "), err, string(exitErr.Stderr))
		}
		return "", fmt.Errorf("running %q: %w", strings.Join(args[:], " "), err)
	}

	var info struct {
		Module struct {
			Path string `json:"Path"`
		} `json:"Module"`
	}
	if err := json.Unmarshal(out, &info); err != nil {
		return "", fmt.Errorf("decoding module info: %w", err)
	}

	return info.Module.Path, nil
}

func writeFile(path, content string) error {
	const header = `// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

// Code generated by go-simpler.org/assert/cmd/cp. DO NOT EDIT.

`
	return os.WriteFile(path, []byte(header+content), 0644)
}
